/*
 
Copyright (C) 2010 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.factory;

import java.sql.Timestamp;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.jms.DeliveryMode;
import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.OutputNotifyGroupInfo;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoContext;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.ejb.entity.NotifyRelationInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyRelationInfoUtil;
//import com.clustercontrol.notify.ejb.entity.NotifyRelationMstLocal;
//import com.clustercontrol.notify.ejb.entity.NotifyRelationMstUtil;
import com.clustercontrol.notify.ejb.mdb.NotifyControlBean;
import com.clustercontrol.notify.queue.NotifyQueueException;
import com.clustercontrol.notify.queue.NotifyQueueSender;
import com.clustercontrol.notify.util.MonitorResultStatusUpdater;

/**
 * 
 * @author takahatat
 *
 */
public class NotifyDispatcher {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( NotifyControlBean.class );
	
	/** 通知情報キャッシュ */
	private static HashMap<String, NotifyInfo> m_notifyMap = null;
	
	/** キュー投げ込みの一貫性を保障するためのロック */
	private static final Object sendQueueLock = new Object();
	
	/** 
	 * 引数で指定された受信メッセージを元に、ログ出力をおこないます。<BR>
	 * Message-Driven Bean で行うビジネスロジックを実装します。特定のメッセージを受信した場合のみ処理を行います。
	 * 受信メッセージは下記の通りです。
	 * 
	 * <p><li>{@link com.clustercontrol.notify.message.OutputNotifyGroupInfo} : 含まれている通知グループIDを元にキャッシュしている通知情報グループを取得します。取得した通知情報に従って、受信メッセージに含まれている通知出力基本情報を基に各種通知を行います。
	 * <p><li>{@link com.clustercontrol.notify.bean.NotifyInfoContext} : 含まれている通知情報を、キャッシュしている通知情報に反映します。
	 *
	 * @see javax.jms.MessageListener#onMessage(javax.jms.Message)
	 *
	 * @see com.clustercontrol.notify.message.OutputNotifyGroupInfo
	 * @see com.clustercontrol.notify.bean.NotifyInfoContext
	 */
	public static void onMessage(javax.jms.Message message, int queueDeliveryMode) {
		m_log.debug("onMessage() start " + message);
		synchronized(sendQueueLock){
			if(m_log.isDebugEnabled()){
				m_log.debug("onMessage start : Message Driven Bean got message " + message);
			}
		
			// 通知情報キャッシュ取得
			if(m_notifyMap == null){
				try{
					m_notifyMap = new SelectNotify().getNotifyMap();
				} catch (Exception e) {
					m_log.error("onMessage(): 通知情報キャッシュ取得時にエラーが発生しました。 " + e.getMessage());
				}
			}
		
			if(message instanceof ObjectMessage)
			{
				ObjectMessage msg = (ObjectMessage)message;
			
				try
				{
					Object objMsg = msg.getObject();
					if(objMsg instanceof OutputNotifyGroupInfo){
						OutputNotifyGroupInfo info = (OutputNotifyGroupInfo)objMsg;
						
						if(info != null){
							String notifyGroupId = info.getNotifyGroupId();
							if(notifyGroupId != null && !"".equals(notifyGroupId.trim())){
							
								notifyAction(info, queueDeliveryMode);
							
							}
						}
					}
					// 通知情報キャッシュ更新
					else if(objMsg instanceof NotifyInfoContext) {
					
						try{
							NotifyInfoContext notifyInfoContext = (NotifyInfoContext)objMsg;
							
							// 通知情報登録 または 更新時 
							if(NotifyInfoContext.TYPE_ADD == notifyInfoContext.getType() || 
									NotifyInfoContext.TYPE_UPDATE == notifyInfoContext.getType()){
							
								m_notifyMap.put(notifyInfoContext.getNotifyId(), notifyInfoContext.getNotifyInfo());
							}
							// 通知情報削除時
							else if(NotifyInfoContext.TYPE_DELETE == notifyInfoContext.getType()){
								m_notifyMap.remove(notifyInfoContext.getNotifyId());
							}
						}
						catch(Exception e){
							m_log.error("onMessage(): 通知情報キャッシュ更新時にエラーが発生しました。" + e.getMessage());
						}
					}
					else
					{
						m_log.debug("onMessage(): ObjectMessage is not an expected instance. " + objMsg.toString());
					}
				} catch(JMSException e){
					m_log.error("onMessage():" + e.getMessage(), e);
				} catch(EJBException e){
					m_log.error("onMessage():" + e.getMessage(), e);
				} catch(Exception e){
					m_log.error("onMessage():" + e.getMessage(), e);
				}	
			}
		
			if(m_log.isDebugEnabled()){
				m_log.debug("onMessage end   : Message Driven Bean got message " + message);
			}
		}
		m_log.debug("onMessage() end   " + message);
	}
	
	
	/**
	 * 引数で指定された情報を各種通知に出力します。<BR>
	 * イベント通知、ステータス通知、メール通知、ジョブ通知、ログエスカレーション通知を行います。
	 * 
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyEventBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyStatusBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyMailBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyJobBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyLogEscalationBean#onMessage(javax.jms.Message)
	 * 
	 * @param info 通知出力情報
	 */
	@SuppressWarnings("unchecked")
	private static void notifyAction(OutputNotifyGroupInfo info, int queueDeliveryMode) {
		// 監視結果重要度ステータスを更新
		boolean prioityChangeFlag = MonitorResultStatusUpdater.update(info);
		
		Date now = new Date();
		Timestamp outputDate = new Timestamp(now.getTime());
		info.setOutputDate(outputDate);
		
		String notifyGroupId = info.getNotifyGroupId();
		try {
//			NotifyRelationMstLocal mst = NotifyRelationMstUtil.getLocalHome().findByPrimaryKey(notifyGroupId);
//
//			// 通知関連情報を取得
//			Collection<NotifyRelationInfoLocal> ct = mst.getNotifyRelationInfos();
			
			// 通知関連情報を取得（通知ID順に）
			Collection<NotifyRelationInfoLocal> ct = NotifyRelationInfoUtil.getLocalHome().findByNotifyGroupId(notifyGroupId);
			
			// 通知グループIDに登録されている数分の通知種別を振り分ける。
			if(ct != null && ct.size() > 0 ) {
				Iterator<NotifyRelationInfoLocal> itr = ct.iterator();
				while (itr.hasNext()) {
					NotifyRelationInfoLocal notifyRelation = itr.next();
					String notifyId = notifyRelation.getNotifyId();
					NotifyInfo notifyInfo = m_notifyMap.get(notifyId);
					int notifyType = notifyRelation.getNotifyType();
					
					try {
						// 通知情報が「通知する」となっている場合
						if (notifyInfo.getValidFlg() == ValidConstant.TYPE_VALID) {
							// 通知IDは設定されていないため、ここで設定する
							info.setNotifyId(notifyId);
							NotifyQueueSender.getInstance(notifyType).send(
									new NotifyRequestMessage(
											info,
											info.getNotifyId(),
											info.getOutputDate(),
											prioityChangeFlag
									), queueDeliveryMode);
							
							if (m_log.isDebugEnabled()) {
								m_log.debug("sending message to jms. ("
										+ (queueDeliveryMode == DeliveryMode.PERSISTENT ? "persistent" : "non-persistent")
										+ " : notifyId=" + info.getNotifyId() + " pluginId=" + info.getPluginId()
										+ " priority=" + info.getPriority() + " generationDate=" + info.getGenerationDate() 
										+ " monitorId=" + info.getMonitorId() + " facilityId=" + info.getFacilityId());
							}
						} else {
							// TODO:通知しない場合は、キャッシュ情報をアップデートしたほうが良い
						}
					} catch (NotifyQueueException e) {
						m_log.warn("notifyAction() : " + info + " : " + e.getMessage(), e);
					} catch (Exception e) {
						m_log.warn("notifyAction() : " + info + " : " + e.getMessage(), e);
					}
				}
			}
		} catch (FinderException e) {
			m_log.warn("notifyAction() : notifyGroupId = " + notifyGroupId +" not found.", e);
		} catch (NamingException e) {
			m_log.warn("notifyAction() : catch NamingExcepiton" + e.getMessage(), e);
		} 	
	}
	
}
