/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.factory;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.jms.DeliveryMode;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.NotifyDuplicateException;
import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.notify.bean.NotifyEventInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoContext;
import com.clustercontrol.notify.bean.NotifyInfoDetail;
import com.clustercontrol.notify.bean.NotifyJobInfo;
import com.clustercontrol.notify.bean.NotifyLogEscalateInfo;
import com.clustercontrol.notify.bean.NotifyMailInfo;
import com.clustercontrol.notify.bean.NotifyStatusInfo;
import com.clustercontrol.notify.bean.NotifyTypeConstant;
import com.clustercontrol.notify.bean.QueueConstant;
import com.clustercontrol.notify.ejb.entity.NotifyCommandInfoData;
import com.clustercontrol.notify.ejb.entity.NotifyCommandInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyCommandInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyHistoryLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyJobInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyJobInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 通知情報を変更するクラスです。
 *
 * @version 3.0.0
 * @since 1.0.0
 */
public class ModifyNotify {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( ModifyNotify.class );
	
	/**
	 * 通知情報を変更します。
	 * <p>
	 * <ol>
	 *  <li>通知IDより、通知情報を取得します。</li>
	 *  <li>通知情報を変更します。</li>
	 *  <li>通知情報に設定されている通知イベント情報を削除します。</li>
	 *  <li>通知イベント情報を作成し、通知情報に設定します。</li>
	 *  <li>キャッシュ更新用の通知情報を生成し、ログ出力キューへ送信します。
	 *      監視管理機能で、監視管理機能で保持している通知情報キャッシュが更新されます。</li>
	 * </ol>
	 * 
	 * @param info 変更対象の通知情報
	 * @return 変更に成功した場合、<code> true </code>
	 * @throws CreateException
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 * @see com.clustercontrol.notify.ejb.entity.NotifyEventInfoBean
	 * @see com.clustercontrol.notify.factory.DeleteNotify#deleteEvents(Collection)
	 * @see com.clustercontrol.notify.bean.NotifyInfoContext#NotifyInfoContext(int, java.lang.String, com.clustercontrol.notify.bean.NotifyInfo)
	 * @see com.clustercontrol.notify.util.SendQueue
	 */
	@SuppressWarnings("unchecked")
	public boolean modify(NotifyInfo info , String user) throws CreateException, NotifyDuplicateException, RemoveException, NamingException, JMSException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.NOTIFY);
    	
		SendQueue queue = null;
		try 
		{
			// 通知情報を取得
			NotifyInfoLocal notify = NotifyInfoUtil.getLocalHome().findByPrimaryKey(info.getNotifyId());
			
			// 通知情報を更新
			if(notify != null){
				notify.setDescription(info.getDescription());
				notify.setValidFlg(info.getValidFlg());
				notify.setInitialCount(info.getInitialCount());
				notify.setRenotifyType(info.getRenotifyType());
				notify.setRenotifyPeriod(info.getRenotifyPeriod());
				notify.setUpdateDate(new Timestamp(new Date().getTime()));
				notify.setUpdateUser(user);
				notify.setValidFlg(info.getValidFlg());
			}
			
			// 通知設定を無効に設定した場合は、関連する通知履歴を削除
			if(notify.getValidFlg().intValue() == ValidConstant.TYPE_INVALID){
				m_log.debug("remove NotifyHistory");
				
				Collection<NotifyHistoryLocal> list = notify.getNotifyHistory();
				for(NotifyHistoryLocal history : list){
					m_log.debug("remove NotifyHistory : " + history);
					
					try {
						history.remove();
					} catch (RemoveException e) {
						m_log.error(e.getMessage(), e);
					}
				}
			}

			// 通知詳細情報を変更
			modifyDetails(info, notify);

			NotifyInfoContext context = new NotifyInfoContext(
					NotifyInfoContext.TYPE_UPDATE, info.getNotifyId(), info);

			// 通知情報キャッシュを更新
			// FIXME Notification Cache will be removed
			queue = new SendQueue(QueueConstant.QUEUE_NAME_NOTIFYCONTROL);
			queue.put(context, DeliveryMode.PERSISTENT);
			queue.terminate();
			
			queue = new SendQueue(QueueConstant.QUEUE_NAME_NOTIFYCONTROL_NON_PERSISTENCE);
			queue.put(context, DeliveryMode.PERSISTENT);
			queue.terminate();

		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { info.getNotifyId() };
			apllog.put("SYS", "002", args);
			m_log.debug("modify():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { info.getNotifyId() };
			apllog.put("SYS", "002", args);
			m_log.debug("modify():" + e.getMessage());
			NotifyDuplicateException e2 = new NotifyDuplicateException(e.getMessage(), e);
			e2.setNotifyId(info.getNotifyId());
			throw e2;
		} catch (RemoveException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { info.getNotifyId() };
			apllog.put("SYS", "002", args);
			m_log.debug("modify():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { info.getNotifyId() };
			apllog.put("SYS", "002", args);
			m_log.debug("modify():" + e.getMessage());
			throw e;
		} catch (JMSException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { info.getNotifyId() };
			apllog.put("SYS", "002", args);
			m_log.debug("modify():" + e.getMessage());
			throw e;
		} finally {
			if (queue != null) {
				queue.terminate();
			}
		}
		return true;
	}
	
	
	public boolean modifyDetails(NotifyInfo info, NotifyInfoLocal notify)
			throws CreateException, NamingException, RemoveException {

		ArrayList<NotifyInfoDetail> eventList = info.getNotifyInfoDetail();

		if (info.getNotifyType() == NotifyTypeConstant.TYPE_STATUS) {
			//通知詳細情報を一度削除
			Collection<NotifyStatusInfoLocal> col = 
				new ArrayList<NotifyStatusInfoLocal>(notify.getNotifyStatusInfo());
			for (NotifyStatusInfoLocal detail : col) {
				detail.remove();
			}

			ArrayList<NotifyStatusInfoLocal> localList = new ArrayList<NotifyStatusInfoLocal>();

			// 通知ステータス情報を挿入
			NotifyStatusInfo status = null;
			for (int index = 0; index < eventList.size(); index++) {
				status = (NotifyStatusInfo) eventList.get(index);
				if (status != null) {
					NotifyStatusInfoLocal local = NotifyStatusInfoUtil
							.getLocalHome().create(info.getNotifyId(),
									status.getPriority(), status.getValidFlg(),
									status.getStatusValidPeriod(),
									status.getStatusInvalidFlg(),
									status.getStatusUpdatePriority());
					localList.add(local);
				}
			}

			notify.setNotifyStatusInfo(localList);
		} else if (info.getNotifyType() == NotifyTypeConstant.TYPE_EVENT) {
			//通知詳細情報を一度削除
			Collection<NotifyEventInfoLocal> col = 
				new ArrayList<NotifyEventInfoLocal>(notify.getNotifyEventInfo());
			for (NotifyEventInfoLocal detail : col) {
				detail.remove();
			}

			ArrayList<NotifyEventInfoLocal> localList = new ArrayList<NotifyEventInfoLocal>();

			// 通知イベント情報を挿入
			NotifyEventInfo event = null;
			for (int index = 0; index < eventList.size(); index++) {
				event = (NotifyEventInfo) eventList.get(index);
				if (event != null) {
					NotifyEventInfoLocal local = NotifyEventInfoUtil
							.getLocalHome().create(info.getNotifyId(),
									event.getPriority(), event.getValidFlg(),
									event.getEventNormalState());
					localList.add(local);
				}
			}

			notify.setNotifyEventInfo(localList);
		} else if (info.getNotifyType() == NotifyTypeConstant.TYPE_MAIL) {
			//通知詳細情報を一度削除
			Collection<NotifyMailInfoLocal> col = 
				new ArrayList<NotifyMailInfoLocal>(notify.getNotifyMailInfo());
			for (NotifyMailInfoLocal detail : col) {
				detail.remove();
			}

			ArrayList<NotifyMailInfoLocal> localList = new ArrayList<NotifyMailInfoLocal>();

			// 通知イベント情報を挿入
			NotifyMailInfo mail = null;
			for (int index = 0; index < eventList.size(); index++) {
				mail = (NotifyMailInfo) eventList.get(index);
				if (mail != null) {
					NotifyMailInfoLocal local = NotifyMailInfoUtil
							.getLocalHome().create(info.getNotifyId(),
									mail.getPriority(),
									mail.getMailTemplateId(),
									mail.getValidFlg(),
									mail.getMailAddress());
					localList.add(local);
				}
			}

			notify.setNotifyMailInfo(localList);
		} else if (info.getNotifyType() == NotifyTypeConstant.TYPE_JOB) {
			//通知詳細情報を一度削除
			Collection<NotifyJobInfoLocal> col = 
				new ArrayList<NotifyJobInfoLocal>(notify.getNotifyJobInfo());
			for (NotifyJobInfoLocal detail : col) {
				detail.remove();
			}

			ArrayList<NotifyJobInfoLocal> localList = new ArrayList<NotifyJobInfoLocal>();

			// 通知イベント情報を挿入
			NotifyJobInfo job = null;
			for (int index = 0; index < eventList.size(); index++) {
				job = (NotifyJobInfo) eventList.get(index);
				if (job != null) {
					NotifyJobInfoLocal local = NotifyJobInfoUtil.getLocalHome()
							.create(info.getNotifyId(), job.getPriority(),
									job.getValidFlg(), job.getJobunitId(), job.getJobId(),
									job.getJobFailurePriority(),
									job.getJobExecFacilityFlg(),
									job.getJobExecFacility());
					localList.add(local);
				}
			}

			notify.setNotifyJobInfo(localList);
		} else if (info.getNotifyType() == NotifyTypeConstant.TYPE_LOG_ESCALATE) {
			//通知詳細情報を一度削除
			Collection<NotifyLogEscalateInfoLocal> col = 
				new ArrayList<NotifyLogEscalateInfoLocal>(notify.getNotifyLogEscalateInfo());
			for (NotifyLogEscalateInfoLocal detail : col) {
				detail.remove();
			}

			ArrayList<NotifyLogEscalateInfoLocal> localList = new ArrayList<NotifyLogEscalateInfoLocal>();

			// 通知イベント情報を挿入
			NotifyLogEscalateInfo log = null;
			for (int index = 0; index < eventList.size(); index++) {
				log = (NotifyLogEscalateInfo) eventList.get(index);
				if (log != null) {
					NotifyLogEscalateInfoLocal local;

					local = NotifyLogEscalateInfoUtil.getLocalHome().create(
							info.getNotifyId(), log.getPriority(),
							log.getValidFlg(),
							log.getEscalateMessage(), log.getSyslogPriority(),
							log.getSyslogFacility(),
							log.getEscalateFacilityFlg(),
							log.getEscalateFacility(), log.getEscalatePort());

					localList.add(local);
				}
			}

			notify.setNotifyLogEscalateInfo(localList);
		} else if (info.getNotifyType() == NotifyTypeConstant.TYPE_COMMAND) {
			//通知詳細情報を一度削除
			Collection<NotifyCommandInfoLocal> col = 
				new ArrayList<NotifyCommandInfoLocal>(notify.getNotifyCommandInfo());
			for (NotifyCommandInfoLocal detail : col) {
				detail.remove();
			}
			
			ArrayList<NotifyCommandInfoLocal> localList = new ArrayList<NotifyCommandInfoLocal>();

			// 通知イベント情報を挿入
			NotifyCommandInfoData command = null;
			for (int index = 0; index < eventList.size(); index++) {
				command = (NotifyCommandInfoData) eventList.get(index);
				if (command != null) {
					NotifyCommandInfoLocal local;

					local = NotifyCommandInfoUtil.getLocalHome().create(command);

					localList.add(local);
				}
			}

			notify.setNotifyCommandInfo(localList);
		}
		return true;
	}
}
