/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.notify.ejb.mdb;

import java.sql.Timestamp;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.jms.DeliveryMode;
import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.OutputBasicInfo;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.commons.util.JmsPersistenceConfig;
import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.QueueConstant;
import com.clustercontrol.notify.ejb.entity.SyslogDuplicationInfoLocal;
import com.clustercontrol.notify.ejb.entity.SyslogDuplicationInfoPK;
import com.clustercontrol.notify.ejb.entity.SyslogDuplicationInfoUtil;

/**
 * ログ重複チェックを行う Message-Driven Bean <BR>
 * <p>重複チェック後、通知へ渡します。<BR>
 *
 * <!-- begin-xdoclet-definition -->
 * @ejb.bean name="SyslogDuplicationBean" 
 *     acknowledge-mode="Auto-acknowledge"
 *     destination-type="javax.jms.Queue"
 *     
 *     transaction-type="Container"
 *     destination-jndi-name="queue/clustercontrolSyslogngDup"
 * 
 * @jboss.depends name="user:service=HinemosService,name=HinemosService"
 * 
 * @jboss.container-configuration
 *  name="Singleton Message Driven Bean"
 * 
 *--
 * Server Runtime Specific Tags
 * If you are not using a specific runtime, you can safely remove the tags below.
 * @jonas.message-driven-destination jndi-name="queue/clustercontrolSyslogngDup"
 * @jboss.destination-jndi-name name="queue/clustercontrolSyslogngDup"
 *
 *--
 * <!-- end-xdoclet-definition -->
 **/
public class SyslogDuplicationBean implements javax.ejb.MessageDrivenBean, javax.jms.MessageListener {
	
	private static final long serialVersionUID = 8122939314944655588L;

	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( SyslogDuplicationBean.class );
	
	/** コンテキスト情報 */
	private javax.ejb.MessageDrivenContext messageContext = null;
	
	/** 残す時間(ミリ秒) */
	private static int TIME_LEFT = 3600000;
	
	/** 削除間隔(ミリ秒) */
	private static int REMOVE_INTERVAL = 3600000;
	
	/** 削除実行時刻 */
	private static long m_removeTime = new Date().getTime();
	
	// プロパティファイルの読み込みと初期設定
	static{
		try {
			TIME_LEFT = 1000 * Integer.parseInt(HinemosProperties.getProperty("monitor.syslog.duplication.time.left"));
			REMOVE_INTERVAL = 1000 * Integer.parseInt(HinemosProperties.getProperty("monitor.syslog.duplication.remove.interval"));
			
			m_removeTime = m_removeTime + REMOVE_INTERVAL;
			
		} catch (Exception e) {
			m_log.error(e.getMessage(), e);
		} 
	}
	
	/**
	 * コンテキスト情報を設定します。<BR>
	 * Message-Driven Bean がインスタンスプールに格納される際に行う処理を実装します。
	 * @see javax.ejb.MessageDrivenBean#setMessageDrivenContext(javax.ejb.MessageDrivenContext)
	 */
	public void setMessageDrivenContext(
			javax.ejb.MessageDrivenContext messageContext)
	throws javax.ejb.EJBException {
		this.messageContext = messageContext;
	}
	
	/** 
	 * Message-Driven Bean が生成される際に行う処理を実装します。
	 * @ejb.create-method 
	 */
	public void ejbCreate() {
		//no specific action required for message-driven beans 
	}
	
	/** 
	 * Message-Driven Bean が削除される際に行う処理を実装します。
	 * @see javax.ejb.MessageDrivenBean#ejbRemove()
	 */
	public void ejbRemove() {
		messageContext = null;
	}
	
	/** 
	 * 引数で指定された受信メッセージを元に、重複チェックをおこないます。<BR>
	 * Message-Driven Bean で行うビジネスロジックを実装します。
	 * 特定のメッセージを受信した場合のみ処理を行います。受信メッセージは下記の通りです。
	 * 
	 * <p><li>{@link com.clustercontrol.notify.message.OutputNotifyGroupInfo} : 
	 * 含まれている出力情報を基にステータスを表示します。
	 *
	 * @see javax.jms.MessageListener#onMessage(javax.jms.Message)
	 * 
	 * @see com.clustercontrol.notify.message.OutputNotifyGroupInfo
	 */
	@SuppressWarnings("unchecked")
	public void onMessage(javax.jms.Message message) {
		m_log.debug("Message Driven Bean got message " + message);
		String queueName = "";

		if(message instanceof ObjectMessage) {
			ObjectMessage msg = (ObjectMessage)message;
			
			try {
				Object objMsg = msg.getObject();
				if(objMsg instanceof NotifyRequestMessage) {
					NotifyRequestMessage outputInfo = (NotifyRequestMessage)objMsg;
					
					// 重複確認の結果、重複していない場合は処理を行う
					if(duplicationCheck(outputInfo.getOutputInfo())) {
						SendQueue queue = null;
						try {
							if (JmsPersistenceConfig.QUEUE_SYSLOG_MODE == DeliveryMode.NON_PERSISTENT) {
								queueName = QueueConstant.QUEUE_NAME_NOTIFYCONTROL_NON_PERSISTENCE;
							} else {
								queueName = QueueConstant.QUEUE_NAME_NOTIFYCONTROL;
							}
							queue = new SendQueue(queueName);
							
							if (m_log.isDebugEnabled()) {
								m_log.debug("sending message to jms. (" + queueName
										+ " " + JmsPersistenceConfig.getPersistenceString(JmsPersistenceConfig.QUEUE_SYSLOG_MODE)
										+ " : outputDate=" + outputInfo.getOutputDate()
										+ " pluginId = " + HinemosModuleConstant.MONITOR_SYSLOGNG + ")");
							}
							queue.put(outputInfo, JmsPersistenceConfig.QUEUE_SYSLOG_MODE);
						} catch (NamingException e) {
							m_log.warn("notifyAction() : catch NamingExcepiton" + e.getMessage(), e);
						}finally{
							if(queue != null){
								try {
									queue.terminate();
								} catch (JMSException e) {
									m_log.warn(e.getMessage(), e);
								}
							}
						}
					}
				}
				else {
					m_log.warn("onMessage(): ObjectMessage is not an expected instance. " + objMsg.toString());
				}
			}
			catch(JMSException e) {
				m_log.error("onMessage():" + e.getMessage(), e);
			}
			catch(EJBException e) {
				m_log.error("onMessage():" + e.getMessage(), e);
			}
			
			// 古いレコード削除用メソッドを呼び出す
			removeOldRecode();
		}
	}
	
	/**
	 * ログ情報の重複を確認するメソッド
	 * 
	 * @param info
	 * @return
	 */
	private boolean duplicationCheck(OutputBasicInfo info) {
		
		boolean ret = true;
		
		SyslogDuplicationInfoLocal duplicationInfo = null;
		
		try {
			try {
				duplicationInfo = SyslogDuplicationInfoUtil.getLocalHome().findByPrimaryKey(
						new SyslogDuplicationInfoPK(
								info.getMonitorId(),
								info.getFacilityId(),
								new Timestamp(info.getGenerationDate().getTime()),
								info.getMessageOrg()));
			} catch (FinderException e) {
				m_log.debug("duplicationCheck : catch FinderException ");
				duplicationInfo = null;
			} 
			
			// すでに重複するログが存在する場合
			if (duplicationInfo != null) {
				
				// マルチIDが同じ場合は、重複カウンタをカウントアップする
				if (duplicationInfo.getMultiId().equals(info.getMultiId())) {
					duplicationInfo.setCount(duplicationInfo.getCount() + 1);
				}
				else {
					m_log.debug("dulicationCheck : duplication !! ");
					ret = false;
					// カウンタが0より大きい場合は、カウントを1減らす
					if (duplicationInfo.getCount() > 0){
						duplicationInfo.setCount(duplicationInfo.getCount() - 1);
						}
					else {
						duplicationInfo.remove();
					}
				}
				
			}
			else {
				m_log.debug("duplicationCheck() : create new");
				SyslogDuplicationInfoUtil.getLocalHome().create(
						info.getMonitorId(),
						info.getFacilityId(),
						new Timestamp(info.getGenerationDate().getTime()),
						info.getMessageOrg(),
						info.getMultiId(),
						0);
			}
			
			
		} catch (NamingException e) {
			m_log.error("duplicationCheck : catch NamingException ",e);
		} catch (EJBException e) {
			m_log.error("duplicationCheck : catch EJBException ",e);
		} catch (RemoveException e) {
			m_log.error("duplicationCheck : catch RemoveException ",e);
		} catch (CreateException e) {
			m_log.error("duplicationCheck : catch CreateException ",e);
		}
		
		return ret;
	}
	
	/**
	 * 指定した間隔で、古いレコードを削除するメソッド
	 */
	private void removeOldRecode(){
		
		Date nowDate = new Date();
		
		// 現在時刻が削除時刻を過ぎていた場合
		if (nowDate.getTime() > m_removeTime) {
			
			try {
				// 現在時刻から残す時間を引く
				long timeLeft = nowDate.getTime() - TIME_LEFT;
				
				Collection<SyslogDuplicationInfoLocal> ct = SyslogDuplicationInfoUtil.getLocalHome().findOldRecode(new Timestamp(timeLeft));
				
				m_log.debug("removeOldRecode() : the number of records ： " + ct.size());

				Iterator<SyslogDuplicationInfoLocal> itr = ct.iterator();
				while(itr.hasNext()){
					itr.next().remove();
				}

				if(ct.size() > 0){
					m_log.info("removeOldRecode() : removed ");
				}
				
			} catch (FinderException e) {
				m_log.error("duplicationCheck : catch FinderException ",e);
			} catch (NamingException e) {
				m_log.error("duplicationCheck : catch NamingException ",e);
			} catch (EJBException e) {
				m_log.error("duplicationCheck : catch EJBException ",e);
			} catch (RemoveException e) {
				m_log.error("duplicationCheck : catch RemoveException ",e);
			}
			
			// 削除時刻に削除間隔を追加
			m_removeTime = m_removeTime + REMOVE_INTERVAL;
		}
		else {
			m_log.debug("removeOldRecode() : It is not deletion time yet. ");
		}
	}
}