/*
Copyright (C) 2010 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
*/

package com.clustercontrol.nodemap.ejb.session;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.BgFileNotFoundException;
import com.clustercontrol.FacilityNotFoundException;
import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.IconFileNotFoundException;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.nodemap.NodeMapException;
import com.clustercontrol.nodemap.ejb.entity.AssociationLocal;
import com.clustercontrol.nodemap.ejb.entity.AssociationPK;
import com.clustercontrol.nodemap.ejb.entity.AssociationUtil;
import com.clustercontrol.nodemap.ejb.entity.BgImageLocal;
import com.clustercontrol.nodemap.ejb.entity.BgImageUtil;
import com.clustercontrol.nodemap.ejb.entity.FacilityPositionLocal;
import com.clustercontrol.nodemap.ejb.entity.FacilityPositionPK;
import com.clustercontrol.nodemap.ejb.entity.FacilityPositionUtil;
import com.clustercontrol.nodemap.ejb.entity.IconImageLocal;
import com.clustercontrol.nodemap.ejb.entity.IconImageUtil;
import com.clustercontrol.nodemap.ejb.entity.MapInfoLocal;
import com.clustercontrol.nodemap.ejb.entity.MapInfoUtil;
import com.clustercontrol.nodemap.bean.Association;
import com.clustercontrol.nodemap.bean.FacilityElement;
import com.clustercontrol.nodemap.bean.NodeMapModel;
import com.clustercontrol.nodemap.bean.NodeElement;
import com.clustercontrol.nodemap.bean.ScopeElement;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityRelationLocal;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.repository.ejb.entity.NodeLocal;
import com.clustercontrol.repository.ejb.entity.NodeUtil;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;

/**
 *
 * <!-- begin-user-doc --> マップ情報の制御を行うsession bean <!-- end-user-doc --> *
 *
 * <!-- begin-xdoclet-definition --> 
 * @ejb.bean name="NodeMapController"	
 *           jndi-name="NodeMapController"
 *           type="Stateless" 
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 * <!-- end-xdoclet-definition --> 
 * @generated
 */
public abstract class NodeMapControllerBean implements javax.ejb.SessionBean {
	protected static Log m_log = LogFactory.getLog( NodeMapControllerBean.class );

	@SuppressWarnings("unused")
	private SessionContext m_context;
	private final int MAX_FILENAME = 64;

	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * リポジトリ情報からマップのデータを生成します。<BR>
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @param parentFacilityId 描画対象スコープの親のファシリティID
	 * @param facilityId 描画対象スコープのファシリティID
	 * @return マップ描画用データ
	 * @throws HinemosUnknownException 
	 * 
	 * @see com.clustercontrol.nodemap.model.NodeMapModel
	 */
	public NodeMapModel createNodeMapModel(String facilityId) throws HinemosUnknownException {
		m_log.debug("createNodeMap() start : " + facilityId);

		NodeMapModel map = null;

		// リポジトリからスコープの情報を取得する
		try {
			// リポジトリからスコープのパスを取得する
			RepositoryControllerLocal rep = RepositoryControllerUtil.getLocalHome().create();
			String facilityPath = rep.getFacilityPath(facilityId, null);
			m_log.debug("facilityPath = " + facilityPath);
			
			// リポジトリからスコープのファシリティ名を取得する
			String facilityName = "";
			if(FacilityConstant.STRING_COMPOSITE.equals(facilityId)){
				facilityName = "";
			} else {
				FacilityLocal bean = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
				m_log.debug("Bean = " + bean);
				facilityName = bean.getFacilityName();
			}
			m_log.debug("facilityName = " + facilityName);
			
			String parentFacilityId = null;
			// 親のファシリティのIDを取得する
			if(FacilityConstant.STRING_COMPOSITE.equals(facilityId)){ // トップスコープの場合は、親はなし
				parentFacilityId = null;
			} else {
				m_log.debug("getParentId = " + facilityId + "," + FacilityConstant.STRING_COMPOSITE);
				parentFacilityId = getParentId(facilityId);
				m_log.debug("parentFacilityId = " + parentFacilityId);
			}
			
			// 新規にマップを生成する
			map = new NodeMapModel(parentFacilityId, facilityId, facilityName, facilityPath);
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
			return new NodeMapModel("", "", "Could not create a map.", "-");
		} catch (Exception e) {
			m_log.error("CreateNodeMapModel " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		m_log.debug("createNodeMapModel() end");
		return map;
	}

	/**
	 * マップのデータをDBに登録します。<BR>
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @param facilityId 描画対象スコープのファシリティID
	 * @return マップ描画用データ
	 * @throws HinemosUnknownException 
	 * 
	 * @see com.clustercontrol.nodemap.model.NodeMapModel
	 */
	public void registerNodeMapModel(NodeMapModel map) throws HinemosUnknownException {
		String mapId = map.getMapId();
		
		// 登録時は、該当のマップの関係（コネクション）と要素を全て削除した後、登録する。
		try {
			// 関係を削除
			Collection<Object> assoList = AssociationUtil.getLocalHome().findByMapId(mapId);
			for(Object bean : assoList){
				try {
					((AssociationLocal)bean).remove();
				} catch (EJBException e) {
					m_log.error(e.getMessage(), e);
				} catch (RemoveException e) {
					m_log.error(e.getMessage(), e);
				}
			}
		} catch (FinderException e) {
			// 何もしない
		} catch (Exception e) {
			m_log.error("regiseterNodeMapModel " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		try {
			// 要素を削除
			Collection<Object> conList = FacilityPositionUtil.getLocalHome().findByMapId(mapId);
			for(Object bean : conList){
				try {
					((FacilityPositionLocal)bean).remove();
				} catch (EJBException e) {
					m_log.error(e.getMessage(), e);
				} catch (RemoveException e) {
					m_log.error(e.getMessage(), e);
				}
			}
		} catch (FinderException e) {
			// 何もしない
		} catch (Exception e) {
			m_log.error("regiseterNodeMapModel " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		try {
			// ファシリティIDをマップIDとしてマップ情報を取得する
			MapInfoLocal bean = MapInfoUtil.getLocalHome().findByPrimaryKey(mapId);
			bean.setBackgroundImage(map.getBgName());
		} catch (FinderException e) {
			// 無い場合は生成
			try {
				MapInfoUtil.getLocalHome().create(
						map.getMapId(),
						map.getBgName()
				);
			} catch (CreateException e1) {
				m_log.error(e,e);
			} catch (NamingException e1) {
				m_log.error(e,e);
			}
		} catch (Exception e) {
			m_log.error("regiseterNodeMapModel " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		// エレメントを更新
		FacilityElement[] elemet = map.getContents();
		for(int i=0; i<elemet.length; i++){
			FacilityPositionLocal posi = null;
			try {
				posi = FacilityPositionUtil.getLocalHome().findByPrimaryKey(
						new FacilityPositionPK(mapId, elemet[i].getFacilityId()));
				posi.setX(elemet[i].getX());
				posi.setY(elemet[i].getY());
			} catch (FinderException e) {
				// 無い場合は新たに生成
				try {
					posi = FacilityPositionUtil.getLocalHome().create(
							mapId,
							elemet[i].getFacilityId(),
							elemet[i].getX(),
							elemet[i].getY()
					);
				} catch (CreateException e1) {
					m_log.error(e,e);
				} catch (Exception e1) {
					m_log.error("regiseterNodeMapModel " + e1.getMessage(), e1);
					throw new HinemosUnknownException(e1.getMessage(), e1);
				}
			} catch (NamingException e) {
				m_log.error(e,e);
			}
		}

		// 関連を更新
		for (Association asso : map.getAssociations()) {
			try {
				AssociationUtil.getLocalHome().findByPrimaryKey(
						new AssociationPK(mapId, asso.getSource(), asso.getTarget()));
			} catch (FinderException e) {
				// 無い場合は新規に作成
				try {
					AssociationUtil.getLocalHome().create(
							mapId,
							asso.getSource(),
							asso.getTarget()
					);
				} catch (CreateException e1) {
					m_log.error(e,e);
				} catch (Exception e1) {
					m_log.error("regiseterNodeMapModel " + e1.getMessage(), e1);
					throw new HinemosUnknownException(e1.getMessage(), e1);
				}
			} catch (NamingException e) {
				m_log.error(e,e);
			}
		}
		
		// アイコンを更新
		for (FacilityElement element : map.getContents()) {
			String facilityId = element.getFacilityId();
			String iconImage = element.getIconImage();
			if (iconImage == null) {
				continue;
			}
			NodeLocal bean = null;
			try {
				bean = NodeUtil.getLocalHome().findByPrimaryKey(facilityId);
			} catch (FinderException e) {
				// スコープの場合はこのルートを通る。
				// m_log.warn("registerNodeMapModel : FinderException, " + facilityId);
				continue;
			} catch (Exception e) {
				m_log.error("regiseterNodeMapModel " + e.getMessage(), e);
				throw new HinemosUnknownException(e.getMessage(), e);
			}
			if (!iconImage.equals(bean.getIconImage())) {
				bean.setIconImage(iconImage);
			}
		}
	}

	/**
	 * マップのデータを取得します。<BR>
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @param facilityId 描画対象スコープのファシリティID
	 * @return マップ描画用データ
	 * @throws NodeMapException 
	 * 
	 * @see com.clustercontrol.nodemap.model.NodeMapModel
	 */
	public NodeMapModel getNodeMapModel(String facilityId) throws HinemosUnknownException, NodeMapException {
		m_log.debug("getNodeMap() start");
		
		// facilityIdが履歴には存在するが、リポジトリに存在しない場合をチェック。
		try {
			if (!FacilityConstant.STRING_COMPOSITE.equals(facilityId)) {
				FacilityLocal bean = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
			}
		} catch (FinderException e) {
			m_log.error(e);
			throw new NodeMapException ("cannot find " + facilityId, e);
		} catch (Exception e) {
			m_log.error("getNodeMapModel " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		
		// リポジトリ情報をもとにマップを生成
		NodeMapModel map = createNodeMapModel(facilityId);
		String mapId = facilityId;

		m_log.debug("MapId = " + mapId);

		MapInfoLocal bean = null;
		try {
			// ファシリティIDをマップIDとしてマップ情報を取得する
			bean = MapInfoUtil.getLocalHome().findByPrimaryKey(mapId);
		} catch (FinderException e) {
			// マップ情報がまだ生成されていない場合
			try {
				m_log.warn(e.getMessage() + " : Map ID = " + mapId);

				// 指定のスコープ配下のスコープおよびノードの一覧を取得する
				ArrayList<String> facilityIdList = RepositoryControllerUtil.
						getLocalHome().create().getFacilityIdList(facilityId, 1);
				for (String fid : facilityIdList) {
					// エレメントを生成
					FacilityElement element = createElementForRepository(facilityId, fid);
					// 座標未登録ノードを追加
					map.addContent(element);
				}
			} catch (FacilityNotFoundException e1) {
				map = new NodeMapModel("", "", "Could not create a map.", "-");
			} catch (CreateException e1) {
				m_log.error(e1.getMessage(), e1);
				map = new NodeMapModel("", "", "Could not create a map.", "-");
			} catch (Exception e1) {
				m_log.error("getNodeMapModel " + e.getMessage(), e1);
				throw new HinemosUnknownException(e.getMessage(), e1);
			}
			return map;
		} catch (Exception e) {
			m_log.error("getodeMapModel " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		// 背景イメージ名を設定する
		map.setBgName(bean.getBackgroundImage());
		
		ArrayList<String> childList = null;
		try {
			childList = RepositoryControllerUtil.
					getLocalHome().create().getFacilityIdList(facilityId, 1);
		} catch (CreateException e3) {
			m_log.error(e3.getMessage(), e3);
			map = new NodeMapModel("", "", "Could not create a map.", "-");
		} catch (Exception e) {
			m_log.error("getodeMapModel " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		
		for (String childId : childList) {
			String parentId = facilityId;

			// 対応するファシリティIDの座標情報がDBにあるか否かを調べる
			FacilityPositionLocal elementPosition = null;
			try {
				elementPosition = FacilityPositionUtil.getLocalHome().findByPrimaryKey(
						new FacilityPositionPK(mapId, childId));
				// リポジトリの情報を元にエレメントを生成
				FacilityElement element = createElementForRepository(parentId, childId);
				element.setPosition(elementPosition.getX(), elementPosition.getY());
				// 座標保持ノードに追加
				map.addContent(element);
			} catch (FinderException e1) {
				try {
					// DBに対象のファシリティの図の情報が無い場合の処理
					// エレメントを生成
					FacilityElement element = createElementForRepository(facilityId, childId);

					// 座標未登録ノードを追加
					map.addContent(element);
				} catch (FacilityNotFoundException e2) {
					// 何もしない。
				} catch (Exception e2) {
					m_log.error("getNodeMapModel " + e2.getMessage(), e2);
					throw new HinemosUnknownException(e2.getMessage(), e2);
				}
			} catch (Exception e) {
				m_log.error("getNodeMapModel " + e.getMessage(), e);
				throw new HinemosUnknownException(e.getMessage(), e);
			}
		}

		// 関連情報を取得して設定
		Collection<AssociationLocal> associations = null;
		m_log.debug("set association");
		try {
			associations = AssociationUtil.getLocalHome().findByMapId(mapId);
			for (AssociationLocal assoBean : associations) {
				String src = assoBean.getSource();
				String trg = assoBean.getTarget();

				// src も trg も両方存在する場合のみマップのモデルに追加する
				if(map.getElement(src) != null && map.getElement(trg) != null){
					Association asso = new Association(src, trg);
					map.addAssociation(asso);
				}
			}
		} catch (FinderException e2) {
			m_log.error(e2,e2);
		} catch (NamingException e) {
			m_log.error(e,e);
		}

		return map;
	}

	/**
	 * 背景データを取得します。<BR>
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @param filename 
	 * @return filedata
	 * @throws BgFileNotFoundException 
	 */
	public byte[] getBgImage(String filename) throws HinemosUnknownException, BgFileNotFoundException {
		// 背景のバイナリデータを取得
		try {
			BgImageLocal bean = BgImageUtil.getLocalHome().findByPrimaryKey(filename);
			return bean.getFiledata();
		} catch (FinderException e) {
			m_log.warn("getBgImage " + e.getMessage(),e);
			throw new BgFileNotFoundException(e.getMessage(),e);
		} catch (Exception e) {
			m_log.error("getBgImage " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
	}


	/**
	 * 背景データをDBに登録します。<BR>
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @param filename
	 * @param filedata
	 * @throws HinemosUnknownException 
	 * @throws NodeMapException 
	 */
	public void setBgImage(String filename, byte[] filedata) throws HinemosUnknownException, NodeMapException {
		if (filename != null && filename.length() > MAX_FILENAME) {
			String[] args = { Integer.toString(MAX_FILENAME) };
			throw new NodeMapException(
					com.clustercontrol.nodemap.util.Messages.getString("file.name.too.long", args));
		}
		try {
			// 上書き
			BgImageLocal bean = BgImageUtil.getLocalHome().findByPrimaryKey(filename);
			bean.setFiledata(filedata);
		} catch (FinderException e) {
			// 無い場合は生成
			try {
				BgImageUtil.getLocalHome().create(filename, filedata);
			} catch (Exception e1) {
				m_log.error("NamingException " + e1.getMessage(),e1);
				throw new HinemosUnknownException(e1.getMessage(),e1);
			}
		} catch (Exception e) {
			m_log.error("NamingException " + e.getMessage(),e);
			throw new HinemosUnknownException(e.getMessage(),e);
		}
	}

	/**
	 * 背景画像のファイル名一覧を取得します。<BR>
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @return Collection<String>
	 * @throws HinemosUnknownException 
	 */
	public Collection<String> getBgImagePK() throws HinemosUnknownException {
		// 背景のPKを取得
		ArrayList<String> filenameArray = new ArrayList<String>();
		Collection<BgImageLocal> beanCollection = null;
		try {
			beanCollection = BgImageUtil.getLocalHome().findAll();
		} catch (Exception e) {
			m_log.error("getBgImagePK " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(),e);
		}
		for (BgImageLocal bean : beanCollection) {
			filenameArray.add((String)bean.getPrimaryKey());
		}
		Collections.sort(filenameArray);
		return filenameArray;
	}

	/**
	 * 背景画像のファイル名の存在有無を取得します。<BR>
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @return boolean
	 */
	public boolean isBgImage(String filename) throws HinemosUnknownException {
		// 背景のバイナリデータを取得
		try {
			BgImageUtil.getLocalHome().findByPrimaryKey(filename);
		} catch (FinderException e) {
			return false;
		} catch (Exception e) {
			m_log.error("isBgImage " + e.getMessage(),e);
			throw new HinemosUnknownException(e.getMessage(),e);
		}
		return true;
	}

	/**
	 * アイコン画像を取得します。<BR>
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @param filename 
	 * @return filedata
	 * @throws HinemosUnknownException 
	 * @throws IconFileNotFoundException 
	 */
	public byte[] getIconImage(String filename) throws HinemosUnknownException, IconFileNotFoundException {
		// 背景のバイナリデータを取得
		IconImageLocal bean = null;
		try {
			bean = IconImageUtil.getLocalHome().findByPrimaryKey(filename);
		} catch (FinderException e) {
			m_log.warn("getIconImage " + e.getMessage());
			throw new IconFileNotFoundException(e.getMessage(),e);
		} catch (Exception e) {
			m_log.error("getIconImage " + e.getMessage(),e);
			throw new HinemosUnknownException(e.getMessage(),e);
		}
		return bean.getFiledata();
	}

	/**
	 * アイコン画像をDBに登録します。<BR>
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @param filename
	 * @param filedata
	 * @throws NodeMapException 
	 * @throws HinemosUnknownException 
	 * 
	 */
	public void setIconImage(String filename, byte[] filedata) throws NodeMapException, HinemosUnknownException {
		if (filename != null && filename.length() > MAX_FILENAME) {
			String[] args = { Integer.toString(MAX_FILENAME) };
			throw new NodeMapException(
					com.clustercontrol.nodemap.util.Messages.getString("file.name.too.long", args));
		}
		try {
			// 上書き
			IconImageLocal bean = IconImageUtil.getLocalHome().findByPrimaryKey(filename);
			bean.setFiledata(filedata);
		} catch (FinderException e) {
			// 無い場合は生成
			try {
				IconImageUtil.getLocalHome().create(filename, filedata);
			} catch (Exception e1) {
				m_log.error("setIconImage " + e1.getMessage(),e1);
				throw new HinemosUnknownException(e1.getMessage(),e1);
			}
		} catch (Exception e) {
			m_log.error("setIconImage " + e.getMessage(),e);
			throw new HinemosUnknownException(e.getMessage(),e);
		}
	}

	/**
	 * アイコンのファイル名一覧を取得します。<BR>
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @return Collection<String>
	 * @throws HinemosUnknownException 
	 */
	public Collection<String> getIconImagePK() throws HinemosUnknownException {
		// アイコンのPKを取得
		ArrayList<String> filenameArray = new ArrayList<String>();
		Collection<IconImageLocal> beanCollection = null;
		try {
			beanCollection = IconImageUtil.getLocalHome().findAll();
		} catch (Exception e) {
			m_log.error("getIconImagePK " + e.getMessage(),e);
			throw new HinemosUnknownException(e.getMessage(),e);
		}
		for (IconImageLocal bean : beanCollection) {
			filenameArray.add((String)bean.getPrimaryKey());
		}
		Collections.sort(filenameArray);
		return filenameArray;
	}

	/**
	 * アイコンのファイル名の存在有無を取得します。<BR>
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @return boolean
	 * @throws HinemosUnknownException 
	 */
	public boolean isIconImage(String filename) throws HinemosUnknownException {
		// アイコンの存在有無を確認
		try {
			IconImageUtil.getLocalHome().findByPrimaryKey(filename);
		} catch (FinderException e) {
			return false;
		} catch (Exception e) {
			m_log.error("isIconImage " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return true;
	}


	/**
	 * 親を探す。
	 * 
	 * @param targetId 対象のスコープID
	 * @param serachId 探索しているスコープID
	 */
	private String getParentId(String targetId) {
		if (FacilityConstant.STRING_COMPOSITE.equals(targetId)) {
			return null;
		} else if (FacilityTreeAttributeConstant.REGISTEREFD_SCOPE.equals(targetId) ||
				FacilityTreeAttributeConstant.INTERNAL_SCOPE.equals(targetId)) {
			return FacilityConstant.STRING_COMPOSITE;
		}
		
		FacilityLocal bean = null;
		try {
			bean = FacilityUtil.getLocalHome().findByPrimaryKey(targetId);
		} catch (FinderException e) {
			m_log.error(e.getMessage(), e);
			return null;
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
			return null;
		}
		
		Collection<FacilityRelationLocal> parentBeanCol =
			bean.getFacilityRelationAsChild();
		
		/*
		 * スコープの親は必ず一個。
		 * 最上位スコープの一段下のみ、親はFacilityRelationLocalから取得できない。
		 */
		if (parentBeanCol.size() == 0) {
			return FacilityConstant.STRING_COMPOSITE;
		}
		if (parentBeanCol.size() != 1) {
			m_log.error("getParentId (" + targetId + ") logic error " +
					parentBeanCol.size());
			return null;
		}
		FacilityRelationLocal parentBean = (FacilityRelationLocal)(parentBeanCol.toArray())[0];
		
		return parentBean.getParentFacilityId();
	}

	private FacilityElement createElementForRepository(String parentId, String fid) throws HinemosUnknownException, FacilityNotFoundException {
		FacilityElement element;
		// 属性値取得用のエンティティビーン
		FacilityLocal facilityEntity;
		try {
			facilityEntity = FacilityUtil.getLocalHome().findByPrimaryKey(fid);
		} catch (FinderException e) {
			m_log.warn("createElementForRepository " + e.getMessage());
			throw new FacilityNotFoundException(e.getMessage(),e);
		} catch (Exception e) {
			m_log.error("createElementForRepository " + e.getMessage(),e);
			throw new HinemosUnknownException(e.getMessage(),e);
		}
		
		// アイコン名を取得
		String facilityName = facilityEntity.getFacilityName();
		String iconImage = null;

		// スコープの場合
		if (facilityEntity.getFacilityType() == FacilityConstant.TYPE_SCOPE) {
			m_log.debug("Scope : " + facilityName + "(" + fid + ")");

			// Hinemos3.1ではスコープのiconImageはnullとなる。
			// Hinemos3.1では有効/無効フラグはスコープでは常に有効となる。
			element = new ScopeElement(parentId, fid, facilityName, "scope", true);
		} else {  // ノードの場合
			m_log.debug("Node : " + facilityName + "(" + fid + ")");
			
			NodeLocal nodeEntity = null;
			try {
				nodeEntity = NodeUtil.getLocalHome().findByPrimaryKey(fid);
			} catch (FinderException e) {
				m_log.warn("createElementForRepository " + e.getMessage());
				throw new FacilityNotFoundException(e.getMessage(),e);
			} catch (Exception e) {
				m_log.error("createElementForRepository " + e.getMessage(),e);
				throw new HinemosUnknownException(e.getMessage(), e);
			}
			boolean valid = facilityEntity.isValid();
			iconImage = nodeEntity.getIconImage();
			element = new NodeElement(parentId, fid, facilityName, iconImage, valid);

			// 表示用の属性値を追加設定
			element.setAttributes("IpProtocolNumber", nodeEntity.getIpAddressVersion());
			element.setAttributes("IpNetworkNumber", nodeEntity.getIpAddressV4());
			element.setAttributes("IpNetworkNumberV6", nodeEntity.getIpAddressV6());
			element.setAttributes("NodeName", nodeEntity.getNodeName());
		}
		// 表示用の属性値を設定
		element.setAttributes("FacilityId", facilityEntity.getFacilityId());
		element.setAttributes("Description", facilityEntity.getDescription());

		return element;
	}
}
