/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.run.factory;

import java.io.File;
import java.io.FileInputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.TransactionRolledbackLocalException;
import javax.jms.DeliveryMode;
import javax.jms.JMSException;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.CalendarNotFountException;
import com.clustercontrol.FacilityNotFoundException;
import com.clustercontrol.bean.MonitorBlockConstant;
import com.clustercontrol.bean.OutputNotifyGroupInfo;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.calendar.ejb.session.CalendarControllerLocal;
import com.clustercontrol.calendar.ejb.session.CalendarControllerUtil;
import com.clustercontrol.commons.util.JmsPersistenceConfig;
import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.monitor.run.bean.MonitorJudgementInfo;
import com.clustercontrol.monitor.run.bean.MonitorRunResultInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoPK;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.monitor.run.util.CallableTask;
import com.clustercontrol.monitor.run.util.MonitorTransactionManager;
import com.clustercontrol.monitor.run.util.ParallelExecution;
import com.clustercontrol.notify.bean.QueueConstant;
import com.clustercontrol.priority.util.PriorityJudgment;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.Messages;

/**
 * 監視を実行する抽象クラス<BR>
 * <p>
 * 監視種別（真偽値，数値，文字列）の各クラスで継承してください。
 *
 * @version 3.0.0
 * @since 2.0.0
 */
abstract public class RunMonitor {
	
	private static final int RETRY_MAX = 1;
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( RunMonitor.class );
	
	/** 通知のメッセージ。 */
	public static final String MESSAGE_INFO = Messages.getString("message.info");
	/** 警告のメッセージ。 */
	public static final String MESSAGE_WARNING = Messages.getString("message.warning");
	/** 危険のメッセージ。 */
	public static final String MESSAGE_CRITICAL = Messages.getString("message.critical");
	/** 不明のメッセージ。 */
	public static final String MESSAGE_UNKNOWN = Messages.getString("message.unknown");
	
	/** 監視結果の確認間隔 */
	public static int RETRY_INTERVAL = 1000;
	
	/** 監視情報のローカルコンポーネント。 */
	protected MonitorInfoLocal m_monitor;
	
	/** 監視対象ID。 */
	protected String m_monitorTypeId;
	
	/** 監視項目ID。 */
	protected String m_monitorId;
	
	/** 通知ID */
	protected String m_notifyGroupId;
	
	/** 監視対象ファシリティID。 */
	protected String m_facilityId;
    
    /** 
     * 値取得の失敗時の重要度。
     * 初期値は値取得失敗を示す値(-1)として、
     * setMonitorInfoがコールされた際に、値取得失敗時の重要度がセットされる。
     * 
 	 * @see #setMonitorInfo(String, String)
     */
	protected int m_failurePriority = PriorityConstant.TYPE_FAILURE;
	
	/** 収集間隔 */
	protected int m_runInterval;
	
	/** 監視開始時刻。 */
	protected Date m_now;
	
	/** スコープ 監視結果取得時刻。 */
	protected long m_scopeDate;
	
	/** ノード 監視結果取得時刻。 */
	protected long m_nodeDate;
	
	/** 応答時間（秒）。 */
	protected int m_responseTime;
	
	/** 通知情報。 */
	protected OutputNotifyGroupInfo m_notifyGroupInfo;
	
	/** リポジトリ情報のローカルコンポーネント。 */
	protected RepositoryControllerLocal m_repository;
		
	/** 監視管理キュー送信。 */
	private SendQueue m_queue;
	
	/** 監視単位。 */
	protected int m_monitorBlock;
	
	/** タイムアウト（秒）。@version 2.1.0 */
	protected int m_timeout;
	
	/** ジョブ実行。 @version 2.1.0 */
	protected int m_jobRun;
	
//	/** 値取得の失敗時のジョブ実行。 */
//	protected int m_failureJobRun = YesNoConstant.TYPE_NO;
	
//	/** 値取得の失敗時のジョブID。 */
//	protected String m_failureJobId;
	
//	/** 値取得の失敗時のジョブ実行 通知抑制と連動。 */
//	protected int m_failureJobInhibitionFlg = YesNoConstant.TYPE_NO;
	
//	/** 値取得の失敗時のジョブ呼出の失敗時の重要度。 */
//	protected int m_failureJobFailurePriority = PriorityConstant.TYPE_UNKNOWN;
	
	/** ノードフラグ。 */
	protected boolean m_isNode;
	
	/** 重要度別ファシリティ名マップ。 */
	protected HashMap<Integer, ArrayList<String>> m_priorityMap;
	
	/** 
	 * 判定情報マップ。
	 * <p>
	 * <dl>
	 *  <dt>キー</dt>
	 *  <dd>真偽値監視：真偽値定数（{@link com.clustercontrol.monitor.run.bean.TruthConstant}）</dd>
	 *  <dd>数値監視：重要度定数（{@link com.clustercontrol.bean.PriorityConstant}）</dd>
	 *  <dd>文字列監視：順序（{@link com.clustercontrol.monitor.run.ejb.entity.MonitorStringValueInfoBean#getOrderNo()}）</dd>
	 * </dl>
	 */
	protected TreeMap<Integer, MonitorJudgementInfo> m_judgementInfoList;
	
	// プロパティファイルの読み込み
	static{
		Properties properties = new Properties();
		String etcdir = System.getProperty("hinemos.manager.etc.dir");
		String propertyFile = etcdir + File.separator + "hinemos.properties";
				
		try {
			properties.load(new FileInputStream(propertyFile));
			
			String retryInterval = properties.getProperty("monitor.common.retry.interval");
			
			RETRY_INTERVAL = Integer.parseInt(retryInterval);
			
			new ParallelExecution(properties);	// スレッド数の設定
			
		} catch (Exception e) {
			m_log.error(e.getMessage(), e);
		} 
	}
	
	
	/**
	 * コンストラクタ。<BR>
	 * 初期処理を呼び出します。
	 * 
	 * @see #initialize()
	 */
	public RunMonitor() throws NamingException, JMSException, CreateException{
		
		// 初期化処理
//		this.initialize();
	}
	
	/**
	 * トランザクションを開始し、引数で指定された監視情報の監視を実行します。
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @param monitorId 監視項目ID
	 * @throws FinderException
	 * @throws CreateException 
	 * @throws RemoveException
	 * @throws JMSException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws SystemException
	 * 
	 * @see #runMonitorInfo()
	 */
	public void run(String monitorTypeId, String monitorId) throws FinderException, RemoveException, JMSException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, InvalidTransactionException, IllegalStateException, SystemException, CreateException {
		
		this.initialize(monitorTypeId);
		
		m_monitorTypeId = monitorTypeId;
		m_monitorId = monitorId;
		
		TransactionManager tm = null;
//		Transaction tx = null;
//		Transactionの利用を取りやめたため、関連箇所ふくめコメントアウトした : uchiyama
		Transaction oldTx = null;
		boolean result = false;
		
		int count = 0;
		boolean deadlock = false;

		try 
		{
			// TransactionManagerを取得
			tm = MonitorTransactionManager.getInstance().getTransactionManager();
			
			// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避 
			if(tm.getTransaction() != null){
				oldTx = tm.suspend();
			}
			
			try{
				
				do {
					try 
					{
						count++;
						result = false;
						deadlock = false;
						
						// トランザクション開始
						if(tm.getTransaction() == null){
							tm.begin();	
//							tx = tm.getTransaction();
						}
						
						// 監視実行
						result = runMonitorInfo();
						
						if(result){
							// コミット
//							tx.commit();
							tm.commit();
						}
						
					} catch (TransactionRolledbackLocalException e) {
						
						// デッドロックが発生した場合はリトライ
						if(e.getCausedByException() instanceof org.jboss.util.deadlock.ApplicationDeadlockException){
							
							deadlock = true;
							if(count <= RETRY_MAX){
								m_log.debug("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
							}
							else{
								m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
							}
							
//							if(tx != null){
//								if(!result){
//									tx.rollback();
//								}
//								tx = null;
//							}
							if(tm.getTransaction() != null){
								if(!result){
									tm.rollback();
								}
							}
						}
						else{
							m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
							throw e;
						}
					}
					
				} while (deadlock && count <= RETRY_MAX);
							
			} catch(NotSupportedException e){
				m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
	            throw e;
			} catch (RollbackException e) {
				m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
	            throw e;
			} catch (HeuristicMixedException e) {
				m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
	            throw e;
			} catch(HeuristicRollbackException e){
				m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
	            throw e;
			} catch(SystemException e){
				m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
	            throw e;
			}
			finally{
				// トランザクション関連の例外が発生した場合は、ロールバック
//				if(tx != null){
//					if(!result){
//						tx.rollback();
//					}
//					tx = null;
//				}
				if(tm.getTransaction() != null){
					if(!result){
						tm.rollback();
					}
				}
			}
		}
		finally{
			
			// 終了処理
			this.terminate();
			
			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);
				
				} catch(InvalidTransactionException e){
					m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
		            throw e;
				} catch(IllegalStateException e){
					m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
		            throw e;
				} catch(SystemException e){
					m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
		            throw e;
				}
			}
		}
	}

	/**
	 * 監視を実行します。（並列処理）
	 * <p>
	 * <ol>
	 * <li>監視情報を取得し、保持します（{@link #setMonitorInfo(String, String)}）。</li>
	 * <li>判定情報を取得し、判定情報マップに保持します（{@link #setJudgementInfo()}）。</li>
	 * <li>チェック条件情報を取得し、保持します（{@link #setCheckInfo()}）。</li>
	 * <li>ファシリティ毎に並列に監視を実行し、値を収集します。 （{@link #collect(String)}）。</li>
	 * <li>監視結果から、判定結果を取得します。 （{@link #getCheckResult(boolean)}）。</li>
	 * <li>監視結果から、重要度を取得します（{@link #getPriority(int)}）。</li>
	 * <li>監視結果を通知します（{@link #notify(boolean, String, int, Date)}）。</li>
	 * </ol>
	 * 
	 * @return 実行に成功した場合、</code> true </code>
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws JMSException
	 * @throws NamingException
	 * @throws CreateException
	 * 
	 * @see #setMonitorInfo(String, String)
	 * @see #setJudgementInfo()
	 * @see #setCheckInfo()
	 * @see #collect(String)
	 * @see #getCheckResult(boolean)
	 * @see #getPriority(int)
	 * @see #notify(boolean, String, int, Date)
	 */
	public boolean runMonitorInfo()throws FinderException, RemoveException, JMSException, NamingException, CreateException {

		m_now = new Date();
		
		m_priorityMap = new HashMap<Integer, ArrayList<String>>();
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_INFO),		new ArrayList<String>());
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_WARNING),	new ArrayList<String>());
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_CRITICAL),	new ArrayList<String>());
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_UNKNOWN),	new ArrayList<String>());
		
		try 
		{
			// 監視基本情報を設定
			boolean run = this.setMonitorInfo(m_monitorTypeId, m_monitorId);
			if(!run){
				// 処理終了
				return true;
			}
			
			// 判定情報を設定
			setJudgementInfo();
			
			// チェック条件情報を設定
			setCheckInfo();
			
			// ファシリティIDの配下全ての一覧を取得
			// 有効/無効フラグがtrueとなっているファシリティIDを取得する
			ArrayList<String> facilityList = m_repository.getExecTargetFacilityIdList(m_facilityId);
//			ArrayList facilityList = m_repository.getNodeFacilityIdList(m_facilityId, RepositoryControllerBean.ALL);
//	
//			//配下にノードがないということはノードの可能性があるので指定されたIDをセット
//			if (facilityList.size() == 0) {
//				if(m_repository.isNode(m_facilityId)){
//					m_isNode = true;
//					facilityList.add(m_facilityId);
//				}
//				else{
//					return true;
//				}
//			}
			if (facilityList.size() == 0) {
				return true;
			}
			
			m_isNode = m_repository.isNode(m_facilityId);
			
			m_log.debug("monitor start : monitorTypeId : " + m_monitorTypeId + ", monitorId : " + m_monitorId);
			
			String facilityId = null;
			// ファシリティIDをキーとしたハッシュマップを作成
			HashMap<String,Future<MonitorRunResultInfo>> futuresMap = new HashMap<String,Future<MonitorRunResultInfo>>(facilityList.size());

			/** 
			 * 監視の実行
			 */
			// ファシリティIDの数だけ、各監視処理を実行する
			Iterator<String> itr = facilityList.iterator();
			while(itr.hasNext()){
				facilityId = itr.next();
				if(facilityId != null && !"".equals(facilityId)){
					
					// マルチスレッド実行用に、RunMonitorのインスタンスを新規作成する
					// インスタンスを新規作成するのは、共通クラス部分に監視結果を保持するため
					RunMonitor runMonitor = this.createMonitorInstance();
					
					// 監視実効に必要な情報を再度セットする
					runMonitor.m_monitorTypeId = this.m_monitorTypeId;
					runMonitor.m_monitorId = this.m_monitorId;
					runMonitor.m_now = this.m_now;
					runMonitor.setMonitorInfo(m_monitorTypeId, m_monitorId);
					runMonitor.m_priorityMap = this.m_priorityMap;
					runMonitor.setJudgementInfo();
					runMonitor.m_repository = this.m_repository;
					runMonitor.setCheckInfo();
					
//					Future<MonitorRunResultInfo> future = ParallelExecution.m_executorService.submit(new CallableTask(this, facilityId));
					Future<MonitorRunResultInfo> future = ParallelExecution.m_executorService.submit(new CallableTask(runMonitor, facilityId));
					futuresMap.put(facilityId, future);
				}
				else {
					facilityList.remove(facilityId);
				}
			}		
			
			
			/** 
			 * 監視結果の集計
			 */
			ArrayList<String> facilityListCopy = null;	// ファシリティリストのコピーを格納（監視を完了したものから削除するためコピーを作成）
			ArrayList<String> facilityListComplete = null;	// 監視を完了したファシリティIDを格納
			boolean isGet = false;	// 監視が完了したかを判定
			MonitorRunResultInfo result = new MonitorRunResultInfo();	// 監視結果を格納
			
			m_log.debug("total start : monitorTypeId : " + m_monitorTypeId + ", monitorId : " + m_monitorId);
			
			// 監視結果出力単位がノードである場合のログ出力情報を送信 
			if(MonitorBlockConstant.TYPE_NODE == m_monitorBlock ||	// 監視単位がノードである場合
					(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock && m_isNode) || 	// 監視単位がスコープかつ、対象のファシリティIDがノードだったら　FIXME：この項目はクライアントで選択できないようにすべき
					 MonitorBlockConstant.TYPE_ALL == m_monitorBlock){	// 監視単位がノードかつスコープである場合（現在未使用）
			
				facilityListCopy = new ArrayList<String>(facilityList);
				facilityListComplete = new ArrayList<String>();
				while(true) {

					facilityListComplete.clear();	// 収集したファシリティID情報を削除する
					itr = facilityListCopy.iterator();	// 収集が完了していない情報のみを確認するためIteratorを取得しなおす
					while(itr.hasNext()) {
						facilityId = itr.next();
						isGet = futuresMap.get(facilityId).isDone();	//監視完了の有無を取得

						// 監視が終了していた場合
						if(isGet){
							m_log.debug("output result : monitorTypeId : " + m_monitorTypeId + ", m_monitorId : " + m_monitorId + ",　m_facilityId : " + facilityId);
							
							result = futuresMap.get(facilityId).get();	// 監視結果を取得
							m_nodeDate = result.getNodeDate();
															
							// 監視結果を通知
							notify(true, facilityId, result.getCheakResult(), new Date(m_nodeDate), result);	
							
							// 監視が完了したファシリティIDをfacilityListCompleteに追加する
							facilityListComplete.add(facilityId);
						}
					}
					
					// 監視が完了したファシリティIDをfacilityListCopyから削除する
					itr = facilityListComplete.iterator();
					while(itr.hasNext()){
						facilityListCopy.remove(itr.next());
					}
					
					// すべてのファシリティIDの監視が完了したかの確認
					if(facilityListCopy.size() == 0) {
						m_log.debug("end collect (node) : monitorTypeId : " + m_monitorTypeId + ", m_monitorId : " + m_monitorId);
						break;
					}
					else {
						m_log.debug("collect (node) : monitorTypeId : " + m_monitorTypeId + ", m_monitorId : " + m_monitorId);
						Thread.sleep(RETRY_INTERVAL);	// まだ結果が出揃っていない場合は「RETRY_INTERVAL」秒sleepする
					}
				}
			}
			
			
			
			// 監視結果出力単位がスコープである場合のログ出力情報を送信 
			if(!m_isNode && // 対象のファシリティIDがノードではない場合
				((MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock || // 監視単位がスコープである場合
					MonitorBlockConstant.TYPE_ALL == m_monitorBlock))){	// 監視単位がノードかつスコープである場合（現在未使用）
				
				ArrayList<Integer> priorityList = new ArrayList<Integer>();	// ノード単位の重要度を格納する
				
				// すべての監視結果が揃っていることを確認する
				facilityListCopy = new ArrayList<String>(facilityList);
				facilityListComplete = new ArrayList<String>();
				while(true){
					
					facilityListComplete.clear();	// 収集したファシリティID情報を削除する
					itr = facilityListCopy.iterator();	// 収集が完了していない情報のみを確認するためIteratorを取得しなおす
					while(itr.hasNext()){
						facilityId = itr.next();
						isGet = futuresMap.get(facilityId).isDone();
						
						// 監視が終了していた場合
						// ノード単位の監視結果を集計し、スコープとしての監視結果を作成する
						if(isGet){
							result = futuresMap.get(facilityId).get();	// 監視結果を取得
							m_nodeDate = result.getNodeDate();
							
							// 判定結果よりノードの重要度を取得
							int priority = result.getPriority();
							// スコープの重要度を設定
							priorityList.add(Integer.valueOf(priority));
							// 重要度別にファシリティIDを設定
							setPriorityMap(Integer.valueOf(priority), facilityId);
							
							// スコープの値取得時刻を設定
			        		if(m_nodeDate > m_scopeDate){
			        			m_scopeDate = m_nodeDate;
			        		}
							
							m_log.debug("output result : monitorTypeId : " + m_monitorTypeId + ", m_monitorId : " + m_monitorId + ",　m_facilityId : " + facilityId);							
							// 監視が完了したファシリティIDをfacilityListCompleteに追加する
							facilityListComplete.add(facilityId);
						}
					}
					
					// 監視が完了したファシリティIDをfacilityListCopyから削除する
					itr = facilityListComplete.iterator();
					while(itr.hasNext()){
						facilityListCopy.remove(itr.next());
					}
					
					// すべてのファシリティIDの監視が完了したかの確認
					if(facilityListCopy.size() == 0) {
						m_log.debug("end collect (scope) : monitorTypeId : " + m_monitorTypeId + ", m_monitorId : " + m_monitorId);
						break;
					}
					else {
						m_log.debug("collect (scope) : monitorTypeId : " + m_monitorTypeId + ", m_monitorId : " + m_monitorId);
						Thread.sleep(RETRY_INTERVAL);	// まだ結果が出揃っていない場合は「RETRY_INTERVAL」秒sleepする
					}
				}
					
				// スコープの重要度を決定する
				Integer priority = PriorityJudgment.judgment(priorityList); 
				if(priority == null){
					priority = Integer.valueOf(m_failurePriority);
				}
				
				// 各ノードの内、最新日付を設定
				Date generationDate = null;
				if(m_scopeDate > 0){
					generationDate = new Date(m_scopeDate);
				}
				else{
					generationDate = m_now;
				}
				
				// 監視結果を通知
				notify(false, m_facilityId, priority.intValue(), generationDate);
					
			}
			
			m_log.debug("monitor end : monitorTypeId : " + m_monitorTypeId + ", monitorId : " + m_monitorId);
			
			return true;
			
		} catch (FacilityNotFoundException e) {
			m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
			throw new FinderException(e.getMessage());
		} catch (FinderException e) {
            m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
            throw e;
		} catch (CreateException e) {
            m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
            throw e;
		} catch (JMSException e) {
			m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
			throw e;
		} catch (NamingException e) {
			m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
			throw e;
		} catch (InterruptedException e) {
			m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
			return false;
		} catch (ExecutionException e) {
			m_log.error("run(): monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ", " + e.getMessage(), e);
			return false;
		}
	}
	
//	/**
//	 * 監視を実行します。
//	 * <p>
//	 * <ol>
//	 * <li>監視情報を取得し、保持します（{@link #setMonitorInfo(String, String)}）。</li>
//	 * <li>判定情報を取得し、判定情報マップに保持します（{@link #setJudgementInfo()}）。</li>
//	 * <li>チェック条件情報を取得し、保持します（{@link #setCheckInfo()}）。</li>
//	 * <li>ファシリティ毎に監視を実行し、値を収集します。 （{@link #collect(String)}）。</li>
//	 * <li>監視結果から、判定結果を取得します。 （{@link #getCheckResult(boolean)}）。</li>
//	 * <li>監視結果から、重要度を取得します（{@link #getPriority(int)}）。</li>
//	 * <li>監視結果を通知します（{@link #notify(boolean, String, int, Date)}）。</li>
//	 * </ol>
//	 * 
//	 * @return 実行に成功した場合、</code> true </code>
//	 * @throws FinderException
//	 * @throws RemoveException
//	 * @throws JMSException
//	 * @throws NamingException
//	 * @throws CreateException
//	 * 
//	 * @see #setMonitorInfo(String, String)
//	 * @see #setJudgementInfo()
//	 * @see #setCheckInfo()
//	 * @see #collect(String)
//	 * @see #getCheckResult(boolean)
//	 * @see #getPriority(int)
//	 * @see #notify(boolean, String, int, Date)
//	 */
//	@SuppressWarnings("unchecked")
//	public boolean runMonitorInfo()throws FinderException, RemoveException, JMSException, NamingException, CreateException {
//
//		m_now = new Date();
//		
//		m_priorityMap = new HashMap();
//		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_INFO),		new ArrayList<String>());
//		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_WARNING),	new ArrayList<String>());
//		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_CRITICAL),	new ArrayList<String>());
//		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_UNKNOWN),	new ArrayList<String>());
//		
//		try 
//		{
//			// 監視基本情報を設定
//			boolean run = this.setMonitorInfo(m_monitorTypeId, m_monitorId);
//			if(!run){
//				// 処理終了
//				return true;
//			}
//			
//			// 判定情報を設定
//			setJudgementInfo();
//			
//			// チェック条件情報を設定
//			setCheckInfo();
//			
//			// ファシリティIDの配下全ての一覧を取得
//			// 有効/無効フラグがtrueとなっているファシリティIDを取得する
//			ArrayList facilityList = m_repository.getNodeFacilityIdList(m_facilityId, RepositoryControllerBean.ALL);
//	
//			//配下にノードがないということはノードの可能性があるので指定されたIDをセット
//			if (facilityList.size() == 0) {
//				if(m_repository.isNode(m_facilityId)){
//					m_isNode = true;
//					facilityList.add(m_facilityId);
//				}
//				else{
//					return true;
//				}
//			}
//	
//			// ファシリティ毎に監視情報を収集
//			ArrayList<Integer> priorityList = new ArrayList<Integer>();
//			String facilityId = null;
//			for(int index=0; index<facilityList.size(); index++){
//				facilityId = (String)facilityList.get(index);
//				if(facilityId != null && !"".equals(facilityId)){
//					
//					// 監視値を収集
//					boolean ret = collect(facilityId);
//					
//					// 監視値より判定結果を取得
//					int checkResult = getCheckResult(ret);
//					
//					// 監視単位がスコープとノードの場合、スコープ用の情報を設定
//					if(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock ||
//							MonitorBlockConstant.TYPE_ALL == m_monitorBlock){
//						
//						// 判定結果よりノードの重要度を取得
//						int priority = getPriority(checkResult);
//						// スコープの重要度を設定
//						priorityList.add(Integer.valueOf(priority));
//						// 重要度別にファシリティIDを設定
//						setPriorityMap(Integer.valueOf(priority), facilityId);
//					}
//					
//					// スコープの値取得時刻を設定
//	        		if(m_nodeDate > m_scopeDate){
//	        			m_scopeDate = m_nodeDate;
//	        		}
//					
//					// ノードのログ出力情報を送信 
//					if(MonitorBlockConstant.TYPE_NODE == m_monitorBlock ||
//							(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock && m_isNode) || 
//							 MonitorBlockConstant.TYPE_ALL == m_monitorBlock){
//						
//						// 監視管理へ通知
//						notify(true, facilityId, checkResult, new Date(m_nodeDate));
//					}
//				}
//			}
//			
//			// スコープのログ出力情報を送信 
//			if(!m_isNode){
//				if(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock || 
//						MonitorBlockConstant.TYPE_ALL == m_monitorBlock){
//					
//					Integer priority = PriorityJudgment.judgment(priorityList); 
//					if(priority == null){
//						priority = Integer.valueOf(m_failurePriority);
//					}
//					
//					// 各ノードの内、最新日付を設定
//					Date generationDate = null;
//					if(m_scopeDate > 0){
//						generationDate = new Date(m_scopeDate);
//					}
//					else{
//						generationDate = m_now;
//					}
//					
//					// 監視管理へ通知
//					notify(false, m_facilityId, priority.intValue(), generationDate);					
//				}
//			}
//			return true;
//			
//		} catch (FinderException e) {
//            m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage(), e);
//            throw e;
//		} catch (CreateException e) {
//            m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage(), e);
//            throw e;
//		} catch (JMSException e) {
//			m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage(), e);
//			throw e;
//		} catch (NamingException e) {
//			m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage(), e);
//			throw e;
//		}
//	}
	
	/**
	 * 判定情報を設定します。
	 * <p>
	 * 各監視種別（真偽値，数値，文字列）のサブクラスで実装します。
	 * 監視情報より判定情報を取得し、判定情報マップに保持します。
	 */
	public abstract void setJudgementInfo();
	
	/**
	 * チェック条件情報を設定します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 監視情報よりチェック条件情報を取得し、保持します。
	 * 
	 * @throws FinderException
	 * @throws NamingException
	 */
	public abstract void setCheckInfo() throws FinderException, NamingException;
	
	/**
	 * 監視対象に対する監視を実行し、値を収集します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 引数で指定されたファシリティIDの監視を実行して値を収集し、
	 * 各監視種別（真偽値，数値，文字列）のサブクラスの監視取得値にセットします。
	 * 
	 * @param facilityId 監視対象のファシリティID
	 * @return 値取得に成功した場合、</code> true </code>
	 * @throws FacilityNotFoundException
	 */
	public abstract boolean collect(String facilityId) throws FacilityNotFoundException;
	
	/**
	 * マルチスレッドを実現するCallableTaskに渡すためのインスタンスを作成するメソッドです。
	 * 監視を実装するクラスでは、RunMonitorクラス（すべての監視実装クラスの親クラス）のインスタンスを返すために、
	 * このメソッドを実装してください。
	 * このメソッドで生成されたインスタンスは、監視実行スレッドごとの監視結果を保持するために利用されます。
	 * 
	 * すべての監視はマルチスレッドで動作しており、監視設定単位でRunMonitorクラスのインスタンスを共有しています。
	 * 監視結果（収集値）は、数値監視、文字列監視、真偽値監視のレベルで共有される変数に格納されるため、
	 * 同一監視設定で複数ノードに対して監視を実行する場合、監視結果に不整合が生じる可能性があります。
	 * 
	 * したがって、本メソッドによって新たにインスタンスを生成し、マルチスレッドを実現するCallableTaskに渡す必要があります。
	 * 
	 * @see com.clustercontrol.monitor.run.factory.RunMonitor#runMonitorInfo()
	 * @see com.clustercontrol.monitor.run.factory.RunMonitorNumericValueType
	 * @see com.clustercontrol.monitor.run.factory.RunMonitorStringValueType
	 * @see com.clustercontrol.monitor.run.factory.RunMonitorTruthValueType
	 * @see com.clustercontrol.monitor.run.util.CallableTask
	 * 
	 * @return
	 * @throws NamingException
	 * @throws JMSException
	 * @throws CreateException
	 */
	public abstract RunMonitor createMonitorInstance() throws NamingException, JMSException, CreateException ;
	
	/**
	 * 判定結果を返します。
	 * <p>
	 * 各監視種別（真偽値，数値，文字列）のサブクラスで実装します。
	 * {@link #collect(String)}メソッドで監視を実行した後に呼ばれます。
	 * 監視取得値と判定情報から、判定結果を返します。
	 * <p>
	 * <dl>
	 *  <dt>判定結果の値</dt>
	 *  <dd>真偽値監視：真偽値定数（{@link com.clustercontrol.monitor.run.bean.TruthConstant}）</dd>
	 *  <dd>数値監視：重要度定数（{@link com.clustercontrol.bean.PriorityConstant}）</dd>
	 *  <dd>文字列監視：順序（{@link com.clustercontrol.monitor.run.ejb.entity.MonitorStringValueInfoBean#getOrderNo()}）</dd>
	 * </dl>
	 * 
	 * @param ret 監視の実行が成功した場合、</code> true </code>
	 * @return 判定結果
	 */
	public abstract int getCheckResult(boolean ret);
	
	/**
	 * 重要度を返します。
	 * <p>
	 * 引数で指定された判定結果のキーに対応する重要度を、判定情報マップから取得します。
	 * 
	 * @param key 各監視種別（真偽値，数値，文字列）の判定結果のキー
	 * @return 重要度
	 * @since 2.0.0
	 */
	public int getPriority(int key) {
		
		MonitorJudgementInfo info = (MonitorJudgementInfo)m_judgementInfoList.get(key);
		if(info != null){
			return info.getPriority();
		}
		else{
			return m_failurePriority;
		}
	}
	
	/**
	 * 通知グループIDを返します。
	 * 
	 * @return 通知グループID
	 * @since 2.1.0
	 */
	public String getNotifyGroupId(){
		return m_notifyGroupId;
	}
	
	/**
	 * ノード用メッセージIDを返します。
	 * <p>
	 * 引数で指定された判定結果のキーに対応するメッセージIDを、判定情報マップから取得します。
	 * 
	 * @param key 各監視種別（真偽値，数値，文字列）の判定結果のキー
	 * @return メッセージID
	 * @since 2.0.0
	 */
	public String getMessageId(int key) {
		
		MonitorJudgementInfo info = (MonitorJudgementInfo)m_judgementInfoList.get(Integer.valueOf(key));
		if(info != null){
			if(info.getMessageId() != null){
				return info.getMessageId();
			}
		}
		return "";
	}
	
	/**
	 * ノード用メッセージを返します。
	 * <p>
	 * 引数で指定された判定結果のキーに対応するメッセージを、判定情報マップから取得します。
	 * 
	 * @param key 各監視種別（真偽値，数値，文字列）の判定結果のキー
	 * @return メッセージ
	 * @since 2.0.0
	 */
	public String getMessage(int key) {
		
		MonitorJudgementInfo info = (MonitorJudgementInfo)m_judgementInfoList.get(Integer.valueOf(key));
		if(info != null){
			if(info.getMessage() != null){
				return info.getMessage();
			}
		}
		return "";
	}
	
	/**
	 * ノード用オリジナルメッセージを返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @param key 各監視種別（真偽値，数値，文字列）の判定結果のキー
	 * @return オリジナルメッセージ
	 * @since 2.0.0
	 */
	public abstract String getMessageOrg(int key);
	
	/**
	 * スコープ用メッセージIDを返します。
	 * <p>
	 * 引数で指定された重要度に対応するメッセージIDを、判定情報マップから取得します。
	 * 
	 * @param priority 重要度
	 * @return メッセージ
	 * @since 2.1.0
	 */
	public String getMessageIdForScope(int priority){
		
		Set<Integer> set = m_judgementInfoList.keySet();
		for (Iterator<Integer> iter = set.iterator(); iter.hasNext();) {
			Integer key = iter.next();
			MonitorJudgementInfo info = (MonitorJudgementInfo) m_judgementInfoList.get(key);
			if(priority == info.getPriority()){
				if(info.getMessageId() != null){
					return info.getMessageId();
				}
			}
		}
		return "";
	}
	
	/**
	 * スコープ用メッセージを返します。
	 * <p>
	 * 引数で指定された重要度に対応するメッセージを返します。
	 * 
	 * @param priority 重要度
	 * @return メッセージ
	 */
	public String getMessageForScope(int priority){
		
		if(priority == PriorityConstant.TYPE_INFO){
			return MESSAGE_INFO;
		}
		else if(priority == PriorityConstant.TYPE_WARNING){
			return MESSAGE_WARNING;
		}
		else if(priority == PriorityConstant.TYPE_CRITICAL){
			return MESSAGE_CRITICAL;
		}
		else{
			return MESSAGE_UNKNOWN;
		}
	}
	
	/**
	 * スコープ用オリジナルメッセージを返します。
	 * <p>
	 * 重要度別の件数，ファシリティ名を表示する文字列を作成し、返します。
	 * 
	 * <dl>
	 *  <dt>通知:X件, 警告:X件, 危険:X件, 不明:X件</dt>
	 *  <dt>通知:</dt>
	 * 	 <dd>NODE1</dd>
	 *   <dd>NODE2</dd>
	 *  <dt>警告:</dt>
	 *   <dd>NODE3</dd>
	 *  <dt>危険:</dt>
	 * 	 <dd>NODE4</dd>
	 *  <dt>不明:</dt>
	 *   <dd>NODE5</dd>
	 * </dl>
	 * 
	 * @param priority 重要度
	 * @return オリジナルメッセージ
	 */
	public String getMessageOrgForScope(int priority){
		
		ArrayList<String> info = m_priorityMap.get(Integer.valueOf(PriorityConstant.TYPE_INFO));
		ArrayList<String> warning = m_priorityMap.get(Integer.valueOf(PriorityConstant.TYPE_WARNING));
		ArrayList<String> critical = m_priorityMap.get(Integer.valueOf(PriorityConstant.TYPE_CRITICAL));
		ArrayList<String> unknown = m_priorityMap.get(Integer.valueOf(PriorityConstant.TYPE_UNKNOWN));
		
		// 重要度別の件数
		StringBuffer summary = new StringBuffer();
		summary.append(Messages.getString("info") + ":" + info.size() + Messages.getString("record") + ", ");
		summary.append(Messages.getString("warning") + ":" + warning.size() + Messages.getString("record") + ", ");
		summary.append(Messages.getString("critical") + ":" + critical.size() + Messages.getString("record") + ", ");
		summary.append(Messages.getString("unknown") + ":" + unknown.size() + Messages.getString("record"));
		
		// 重要度別のファシリティ名 
		StringBuffer detail = new StringBuffer();
		detail.append(getItemListString("\n" + Messages.getString("info"), info));
		detail.append(getItemListString("\n" + Messages.getString("warning"), warning));
		detail.append(getItemListString("\n" + Messages.getString("critical"), critical));
		detail.append(getItemListString("\n" + Messages.getString("unknown"), unknown));
		
		return summary.toString() + detail.toString();
	}
	
	/**
	 * ノードの監視結果取得時刻を返します。
	 * 
	 * @return ノードの監視結果取得時刻
	 * @since 3.0.0
	 */
	public long getNodeDate() {
		return m_nodeDate;
	}
	
//	/**
//	 * ログ出力ジョブ実行情報を返します。
//	 * <p>
//	 * 引数で指定された判定結果のキーに対応するログ出力ジョブ実行情報を、判定情報マップから取得します。
//	 * ジョブ実行をしない場合は、<code>null</code>を返します。
//	 * 
//	 * @param key 各監視種別（真偽値，数値，文字列）の判定結果のキー
//	 * @return ログ出力ジョブ実行情報
//	 * @since 2.1.0
//	 */
//	public LogOutputJobRunInfo getJobRunInfo(int key) {
//		
//		// ジョブ実行をしない場合
//		if(m_jobRun == YesNoConstant.TYPE_NO){
//			return null;
//		}
//		
//		LogOutputJobRunInfo jobRunInfo = null;
//		if(key != PriorityConstant.TYPE_FAILURE){
//			MonitorJudgementInfo info = (MonitorJudgementInfo)m_judgementInfoList.get(Integer.valueOf(key));
//		/*	if(info != null){
//				jobRunInfo = new LogOutputJobRunInfo();
//				jobRunInfo.setJobRun(info.getJobRun());
//				jobRunInfo.setJobId(info.getJobId());
//				jobRunInfo.setJobInhibitionFlg(info.getJobInhibitionFlg());
//				jobRunInfo.setJobFailurePriority(info.getJobFailurePriority());
//			}*/
//		}
//
//		return jobRunInfo;
//	}
	

//	/**
//	 * スコープ用ログ出力ジョブ実行情報を返します。
//	 * <p>
//	 * 引数で指定された重要度に対応するログ出力ジョブ実行情報を、判定情報マップから取得します。
//	 * 
//	 * @param priority 重要度
//	 * @return ログ出力ジョブ実行情報
//	 * @since 2.1.0
//	 */
//	public LogOutputJobRunInfo getJobRunInfoForScope(int priority) {
//
//		// ジョブ実行をしない場合
//		if(m_jobRun == YesNoConstant.TYPE_NO){
//			return null;
//		}
//		
//		LogOutputJobRunInfo jobRunInfo = null;
//		if(priority != PriorityConstant.TYPE_FAILURE){
//			Set set = m_judgementInfoList.keySet();
//			for (Iterator iter = set.iterator(); iter.hasNext();) {
//				Integer key = (Integer)iter.next();
//				MonitorJudgementInfo info = (MonitorJudgementInfo)m_judgementInfoList.get(key);
//			}
//		}
//			
//		return jobRunInfo;
//	}
	
	/**
	 * 監視情報を設定します。
	 * <p>
	 * 引数で指定された監視対象と監視項目の監視情報を取得し、保持します。<BR>
	 * 通知IDが指定されていない場合は、処理を終了します。<BR>
	 * カレンダIDが指定されていた場合は、稼動日か否かチェックします（{@link com.clustercontrol.calendar.ejb.session.CalendarControllerBean#isRun(java.lang.String, java.util.Date)}）。非稼動日の場合は、処理を終了します。
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @param monitorId 監視項目ID
	 * @return 監視を実行する場合、</code> true </code>
	 * @throws NamingException
	 * @throws CreateException
	 * @throws FinderException 
	 * @since 2.0.0
	 * 
	 * @see com.clustercontrol.calendar.ejb.session.CalendarControllerBean#isRun(java.lang.String, java.util.Date)
	 */
	protected boolean setMonitorInfo(String monitorTypeId, String monitorId) throws NamingException, CreateException, FinderException{

		// 監視基本情報を取得
		MonitorInfoPK pk = new MonitorInfoPK(monitorId, monitorTypeId);
		m_monitor = MonitorInfoUtil.getLocalHome().findByPrimaryKey(pk);
		
		// 通知ID
		m_notifyGroupId = m_monitor.getNotifyGroupId();
		if(m_notifyGroupId == null || "".equals(m_notifyGroupId)){
			if(m_monitor.getMonitorType().intValue() != MonitorTypeConstant.TYPE_STRING){
				// 通知しない場合は、処理終了
				return false;
			}
		}
		
		// カレンダID
		String calendarId = m_monitor.getCalendarId();
		if(calendarId != null && !"".equals(calendarId)){
			// 稼働日か否かチェック
			CalendarControllerLocal calendar = CalendarControllerUtil.getLocalHome().create();
			
			try {
				if(!calendar.isRun(calendarId, m_now).booleanValue()){
					// 非稼働日の場合は、処理終了
					m_log.debug("setMonitorInfo() : monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + ",カレンダID：" + calendarId + "。非稼働日の為、監視は実行されません。");
					return false;
				}
			} catch (CalendarNotFountException e) {
				// 指定されたカレンダIDがすでに存在しない場合は、処理終了
				m_log.info("setMonitorInfo() : monitorTypeId:" + m_monitorTypeId + ",monitorId：" + m_monitorId  + "。カレンダID「" + calendarId + "」は存在しません。");
				return false;
			}
		}
		
		// 監視対象ファシリティID
		m_facilityId = m_monitor.getFacilityId();
		// 監視単位
		m_monitorBlock = m_monitor.getMonitorBlock().intValue();
		// タイムアウト
		m_timeout = m_monitor.getTimeout().intValue();
		// 値取得失敗時の重要度
		m_failurePriority = m_monitor.getFailurePriority().intValue();
		// 収集間隔
		m_runInterval = m_monitor.getRunInterval().intValue();
		// ジョブ実行
		//m_jobRun = m_monitor.getJobRun().intValue();
		
		return true;
	}
	
	/**
	 * 監視管理に通知します。
	 * <p>
	 * 通知に必要となる情報をログ出力情報にセットし、キューへメッセージを送信します。
	 * <ol>
	 * <li>通知IDを取得し、<code>null</code>の場合は、処理を終了します。</li>
	 * <li>各情報を取得し、ログ出力情報（{@link com.clustercontrol.monitor.message.LogOutputNotifyInfo }）にセットします。
	 * 監視単位がノードの場合とスコープの場合では、通知する情報が異なります。
	 * </li>
	 * <li>ログ出力情報を、ログ出力キューへ送信します。</li>
	 * </ol>
	 * 
	 * @param isNode ノードの場合、</code> true </code>
	 * @param facilityId 通知対象のファシリティID
	 * @param result 判定結果
	 * @param generationDate ログ出力日時（監視を実行した日時）
	 * @throws FinderException
	 * @throws JMSException
	 * @throws NamingException
	 * @since 2.0.0
	 * 
	 * @see com.clustercontrol.monitor.message.LogOutputNotifyInfo 
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityPath(java.lang.String, java.lang.String)
	 * @see com.clustercontrol.monitor.run.util.SendQueue#put(Serializable)
	 * @see #getNotifyId()
	 * @see #getPriority(int)
	 * @see #getMessageId(int)
	 * @see #getMessage(int)
	 * @see #getMessageOrg(int)
	 * @see #getJobRunInfo(int)
	 * @see #getMessageForScope(int)
	 * @see #getMessageIdForScope(int)
	 * @see #getMessageOrgForScope(int)
	 * @see #getJobRunInfoForScope(int)
	 */
	protected void notify(boolean isNode, 
								String facilityId, 
								int result, 
								Date generationDate) throws FinderException, JMSException, NamingException{
		
		// for debug
		if (m_log.isDebugEnabled()) {
			m_log.debug("notify() isNode = " + isNode + ", facilityId = " + facilityId 
					+ ", result = " + result + ", generationDate = " + generationDate.toString());
		}
		// 通知IDが指定されていない場合、通知しない
		String notifyGroupId = getNotifyGroupId();
		if(notifyGroupId == null || "".equals(notifyGroupId)){
			return;
		}
		
		// 通知情報を設定 
		if(m_notifyGroupInfo == null){
			m_notifyGroupInfo = new OutputNotifyGroupInfo();
			m_notifyGroupInfo.setPluginId(m_monitorTypeId);
			m_notifyGroupInfo.setMonitorId(m_monitorId);
			m_notifyGroupInfo.setApplication(m_monitor.getApplication());
		}
		
		m_notifyGroupInfo.setNotifyGroupId(notifyGroupId);
		
		String facilityPath = m_repository.getFacilityPath(facilityId, null);
		m_notifyGroupInfo.setFacilityId(facilityId);
		m_notifyGroupInfo.setScopeText(facilityPath);
		
		
		int priority = -1;
		String messageId = "";
		String message = "";
		String messageOrg = "";
//		LogOutputJobRunInfo jobRunInfo = null;
		
		if(isNode){
			// ノードの場合
			priority = getPriority(result);
			messageId = getMessageId(result);
			message = getMessage(result);
			messageOrg = getMessageOrg(result);
//			jobRunInfo = getJobRunInfo(result);
		}
		else{
			// スコープの場合
			priority = result;
			messageId = getMessageIdForScope(result);
			message = getMessageForScope(result);
			messageOrg = getMessageOrgForScope(result);
//			jobRunInfo = getJobRunInfoForScope(result);
		}
		m_notifyGroupInfo.setPriority(priority);
		m_notifyGroupInfo.setMessageId(messageId);
		m_notifyGroupInfo.setMessage(message);
		m_notifyGroupInfo.setMessageOrg(messageOrg);
//		m_notifyInfo.setJobRun(jobRunInfo);
		
		m_notifyGroupInfo.setGenerationDate(generationDate);
		
		// for debug
		if (m_log.isDebugEnabled()) {
			m_log.debug("notify() priority = " + priority
					+ " , messageId = " + messageId
					+ " , message = " + message
					+ " , messageOrg = " + messageOrg
					+ ", generationDate = " + generationDate);
		}
		
		// ログ出力情報を送信
		if (m_log.isDebugEnabled()) {
			m_log.debug("sending message to jms. (" + getJmsPersistence(m_notifyGroupInfo.getPluginId())
					+ " " + JmsPersistenceConfig.getPersistenceString(getJmsPersistence(m_notifyGroupInfo.getPluginId()))
					+ " : notifyGroupId=" + m_notifyGroupInfo.getNotifyGroupId() + " priority=" + m_notifyGroupInfo.getPriority()
					+ " generationDate=" + m_notifyGroupInfo.getGenerationDate() + " pluginId=" + m_notifyGroupInfo.getPluginId()
					+ " monitorId=" + m_notifyGroupInfo.getMonitorId() + " facilityId=" + m_notifyGroupInfo.getFacilityId() + ")");
		}
		m_queue.put(m_notifyGroupInfo, getJmsPersistence(m_notifyGroupInfo.getPluginId()));
	}

	/**
	 * 監視管理に通知します。(並列処理時)
	 * <p>
	 * 通知に必要となる情報をログ出力情報にセットし、キューへメッセージを送信します。
	 * <ol>
	 * <li>通知IDを取得し、<code>null</code>の場合は、処理を終了します。</li>
	 * <li>各情報を取得し、ログ出力情報（{@link com.clustercontrol.monitor.message.LogOutputNotifyInfo }）にセットします。
	 * 監視単位がノードの場合とスコープの場合では、通知する情報が異なります。
	 * </li>
	 * <li>ログ出力情報を、ログ出力キューへ送信します。</li>
	 * </ol>
	 * 
	 * @param isNode ノードの場合、</code> true </code>
	 * @param facilityId 通知対象のファシリティID
	 * @param result 判定結果
	 * @param generationDate ログ出力日時（監視を実行した日時）
	 * @param resultList 並列実行時の情報
	 * @throws FinderException
	 * @throws JMSException
	 * @throws NamingException
	 * @since 2.0.0
	 * 
	 * @see com.clustercontrol.monitor.message.LogOutputNotifyInfo 
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getFacilityPath(java.lang.String, java.lang.String)
	 * @see com.clustercontrol.monitor.run.util.SendQueue#put(Serializable)
	 * @see #getNotifyId()
	 * @see #getPriority(int)
	 * @see #getMessageId(int)
	 * @see #getMessage(int)
	 * @see #getMessageOrg(int)
	 * @see #getJobRunInfo(int)
	 * @see #getMessageForScope(int)
	 * @see #getMessageIdForScope(int)
	 * @see #getMessageOrgForScope(int)
	 * @see #getJobRunInfoForScope(int)
	 */
	protected void notify(boolean isNode, 
								String facilityId, 
								int result, 
								Date generationDate,
								MonitorRunResultInfo resultInfo) throws FinderException, JMSException, NamingException{
		
		// for debug
		if (m_log.isDebugEnabled()) {
			m_log.debug("notify() isNode = " + isNode + ", facilityId = " + facilityId 
					+ ", result = " + result + ", generationDate = " + generationDate.toString() 
					+ ", resultInfo = " + resultInfo.getMessage());
		}

		// 通知IDが指定されていない場合、通知しない
//		String notifyGroupId = getNotifyGroupId();
		String notifyGroupId = resultInfo.getNotifyGroupId();
		if(notifyGroupId == null || "".equals(notifyGroupId)){
			return;
		}
		
		// 通知情報を設定 
		if(m_notifyGroupInfo == null){
			m_notifyGroupInfo = new OutputNotifyGroupInfo();
			m_notifyGroupInfo.setPluginId(m_monitorTypeId);
			m_notifyGroupInfo.setMonitorId(m_monitorId);
			m_notifyGroupInfo.setApplication(m_monitor.getApplication());
		}
		
		m_notifyGroupInfo.setNotifyGroupId(notifyGroupId);
		
		String facilityPath = m_repository.getFacilityPath(facilityId, null);
		m_notifyGroupInfo.setFacilityId(facilityId);
		m_notifyGroupInfo.setScopeText(facilityPath);
		
		
		int priority = -1;
		String messageId = "";
		String message = "";
		String messageOrg = "";
//		LogOutputJobRunInfo jobRunInfo = null;
		
		if(isNode){
			// ノードの場合
			priority = resultInfo.getPriority();
			messageId = resultInfo.getMessageId();
			message = resultInfo.getMessage();
			messageOrg = resultInfo.getMessageOrg();
//			jobRunInfo = resultList.getJobRunInfo();
		}
		else{
			// スコープの場合
			priority = result;
			messageId = getMessageIdForScope(result);
			message = getMessageForScope(result);
			messageOrg = getMessageOrgForScope(result);
//			jobRunInfo = getJobRunInfoForScope(result);
		}
		m_notifyGroupInfo.setPriority(priority);
		m_notifyGroupInfo.setMessageId(messageId);
		m_notifyGroupInfo.setMessage(message);
		m_notifyGroupInfo.setMessageOrg(messageOrg);
//		m_notifyInfo.setJobRun(jobRunInfo);
		
		m_notifyGroupInfo.setGenerationDate(generationDate);
		
		// for debug
		if (m_log.isDebugEnabled()) {
			m_log.debug("notify() priority = " + priority
					+ " , messageId = " + messageId
					+ " , message = " + message
					+ " , messageOrg = " + messageOrg
					+ ", generationDate = " + generationDate);
		}
		
		// ログ出力情報を送信
		if (m_log.isDebugEnabled()) {
			m_log.debug("sending message to jms. (" + m_notifyGroupInfo.getPluginId()
					+ " " + JmsPersistenceConfig.getPersistenceString(getJmsPersistence(m_notifyGroupInfo.getPluginId()))
					+ " : notifyGroupId=" + m_notifyGroupInfo.getNotifyGroupId() + " priority=" + m_notifyGroupInfo.getPriority()
					+ " generationDate=" + m_notifyGroupInfo.getGenerationDate() + " pluginId=" + m_notifyGroupInfo.getPluginId()
					+ " monitorId=" + m_notifyGroupInfo.getMonitorId() + " facilityId=" + m_notifyGroupInfo.getFacilityId() + ")");
		}
		m_queue.put(m_notifyGroupInfo, getJmsPersistence(m_notifyGroupInfo.getPluginId()));
	}
	
	/**
	 * 重要度別ファシリティ名を設定します。
	 * <p>
	 * <ol>
	 * <li>引数で指定されたファシリティIDのファシリティ名を取得します。</li>
	 * <li>重要度別のファシリティマップの引数で指定された重要度に、ファシリティ名を追加します。</li>
	 * 
	 * @param priority 重要度
	 * @param facilityId ファシリティ名
	 * @throws NamingException
	 * @throws FinderException
	 */
	@SuppressWarnings("unchecked")
	protected void setPriorityMap(Integer priority, String facilityId) throws NamingException, FinderException{
		
		ArrayList list = (ArrayList)m_priorityMap.get(priority);
		if(list != null){
			
			// ファシリティ名を取得
			String facilityName = m_repository.getFacilityPath(facilityId, null);
			
			list.add(facilityName);
			m_priorityMap.put(priority, list);
		}
	}
	
	/**
	 * 項目一覧の文字列を返します。
	 * 
	 * <dl>
	 *  <dt>項目名:</dt>
	 * 	 <dd>リスト[0]</dd>
	 *   <dd>リスト[1]</dd>
	 *   <dd>リスト[2]</dd>
	 *   <dd>　 ：　</dd>
	 * </dl>
	 * 
	 * @param item 項目名
	 * @param list リスト
	 * @return 項目一覧の文字列
	 */
	protected String getItemListString(String item, ArrayList<String> list){
		
		int length = list.size(); 
		if(length > 0){
			StringBuffer result = new StringBuffer();
			result.append(item + ":" + "\n");
			for (int i = 0; i < length; i++) {
				result.append("\t" + list.get(i));
				if(i < length-1){
					result.append("\n");
				}
			}
			return result.toString();
		}
		else{
			return "";
		}
	}
	
	/**
	 * 監視実行の初期処理を行います。
	 * 
	 * 監視を実行するrunメソッドの最初の部分でcallしてください。
	 * runメソッド終了部分で、terminatteメソッドをcallし、キューのコネクションをクローズする必要があります。
	 * 
	 * @throws NamingException
	 * @throws JMSException
	 * @throws CreateException
	 */
//	private void initialize() throws NamingException, JMSException, CreateException{
	protected void initialize(String monitorTypeId) throws NamingException, JMSException, CreateException{
		
		String queueName = "";
		int persistMode = DeliveryMode.PERSISTENT;
		
		persistMode = getJmsPersistence(monitorTypeId);
		
		// ログ出力
		if (persistMode == DeliveryMode.NON_PERSISTENT) {
			queueName = QueueConstant.QUEUE_NAME_NOTIFYCONTROL_NON_PERSISTENCE;
		} else {
			queueName = QueueConstant.QUEUE_NAME_NOTIFYCONTROL;
		}
		if (m_log.isDebugEnabled()) {
			m_log.debug("initializing jms queue. (" + queueName + " for " + monitorTypeId + ")");
		}
		m_queue = new SendQueue(queueName);

		try{
			// リポジトリ
			m_repository = RepositoryControllerUtil.getLocalHome().create();
			
		} catch(CreateException e){
			m_log.error("initialize():" + e.getMessage(), e);
			throw e;
		} catch(NamingException e){
			m_log.error("initialize():" + e.getMessage(), e);
			throw e;
		}
	}
	
	/**
	 * 該当の監視タイプの結果を永続化させるかどうか
	 */
	private static int getJmsPersistence(String monitorTypeId) {
		int jmsDeliveryMode = DeliveryMode.PERSISTENT;
		
		if ( HinemosModuleConstant.MONITOR_AGENT.equals(monitorTypeId) ) {
			jmsDeliveryMode = JmsPersistenceConfig.QUEUE_AGENT_MODE;
		} else if ( HinemosModuleConstant.MONITOR_HTTP.equals(monitorTypeId) ) {
			jmsDeliveryMode = JmsPersistenceConfig.QUEUE_HTTP_MODE;
		} else if ( HinemosModuleConstant.MONITOR_PERFORMANCE.equals(monitorTypeId) ) {
			jmsDeliveryMode = JmsPersistenceConfig.QUEUE_PERFORMANCE_MODE;
		} else if ( HinemosModuleConstant.MONITOR_PING.equals(monitorTypeId) ) {
			jmsDeliveryMode = JmsPersistenceConfig.QUEUE_PING_MODE;
		} else if ( HinemosModuleConstant.MONITOR_PORT.equals(monitorTypeId) ) {
			jmsDeliveryMode = JmsPersistenceConfig.QUEUE_PORT_MODE;
		} else if ( HinemosModuleConstant.MONITOR_PROCESS.equals(monitorTypeId) ) {
			jmsDeliveryMode = JmsPersistenceConfig.QUEUE_PROCESS_MODE;
		} else if ( HinemosModuleConstant.MONITOR_SNMP.equals(monitorTypeId) ) {
			jmsDeliveryMode = JmsPersistenceConfig.QUEUE_SNMP_MODE;
		} else if ( HinemosModuleConstant.MONITOR_SQL.equals(monitorTypeId) ) {
			jmsDeliveryMode = JmsPersistenceConfig.QUEUE_SQL_MODE;
		}
		
		return jmsDeliveryMode;
	}
	
	
	/**
	 * 監視実行の終了処理を行います。
	 * 
	 * @throws JMSException
	 * @throws RemoveException
	 */
	protected void terminate() throws JMSException, RemoveException{
		
		try{
			if(m_queue != null){
				m_queue.terminate();
			}
			m_queue = null;
			
			if(m_repository != null){
				m_repository.remove();
			}
			m_repository = null;
			
		} catch (JMSException e) {
			m_log.error("terminate():" + e.getMessage(), e);
			throw e;
		} catch (RemoveException e) {
			m_log.error("terminate():" + e.getMessage(), e);
			throw e;
		}
	}
}
