/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.ejb.session;

import java.rmi.RemoteException;
import java.text.ParseException;
import java.util.ArrayList;

import javax.ejb.EJBException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.SchedulerException;
import org.quartz.jobs.ee.ejb.EJBInvokerJob;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.JobInfoNotFoundException;
import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.jobmanagement.bean.NodeOperationInfo;
import com.clustercontrol.jobmanagement.bean.OperationConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodePK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeUtil;
import com.clustercontrol.jobmanagement.factory.JobOperationJudgment;
import com.clustercontrol.jobmanagement.factory.OperateForceStopOfJob;
import com.clustercontrol.jobmanagement.factory.OperateMaintenanceOfJob;
import com.clustercontrol.jobmanagement.factory.OperateSkipOfJob;
import com.clustercontrol.jobmanagement.factory.OperateStartOfJob;
import com.clustercontrol.jobmanagement.factory.OperateStopOfJob;
import com.clustercontrol.jobmanagement.factory.OperateSuspendOfJob;
import com.clustercontrol.jobmanagement.factory.OperateWaitOfJob;
import com.clustercontrol.jobmanagement.factory.OperationJob;
import com.clustercontrol.jobmanagement.message.RunResultInfo;
import com.clustercontrol.quartzmanager.ejb.session.QuartzManager;
import com.clustercontrol.quartzmanager.util.QuartzUtil;
import com.clustercontrol.util.apllog.AplLogger;


/**
 * ジョブ管理機能の実行管理を行う Session Bean クラス<BR>
 * 
 * @ejb.bean name="JobRunManagement"	
 *           jndi-name="JobRunManagement"
 *           type="Stateless" 
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobCommandInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobCommandMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobEndInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobEndMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobFileInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobFileMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobNoticeInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobNoticeMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobParamInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobParamMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobRelationInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobRelationMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobSessionLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobSessionJobLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobSessionNodeLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartJobInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartJobMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartTimeInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartTimeMasterLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 */
public abstract class JobRunManagementBean implements javax.ejb.SessionBean {
	/** ログ出力のインスタンス<BR> */
	protected static Log m_log = LogFactory.getLog( JobRunManagementBean.class );
    
	/** Quartzに設定するグループ名<BR> */
	public static final String GROUP_NAME = "JOB_MANAGEMENT";
    
    /**
     * Quartzへジョブ管理の定周期処理を登録します。<BR>
	 * Quartzからは、{@link com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#run()} が呼び出されます。<BR>
	 * 
     * @ejb.interface-method
     * 
     * @param cronString cronの定義文字列
     * @throws ParseException
     * @throws SchedulerException
     * @throws HinemosUnknownException 
     * 
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#addSchedule(org.quartz.JobDetail, org.quartz.Trigger)
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#deleteSchedule(java.lang.String, java.lang.String)
     * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementHome#JNDI_NAME
     * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#GROUP_NAME
     */
    public void addQuartz(String cronString) throws ParseException, SchedulerException, HinemosUnknownException {

        try {
            //QuartzのSchedulerをルックアップ
    		QuartzManager manager = QuartzUtil.getQuartzManager();
            //ジョブ削除
        	manager.deleteSchedule(JobRunManagementHome.JNDI_NAME, 
                    GROUP_NAME);
        
	        //JobDetail作成
	        JobDetail job = new JobDetail(
	        		JobRunManagementHome.JNDI_NAME, 
	                GROUP_NAME, 
	                EJBInvokerJob.class);
	        
	        String serverName = System.getProperty("jboss.server.name");
		    if(serverName.equals("all")){
		        job.getJobDataMap().put(
		                EJBInvokerJob.PROVIDER_URL, 
		                "jnp://localhost:1100");
		    }
		    
	        //ジョブ完了時に削除されないようにする。
	        job.setDurability(true);
		    //ジョブ実行失敗時に再実行するようにする。
		    //JBoss再起動時に複数同時に再実行される可能性があるためこのオプションは設定しないよう変更する
	//	    job.setRequestsRecovery(true);
		    
	        //JobDetailに呼び出すクラスとメソッドを設定
	        job.getJobDataMap().put(
	                EJBInvokerJob.EJB_JNDI_NAME_KEY, 
	                JobRunManagementHome.JNDI_NAME);
	        job.getJobDataMap().put(
	                EJBInvokerJob.EJB_METHOD_KEY, 
	                "run");
	        
	        //JobDetailに呼び出すメソッドの引数を設定
	        Object[] jdArgs = new Object[0];
	        job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
	        
	        //CronTriggerを作成
	        CronTrigger cronTrigger = new CronTrigger(JobRunManagementHome.JNDI_NAME, GROUP_NAME);
	        
	        //起動失敗した場合は、次の起動予定時刻をセットするように設定
	        cronTrigger.setMisfireInstruction(CronTrigger.MISFIRE_INSTRUCTION_DO_NOTHING);        
	        
	        //スケジュールを設定
	        cronTrigger.setCronExpression(cronString);
	        
	        //Schedulerにジョブを登録する
			manager.addSchedule(job, cronTrigger);
			
		} catch (RemoteException e) {
			m_log.error("addQuartz() : " + e.getMessage(), e ); 
			
		} catch (NamingException e) {
			m_log.error("addQuartz() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
    }
    
    /**
     * Quartzからジョブ管理の定周期処理を削除します。<BR>
     * 
     * @ejb.interface-method
     * 
     * @throws ParseException
     * @throws SchedulerException
     * @throws HinemosUnknownException 
     * 
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#deleteSchedule(java.lang.String, java.lang.String)
     * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementHome#JNDI_NAME
     * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#GROUP_NAME
     */
    public void deleteQuartz() throws ParseException, SchedulerException, HinemosUnknownException {
		
        try {
            //QuartzのSchedulerをルックアップ
        	QuartzManager manager = QuartzUtil.getQuartzManager();
            //ジョブ削除
        	manager.deleteSchedule(JobRunManagementHome.JNDI_NAME, 
                    GROUP_NAME);
        } catch (SchedulerException e) {
            m_log.debug("deleteQuartz() : deleteJob error = " + e.getMessage());
        } catch (RemoteException e) {
			//TODO RemoteException未実装
		} catch (NamingException e) {
			m_log.error(" : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
    }
    
    /**
     * Quartzからのコールバックメソッド<BR>
     * <P>
     * Quartzから定周期で呼び出されます。<BR>
     * <BR>
     * 実行状態が実行中のセッションをチェックし、実行可能なジョブを開始する。<BR>
     * 実行状態が待機のセッションをチェックし、ジョブを開始する。<BR>
     * @throws HinemosUnknownException 
     * @throws JobInfoNotFoundException 
     * 
     * @ejb.interface-method
     * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 *     
     * @see com.clustercontrol.jobmanagement.factory.OperationJob#runJob()
     * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#runWaitJob()
     */
    public void run() throws HinemosUnknownException, JobInfoNotFoundException {
    	m_log.debug("run() start");

    	OperationJob operation = new OperationJob();
    	ArrayList<String> list = null;
    	
    	try {
    		list = operation.getUnendSessionList();
    		
        	for (int i = 0; i < list.size(); i++) {
        		runSession(list.get(i));
    		}
        	
		} catch (NamingException e) {
			m_log.error("run() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
	}
    
    /**
     * 指定されたセッションIDの実行中、待機セッションに対して、実行可能なジョブを開始する
     * 
     * @ejb.transaction
	 *     type="RequiresNew"
	 *     
     * @param sessionId セッションID
     * @throws HinemosUnknownException 
     * @throws JobInfoNotFoundException 
     */
    public void runSession(String sessionId) throws HinemosUnknownException, JobInfoNotFoundException {
    	m_log.debug("runSession() start : sessionid = " + sessionId);

    	OperationJob operation = new OperationJob();
    	try {
        	operation.runJob(sessionId);
		} catch (NamingException e) {
			m_log.error("runSession() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
    }
    
    
    /**
     * 実行状態が停止処理中のセッションのジョブをチェックします。<BR>
     * <P>
     * 実行状態の停止処理中のセッションをチェックし、ジョブを停止する。<BR>
     * @throws HinemosUnknownException 
     * @throws JobInfoNotFoundException 
     * 
     * @ejb.interface-method
     * 
     * @ejb.transaction
	 *     type="RequiresNew"
     * 
     * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkStoppingJob()
     */
    public void checkStoppingJob() throws HinemosUnknownException, JobInfoNotFoundException {
        m_log.debug("checkStoppingJob()");
        
        try {
			OperationJob operation = new OperationJob();
			operation.checkStoppingJob();
		} catch (NamingException e) {
			m_log.error("checkStoppingJob() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
    }
    
    /**
     * ジョブを実行します。<BR>
     * 
     * @ejb.interface-method
     * 
     * @ejb.transaction
	 *     type="RequiresNew"
     * 
     * @param sessionId セッションID
     * @param jobunitId 所属ジョブユニットのジョブID
     * @param jobId ジョブID
     * @throws HinemosUnknownException 
     * @throws JobInfoNotFoundException 
     * @see com.clustercontrol.jobmanagement.factory.OperationJob#runJob(String, String, String)
     */
    public void runJob(String sessionId, String jobunitId, String jobId) throws HinemosUnknownException, JobInfoNotFoundException {
        m_log.debug("runJob() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId);
        
        try {
			OperationJob operation = new OperationJob();
			operation.runJob(sessionId, jobunitId, jobId);
		} catch (NamingException e) {
			m_log.error("runJob() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
    }

	/**
	 * ジョブの操作を行います。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param property ジョブ操作用プロパティ
	 * @throws HinemosUnknownException 
	 * @throws JobInfoNotFoundException 
	 * @see com.clustercontrol.jobmanagement.factory.OperateStartOfJob#startJob(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateSuspendOfJob#suspendJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateSuspendOfJob#releaseSuspendJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateWaitOfJob#waitJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateWaitOfJob#releaseWaitJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateSkipOfJob#skipJob(String, String, Integer)
	 * @see com.clustercontrol.jobmanagement.factory.OperateSkipOfJob#releaseSkipJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateStopOfJob#stopJob(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateMaintenanceOfJob#maintenanceJob(String, String, String, Integer)
	 * @see com.clustercontrol.jobmanagement.factory.OperateForceStopOfJob#forceStopJob(String, String, String, Integer)
	 */
	public void operationJob(NodeOperationInfo property) throws HinemosUnknownException, JobInfoNotFoundException {
	    m_log.debug("operationJob()");
	    
		String sessionId = property.getSessionId();
		String jobunitId = property.getJobunitId();
		String jobId = property.getJobId();
		String facilityId = property.getFacilityId();
		Integer control = property.getControl();
		Integer endValue = property.getEndValue();
		
	    int status = 0;
	    int jobType = 0;
	    
	    try {
            if(facilityId != null && facilityId.length() > 0){
                //セッションIDとジョブIDから、セッションジョブを取得
                JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
                            new JobSessionJobPK(sessionId, jobunitId, jobId));
                
                //ジョブタイプを取得
                if(sessionJob.getJobInfo().getJob_type().intValue() == JobConstant.TYPE_FILEJOB){
                	jobId = jobId + "_" + facilityId;
                	facilityId = null;
                	
                	//セッションIDとジョブIDから、セッションジョブを取得
                    JobSessionJobLocal childSessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
                            new JobSessionJobPK(sessionId, jobunitId, jobId));
	                
	                //実行状態を取得
	                status = childSessionJob.getStatus().intValue();
	                
	                //ジョブタイプを取得
	                jobType = JobOperationJudgment.TYPE_JOBNET;
                }
                else{
	                //セッションIDとジョブIDから、セッションジョブを取得
	                JobSessionNodeLocal sessionNode = 
	                    JobSessionNodeUtil.getLocalHome().findByPrimaryKey(
	                            new JobSessionNodePK(sessionId, jobunitId, jobId, facilityId));
	                
	                //実行状態を取得
	                status = sessionNode.getStatus().intValue();
	                
	                //ジョブタイプを取得
	                jobType = JobOperationJudgment.TYPE_NODE;
                }
            }
            else{
                //セッションIDとジョブIDから、セッションジョブを取得
                JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
                            new JobSessionJobPK(sessionId, jobunitId, jobId));
                
                //実行状態を取得
                status = sessionJob.getStatus().intValue();
                
                //ジョブタイプを取得
                if(sessionJob.getJobInfo().getJob_type().intValue() == JobConstant.TYPE_JOB){
                    jobType = JobOperationJudgment.TYPE_JOB;
                }
                else{
                    jobType = JobOperationJudgment.TYPE_JOBNET;
                }
            }
        } catch (Exception e) {
        	//TODO 
        }
        
        //ジョブタイプとジョブの実行状態から操作可能かチェック
        if(JobOperationJudgment.judgment(control.intValue(), jobType, status)){
            if(control.intValue() == OperationConstant.TYPE_START_AT_ONCE){
                //開始[即時]
        	    try {
                    new OperateStartOfJob().startJob(sessionId, jobunitId, jobId, facilityId);
        	    } catch (JobInfoNotFoundException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "007", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "007", args);
                    m_log.error("operationJob() : " + e.getMessage(), e ); 
                    throw new HinemosUnknownException(e.getMessage(), e);
                } catch (EJBException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "007", args);
                    m_log.error("operationJob() : " + e.getMessage(), e ); 
                    throw new HinemosUnknownException(e.getMessage(), e);
				} catch (RemoveException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "007", args);
                    m_log.error("operationJob() : " + e.getMessage(), e ); 
                    throw new HinemosUnknownException(e.getMessage(), e);
				}
            }
            else if(control.intValue() == OperationConstant.TYPE_START_SUSPEND){
                //開始[中断解除]
                try {
                    new OperateSuspendOfJob().releaseSuspendJob(sessionId, jobunitId, jobId);
                } catch (JobInfoNotFoundException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "008", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "008", args);
                    m_log.error("operationJob() : " + e.getMessage(), e ); 
                    throw new HinemosUnknownException(e.getMessage(), e);
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_START_WAIT){
                //開始[保留解除]
                try {
                    new OperateWaitOfJob().releaseWaitJob(sessionId, jobunitId, jobId);
                } catch (JobInfoNotFoundException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "009", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "009", args);
                    m_log.error("operationJob() : " + e.getMessage(), e ); 
                    throw new HinemosUnknownException(e.getMessage(), e);
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_START_SKIP){
                //開始[スキップ解除]
                try {
                    new OperateSkipOfJob().releaseSkipJob(sessionId, jobunitId, jobId);
                } catch (JobInfoNotFoundException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "010", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "010", args);
                    m_log.error("operationJob() : " + e.getMessage(), e ); 
                    throw new HinemosUnknownException(e.getMessage(), e);
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_STOP_AT_ONCE){
                try {
                    //停止[コマンド]
                    new OperateStopOfJob().stopJob(sessionId, jobunitId, jobId, facilityId);
                } catch (JobInfoNotFoundException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "011", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "011", args);
                    m_log.error("operationJob() : " + e.getMessage(), e ); 
                    throw new HinemosUnknownException(e.getMessage(), e);
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_STOP_SUSPEND){
                try {
                    //停止[中断]
                    new OperateSuspendOfJob().suspendJob(sessionId, jobunitId, jobId);
                } catch (JobInfoNotFoundException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "012", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "012", args);
                    m_log.error("operationJob() : " + e.getMessage(), e ); 
                    throw new HinemosUnknownException(e.getMessage(), e);
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_STOP_WAIT){
                try {
                    //停止[保留]
                    new OperateWaitOfJob().waitJob(sessionId, jobunitId, jobId);
                } catch (JobInfoNotFoundException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "012", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "012", args);
                    m_log.error("operationJob() : " + e.getMessage(), e ); 
                    throw new HinemosUnknownException(e.getMessage(), e);
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_STOP_SKIP){
                //停止[スキップ]
                if(endValue == null){
                    throw new NullPointerException();
                }
                try {
                    new OperateSkipOfJob().skipJob(sessionId, jobunitId, jobId, endValue);
                } catch (JobInfoNotFoundException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "013", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId};
                    apllog.put("SYS", "013", args);
                    m_log.error("operationJob() : " + e.getMessage(), e ); 
                    throw new HinemosUnknownException(e.getMessage(), e);
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_STOP_MAINTENANCE){
                //停止[終了値の変更]
                if(endValue == null){
                    throw new NullPointerException();
                }
                try {
                    new OperateMaintenanceOfJob().maintenanceJob(sessionId, jobunitId, jobId, facilityId, endValue);
                } catch (JobInfoNotFoundException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "014", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "014", args);
                    m_log.error("operationJob() : " + e.getMessage(), e ); 
                    throw new HinemosUnknownException(e.getMessage(), e);
                }
            }
            else if(control.intValue() == OperationConstant.TYPE_STOP_FORCE){
                //停止[強制]
                if(endValue == null){
                    throw new NullPointerException();
                }
                try {
                    new OperateForceStopOfJob().forceStopJob(sessionId, jobunitId, jobId, facilityId, endValue);//修正予定
                } catch (JobInfoNotFoundException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "014", args);
                    m_log.debug("operationJob() : " + e.getMessage());
                    throw e;
                } catch (NamingException e) {
                    AplLogger apllog = new AplLogger("JOB", "job");
                    String[] args = {sessionId, jobId, facilityId};
                    apllog.put("SYS", "014", args);
                    m_log.error("operationJob() : " + e.getMessage(), e ); 
                    throw new HinemosUnknownException(e.getMessage(), e);
                }
            }
        }
        else{
            throw new IllegalStateException();
        }
	}
	
	/**
	 * ノード終了処理を行います。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @param info 実行結果情報
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#endNode(RunResultInfo)
	 */
	public void endNode(RunResultInfo info) throws NamingException, JobInfoNotFoundException {
	    m_log.debug("endNode() : sessionId=" + info.getSessionId() + ", jobId=" + info.getJobId() + ", facilityId=" + info.getFacilityId());
	    
	    OperationJob operation = new OperationJob();
	    operation.endNode(info);
	}
}
