/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.sql.SQLException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.SchedulerException;

import com.clustercontrol.CalendarNotFountException;
import com.clustercontrol.FacilityNotFoundException;
import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.JobInfoNotFoundException;
import com.clustercontrol.JobInvalidException;
import com.clustercontrol.JobMasterNotFoundException;
import com.clustercontrol.NotifyNotFoundException;
import com.clustercontrol.UsedFacilityException;
import com.clustercontrol.UserNotFoundException;
import com.clustercontrol.bean.CommonTableTreeItem;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.bean.OutputBasicInfo;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.calendar.ejb.session.CalendarControllerLocal;
import com.clustercontrol.calendar.ejb.session.CalendarControllerUtil;
import com.clustercontrol.commons.ejb.session.CheckFacility;
import com.clustercontrol.jobmanagement.bean.JobForwardFile;
import com.clustercontrol.jobmanagement.bean.JobHistoryFilter;
import com.clustercontrol.jobmanagement.bean.JobHistoryList;
import com.clustercontrol.jobmanagement.bean.JobNodeDetail;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobTriggerInfo;
import com.clustercontrol.jobmanagement.bean.JobTriggerTypeConstant;
import com.clustercontrol.jobmanagement.bean.NodeOperationInfo;
import com.clustercontrol.jobmanagement.bean.JobSchedule;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionUtil;
import com.clustercontrol.jobmanagement.factory.JobOperationProperty;
import com.clustercontrol.jobmanagement.factory.ModifyJob;
import com.clustercontrol.jobmanagement.factory.ModifySchedule;
import com.clustercontrol.jobmanagement.factory.SelectJob;
import com.clustercontrol.jobmanagement.factory.SelectRunSchedule;
import com.clustercontrol.jobmanagement.factory.SelectSchedule;
import com.clustercontrol.jobmanagement.util.JobUtil;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ジョブ管理機能の管理を行う Session Bean クラス<BR>
 * <p>クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。<BR>
 * 
 * @ejb.bean name="JobController"	
 *           jndi-name="JobController"
 *           type="Stateless" 
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobCommandInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobCommandMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobEndInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobEndMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobFileInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobFileMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobNoticeInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobNoticeMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobParamInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobParamMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobRelationInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobRelationMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobSessionLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobSessionJobLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobSessionNodeLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartJobInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartJobMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartTimeInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=JobStartTimeMasterLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 */
public abstract class JobControllerBean implements javax.ejb.SessionBean, CheckFacility {
	/** ログ出力のインスタンス<BR> */
	protected static Log m_log = LogFactory.getLog( JobControllerBean.class );
    
	/** コンテキスト情報<BR> */
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean がインスタンスプールに格納される際に行う処理を実装します。<BR>
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	
	
	/**
	 * セパレータ文字列を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @return セパレータ文字列
	 */
	public String getSeparator() {
		return SelectJob.SEPARATOR;
	}
	
	/**
	 * ジョブツリー情報を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @ejb.permission
     *     role-name="JobManagementExecute"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 *     
	 * @param treeOnly true=ジョブ情報を含まない, false=ジョブ情報含む
	 * @param locale ロケール情報
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws NotifyNotFoundException 
	 * @throws HinemosUnknownException 
	 * @throws JobMasterNotFoundException 
	 * @throws UserNotFoundException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getJobTree(boolean, Locale)
	 */
	public JobTreeItem getJobTree(boolean treeOnly, Locale locale) throws NotifyNotFoundException, HinemosUnknownException, JobMasterNotFoundException, UserNotFoundException {
	    m_log.debug("getJobTree() : treeOnly=" + treeOnly + ", locale=" + locale);
	    
		String loginUser = m_context.getCallerPrincipal().getName();
	    
		JobTreeItem item = null;
		try {
			//ジョブツリーを取得
			SelectJob select = new SelectJob();
			item = select.getJobTree(treeOnly, locale, loginUser);
		} catch (NamingException e) {
			m_log.error("SelectJob.getJobTree() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (JobMasterNotFoundException e) {
			throw e;
		} catch (UserNotFoundException e) {
			throw e;
		}

		return item;
	}
	
	/**
	 * 同一階層のジョブツリー情報を取得する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param parentJobunitId 親ジョブのジョブユニットID
	 * @param parentJobId 親ジョブID
	 * @param locale ロケール情報
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws JobMasterNotFoundException 
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getJobTreeOneLevel(String, Locale)
	 */
	public JobTreeItem getJobTreeOneLevel(String parentJobunitId, String parentJobId, Locale locale) throws JobMasterNotFoundException, HinemosUnknownException {
	    m_log.debug("getJobTreeOneLevel() : parentJobId=" + parentJobId + ", locale=" + locale);
	    
		//ジョブツリーを取得
	    JobTreeItem item = null;
		try {
			SelectJob select = new SelectJob();
			item = select.getJobTreeOneLevel(parentJobunitId, parentJobId, locale);
		} catch (NamingException e) {
			m_log.error("getJobTreeOneLevel() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (JobMasterNotFoundException e) {
			throw e;
		}

		return item;
	}
	
	/**
	 * ジョブツリー情報を登録する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementWrite"
     *     method-intf="Remote"
	 * 
	 * @ejb.transaction
	 * 		type="NotSupported"
	 * 
	 * @param item ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws HinemosUnknownException 
	 * @throws JobMasterNotFoundException 
	 * @throws JobInvalidException
	 * @throws JobMasterNotFoundException 
	 * @see com.clustercontrol.jobmanagement.factory.ModifyJob#registerJob(JobTreeItem, String)
	 */
	public void registerJob(JobTreeItem item) throws HinemosUnknownException, JobMasterNotFoundException, JobInvalidException {
		m_log.debug("registerJob() start");
		
		Principal pri = m_context.getCallerPrincipal();
		String loginUser = pri.getName();
		
		////////////////
		// Check
		////////////////
		// ジョブユニットID重複チェック
		JobUtil.findDuplicateJobunitId(item);
		// ジョブユニット内ジョブID重複チェック
		JobUtil.findDuplicateJobId(item, true);
		// 待ち条件チェック
		JobUtil.checkWaitRule(item);

		////////////////
		// Main
		// 0. 内部DBのジョブマスタ取得：クライアントのジョブツリーと比較用に取得する
		// 1. ジョブユニットの削除：内部DBに存在 & クライアントのジョブツリーに存在しない、ジョブユニットを削除する
		// 2. ジョブユニットの登録：削除対象以外のジョブユニットをリプレイス(削除 & 再登録)する
		////////////////
		
		// Exceptionが発生したかどうかのフラグ
		boolean error = false;
		// 失敗したジョブユニットIDのリスト
		ArrayList<String> errorJobunitIdList = new ArrayList<String>();
		
		// topを取得
		JobTreeItem top = item.getChildren(0);
		JobTreeItem[] jobunits = top.getChildrenArray();

		////////////////
		// 0. 内部DBのジョブマスタ取得：クライアントのジョブツリーと比較用に取得する
		////////////////
		
		//登録済みジョブユニット一覧を取得
		//一覧取得に失敗した場合は全件失敗とするため、リターン
		Collection<JobMasterLocal> registeredJobunits;
		try {
			registeredJobunits = JobMasterUtil.getLocalHome().findByJobType(JobConstant.TYPE_JOBUNIT);
		} catch (FinderException e1) {
			m_log.warn("JobMasterUtil.getLocalHome().findByJobType() : " + e1.getMessage());
			JobMasterNotFoundException je = new JobMasterNotFoundException(e1.getMessage(), e1);
			je.setJobType(JobConstant.TYPE_JOBUNIT);
			throw je;
		} catch (NamingException e1) {
			m_log.error("registerJob() : " + e1.getMessage(), e1 ); 
			throw new HinemosUnknownException(e1.getMessage(), e1);
		}
		
		////////////////
		// 1. ジョブユニットの削除：内部DBに存在 & クライアントのジョブツリーに存在しない、ジョブユニットを削除する
		// 
		// 注意事項(マルチクライアントアクセス時)
		// ・「ジョブユニットの削除」時に他クライアントから管理ユーザを変更され権限がない場合は、削除を行わない
		////////////////
		
		//itemに含まれないものは、クライアントサイドで削除されたものとする
		for (JobMasterLocal mst : registeredJobunits) {
			
			boolean deleteFlg = true;
			for (JobTreeItem jobunit : jobunits) {
				if (jobunit.getData().getType() == JobConstant.TYPE_JOBUNIT){
					m_log.debug("registerJob() : registeredJobunitId=" + mst.getJobunit_id() + ", jobunitId=" + jobunit.getData().getJobunitId());
					if (mst.getJobunit_id().equals(jobunit.getData().getJobunitId())) {
						deleteFlg = false;
						break;
					}
				}
			}
			
			m_log.debug("registerJob() : registeredJobunitId=" + mst.getJobunit_id() + ", deleteFlg=" + deleteFlg);
			
			if (deleteFlg) {
				
				// 削除に失敗した場合、ジョブユニットIDを記録する
				try {
					JobControllerUtil.getLocalHome().create().deleteJobunit(mst.getJobunit_id());
				} catch (Exception e) {
					m_log.error("deleteJobunit : " + mst.getJobunit_id() + " : " + e.getMessage(), e);
					error = true;
					errorJobunitIdList.add(mst.getJobunit_id());
				}
			}
		}
		
		////////////////
		// 2. ジョブユニットの登録：削除対象以外のジョブユニットをリプレイス(削除 & 再登録)する
		//
		// 注意事項(マルチクライアントアクセス時)
		// ・「ジョブユニットの登録」時に他クライアントから管理ユーザを変更され権限がない場合は、リプレイスを行わない
		// ・「ジョブユニットの登録」時に他クライアントから管理ユーザを変更され権限を付与されている場合は、リプレイスを行わない（クライアント側にジョブツリーを持たないため）
		////////////////
		
		//ジョブユニット単位でregisterJobunitメソッドをコールする
		for (JobTreeItem jobunit : jobunits) {
			try {
				// 登録処理に失敗した場合、ジョブユニットIDを記録する
				if(JobUtil.isReferableJobunit(jobunit, loginUser)){
					m_log.debug("registerJobunit : replace jobunit " + jobunit.getData().getJobunitId());
					JobControllerUtil.getLocalHome().create().registerJobunit(jobunit);
				}
				else {
					m_log.debug("registerJobunit : not replace jobunit " + jobunit.getData().getJobunitId() 
							+ ", because this jobunit is not referable by " + loginUser);
				}
			} catch (Exception e) {
				m_log.error("registerJobunit : " + jobunit.getData().getJobunitId() + " : " + e.getMessage(), e);
				error = true;
				errorJobunitIdList.add(jobunit.getData().getJobunitId());
			}
		}
		
		// 登録処理に失敗していたらExceptionをスローする
		if (error) {
			String[] args = {errorJobunitIdList.toString()};
			throw new JobInvalidException(Messages.getString("message.job.67", args));
		}
	}
	
	/**
	 * ジョブユニット単位でジョブツリー情報を登録する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="JobManagementWrite"
	 *     method-intf="Remote"
	 * 
	 * @ejb.transaction
	 * 		type="RequiresNew"
	 * 
	 * @param item ジョブユニットのジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws NotifyNotFoundException 
	 * @throws HinemosUnknownException 
	 * @throws JobMasterNotFoundException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifyJob#registerJobunit(JobTreeItem, String)
	 */
	public void registerJobunit(JobTreeItem jobunit) throws NotifyNotFoundException, HinemosUnknownException, JobMasterNotFoundException {
		m_log.debug("registerJobunit() : jobunitId=" + jobunit.getData().getJobunitId());
		
		Principal pri = m_context.getCallerPrincipal();
		String loginUser = pri.getName();
		
		try {
			ModifyJob modify = new ModifyJob();
			
			//削除->登録を同一トランザクションで実行するため、factoryの削除メソッドをcallする
			modify.deleteJobunit(jobunit.getData().getJobunitId(), loginUser);
			modify.registerJobunit(jobunit, loginUser);
		} catch (EJBException e) {
			m_log.error("registerJobunit() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("registerJobunit() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (RemoveException e) {
			m_log.error("registerJobunit() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.error("registerJobunit() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (JobInvalidException e) {
			m_log.warn("registerJobunit() : " + e.getMessage()); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		
	}
	
	/**
	 * ジョブユニット単位でジョブツリー情報を削除する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="JobManagementWrite"
	 *     method-intf="Remote"
	 * 
	 * @ejb.transaction
	 * 		type="RequiresNew"
	 * 
	 * @param jobunitId 削除対象ジョブユニットのジョブID
	 * @throws NotifyNotFoundException 
	 * @throws HinemosUnknownException 
	 * @throws JobMasterNotFoundException 
	 */
	public void deleteJobunit(String jobunitId) throws NotifyNotFoundException, HinemosUnknownException, JobMasterNotFoundException {
		m_log.debug("deleteJobunit() : jobunitId=" + jobunitId);
		
		Principal pri = m_context.getCallerPrincipal();
		String loginUser = pri.getName();
		
		try {
			ModifyJob modify = new ModifyJob();
			modify.deleteJobunit(jobunitId, loginUser);
		} catch (EJBException e) {
			m_log.error("deleteJobunit() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("deleteJobunit() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (RemoveException e) {
			m_log.error("deleteJobunit() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.error("deleteJobunit() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (JobInvalidException e) {
			m_log.warn("registerJobunit() : " + e.getMessage()); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		
	}
	
	
	/**
	 * ジョブ操作開始用プロパティを返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementExecute"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @param locale ロケール情報
	 * @return ジョブ操作開始用プロパティ
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.JobOperationProperty#getStartProperty(String, String, String, String, Locale)
	 */
	public ArrayList<String> getAvailableStartOperation(String sessionId, String jobunitId, String jobId, String facilityId, Locale locale) {
	    JobOperationProperty jobProperty = new JobOperationProperty();
	    return jobProperty.getAvailableStartOperation(sessionId, jobunitId, jobId, facilityId, locale);
	}
	
	/**
	 * ジョブ操作停止用プロパティを返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementExecute"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @param locale ロケール情報
	 * @return ジョブ操作停止用プロパティ
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.JobOperationProperty#getStopProperty(String, String, String, String, Locale)
	 */
	public ArrayList<String> getAvailableStopOperation(String sessionId, String jobunitId,  String jobId, String facilityId, Locale locale) {
	    JobOperationProperty jobProperty = new JobOperationProperty();
	    return jobProperty.getAvailableStopOperation(sessionId, jobunitId, jobId, facilityId, locale);
	}

	/**
	 * ジョブを実行します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementExecute"
     *     method-intf="Remote"
	 * 
	 * @param jobunitId 
	 * @param jobId ジョブID
	 * @param triggerInfo 実行契機情報
	 * @throws FacilityNotFoundException 
	 * @throws HinemosUnknownException 
	 * @throws JobInfoNotFoundException 
	 * @throws JobMasterNotFoundException 
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#createJobInfo(String, String, NotifyRequestMessage, JobTriggerInfo)
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#runJob(String, String)
	 */
	public void runJob(String jobunitId, String jobId, JobTriggerInfo triggerInfo) throws FacilityNotFoundException, HinemosUnknownException, JobMasterNotFoundException, JobInfoNotFoundException {
	    m_log.debug("runJob() : jobId=" + jobId);
	    
	    // triggerInfo の実行契機種別が「TRIGGER_MANUAL」の場合、ユーザ名を登録
	    if (triggerInfo.getTrigger_type() == JobTriggerTypeConstant.TYPE_MANUAL) {
		    Principal pri = m_context.getCallerPrincipal();
		    triggerInfo.setTrigger_info(pri.getName());
	    }
		
        String sessionId = null;
        try {
    	    //ジョブ情報作成 新たなトランザクションを生成して実行するため、EJBメソッドで実行
            JobControllerLocal jobController = JobControllerUtil.getLocalHome().create();
            sessionId = jobController.createJobInfo(jobunitId, jobId, null, triggerInfo);
        } catch (CreateException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {jobId};
            apllog.put("SYS", "003", args);
            m_log.debug("runJob() : " + e.getMessage());
            m_log.error("runJob() : " + e.getMessage(), e);
            throw new HinemosUnknownException(e.getMessage(), e);
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {jobId};
            apllog.put("SYS", "003", args);
            m_log.debug("runJob() : " + e.getMessage());
            m_log.error("runJob() : " + e.getMessage(), e);
            throw new HinemosUnknownException(e.getMessage(), e);
        } catch (FacilityNotFoundException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {jobId};
            apllog.put("SYS", "003", args);
            m_log.debug("runJob() : " + e.getMessage());
            throw e;
		}
        
	    try {
	        //ジョブ実行
		    JobRunManagementLocal jobRunManagement = JobRunManagementUtil.getLocalHome().create();
            jobRunManagement.runJob(sessionId, jobunitId, jobId);
         } catch (NamingException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {sessionId, jobId};
            apllog.put("SYS", "002", args);
            m_log.debug("runJob() : " + e.getMessage());
            m_log.error("runJob() : " + e.getMessage(), e);
            throw new HinemosUnknownException(e.getMessage(), e);
        } catch (CreateException e) {
        	m_log.error("runJob() : " + e.getMessage(), e);
            throw new HinemosUnknownException(e.getMessage(), e);
		}
	}
	
	/**
	 * ジョブを実行します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementExecute"
     *     method-intf="Remote"
	 * 
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param info ログ出力情報
	 * @param triggerInfo 実行契機情報
	 * @throws HinemosUnknownException 
	 * @throws JobInfoNotFoundException 
	 * @throws JobMasterNotFoundException 
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#createJobInfo(String, String, NotifyRequestMessage, JobTriggerInfo}
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#runJob(String, String)
	 */
	public void runJob(String jobunitId, String jobId, OutputBasicInfo info, JobTriggerInfo triggerInfo) 
			throws FacilityNotFoundException, HinemosUnknownException, JobInfoNotFoundException, JobMasterNotFoundException {
	    m_log.debug("runJob() : jobId=" + jobId);
	    
	    String sessionId = null;
	    try {
		    //ジョブ情報作成 新たなトランザクションを生成して実行するため、EJBメソッドで実行
		    JobControllerLocal jobController = JobControllerUtil.getLocalHome().create();
	    	sessionId = jobController.createJobInfo(jobunitId, jobId, info, triggerInfo);
	    } catch (CreateException e) {
	    	AplLogger apllog = new AplLogger("JOB", "job");
	    	String[] args = {jobId};
	    	apllog.put("SYS", "003", args);
	    	m_log.debug("runJob() : " + e.getMessage());
	    	m_log.error("runJob() : " + e.getMessage(), e ); 
	    	throw new HinemosUnknownException(e.getMessage(), e);
	    } catch (NamingException e) {
	    	AplLogger apllog = new AplLogger("JOB", "job");
	    	String[] args = {jobId};
	    	apllog.put("SYS", "003", args);
	    	m_log.debug("runJob() : " + e.getMessage());
	    	m_log.error("runJob() : " + e.getMessage(), e ); 
	    	throw new HinemosUnknownException(e.getMessage(), e);
	    } catch (FacilityNotFoundException e) {
	    	AplLogger apllog = new AplLogger("JOB", "job");
	    	String[] args = {jobId};
	    	apllog.put("SYS", "003", args);
	    	m_log.debug("runJob() : " + e.getMessage());
	    	throw e;
		}
	    
	    try {
	        //ジョブ実行
		    JobRunManagementLocal JobRunManagement = JobRunManagementUtil.getLocalHome().create();
            JobRunManagement.runJob(sessionId, jobunitId, jobId);
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {sessionId, jobId};
            apllog.put("SYS", "002", args);
            m_log.debug("runJob() : " + e.getMessage());
            m_log.error("runJob() : " + e.getMessage(), e ); 
	    	throw new HinemosUnknownException(e.getMessage(), e);
        } catch (CreateException e) {
        	m_log.error("runJob() : " + e.getMessage(), e ); 
	    	throw new HinemosUnknownException(e.getMessage(), e);
		}
	}
	
	/**
	 * ジョブをスケジュール実行します。<BR>
	 * Quartzからスケジュール実行時に呼び出されます。<BR>
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param calendarId カレンダID
	 * @param triggerInfo 実行契機情報
	 * @throws FacilityNotFoundException
	 * @throws HinemosUnknownException 
	 * @throws JobInfoNotFoundException 
	 * @throws JobMasterNotFoundException 
	 * @throws CalendarNotFountException 
	 * @throws JobMasterNotFoundException 
	 * @see com.clustercontrol.calendar.ejb.session.CalendarControllerBean#isRun(String, Date)}
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#runJob(String, String, JobTriggerInfo)
	 */
	public void scheduleRunJob(String jobunitId, String jobId, String calendarId, JobTriggerInfo triggerInfo)
			throws FacilityNotFoundException, HinemosUnknownException, JobInfoNotFoundException, CalendarNotFountException, JobMasterNotFoundException {
	    m_log.debug("runJob() : jobId=" + jobId + ", calendarId=" + calendarId);
	    
    	//カレンダをチェック
	    try {
			boolean check = false;
			if(calendarId != null && calendarId.length() > 0){
				CalendarControllerLocal calendar = 
					CalendarControllerUtil.getLocalHome().create();
					
				//カレンダによる実行可/不可のチェック
				if(calendar.isRun(calendarId, new Date()).booleanValue()){
					check = true;
				}
			}
			else{
				check = true;
			}
			
			if(!check)
				return;
			
			//ジョブ実行
			runJob(jobunitId, jobId, triggerInfo);
		} catch (CreateException e) {
			m_log.error("scheduleRunJob() : " + e.getMessage(), e ); 
	    	throw new HinemosUnknownException(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("scheduleRunJob() : " + e.getMessage(), e ); 
	    	throw new HinemosUnknownException(e.getMessage(), e);
		} catch (FacilityNotFoundException e) {
			m_log.debug("scheduleRunJob() -> " + e.getMessage());
			throw e;
		} catch (CalendarNotFountException e) {
			m_log.debug("scheduleRunJob() -> " + e.getMessage());
			throw e;
		} catch (JobMasterNotFoundException e) {
			AplLogger apllog = new AplLogger(HinemosModuleConstant.JOB, "job");
			String[] args = { jobId, triggerInfo.getTrigger_info() };
			apllog.put("SYS", "016", args);
			throw e;
		}
	}
	
	/**
	 * ジョブの実行用情報を作成します。<BR>
	 * 
     * @ejb.interface-method
     *     view-type="local"
     *     
	 * @ejb.permission
     *     role-name="JobManagementWrite"
     *     method-intf="Remote"
     *     
	 * @ejb.transaction
	 *     type="RequiresNew"
	 * 
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param info ログ出力情報
	 * @param triggerInfo 実行契機情報
	 * @return セッションID
	 * @throws FacilityNotFoundException
	 * @throws HinemosUnknownException 
	 * @throws JobInfoNotFoundException 
	 * @throws JobMasterNotFoundException 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionLocalHome#makeSession(java.lang.String, com.clustercontrol.notify.message.NotifyRequestMessage)
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionBean#ejbHomeMakeSession(String, NotifyRequestMessage)
	 */
	public String createJobInfo(String jobunitId, String jobId, OutputBasicInfo info, JobTriggerInfo triggerInfo)
			throws FacilityNotFoundException, HinemosUnknownException, JobMasterNotFoundException, JobInfoNotFoundException {
	    m_log.debug("createJobInfo() : jobId=" + jobId);

	    try {
			return JobSessionUtil.getLocalHome().makeSession(jobunitId, jobId, info, triggerInfo);
		} catch (FacilityNotFoundException e) {
			m_log.debug("createJobInfo() -> " + e.getMessage());
			throw e;
		} catch (CreateException e) {
			m_log.error("createJobInfo() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("createJobInfo() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
	}

	/**
	 * ジョブ操作を行います。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementExecute"
     *     method-intf="Remote"
	 * 
	 * @param property ジョブ操作用プロパティ
	 * @throws HinemosUnknownException 
	 * @throws JobInfoNotFoundException
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobRunManagementBean#operationJob(Property)
	 */
	public void operationJob(NodeOperationInfo property) throws HinemosUnknownException, JobInfoNotFoundException {
	    m_log.debug("operationJob()");
	    
	    try {
			JobRunManagementLocal JobRunManagement = JobRunManagementUtil.getLocalHome().create();
			JobRunManagement.operationJob(property);
		} catch (CreateException e) {
			m_log.error("createJobInfo() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("createJobInfo() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
	}
	
	/**
	 * ジョブ履歴一覧情報を返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param histories 表示履歴数
	 * @return ジョブ履歴一覧情報（Objectの2次元配列）
	 * @throws JobInfoNotFoundException
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getHistoryList(int)
	 */
	public JobHistoryList getHistoryList(int histories) throws JobInfoNotFoundException, HinemosUnknownException {
	    m_log.debug("getHistoryList()");
	    
	    String userId = m_context.getCallerPrincipal().getName();
	    
	    JobHistoryList list = null;
		try {
			SelectJob select = new SelectJob();
			list = select.getHistoryList(userId, null, histories);
		} catch (NamingException e) {
			m_log.error("getHistoryList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (SQLException e) {
			m_log.error("getHistoryList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (JobInfoNotFoundException e) {
			throw e;
		}
	    
	    return list;
	}
	
	/**
	 * ジョブ履歴一覧情報を返します。<BR>
	 * 
	 * @ejb.interface-method
 	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param property 履歴フィルタ用プロパティ
	 * @param histories 表示履歴数
	 * @return ジョブ履歴一覧情報（Objectの2次元配列）
	 * @throws JobInfoNotFoundException
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getHistoryList(Property, int)
	 */
	public JobHistoryList getHistoryList(JobHistoryFilter property, int histories) throws JobInfoNotFoundException, HinemosUnknownException {
	    m_log.debug("getHistoryList()");
	    
	    String userId = m_context.getCallerPrincipal().getName();
	    
	    JobHistoryList list = null;
		try {
			SelectJob select = new SelectJob();
			list = select.getHistoryList(userId, property, histories);
		} catch (NamingException e) {
			m_log.error("getHistoryList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (SQLException e) {
			m_log.error("getHistoryList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (JobInfoNotFoundException e) {
			throw e;
		}
	    
	    return list;
	}
	
	/**
	 * ジョブ詳細一覧情報を返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @return ジョブ詳細一覧情報（Objectの2次元配列）
	 * @throws JobInfoNotFoundException
	 * @throws HinemosUnknownException 
	 * @throws JobMasterNotFoundException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getDetailList(String)
	 */
	public CommonTableTreeItem getDetailList(String sessionId) throws JobInfoNotFoundException, HinemosUnknownException, JobMasterNotFoundException {
	    m_log.debug("getDetailList() : sessionId=" + sessionId);
	    
	    CommonTableTreeItem item = null;
		try {
			SelectJob select = new SelectJob();
			item = select.getDetailList(sessionId);
		} catch (NamingException e) {
			m_log.error("getDetailList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
	    
	    return item;
	}
	
	/**
	 * ノード詳細一覧情報を返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 *     
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param locale ロケール情報
	 * @return ノード詳細一覧情報（Objectの2次元配列）
	 * @throws JobInfoNotFoundException
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getNodeDetailList(String, String, String, Locale)
	 */
	public ArrayList<JobNodeDetail> getNodeDetailList(String sessionId, String jobunitId, String jobId, Locale locale) throws JobInfoNotFoundException, HinemosUnknownException {
	    m_log.debug("getNodeDetailList() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId);
	    
	    ArrayList<JobNodeDetail> list;
		try {
			SelectJob select = new SelectJob();
			list = select.getNodeDetailList(sessionId, jobunitId, jobId, locale);
		} catch (NamingException e) {
			m_log.error("getNodeDetailList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.error("getNodeDetailList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (JobInfoNotFoundException e) {
			throw e;
		}

	    return list;
	}
	
	/**
	 * ファイル転送一覧情報を返します。<BR>
	 * 
	 * @ejb.interface-method
 	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 *     
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @return ファイル転送一覧情報（Objectの2次元配列）
	 * @throws JobInfoNotFoundException
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getForwardFileList(String, String)
	 */
	public ArrayList<JobForwardFile> getForwardFileList(String sessionId, String jobunitId, String jobId) throws JobInfoNotFoundException, HinemosUnknownException {
	    m_log.debug("getForwardFileList() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId);
	    
	    ArrayList<JobForwardFile> list = null;
		try {
			SelectJob select = new SelectJob();
			list = select.getForwardFileList(sessionId, jobunitId, jobId);
		} catch (NamingException e) {
			m_log.error("getForwardFileList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.error("getForwardFileList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (JobInfoNotFoundException e) {
			throw e;
		}

	    return list;
	}
	
	/**
	 * スケジュール情報を登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementWrite"
     *     method-intf="Remote"
	 * 
	 * @param info スケジュール情報
	 * @throws HinemosUnknownException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#addSchedule(JobSchedule, String)
	 */
	public void addSchedule(JobSchedule info) throws HinemosUnknownException {
	    m_log.debug("addSchedule() : scheduleId=" + info.getId());
	    
	    Principal pri = m_context.getCallerPrincipal();
	    
	    ModifySchedule modify = new ModifySchedule();
	    try {
            modify.addSchedule(info, pri.getName());
        } catch (ParseException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {info.getId(), info.getJobId()};
            apllog.put("SYS", "004", args);
            m_log.debug("addSchedule() : " + e.getMessage());
            throw new HinemosUnknownException(e.getMessage(), e);
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {info.getId(), info.getJobId()};
            apllog.put("SYS", "004", args);
            m_log.error("addSchedule() : " + e.getMessage(), e ); 
            throw new HinemosUnknownException(e.getMessage(), e);
        } catch (SchedulerException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {info.getId(), info.getJobId()};
            apllog.put("SYS", "004", args);
            m_log.debug("addSchedule() : " + e.getMessage());
            throw new HinemosUnknownException(e.getMessage(), e);
        }
	}
	
	/**
	 * スケジュール情報を変更します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementWrite"
     *     method-intf="Remote"
	 * 
	 * @param info スケジュール情報
	 * @throws HinemosUnknownException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#modifySchedule(JobSchedule, String)
	 */
	public void modifySchedule(JobSchedule info) throws HinemosUnknownException {
	    m_log.debug("modifySchedule() : scheduleId=" + info.getId());
	    
	    Principal pri = m_context.getCallerPrincipal();
	    
	    ModifySchedule modify = new ModifySchedule();
	    try {
            modify.modifySchedule(info, pri.getName());
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {info.getId()};
            apllog.put("SYS", "006", args);
            m_log.error("modifySchedule() : " + e.getMessage(), e ); 
            throw new HinemosUnknownException(e.getMessage(), e);
        } catch (SchedulerException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {info.getId()};
            apllog.put("SYS", "006", args);
            m_log.debug("modifySchedule() : " + e.getMessage());
            throw new HinemosUnknownException(e.getMessage(), e);
        } catch (ParseException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {info.getId()};
            apllog.put("SYS", "006", args);
            m_log.debug("modifySchedule() : " + e.getMessage());
            throw new HinemosUnknownException(e.getMessage(), e);
        }
	}
	
	/**
	 * スケジュール情報を削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementWrite"
     *     method-intf="Remote"
	 * 
	 * @param scheduleId スケジュールID
	 * @throws HinemosUnknownException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#deleteSchedule(String)
	 */
	public void deleteSchedule(String scheduleId) throws HinemosUnknownException {
		m_log.debug("deleteSchedule() : scheduleId=" + scheduleId);
	    
	    ModifySchedule modify = new ModifySchedule();
	    try {
            modify.deleteSchedule(scheduleId);
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {scheduleId};
            apllog.put("SYS", "005", args);
            m_log.error("deleteSchedule() : " + e.getMessage(), e ); 
            throw new HinemosUnknownException(e.getMessage(), e);
        } catch (SchedulerException e) {
            AplLogger apllog = new AplLogger("JOB", "job");
            String[] args = {scheduleId};
            apllog.put("SYS", "005", args);
            m_log.debug("deleteSchedule() : " + e.getMessage());
            throw new HinemosUnknownException(e.getMessage(), e);
        }
	}
	
	/**
	 * スケジュール一覧情報を返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return スケジュール一覧情報（Objectの2次元配列）
	 * @throws JobMasterNotFoundException
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectSchedule#getScheduleList()
	 */
	public ArrayList<JobSchedule> getScheduleList() throws JobMasterNotFoundException, HinemosUnknownException  {
	    m_log.debug("getScheduleList()");
	    
	    ArrayList<JobSchedule> list;
		try {
			SelectSchedule select = new SelectSchedule();
			list = select.getScheduleList(m_context.getCallerPrincipal().getName());
		} catch (NamingException e) {
			m_log.error("getScheduleList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (SchedulerException e) {
			m_log.error("getScheduleList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}

	    return list;
	}
	
	/**
	 * 実行予定一覧情報を返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return 実行予定一覧情報（Objectの2次元配列）
	 * @throws JobMasterNotFoundExceptionException 
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectRunSchedule#getList()
	 */
	public ArrayList<ArrayList<Object>> getRunScheduleList() throws JobMasterNotFoundException, HinemosUnknownException {
	    m_log.debug("getRunScheduleList()");
	    
	    ArrayList<ArrayList<Object>> list = null;
		try {
			SelectRunSchedule select = new SelectRunSchedule();
			list = select.getList(m_context.getCallerPrincipal().getName());
		} catch (NamingException e) {
			m_log.error("getRunScheduleList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (SchedulerException e) {
			m_log.error("getRunScheduleList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}

	    return list;
	}
	
	/**
	 * 実行予定情報を返します。<BR>
	 * 
	 * @ejb.interface-method
     *     
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param base 基準日
	 * @return 実行予定情報（Objectの2次元配列）
	 * @throws JobMasterNotFoundException 
	 * @throws HinemosUnknownException 
	 * @throws CalendarNotFountException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectRunSchedule#getSchedule(Date)
	 */
	public ArrayList<ArrayList<Object>> getRunSchedule(Date base) throws JobMasterNotFoundException, HinemosUnknownException, CalendarNotFountException {
		m_log.debug("getRunSchedule() : base=" + base);
		
		ArrayList<ArrayList<Object>> list = null;
		try {
			SelectRunSchedule select = new SelectRunSchedule();
			list = select.getSchedule(base, m_context.getCallerPrincipal().getName());
		} catch (NamingException e) {
			m_log.error("getRunSchedule() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (SchedulerException e) {
			m_log.error("getRunSchedule() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		return list;
	}
	
	/**
	 * 実行予定詳細一覧情報を返します。<BR>
	 * 
	 * @ejb.interface-method
     *     
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param scheduleId スケジュールID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @return 実行予定詳細一覧情報（Objectの2次元配列）
	 * @throws JobMasterNotFoundException 
	 * @throws HinemosUnknownException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectRunSchedule#getDetailList(String, String)
	 */
	public CommonTableTreeItem getRunScheduleDetailList(String scheduleId, String jobunitId) throws JobMasterNotFoundException, HinemosUnknownException {
		m_log.debug("getRunScheduleDetailList() : scheduleId=" + scheduleId + ", jobunitId=" + jobunitId);
		
		CommonTableTreeItem item = null;
		try {
			SelectRunSchedule select = new SelectRunSchedule();
			item = select.getDetailList(scheduleId, jobunitId);
		} catch (NamingException e) {
			m_log.error("getRunScheduleDetailList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (SchedulerException e) {
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		return item;
	}
	
	/**
	 * 実行予定詳細情報を返します。<BR>
	 * 
	 * @ejb.interface-method
     *     
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param scheduleId スケジュールID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param base 基準日
	 * @return 実行予定詳細情報（Objectの2次元配列）
	 * @throws JobMasterNotFoundException 
	 * @throws HinemosUnknownException 
	 * @throws CalendarNotFountException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectRunSchedule#getDetailSchedule(String, Date)
	 */
	public CommonTableTreeItem getRunScheduleDetail(String scheduleId, String jobunitId, Date base)
			throws JobMasterNotFoundException, HinemosUnknownException, CalendarNotFountException {
		m_log.debug("getRunScheduleDetail() : scheduleId=" + scheduleId + ", jobunitId=" + jobunitId + ", base=" + base);
		
		CommonTableTreeItem item = null;
		try {
			SelectRunSchedule select = new SelectRunSchedule();
			item = select.getDetailSchedule(scheduleId, jobunitId, base);
		} catch (NamingException e) {
			m_log.error("getRunScheduleDetail() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (SchedulerException e) {
			m_log.error("getRunScheduleDetail() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (RemoteException e) {
			m_log.error("getRunScheduleDetail() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return item;
	}
	
	/**
	 * カレンダIDリストを返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return カレンダIDのリスト
	 * @throws HinemosUnknownException 
	 * @throws CalendarNotFountException 
	 * @throws  
	 * 
	 * @see com.clustercontrol.calendar.ejb.session.CalendarControllerBean#getCalendarIdList()
	 */
	public ArrayList<String> getCalendarIdList() throws HinemosUnknownException, CalendarNotFountException {
		m_log.debug("getCalendarIdList()");
		
		ArrayList<String> list = null;
		try {
			CalendarControllerLocal calendar = 
				CalendarControllerUtil.getLocalHome().create();
			
			list = calendar.getCalendarIdList();
		} catch (CreateException e) {
			m_log.error("getCalendarIdList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getCalendarIdList() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		
	    return list;
	}
	
	/**
	 * セッションジョブ情報を返します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @ejb.permission
     *     role-name="JobManagementRead"
     *     method-intf="Remote"
     *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 * @throws JobInfoNotFoundException 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getSessionJobInfo(String, String, String)
	 */
	public JobTreeItem getSessionJobInfo(String sessionId, String jobunitId, String jobId) throws HinemosUnknownException, NotifyNotFoundException, JobInfoNotFoundException {
	    m_log.debug("getSessionJobInfo() : sessionId=" + sessionId + ", jobId=" + jobId);
	    
	    JobTreeItem item = null;
		try {
			SelectJob select = new SelectJob();
			item = select.getSessionJobInfo(sessionId, jobunitId, jobId);
		} catch (NamingException e) {
			m_log.error("getSessionJobInfo() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		} catch (CreateException e) {
			m_log.error("getSessionJobInfo() : " + e.getMessage(), e ); 
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		
	    return item;
	}
	
	/**
	 * ファシリティIDが使用されているかチェックします。<BR>
	 * <P>
	 * 使用されていたら、UsedFacilityException がスローされる。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="JobManagementRead"
	 *     method-intf="Remote"
	 *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId ファシリティ
	 * @throws UsedFacilityException
	 * 
	 * @see com.clustercontrol.commons.ejb.session.CheckFacility#isUseFacilityId(java.lang.String)
	 * @see com.clustercontrol.bean.PluginConstant;
	 */
	public void isUseFacilityId(String facilityId) throws UsedFacilityException {
		try {
			Collection<JobCommandInfoLocal> ct = null;
			
			// ファシリティIDが使用されているジョブコマンドマスタを取得する。
			ct = JobCommandMasterUtil.getLocalHome().findByFacilityId(facilityId);
			if(ct != null && ct.size() > 0) {
				throw new UsedFacilityException(PluginConstant.TYPE_JOBMANAGEMENT);
			}
			
			// ファシリティIDが使用されているジョブコマンド情報を取得する。
			ct = null;
			ct = JobCommandInfoUtil.getLocalHome().findByFacilityId(facilityId);
			if(ct != null && ct.size() > 0) {
				Iterator<JobCommandInfoLocal> itr = ct.iterator();
				while(itr.hasNext()){
					//ジョブコマンド情報を取得
					JobCommandInfoLocal command = (JobCommandInfoLocal)itr.next();
					JobInfoLocal job = command.getJobInfo();
					JobSessionJobLocal sessionJob = job.getJobSessionJob();
					//セッションジョブの実行状態が終了または変更済以外の場合、UsedFacilityExceptionをスロー
					if(sessionJob.getStatus().intValue() != StatusConstant.TYPE_END && 
							sessionJob.getStatus().intValue() != StatusConstant.TYPE_MODIFIED){
						throw new UsedFacilityException(PluginConstant.TYPE_JOBMANAGEMENT);
					}
				}
			}
			
			// ファシリティIDが使用されているセッションノードを取得する。
			ct = null;
			ct = JobSessionNodeUtil.getLocalHome().findByFacilityId(facilityId);
			if(ct != null && ct.size() > 0) {
				Iterator<JobCommandInfoLocal> itr = ct.iterator();
				while(itr.hasNext()){
					//セッションノードを取得
					JobSessionNodeLocal sessionNode = (JobSessionNodeLocal)itr.next();
					//セッションノードの実行状態が終了または変更済以外の場合、UsedFacilityExceptionをスロー
					if(sessionNode.getStatus().intValue() != StatusConstant.TYPE_END && 
							sessionNode.getStatus().intValue() != StatusConstant.TYPE_MODIFIED){
						throw new UsedFacilityException(PluginConstant.TYPE_JOBMANAGEMENT);
					}
				}
			}
			
		} catch (FinderException e) {
			m_log.warn("isUseFacilityId() : " + e.getMessage());
		} catch (NamingException e) {
			m_log.warn("isUseFacilityId() : " + e.getMessage());
		}
	}
}
