/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.forward;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;
import javax.naming.InitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.repository.bean.TopicConstant;
import com.clustercontrol.repository.message.UpdateRepositoryInfo;
import com.clustercontrol.syslogng.forward.util.LogForwardProperties;
import com.clustercontrol.syslogng.message.UpdateMonitorRuleInfo;

/**
 * リポジトリ更新トピックを受信するクラス<BR>
 * トピックへの接続と、メッセージの受信を行います。
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class UpdateRepositoryInfoReceiveTopic implements MessageListener, ExceptionListener {
	
	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog(UpdateRepositoryInfoReceiveTopic.class);
	
	/** コネクションファクトリー。 */
	private static final String TOPIC_CON_FACTORY = "ConnectionFactory";
	
	/** 再接続間隔（秒）。 */
	private static long m_repositoryTopicReconnectionInterval = 10;
	
	/** 多重化ID。 */
	private final static String MULTI_ID = "multi.id";
	
	/** メッセージ管理のインスタンス。 */
	protected LogManager m_logmanager;
	
	/** コネクションファクトリー。 */
	private TopicConnectionFactory m_factory;
	/** コネクション。 */
	protected TopicConnection m_con;
	/** トピック。 */
	protected Topic m_topic;
	/** セッション。 */
	protected TopicSession m_session;
	/** サブスクライバ。 */
	protected TopicSubscriber m_subscriber;
	
	/** EJBアクセスのインスタンス。 */
	protected EJBContoroller m_ejbContoroler;
	
	/** エラーフラグ。 */
	private boolean m_errFlg = false;
	
	/** UpdateRepositoryInfoReceiveTopicクラス用スケジューラ * */
	private ScheduledExecutorService m_scheduler = Executors.newScheduledThreadPool(10);
    private ScheduledFuture<?> m_future;
	
	/** Lockオブジェクト */
	private Object reconnectLock = new Object();
	
	/**
	 * コンストラクタ。<BR>
	 * 初期処理を呼び出します。
	 * 
	 * @param ejbContoroler EJBアクセス
	 * @param props　プロパティファイル情報
	 * 
	 * @see #initial()
	 */
	@SuppressWarnings("unchecked")
	public UpdateRepositoryInfoReceiveTopic(LogManager logmanager, EJBContoroller ejbContoroler) {
		super();
		 
		m_logmanager = logmanager;
		m_ejbContoroler = ejbContoroler;
		 
		
		String interval = LogForwardProperties.getProperty("repository.topic.reconnection.interval");
		if (interval != null) {
			try {
				// プロパティファイルには秒で記述
				m_repositoryTopicReconnectionInterval = Integer.parseInt(interval);
				m_log.info("repository.topic.reconnection.interval = " + m_repositoryTopicReconnectionInterval + " sec");
			} catch (NumberFormatException e) {
				m_log.error("repository.topic.reconnection.interval",e);
			}
		}
		//接続処理
		initial();
	}
	
	/** 
	 * 引数で指定された受信メッセージを元に、キャッシュの更新を行います。<BR>
	 * 特定のメッセージを受信した場合のみ処理を行います。受信メッセージは下記の通りです。
	 * 
	 * <p><li>{@link com.clustercontrol.repository.message.UpdateRepositoryInfo} : リポジトリ情報を更新します。
	 * <p><li>{@link com.clustercontrol.syslogng.message.UpdateMonitorRuleInfo} : フィルタ情報を更新します。
	 * 
	 * @see javax.jms.MessageListener#onMessage(javax.jms.Message)
	 * 
	 * @see com.clustercontrol.syslogng.forward.EJBContoroller#loadFilterInfo()
	 */
	public void onMessage(Message message) {
		
		if(message instanceof ObjectMessage){
			ObjectMessage objectMessage = (ObjectMessage)message;
			Object obj;
			try {
				obj = objectMessage.getObject();
			} catch (JMSException e) {
				return;
			}
			
			// リポジトリ更新時
			if(obj instanceof UpdateRepositoryInfo){
				m_log.debug("onMessage : Receive UpdateRepositoryInfo");
				
				m_ejbContoroler.loadFilterInfo();
			
			}
			// フィルタ条件更新時
			else if(obj instanceof UpdateMonitorRuleInfo){
				m_log.debug("onMessage : Receive UpdateMonitorRuleInfo");
				
				m_ejbContoroler.loadFilterInfo();
			
			}
			else{
				m_log.error("onMessage()：Invalid message type received. " + obj.getClass().getCanonicalName());
			}
		}else{
			m_log.error("onMessage()：Invalid message type received. " + message.getClass());
		}
	}
	
	/**
	 * 通信エラーハンドラです。
	 * 
	 * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
	 * 
	 * @see #setErrFlg(boolean) 
	 */
	public void onException(JMSException arg0) {
		
		m_log.error(arg0.getMessage(), arg0);

		m_logmanager.reConnection();

	}
	
	/**
	 * エラーフラグを設定します。<BR>
	 * EJB再接続タイマータスクを登録します。
	 * 
	 * @param errFlg エラーの場合、<code> true </code>
	 */
	synchronized private void setErrFlg(boolean errFlg) {
		m_log.debug("setErrFlg() start : errFlg = " + errFlg);
		if (m_errFlg == false && errFlg == true) {
			
			m_log.debug("setErrFlg() set ReSetupTask to timer");
			m_future = m_scheduler.scheduleWithFixedDelay(
					new ReSetupTask(), 
					m_repositoryTopicReconnectionInterval, 
					m_repositoryTopicReconnectionInterval, 
					TimeUnit.SECONDS);
		}
		m_errFlg = errFlg;
		m_log.debug("setErrFlg() finish");
	}

	/**
	 * 再接続処理を行います。
	 * <p>
	 * <ol>
	 *  <li>終了処理を行います。</li>
	 *  <li>初期処理を行います。</li>
	 *  <li>エラーフラグを解除します。</li>
	 * </ol>
	 * 
	 * @see #terminate()
	 * @see #initial()
	 * @see #setErrFlg(boolean)
	 */
	synchronized private boolean reInitial() {
		boolean ret = false;
		
		m_log.info("reInitial() start");
		
		terminate();
		
		if (initial()) {
			
			m_log.info("reInitial() success!");
			ret = true;
			
			//エラーフラグ解除
			setErrFlg(false);
			
		} else {
			m_log.warn("reInitial() fail!");
		}
		
		return ret;
	}
	
	/**
	 * JMS接続の終了処理を行います。
	 * <p>
	 * <ol>
	 * <li>セッションをクローズします。</li>
	 * <li>コネクションをクローズします。</li>
	 * </ol>
	 */
	public void terminate() {
		m_log.debug("terminate() start");

		terminateSumscriber();
		
		try {
			if (m_session != null)
				m_session.close();
		} catch (JMSException e) {
			m_log.debug("terminate() session close error", e);
		}
		
		try {
			if (m_con != null)
				m_con.close();
		} catch (JMSException e) {
			m_log.debug("terminate() connection close error", e);
		}
		
		m_log.debug("terminate() finish");
	}
	
	/**
	 * トピック受信の終了処理を行います。
	 */
	private void terminateSumscriber() {
		try {
			if (m_subscriber != null)
				m_subscriber.close();
		} catch (JMSException e) {
			m_log.debug("terminateSumscriber() subscriber close error", e);
		}
	}
	
	
	
	/**
	 * JMS接続の初期処理を行います。
	 * <p>
	 * <ol>
	 *  <li>コネクションファクトリと宛先を取得します。</li>
	 *  <li>コネクションファクトリより、JMSプロバイダとのコネクションを生成します。</li>
	 *  <li>コネクションよりセッションを生成します。</li>
	 *  <li>トピック接続を開始します。</li>
	 * </ol>
	 * 
	 * @see #initialTopic()
	 * 
	 * @return 成功した場合、<code> true </code>
	 */
	private boolean initial() {
		
		InitialContext con = null;
		
		try {
			//InitialContextの生成
			con = new InitialContext(LogForwardProperties.getProperties());

			//コネクションファクトリ生成
			m_factory = (TopicConnectionFactory)con.lookup(TOPIC_CON_FACTORY);
			
			//コネクション生成
			m_con = m_factory.createTopicConnection();
			
			//セッション生成
			m_session = m_con.createTopicSession(false,
					Session.AUTO_ACKNOWLEDGE);
			
			//メッセージTopic取得
			m_topic = (Topic)con.lookup(TopicConstant.TOPIC_NAME_EXECUTE);
			
			
			//エラーハンドらセット
			m_con.setExceptionListener(this);
			
			m_con.start();
			
			//トピック接続開始
			initialTopic();
			
			
		} catch (Exception e) {
			setErrFlg(true);
			return false;
		} finally {
			try {
				if (con != null)
					con.close();
			} catch (Exception e1) {
			}
		}
		return true;
		
	}
	
	/**
	 * トピック受信の初期処理を行います。
	 * 
	 * @return 成功した場合、<code> true </code>
	 */
	private boolean initialTopic() {
		
		
		//現在のTopic受信を終了
		terminateSumscriber();
		
		//新しいファシリティIDでトピック受信開始
		try {
	        //ファシリティIDでメッセージセレクターを作成
	        StringBuffer msgSelector = new StringBuffer();
	        
	        String multiId = LogForwardProperties.getProperty(MULTI_ID, null);
	        
			if( multiId != null && multiId.length() > 0 ){
				msgSelector.append("MultiId='");
				msgSelector.append(multiId);
				msgSelector.append("' OR All='true' OR Syslog='true'");
			}else{
				msgSelector.append("All='true' OR Syslog='true'");
			}
	        
			m_subscriber = m_session.createSubscriber(m_topic, msgSelector.toString(), false);
			
			//コールバックを登録する
			m_subscriber.setMessageListener(this);
			
		} catch (Exception e) {
			setErrFlg(true);
			return false;
		} finally {
		}
		return true;
		
	}
	
	/**
	 * 再接続処理を行う
	 */
	public void reconnect() {
		
		synchronized (reconnectLock) {
		
			setErrFlg(true);
			
			terminate();
			
		}
	}
	
	/**
	 * EJB再接続タイマータスクです。<BR>
	 * 通信エラーとなった場合に定周期で呼ばれ、再接続を行います。
	 */
	protected class ReSetupTask implements Runnable {

		/**
		 * デフォルトコンストラクタ
		 */
		public ReSetupTask() {
		}
		
		/**
		 * 再接続を行い、成功した場合、このタスクをタイマーから削除します。
		 * 
		 * @see com.clustercontrol.syslogng.forward.UpdateRepositoryInfoReceiveTopic#reInitial()
		 */
		public void run() {
			m_log.debug("UpdateRepositoryInfoReceiveTopic.ReSetupTask.run() start");
			try {
				if (reInitial()) {
					//このタスクをタイマーから解除
					m_log.debug("UpdateRepositoryInfoReceiveTopic.ReSetupTask.run() task cancel");
					if(m_future != null){
						m_future.cancel(true);
					}
				}				
			} catch (Exception e) {
				m_log.error("UpdateRepositoryInfoReceiveTopic.ReSetupTask.run()", e);
			}
		}
		
	}
	
}
