/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.syslogng.forward;

import java.io.File;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.CalendarNotFountException;
import com.clustercontrol.bean.OutputBasicInfo;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.ejb.session.CalendarController;
import com.clustercontrol.calendar.ejb.session.CalendarControllerHome;
import com.clustercontrol.notify.bean.QueueConstant;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryController;
import com.clustercontrol.repository.ejb.session.RepositoryControllerHome;
import com.clustercontrol.syslogng.bean.LogFilterInfo;
import com.clustercontrol.syslogng.ejb.session.MonitorSyslogNGController;
import com.clustercontrol.syslogng.ejb.session.MonitorSyslogNGControllerHome;
import com.clustercontrol.syslogng.forward.util.LogForwardProperties;

/**
 * EJBアクセスを行うクラス<BR>
 * syslog処理のためのEJBアクセスを行います。
 * 
 * @version 3.0.0
 * @since 1.0.0
 */
public class EJBContoroller {
	
	/** プロパティのキー：コネクションファクトリー。 */
	private static final String QUEUE_CON_FACTORY = "ConnectionFactory";
	
	/** プロパティのキー：キューへの接続ユーザ。 */
	private static final String QUEUE_USER_NAME = "queue.user.name";
	
	/** プロパティのキー：キューへの接続パスワード。 */
	private static final String QUEUE_USER_PASSWORD = "queue.user.password";
	
	/** プロパティのキー：リトライ間隔（ミリ秒）。 */
	private final static String MSG_RETRY_INTERVAL = "msg.retry.interval";
	
	/** プロパティのキー：キャッシュ更新間隔（ミリ秒）。 */
	private final static String FILTER_CHACHE_INTERVAL = "filter.cache.interval";
	
	/** プロパティのキー：メッセージファイルのパス。 */
	private final static String MSG_FILE_PATH = "msg.file.msg.path";
	
	/** プロパティのキー：メッセージファイルの拡張子。 */
	private final static String MSG_FILE_SUFFIX = "msg.file.msg.syffix";
	
	/** ディレクトリ下に作成するファイルの上限 */
	private final static String MAX_FILE_COUNT = "msg.max.file.count";
	
	/** プロパティのキー：キューの名前。 */
	private final static String QUEUE_NAME = "queue.name";
	
	/** コネクション。 */
	private QueueConnection m_con;
	
	/** キュー。 */
	private Queue m_queue;
	
	/** セッション。 */
	private QueueSession m_session;
	
	/** コネクションファクトリー。 */
	private QueueConnectionFactory m_factory;
	
	/** エラーフラグ。 */
	private boolean m_errFlg = false;
	
	/** リトライ間隔（秒）。 */
	private long m_msgRetryInterval = 10;
	
	/** キャッシュ更新間隔（秒）。 */
	private long m_filterCacheInterval = 600;
	
	/** メッセージ管理のインスタンス。 */
	private LogManager m_logManager;
	
	/** ファイル入出力のインスタンス。 */
	private FileUtil m_fileUtil;
	
	/** EJBContorollerクラス用スケジューラ * */
	private ScheduledExecutorService m_scheduler = Executors.newScheduledThreadPool(10);
    private ScheduledFuture<?> m_future;
	
	/** ログ出力のインスタンス。 */
	static Log log = LogFactory.getLog(EJBContoroller.class);
	
	/** リポジトリ情報のローカルコンポーネント。 */
	private RepositoryControllerHome m_repositoryHome;
	
	/** syslog-ng情報のホーム。 */
	private MonitorSyslogNGControllerHome m_logFilterHome;
	
	/** Lockオブジェクト */
	private Object reconnectLock = new Object();
	
	/**
	 * コンストラクタ。
	 * 
	 * @param logManager メッセージ管理
	 * @param props プロパティファイル情報
	 */
	public EJBContoroller(LogManager logManager) {
		
		//インスタンス変数の初期化
		
		//ログマネージャ
		m_logManager = logManager;
		
		//エラーリトライインターバル(sec)
		String retryInterval = LogForwardProperties.getProperty(MSG_RETRY_INTERVAL);
		if (retryInterval != null) {
			try {
				m_msgRetryInterval = Long.parseLong(retryInterval);
				log.info(MSG_RETRY_INTERVAL + " = " + m_msgRetryInterval + " sec");
			} catch (NumberFormatException e) {
				log.error(MSG_RETRY_INTERVAL,e);
			}
		}
		
		//キャッシュ更新インターバル
		String casheInterval = LogForwardProperties.getProperty(FILTER_CHACHE_INTERVAL);
		if (casheInterval != null) {
			try {
				m_filterCacheInterval = Long.parseLong(casheInterval);
				log.info(FILTER_CHACHE_INTERVAL + " = " + m_filterCacheInterval + " sec");
			} catch (NumberFormatException e) {
				log.error(FILTER_CHACHE_INTERVAL,e);
			}
		}
		
		//ファイル入出力ユーティリティ
		String filePath = LogForwardProperties.getProperty(MSG_FILE_PATH, ".");
		log.info(MSG_FILE_PATH + " = " + filePath);
		String msgSuffix = LogForwardProperties.getProperty(MSG_FILE_SUFFIX, ".msg");
		log.info(MSG_FILE_SUFFIX + " = " + msgSuffix);
		
		int maxFileCount;
		try {
			maxFileCount = Integer.parseInt(LogForwardProperties.getProperty(MAX_FILE_COUNT, ".txt"));
			log.info(MAX_FILE_COUNT + " = " + maxFileCount);
		} catch (Exception e){
			log.error(MAX_FILE_COUNT,e);
			maxFileCount = -1;
		}
		
		m_fileUtil = new FileUtil(filePath, msgSuffix, maxFileCount);
		
		
		//サーバー接続処理
		if (initial()) {
			sendMsgRetry();
		}
		
		//フィルタ条件の読み込み
		loadFilterInfo();
		
		//キャッシュ更新タイマー開始
		m_scheduler.scheduleWithFixedDelay(
				new ReflashFilterTask(), 
				m_filterCacheInterval, 
				m_filterCacheInterval, 
				TimeUnit.SECONDS);
		
	}
	
	/**
	 * メッセージ送信の初期処理を行います。
	 * <p>
	 * <ol>
	 *  <li>コネクションファクトリと宛先を取得します。</li>
	 *  <li>コネクションファクトリより、JMSプロバイダとのコネクションを生成します。</li>
	 *  <li>コネクションよりセッションを生成します。</li>
	 * </ol>
	 * 
	 * @return 成功した場合、<code> true </code>
	 */
	private boolean initial() {
		
		log.info("initial() start");
		
		InitialContext con = null;
		
		try {
			//InitialContextの生成
			con = new InitialContext(LogForwardProperties.getProperties());
			
			//コネクションファクトリ生成
			m_factory = (QueueConnectionFactory) con.lookup(QUEUE_CON_FACTORY);
			
			//コネクション生成
			if (LogForwardProperties.getProperty(QUEUE_USER_NAME) != null) {
				//ユーザ認証
				m_con = m_factory.createQueueConnection(
						LogForwardProperties.getProperty(QUEUE_USER_NAME),
						LogForwardProperties.getProperty(QUEUE_USER_PASSWORD));
			} else {
				//ユーザ認証なし
				m_con = m_factory.createQueueConnection();
			}
			
			//コネクションが切れた際のハンドリング
			m_con.setExceptionListener(new JMSExceptionListener());
			
			//セッション生成
			m_session = m_con.createQueueSession(false,
					Session.AUTO_ACKNOWLEDGE);
			
			String queueName = QueueConstant.QUEUE_NAME_NOTIFYCONTROL;
			//キューの名前の取得
			if (LogForwardProperties.getProperty(QUEUE_NAME) != null && !LogForwardProperties.getProperty(QUEUE_NAME).equals("")) {
				log.info("initial() queue name : " + LogForwardProperties.getProperty(QUEUE_NAME));
				queueName = LogForwardProperties.getProperty(QUEUE_NAME);
			}
			//メッセージQueue取得
			m_queue = (Queue) con.lookup(queueName);
			
		} catch (Exception e) {
			log.error("initial() fail!", e);
			setErrFlg(true);
			return false;
		} finally {
			try {
				if (con != null)
					con.close();
			} catch (Exception e1) {
				log.warn("initial() close fail!", e1);
			}
		}
		
		log.info("initial() success!");
		return true;
	}
	
	/**
	 * フィルタとリポジトリ情報を読み込みます。
	 * <p>
	 * <ol>
	 * <li>フィルタ情報を読み込みます。</li>
	 * <li>フィルタに設定されているスコープのノード情報を読み込みます。</li>
	 * <li>結果をフィルタとファシリティ情報に格納します。</li>
	 * </ol>
	 * 
	 * @see com.clustercontrol.syslogng.forward.LoginManager#getNamingContext()
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getExecTargetFacilityIdList(String)
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getNodeFacilityIdList(java.lang.String, int, boolean)
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getNodeDetail(java.util.ArrayList, java.util.ArrayList)
	 */
	public void loadFilterInfo() {
		
		log.debug("loadFilterInfo() start");
		
		//
		// syslog-ng監視条件取得
		//
		ArrayList<LogFilterRepositoryInfo> filterRepList = null;
		
		NamingContext con;
		try {
			con = LoginManager.getContextManager(LogForwardProperties.getProperties()).getNamingContext();
			
			//リポジトリ情報ホームインターフェース
			m_repositoryHome = (RepositoryControllerHome) con
			.lookup(RepositoryControllerHome.JNDI_NAME);
			
			//syslog-ng監視情報ホームインターフェース
			m_logFilterHome =  (MonitorSyslogNGControllerHome) con
			.lookup(MonitorSyslogNGControllerHome.JNDI_NAME);
		} catch (NamingException e) {
			// login & lookupに失敗すると諦める
			log.warn("loadFilterInfo() login error",e);
			return;
		}
		
		RepositoryController repositoryController = null;
		MonitorSyslogNGController logFilter = null;
		
		
		try {
			//
			// EJB初期処理
			//
			log.debug("loadFilterInfo() initial EJB Interface");
			logFilter = m_logFilterHome.create();
			repositoryController = m_repositoryHome.create();
			
			
			//フィルタ情報取得（ダイアログで見えるものリストと同じ？）
			ArrayList filterList = logFilter.getFilterInfoList();
			
			//実際に実行するときのフィルタ情報
			filterRepList = new ArrayList<LogFilterRepositoryInfo>();
			
			// syslog-ng監視条件には対象のfacilityIDしか含まないので、
			//　リポジトリにアクセスを行い、フィルタ条件情報（ホスト名）を作成
			for (Iterator iter = filterList.iterator(); iter.hasNext();) {
				
				//１つ１つのフィルタ条件を有効フラグを確認して、
				//実行用のフィルタ条件としていく
				LogFilterInfo filterInfo = (LogFilterInfo) iter.next();
				if (ValidConstant.typeToBoolean(filterInfo.getValid())) {
					
					LogFilterRepositoryInfo filterRepInfo = 
						new LogFilterRepositoryInfo(filterInfo);
					
					String facilityID = filterInfo.getFacilityId();
					log.debug("loadFilterInfo() getList For:" + facilityID);
					
					
					//未登録ノード対応
					//登録ノードのホスト名のリストをここでは入れておき、
					//containsのメソッドで除外する
					
					if(facilityID.equals(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE)){
						
						facilityID=FacilityTreeAttributeConstant.REGISTEREFD_SCOPE;
						
					}
						
					//配下の一覧
					ArrayList facilityList = repositoryController.getExecTargetFacilityIdList(facilityID);
					
					//ノード名/IPアドレス取得
					ArrayList<String> attributeList = new ArrayList<String>();
					attributeList.add(FacilityAttributeConstant.NODENAME);
					attributeList.add(FacilityAttributeConstant.CN);
					attributeList.add(FacilityAttributeConstant.IPPROTOCOLNUMBER);
					attributeList.add(FacilityAttributeConstant.IPNETWORKNUMBER);
					attributeList.add(FacilityAttributeConstant.IPNETWORKNUMBERV6);
					
					HashMap facilityAttrMap = repositoryController
					.getNodeDetail(facilityList, attributeList);
					
					filterRepInfo.setMap(facilityAttrMap);
					
					filterRepList.add(filterRepInfo);
					
				}
				
			}
			
			log.info("update filter info");
		} catch (Exception e) {
			// 処理失敗で諦める
			log.error("loadFilterInfo() fail!",e);
			return;
		} finally {
			//ログアウト
			try {
				LoginManager.getContextManager(LogForwardProperties.getProperties()).logout();
			} catch (NamingException e) {
				log.warn("loadFilterInfo() logout error",e);
			}
		}
		
		//
		// フィルタ条件を設定
		//
		m_logManager.add(filterRepList);
		
		log.debug("loadFilterInfo() finish");
		return;
	}
	
	/**
	 * 引数で指定されたログ出力メッセージが、指定されたカレンダIDの稼働日に
	 * 出力されたメッセージかチェックします。
	 * 
	 * @param id カレンダID
	 * @param msg ログ出力メッセージ
	 * @return 稼働日の場合、<code> true </code>
	 * 
	 * @see com.clustercontrol.syslogng.forward.LoginManager#getNamingContext()
	 * @see com.clustercontrol.calendar.ejb.session.CalendarControllerBean#isRun(java.lang.String, java.util.Date)
	 */
	public boolean isRun(String id, OutputBasicInfo msg) {
		
		if(log.isDebugEnabled()){
			log.debug("isRun() start id = " + id + ", msg = " + msg.getMessage());
		}
		
		CalendarController calendarController = null;
		
		try {
			NamingContext con = LoginManager.getContextManager(LogForwardProperties.getProperties()).getNamingContext();
			
			//カレンダ情報ホームインターフェース
			CalendarControllerHome calendarHome = (CalendarControllerHome) con.lookup(CalendarControllerHome.JNDI_NAME);
			calendarController = calendarHome.create();
			
			try {
				if(calendarController.isRun(id, msg.getGenerationDate()).booleanValue()){
					// 稼動日
					return true;
				}
				else{
					// 非稼働日
					if(log.isDebugEnabled()){
						log.debug("isRun() : calendarId：" + id + ". It is non-opetating day.");
					}
				}
			} catch (CalendarNotFountException e) {
				// 指定されたカレンダIDがすでに存在しない
				log.info("isRun() : calendarID [ " + id + " ] does not exists");
			}
			
		} catch (Exception e) {
			log.error("isRun() error",e);
			
			// 失敗はファイル出力
			log.info("isRun() write message to local disk msg = " + msg.getMessage());
			m_logManager.addMsg(msg.getMessageOrg());
			
			setErrFlg(true);
			
		} finally {
			//ログアウト
			try {
				LoginManager.getContextManager(LogForwardProperties.getProperties()).logout();
			} catch (NamingException e) {
				log.error("isRun() logout error", e);
			}
		}
		return false;		
	}
	
	/**
	 * ログ出力メッセージを送信します。<BR>
	 * キューへの送信を試みます。
	 * 失敗した場合、メッセージファイルを出力します。
	 * 
	 * @param msg ログ出力メッセージ
	 * 
	 * @see #_sendMsg(Object)
	 * @see com.clustercontrol.syslogng.forward.FileUtil#write(Object)
	 */
	synchronized public void sendMsg(OutputBasicInfo msg) {
		
		
		if (!_sendMsg(msg)) {
			//送信失敗はファイル出力
			log.debug("sendMsg() : write message to file ");
			m_fileUtil.write(msg);
		}
	}
	
	/**
	 * 引数で指定されたメッセージをキューへ送信します。
	 * 
	 * @param msg メッセージ
	 * @return 成功した場合、<code> true </code>
	 */
	private boolean _sendMsg(Object msg) {
		log.debug("_sendMsg() start");

		if (isErrFlg()) {
			return false;
		}
		
		QueueSender sender = null;
		try {
			//Sender作成
			sender = m_session.createSender(m_queue);
			//メッセージ作成
			ObjectMessage mess = m_session.createObjectMessage((Serializable)msg);
			//送信
			if(log.isDebugEnabled()){
				log.debug("_sendMsg() Send Object");
			}
			sender.send(mess);
			
		} catch (JMSException e) {
			log.error("_sendMsg() error", e);
			setErrFlg(true);
			return false;
		} finally {
			try {
				if (sender != null)
					sender.close();
			} catch (Exception e1) {
				log.error("_sendMsg() sender close error", e1);
			}
		}
		return true;
		
	}
	
	/**
	 * メッセージをキューへ再送信します。
	 * <p>
	 * <ol>
	 *  <li>メッセージファイル一覧を取得します。</li>
	 *  <li>メッセージファイルを読み込みます。</li>
	 *  <li>読み込んだメッセージをキューへ送信します。</li>
	 * </ol>
	 * 
	 * @see com.clustercontrol.syslogng.forward.FileUtil#getFileList()
	 * @see com.clustercontrol.syslogng.forward.FileUtil#readMsgFile(File)
	 * @see #_sendMsg(Object)
	 */
	synchronized private void sendMsgRetry() {
		
		log.debug("sendMsgRetry() start");
		
		//ファイル一覧取得
		Collection col = m_fileUtil.getFileList();
		
		for (Iterator iter = col.iterator(); iter.hasNext();) {
			File file = (File) iter.next();
			//ファイルを読み込み
			Object msg = m_fileUtil.readMsgFile(file);
			
			//送信
			if (!_sendMsg(msg)) {
				//失敗したら処理中止
				break;
			}
			//送信したファイルを削除
			file.delete();
		}
		
		log.debug("sendMsgRetry() finish");	}
	
	/**
	 * サーバ接続の終了処理を行います。
	 * <p>
	 * <ol>
	 * <li>メッセージサービス再接続用タイマーを解除します。</li>
	 * <li>メッセージ送信の終了処理を行います。</li>
	 * </ol>
	 * 
	 * @see #_terminate()
	 */
	public void terminate() {
		
		m_scheduler.shutdownNow();
		
		_terminate();
	}
	
	/**
	 * メッセージ送信の終了処理を行います。
	 * <p>
	 * <ol>
	 * <li>セッションをクローズします。</li>
	 * <li>コネクションをクローズします。</li>
	 * </ol>
	 */
	public void _terminate() {
		log.debug("_terminate() start");

		try {
			if (m_session != null)
				m_session.close();
		} catch (JMSException e) {
			log.debug("_terminate() session close error", e);
		}
		
		try {
			if (m_con != null)
				m_con.close();
		} catch (JMSException e) {
			log.debug("_terminate() connection close error", e);
		}
		log.debug("_terminate() finish");
	}
	
	
	
	/**
	 * エラーフラグを設定します。
	 * 
	 * @param errFlg エラーの場合、<code> true </code>
	 */
	synchronized private void setErrFlg(boolean errFlg) {
		log.debug("setErrFlg() start : errFlg = " + errFlg);
		if (m_errFlg == false && errFlg == true) {
			
			log.debug("setErrFlg() set ReSetupTask to timer");
			m_future = m_scheduler.scheduleWithFixedDelay(
					new ReSetupTask(), 
					m_msgRetryInterval, 
					m_msgRetryInterval, 
					TimeUnit.SECONDS);
			
		}
		m_errFlg = errFlg;
		log.debug("setErrFlg() finish");
	}
	
	/**
	 * エラーフラグを返します。
	 * 
	 * @return エラーフラグ
	 */
	synchronized private boolean isErrFlg() {
		return m_errFlg;
	}
	
	/**
	 * 再接続処理を行います。
	 * <p>
	 * <ol>
	 *  <li>終了処理を行います。</li>
	 *  <li>初期処理を行います。</li>
	 *  <li>エラーフラグを解除します。</li>
	 *  <li>フィルタとリポジトリ情報を読み込みます。</li>
	 *  <li>メッセージをキューへ再送信します。</li>
	 * </ol>
	 * 
	 * @see #_terminate()
	 * @see #initial()
	 * @see #setErrFlg(boolean)
	 * @see #loadFilterInfo()
	 * @see #sendMsgRetry()
	 */
	synchronized private boolean reInitial() {
		
		boolean ret = false;
		
		log.info("reInitial() start");
		
		_terminate();
		
		if (initial()) {
			
			log.info("reInitial() success!");
			ret = true;
			
			//エラーフラグ解除
			setErrFlg(false);
			
			loadFilterInfo();
			sendMsgRetry();
			
		} else {
			log.warn("reInitial() fail!");
		}
		
		return ret;
	}
	
	/**
	 * 再接続処理を行う
	 */
	public void reconnect() {
		
		synchronized (reconnectLock) {
			
			setErrFlg(true);
        
			_terminate();
		}
        
	}
	
	/**
	 * フィルタ情報取得タイマータスクです。<BR>
	 * 定周期でフィルタ情報を取得し、キャッシュを更新します。
	 */
	protected class ReflashFilterTask implements Runnable {

		/**
		 * デフォルトコンストラクタ
		 */
		public ReflashFilterTask() {
		}
		
		/**
		 * キャッシュを更新します。
		 * 
		 * @see com.clustercontrol.syslogng.forward.EJBContoroller#loadFilterInfo()
		 */
		public void run() {
			log.debug("ReflashFilterTask.run() start");
			try{
				loadFilterInfo();
			} catch (Exception e) {
				log.error("ReflashFilterTask.run() catche RuntimeException", e);
			}
		}
	}
	
	/**
	 * EJB再接続タイマータスクです。
	 * <BR>
	 * 通信エラーとなった場合に定周期で呼ばれ、再接続を行います。
	 */
	protected class ReSetupTask implements Runnable {

		/**
		 * デフォルトコンストラクタ
		 */
		public ReSetupTask() {
		}
		
		/**
		 * 再接続を行い、成功した場合、このタスクをタイマーから削除します。
		 * 
		 * @see com.clustercontrol.syslogng.forward.EJBContoroller#reInitial()
		 */
		public void run() {
			log.debug("ReSetupTask.run() start");
			try{
				if (reInitial()) {
					//このタスクをタイマーから解除
					log.debug("ReSetupTask.run() task cancel");
					if(m_future != null){
						m_future.cancel(true);
					}
				}				
			} catch (Exception e) {
				log.error("ReSetupTask.run() catche RuntimeException", e);
			}
		}
		
	}
	
	
	/** 
	 * 通信エラーハンドラです。
	 * <BR>
	 * JMSException発生時に再接続用のタイマースレッド（ReSetupTask）を起動します。
	 * 
	 */
	protected class JMSExceptionListener implements ExceptionListener {
		
		/**
		 * 接続をクローズし、再接続のタイマータスクを起動します。
		 *
		 * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
		 */
		public void onException(JMSException e) {
			log.info("onException() : try reconnect : ", e);
			
			m_logManager.reConnection();
		}
	}
	
}
