/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.swt.SWT;

import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.bean.TableColumnInfo;
import com.clustercontrol.commons.ejb.UsedFacilityException;
import com.clustercontrol.commons.ejb.session.CheckFacility;
import com.clustercontrol.syslogng.bean.LogListTabelDefine;
import com.clustercontrol.syslogng.ejb.entity.SyslogMonitorRuleUtil;
import com.clustercontrol.syslogng.factory.ModifyMonitorRule;
import com.clustercontrol.syslogng.factory.SelectMonitorRule;
import com.clustercontrol.util.Messages;

/**
 * syslog-ng監視の管理を行う Session Bean <BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 * 
 * @ejb.bean name="SyslogNGController"	
 *           jndi-name="SyslogNGController"
 *           type="Stateless" 
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=RepositoryController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorController"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 */
public abstract class SyslogNGControllerBean implements javax.ejb.SessionBean, CheckFacility {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( SyslogNGControllerBean.class );
	
	/** コンテキスト情報。 */
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean の生成時に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * フィルタ情報一覧のテーブル定義情報を取得します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.syslogng.bean.LogListTabelDefine
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getLogFilterListTableDefine(Locale locale) {
		
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();
		
		tableDefine.add(LogListTabelDefine.ORDER_NO,
				new TableColumnInfo(Messages.getString("order", locale), TableColumnInfo.NONE, 50, SWT.RIGHT));
		tableDefine.add(LogListTabelDefine.PROCESS_TYPE,
				new TableColumnInfo(Messages.getString("process", locale), TableColumnInfo.PROCESS, 50, SWT.LEFT));		
		tableDefine.add(LogListTabelDefine.MONITOR_ID,
				new TableColumnInfo(Messages.getString("monitor.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));	
		tableDefine.add(LogListTabelDefine.DESCRIPTION, 
				new TableColumnInfo(Messages.getString("description", locale), TableColumnInfo.NONE, 200, SWT.LEFT));
		tableDefine.add(LogListTabelDefine.FACILITY_ID, 
				new TableColumnInfo(Messages.getString("scope", locale), TableColumnInfo.FACILITY, 200, SWT.LEFT));
		tableDefine.add(LogListTabelDefine.CALENDAR_ID, 
				new TableColumnInfo(Messages.getString("calendar", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(LogListTabelDefine.VALID_FLG, 
				new TableColumnInfo(Messages.getString("valid", locale) + "/" + Messages.getString("invalid", locale), TableColumnInfo.VALID, 80, SWT.LEFT));
		tableDefine.add(LogListTabelDefine.CREATE_USER, 
				new TableColumnInfo(Messages.getString("creator.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(LogListTabelDefine.CREATE_TIME, 
				new TableColumnInfo(Messages.getString("create.time", locale), TableColumnInfo.NONE, 130, SWT.LEFT));
		tableDefine.add(LogListTabelDefine.UPDATE_USER, 
				new TableColumnInfo(Messages.getString("modifier.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(LogListTabelDefine.UPDATE_TIME, 
				new TableColumnInfo(Messages.getString("update.time", locale), TableColumnInfo.NONE, 130, SWT.LEFT));
		
		return tableDefine;
	}
	
	/**
	 * フィルタ情報一覧を取得します。<BR>
	 * 
	 * ArrayListにはLogFilterInfoが格納されています。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SyslogNgMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @return フィルタ情報一覧 （{@link com.clustercontrol.syslogng.bean.LogFilterInfo}のリスト）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.syslogng.factory.SelectMonitorRule#getFilterInfoList()
	 */
	public ArrayList getFilterInfoList() throws CreateException, FinderException, NamingException {
		
	    SelectMonitorRule select = new SelectMonitorRule();
	    return select.getFilterInfoList();
	}

	/**
	 * フィルタ情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SyslogNgMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param filterInfoList フィルタ情報一覧 （{@link com.clustercontrol.syslogng.bean.LogFilterInfo}のリスト）
	 * @return 登録に成功した場合、<code> true </code>
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.syslogng.factory.ModifyMonitorRule#deleteMonitorRule()
	 * @see com.clustercontrol.syslogng.factory.ModifyMonitorRule#createMonitorRule(ArrayList, String)
	 */
	public boolean createMonitorRuleList(ArrayList filterInfoList) throws CreateException, RemoveException, FinderException, NamingException {
		
		Principal pri = m_context.getCallerPrincipal();

		ModifyMonitorRule modify = new ModifyMonitorRule();
		
		// 監視条件削除 
		modify.deleteMonitorRule();
		// 監視条件登録
	    return modify.createMonitorRule(filterInfoList, pri.getName());
	}
	
	/**
	 * ファシリティIDが使用されているかチェックします。<BR>
	 * <P>
	 * 使用されていたら、UsedFacilityException がスローされる。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="SyslogNgMonitorRead"
	 *     method-intf="Remote"
	 *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId ファシリティ
	 * @throws UsedFacilityException
	 * 
	 * @see com.clustercontrol.commons.ejb.session.CheckFacility#isUseFacilityId(java.lang.String)
	 * @see com.clustercontrol.bean.PluginConstant;
	 */
	public void isUseFacilityId(String facilityId) throws UsedFacilityException {
		try {
			// ファシリティIDが監視対象になっている監視情報を取得する。
			Collection ct = SyslogMonitorRuleUtil.getLocalHome().findByFacilityId(facilityId);
			if(ct != null && ct.size() > 0) {
				throw new UsedFacilityException(PluginConstant.TYPE_SYSLOGNG_MONITOR);
			}
		} catch (FinderException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		} catch (NamingException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		}
	}
}
