/*
                                                                                                
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
*/

package com.clustercontrol.poller.job;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.Job;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;

import com.clustercontrol.poller.PollingController;
import com.clustercontrol.poller.PollerManager;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * SNMPポーリングを実行するジョブクラス（Quartzに登録します）
 * 
 * @version 3.0.0
 * @since 2.0.0
 */
public class PollingJob implements Job {
	private static Log m_log = LogFactory.getLog( PollingJob.class );

	// Quartzに起動されたスレッドの情報を格納するクラス
	private class PollingThreadInfo {
		public PollingThreadInfo(String pollerGroup, String pollerName) {
			this.pollerGroup = pollerGroup;
			this.pollerName = pollerName;
			threadName = Thread.currentThread().getName();
		}
		private final long startTime = System.currentTimeMillis();
		public long getStartTime() {
			return startTime;
		}
		private final String threadName;
		public String getThreadName() {
			return threadName;
		}
		private final String pollerGroup;
		public String getPollerGroup() {
			return pollerGroup;
		}
		private final String pollerName;
		public String getPollerName() {
			return pollerName;
		}
		private volatile boolean isLogged = false;
		public boolean getIsLogged() {
			return isLogged;
		}
		public void setIsLogged() {
			isLogged = true;
		}
	}
	// quartzより起動されて現在アクティブなスレッドに関する情報をスレッドIDをキーに保持するマップ
	private static final ConcurrentHashMap<Long, PollingThreadInfo> activePollingThreadInfoMap = 
		new ConcurrentHashMap<Long, PollingThreadInfo>();
	
	/**
	 * Qurtzからコールバックされるメソッド
	 * ノード毎にスレッドを起動しポーリングを実行します。
	 */
	public void execute(JobExecutionContext context) throws JobExecutionException {
		// デバッグログ出力
		m_log.debug("execute start : ");
		
		// 引数を取得します
		JobDataMap dataMap = context.getJobDetail().getJobDataMap();

		// ポーラマネージャを参照するためのJNDI名
		String jndiName = dataMap.getString("jndiName");

		// ポーラグループ名
		String pollerGroup = dataMap.getString("pollerGroup");

		// ポーラ名
		String pollerName = dataMap.getString("pollerName");

		try {
			// 指定のポーラを取得する
			InitialContext ctx = new InitialContext();
			Object obj = ctx.lookup(jndiName);
			
			PollerManager manager = 
				(PollerManager)PortableRemoteObject.narrow(obj, PollerManager.class);

			PollingController poller = manager.getPoller(pollerGroup, pollerName);

			// 現在走行中のスレッド情報を走査して、長時間（ここでは60*60*1000(ms)=1h）
			// 経過していて、 なおかつ今回初めて長いと判定されたスレッド情報を探し出す。
			final int timeout_min = 60;
			for (final PollingThreadInfo threadInfo : activePollingThreadInfoMap.values()) {
				if (System.currentTimeMillis() - (timeout_min*60*1000) > threadInfo.getStartTime() &&
						threadInfo.getIsLogged() == false) {
					
					// 今回リークした可能性があるスレッドについて情報を取りだし、ログ出力済みフラグを立てる
					final String longPollerGroup = threadInfo.getPollerGroup();
					final String longPollerName = threadInfo.getPollerName();
					threadInfo.setIsLogged();
					
					// 全スレッドに関する統計情報を取得する
					final int currentThreadCount = activePollingThreadInfoMap.size();
					int longThreadCount = 0;
					for (final PollingThreadInfo tmpInfo : activePollingThreadInfoMap.values()) {
						if (tmpInfo.getIsLogged()) {
							longThreadCount++;
						}
					}
					
					// 今回新規にリークした可能性があるスレッドの情報と、全スレッドの統計情報をログ出力する
					AplLogger apllog = new AplLogger("COMMON", "common");
					String[] args = {
							Integer.toString(timeout_min),
							longPollerGroup,
							longPollerName,
							Integer.toString(longThreadCount),
							Integer.toString(currentThreadCount)
					};
					apllog.put("SYS", "001", args);
				}
			}
			
			// ポーリング開始前に自身のスレッド情報をマップに登録
			activePollingThreadInfoMap.put(
					Thread.currentThread().getId(),
					new PollingThreadInfo(pollerGroup, pollerName)
			);
			
			// ポーリングを実行
			poller.run();
		} catch (NamingException e) {
			m_log.error(e.getMessage(), e);
		} finally {
			// 自身のスレッド情報を登録解除
			activePollingThreadInfoMap.remove(Thread.currentThread().getId());
		}
		
		// デバッグログ出力
		m_log.debug("execute end   : " + jndiName);
	}
}
