/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.ejb.session;

import java.net.UnknownHostException;
import java.rmi.RemoteException;
import java.rmi.dgc.VMID;
import java.security.Principal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.swt.SWT;

import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.TableColumnInfo;
import com.clustercontrol.commons.ejb.UsedFacilityException;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.bean.NodeListTabelDefine;
import com.clustercontrol.repository.bean.NodeScopeTabelDefine;
import com.clustercontrol.repository.bean.ScopeListTableDefine;
import com.clustercontrol.repository.bean.TopicConstant;
import com.clustercontrol.repository.ejb.entity.FacilityData;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.repository.factory.DeleteFacility;
import com.clustercontrol.repository.factory.ModifyFacility;
import com.clustercontrol.repository.factory.NodeFilterProperty;
import com.clustercontrol.repository.factory.NodeProperty;
import com.clustercontrol.repository.factory.ScopeProperty;
import com.clustercontrol.repository.factory.SearchNodeBySNMP;
import com.clustercontrol.repository.factory.SelectFacility;
import com.clustercontrol.repository.util.SendTopic;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;

/**
 *
 * <!-- begin-user-doc --> リポジトリ情報（ノード、スコープ）の生成、変更、削除、
 * 参照を行うSessionBean<BR>　
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 * <!-- end-user-doc --> *
 *
 * <!-- begin-xdoclet-definition --> 
 * @ejb.bean name="RepositoryController"	
 *           jndi-name="RepositoryController"
 *           type="Stateless" 
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.jndi:jndiName=external/hinemos/ldap/provider,service=ExternalContext"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=DeviceInfo"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=Facility"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=FacilityTree"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 *           
 * <!-- end-xdoclet-definition --> 
 * @generated
 */
public abstract class RepositoryControllerBean implements javax.ejb.SessionBean {
	protected static Log m_log = LogFactory.getLog( RepositoryControllerBean.class ); 
	
	private SessionContext m_context;
	
	public static final int ALL = 0;
	public static final int ONE_LEVEL = 1;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * ファシリティツリー（スコープツリー）取得を取得します。
	 * <BR>
	 * 取得したファシリティツリーには割り当てられたノードを含みます。<BR>
	 * このメソッドはクライアントの画面情報を作成するために
	 * 呼び出されます。クライアントのロケールを引数をして必要とします。<BR>
	 * （最上位のスコープという表記をscopeをいう表記を切り替えるため。）
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param locale クライアントのロケール
	 * @return FacilityTreeItemの階層オブジェクト
	 * @throws FinderException
	 * @throws NamingException
	 */
	public FacilityTreeItem getFacilityTree(Locale locale) throws FinderException, NamingException {

		//ファシリティツリー(ノード含む)を取得
		FacilityTreeItem item = null;
		try {
			SelectFacility select = new SelectFacility();
			item = select.getFacilityTree(locale, null);
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getFacilityTree() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}

		return item;
	}
	
	/**
	 * ファシリティツリー（スコープツリー）取得を取得します。(有効なノードのみ)
	 * <BR>
	 * 取得したファシリティツリーには割り当てられたノードを含みます。<BR>
	 * このメソッドはクライアントの画面情報を作成するために
	 * 呼び出されます。クライアントのロケールを引数をして必要とします。<BR>
	 * （最上位のスコープという表記をscopeをいう表記を切り替えるため。）
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param locale クライアントのロケール
	 * @return FacilityTreeItemの階層オブジェクト
	 * @throws FinderException
	 * @throws NamingException
	 */
	public FacilityTreeItem getExecTargetFacilityTree(Locale locale) throws FinderException, NamingException {

		//ファシリティツリー(ノード含む)を取得
		FacilityTreeItem item = null;
		try {
			SelectFacility select = new SelectFacility();
			item = select.getFacilityTree(locale, Boolean.TRUE);
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getFacilityTree() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}

		return item;
	}
	
	/**
	 * ファシリティツリー（スコープツリー）取得を取得します。<BR>
	 * 取得したファシリティツリーには割り当てられたノードを含みません。<BR>
	 * このメソッドはクライアントの画面情報を作成するために
	 * 呼び出されます。クライアントのロケールを引数をして必要とします。<BR>
	 * （最上位のスコープという表記をscopeをいう表記を切り替えるため。）
	 * 	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param locale　クライアントのロケール
	 * @return FacilityTreeItemの階層オブジェクト
	 * @throws NamingException
	 * @throws FinderException
	 */
	public FacilityTreeItem getScopeFacilityTree(Locale locale) throws FinderException, NamingException {
		
		FacilityTreeItem item = null;
		try {
			//ファシリティツリー(スコープのみ)を取得
			SelectFacility select = new SelectFacility();
			item = select.getScopeFacilityTree(locale);
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getScopeFacilityTree() -> " + e.getMessage());
		} catch (NamingException e) {
			throw e;
		}

		return item;
	}
	
	/**
	 * ノード一覧テーブル定義を取得取得します。<BR>
	 * リポジトリのノードビューにおいてテーブルの表示項目名（カラム名）を
	 * 取得します。このメソッドはクライアントから呼ばれることを想定しており、
	 * 引数にクライアントのロケールを要求します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param locale　クライアントのロケール
	 * @return TableColumnInfoの配列
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getNodeListTableDefine(Locale locale) {
		
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();

		tableDefine.add(NodeListTabelDefine.FACILITY_ID, 
				new TableColumnInfo(Messages.getString("facility.id", locale), TableColumnInfo.FACILITY_ID, 100, SWT.LEFT));
		tableDefine.add(NodeListTabelDefine.FACILITY_NAME, 
				new TableColumnInfo(Messages.getString("facility.name", locale), TableColumnInfo.FACILITY_NAME, 150, SWT.LEFT));
		tableDefine.add(NodeListTabelDefine.DESCRIPTION, 
				new TableColumnInfo(Messages.getString("description", locale), TableColumnInfo.NONE, 200, SWT.LEFT));

		return tableDefine;
	}
	
	/**
	 * 割当スコープ一覧テーブル定義を取得します。<BR>
	 * リポジトリの割り当てスコープにおいてテーブルの表示項目名（カラム名）を
	 * 取得します。このメソッドはクライアントから呼ばれることを想定しており、
	 * 引数にクライアントのロケールを要求します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param locale　クライアントのロケール
	 * @return TableColumnInfoの配列
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getNodeScopeTableDefine(Locale locale) {
		
		//テーブル定義配列
		ArrayList tableDefine = new ArrayList();

		tableDefine.add(NodeScopeTabelDefine.SCOPE, 
				new TableColumnInfo(Messages.getString("scope", locale), TableColumnInfo.NONE, 300, SWT.LEFT));

		return tableDefine;
	}
	
	/**
	 * 割当スコープ一覧を取得します。<BR>
	 * 割り当てスコープ一覧とは、あるノードが属しているスコープすべてを
	 * 一覧表示したものです。
	 * クライアントの割り当てスコープビューの表示データとなります。
	 * 戻り値はArrayListのArrayListで中のArrayListには"スコープ"が最上位からの
	 * スコープパス表記で（Stringで）格納されています。
	 * 外のArrayListには、その(ArrayListの）レコードが順に格納されています。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ノードのファシリティID
	 * @return オブジェクトの2次元配列
	 * @throws NamingException
	 * @throws FinderException
	 */
	public ArrayList getNodeScopeList(String facilityId) throws FinderException, NamingException {
		
		try {
			SelectFacility select = new SelectFacility();
			ArrayList list = select.getNodeScopeList(facilityId);
			
			return list;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getNodeScopeList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * スコープ一覧テーブル定義を取得します。<BR>
	 * リポジトリのスコープにおいてテーブルの表示項目名（カラム名）を
	 * 取得します。このメソッドはクライアントから呼ばれることを想定しており、
	 * 引数にクライアントのロケールを要求します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param locale クライアントのロケール
	 * @return TableColumnInfoの配列
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getScopeListTableDefine(Locale locale) {
		
		/** テーブル情報定義配列 */
		ArrayList tableDefine = new ArrayList();

		tableDefine.add(ScopeListTableDefine.FACILITY_ID, 
				new TableColumnInfo(Messages.getString("facility.id", locale), TableColumnInfo.FACILITY_ID, 100, SWT.LEFT));
		tableDefine.add(ScopeListTableDefine.FACILITY_NAME, 
				new TableColumnInfo(Messages.getString("facility.name", locale), TableColumnInfo.FACILITY_NAME, 150, SWT.LEFT));
		tableDefine.add(ScopeListTableDefine.DESCRIPTION, 
				new TableColumnInfo(Messages.getString("description", locale), TableColumnInfo.NONE, 200, SWT.LEFT));
	
		return tableDefine;
	}
	

	/**
	 * スコープ一覧を取得します。<BR>
	 * あるスコープを指定してその直下にあるスコープを取得します。<BR>
	 * このメソッドは引数としてそのスコープのファシリティIDを要求します。<BR>
	 * 戻り値はArrayListのArrayListで中のArrayListには子スコープの
	 * "ファシリティID"、"ファシリティ名"、"説明"がStringとして格納されています。<BR>
	 * 外のArrayListには、その(ArrayListの）レコードが順に格納されています。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId　スコープのファシリティID
	 * @return オブジェクトの2次元配列
	 * @throws NamingException
	 * @throws FinderException
	 */
	public ArrayList getScopeList(String parentFacilityId) throws FinderException, NamingException {
		
		try {
			SelectFacility select = new SelectFacility();
			ArrayList list = select.getScopeList(parentFacilityId);
			
			return list;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getScopeList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ノード一覧を取得します。<BR>
	 * リポジトリに登録されているすべてのノードを取得します。<BR>
	 * 戻り値はArrayListのArrayListで、中のArrayListには
	 * ノードの"ファシリティID"、"ファシリティ名"、"説明"がStringとして格納されています。
	 * 外のArrayListには、その(ArrayListの）レコードが順に格納されています。
	 * 
	 * @ejb.interface-method
     *     
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @return オブジェクトの2次元配列
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getNodeList() throws FinderException, NamingException {
	
		try {
			//ファシリティツリーを取得
			SelectFacility select = new SelectFacility();
			ArrayList list = select.getNodeList();

			return list;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getNodeList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ノード一覧を取得します。<BR>
	 * 
	 * あるファシリティIDの配下または直下のノード一覧を取得します。<BR>
	 * このメソッドでは、引数levelで直下または配下を制御します。<BR>
	 * 戻り値はArrayListのArrayListで、中のArrayListには
	 * ノードの"ファシリティID"、"ファシリティ名"、"説明"がStringとして格納されています。<BR>
	 * 外のArrayListには、その(ArrayListの）レコードが順に格納されています。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId
	 * @param level　取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return オブジェクトの2次元配列
	 * @throws NamingException
	 * @throws FinderException
	 */
	public ArrayList getNodeList(String parentFacilityId, int level) throws FinderException, NamingException {
		
		try {
			//ファシリティツリーを取得
			SelectFacility select = new SelectFacility();
			ArrayList list = select.getNodeList(parentFacilityId, level);
			return list;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getNodeList() -> " + e.getMessage()); 
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ノード一覧を取得します。<BR>
	 * 
	 * クライアントなどで検索条件行った場合に呼ばれ、該当するノード一覧を取得します。<BR>
	 * 引数はノードPropertyであり、"ファシリティID"、"ファシリティ名"、"説明"、
	 * "IPアドレス"、"OS名"、"OSリリース"、"管理者"、"連絡先"が１つ以上含まれており、
	 * その条件を元に該当するノードを戻り値とします。<BR>
	 * 戻り値はArrayListのArrayListで、中のArrayListには
	 * ノードの"ファシリティID"、"ファシリティ名"、"説明"がStringとして格納されています。<BR>
	 * 外のArrayListには、その(ArrayListの）レコードが順に格納されています。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param property　検索条件のプロパティ
	 * @return オブジェクトの2次元配列
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getNodeList(Property property) throws FinderException, NamingException {
		
		try {
			//ファシリティツリーを取得
			SelectFacility select = new SelectFacility();
			ArrayList list = select.getNodeList(property);
			return list;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getNodeList() -> " + e.getMessage()); 
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ファシリティIDリストを取得します。<BR>
	 * 
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（スコープ、ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 2.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ファシリティID
	 * @param level　取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return ファシリティIDの配列
	 * @throws NamingException
	 * @throws FinderException
	 */
	public ArrayList getFacilityIdList(String parentFacilityId, int level) throws FinderException, NamingException {
		
		try {
			//ファシリティツリーを取得
			SelectFacility select = new SelectFacility();
			ArrayList list = select.getFacilityIdList(parentFacilityId, level, false);
			return list;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getFacilityIdList() -> " + e.getMessage()); 
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ファシリティIDリストを取得します。<BR>
	 * 
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（スコープ、ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 引数のsortにtrueをセットした場合には、listがCollator.compare()にしたがってソートされる。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 2.1.0
	 * @since 2.1.0
	 * 
	 * @param parentFacilityId ファシリティID
	 * @param level 取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @param sort ソートするか？(する:true しない:false)
	 * @return ファシリティIDの配列
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getFacilityIdList(String parentFacilityId, int level, boolean sort) throws FinderException, NamingException {
		
		try {
			//ファシリティツリーを取得
			SelectFacility select = new SelectFacility();
			ArrayList list = select.getFacilityIdList(parentFacilityId, level, sort);
			return list;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getFacilityIdList() -> " + e.getMessage()); 
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * 
	 * 監視・ジョブ等の処理を実行する対象となる、ファシリティIDのリストを取得します。
	 * 引数で指定されたファシリティIDが、ノードかスコープによって、以下のようなリストを取得します。
	 * 
	 * ノードの場合
	 *   引数で指定されたfacilityIdが格納されたArrauList
	 *   ただし、管理対象（有効/無効フラグが真）の場合のみ
	 * 
	 * スコープの場合
	 *   配下に含まれるノードのファシリティIDが格納されたArrayList
	 *   ただし、管理対象（有効/無効フラグが真）のみ
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 3.0.0
	 * @since 3.0.0
	 * 
	 * 
	 * @param facilityId 処理を実行する対象となるファシリティID
	 * @return 有効なノードのリスト（有効なノードがひとつも含まれない場合は空のリスト）
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList<String> getExecTargetFacilityIdList(String facilityId) throws FinderException, NamingException{
		ArrayList<String> ret = null;
		
		if (this.isNode(facilityId)) {
			ret = new ArrayList<String>();
			
			FacilityLocal facility = FacilityUtil.getLocalHome().findByFacilityId(facilityId);
			// 対象がノードかつ、有効無効フラグが真なら、facilityIdをArrayListに追加して返す
			if (facility.getValid()){
				ret.add(facilityId);
			}
			
		} else {
			ret = this.getNodeFacilityIdList(facilityId, RepositoryControllerBean.ALL);
		}
		
		return ret;
	}
	
	
	/**
     * ノードのファシリティIDリストを取得します。<BR>
	 * 
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 2.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ファシリティID
	 * @param level  取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @return ファシリティIDの配列
	 * @throws NamingException
	 * @throws FinderException
	 */
	public ArrayList getNodeFacilityIdList(String parentFacilityId, int level) throws FinderException, NamingException {
		
		try {
			//ファシリティツリーを取得
			SelectFacility select = new SelectFacility();
			ArrayList list = select.getNodeFacilityIdList(parentFacilityId, level, false, new Boolean(true));
			return list;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getNodeFacilityIdList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ノードのファシリティIDリスト取得<BR>
	 * 引数に指定した親ファシリティIDの配下または直下のファシリティ（ノード）の
	 * リストを取得します。<BR>
	 * 戻り値は ファシリティID（String）のArrayList
	 * 引数のsortにtrueをセットした場合には、listがCollator.compare()にしたがってソートされる。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 2.1.0
	 * @since 2.1.0
	 * 
	 * @param parentFacilityId
	 * @param level   取得レベル 0:ALL(配下) 1:ONE_LEVEL（直下）
	 * @param sort sort ソートするか？(する:true しない:false)
	 * @return ファシリティIDの配列
	 * @throws NamingException
	 * @throws FinderException
	 */
	public ArrayList getNodeFacilityIdList(String parentFacilityId, int level, boolean sort, Boolean valid) throws FinderException, NamingException {
		
		try {
			//ファシリティツリーを取得
			SelectFacility select = new SelectFacility();
			ArrayList list = select.getNodeFacilityIdList(parentFacilityId, level, sort, valid);
			return list;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getNodeFacilityIdList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ノードのファシリティIDリストを取得します。<BR>
	 * 
	 * リポジトリにあるすべてのノードのリストを取得します。<BR>
	 * 戻り値は ファシリティID(String)のArrayList<BR>
	 * 
	 * getNodeList() との違いはこちらの戻り値はArrayListの２次元ではなく、
	 * 単純にファシリティID（String）のみのArrayList
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @return ファシリティIDの配列
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getNodeFacilityIdList() throws FinderException, NamingException {
		
		try {
			//ファシリティツリーを取得
			SelectFacility select = new SelectFacility();
			ArrayList list = select.getNodeFacilityIdList(false);
			return list;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getNodeFacilityIdList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ノードのファシリティIDリストを取得します。<BR>
	 * 
	 * リポジトリにあるすべてのノードのリストを取得します。<BR>
	 * 戻り値は ファシリティID(String)のArrayList
	 * 引数のsortにtrueをセットした場合には、listがCollator.compare()にしたがってソートされる。<BR>
	 * 
	 * getNodeList() との違いはこちらの戻り値はArrayListの２次元ではなく、
	 * 単純にファシリティID（String）のみのArrayList
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 2.1.0
	 * @since 2.1.0
	 * 
	 * @param sort sort ソートするか？(する:true しない:false)
	 * @return ファシリティIDの配列
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getNodeFacilityIdList(boolean sort) throws FinderException, NamingException {
		
		try {
			//ファシリティツリーを取得
			SelectFacility select = new SelectFacility();
			ArrayList list = select.getNodeFacilityIdList(sort);
			return list;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getNodeFacilityIdList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * デバイス情報リストを取得します。<BR>
	 * 指定したファシリティIDのデバイスのリストを取得します。<BR>
	 * 戻り値はcom.clustercontrol.performanceMGR.bean.DeviceDataのArrayList
     *
	 * @see com.clustercontrol.performanceMGR.bean.DeviceData
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 * 
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId
	 * @return DeviceDataの配列
	 * @throws NamingException
	 * @throws FinderException
	 */
	public ArrayList getDeviceList(String facilityId) throws FinderException, NamingException {
		
		try {
			SelectFacility select = new SelectFacility();
			ArrayList list = select.getDeviceList(facilityId);
			return list;			
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getDeviceFacilityIdList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ノード詳細情報を取得します。<BR>
	 * 
	 * ファシリティIDに対応するノードの詳細情報を取得します。<BR>
	 * 引数のattributesには、com.clustercontrol.repository.bean.FacilityAttributeConstantに
	 * 定義される属性を1つ以上定義します。<BR>
	 * 戻り値のHashMapには属性とその時の値セットが格納されます。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId　ファシリティID
	 * @param attributes  取得したい属性のリスト
	 * @return 属性名と値のHashMapとして返す
	 * @throws NamingException
	 * @throws FinderException
	 */
	public HashMap getNodeDetail(String facilityId, ArrayList attributes) throws FinderException, NamingException {
		
		try {
			SelectFacility select = new SelectFacility();
			HashMap map = select.getNodeDetail(facilityId, attributes);
			return map;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getNodeDetail() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 *　複数ノードのノード詳細を一度に取得します。<BR>
	 * 
	 * 指定したファシリティIDのリストに対応する属性の集合を取得します。<BR>
	 * 戻り値は ファシリティIDとその属性のHashMap
	 * HashMapからファシリティIDをkeyにして取り出したObjectはさらにHashMap
	 * そのHashMapの中身は引数に指定したattributesとそれに対応する値
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityIdList
	 * @param attributes
	 * @return ファシリティIDと属性値(属性名と値のHashMap)のHashMap
	 * @throws NamingException
	 * @throws FinderException
	 */
	public HashMap getNodeDetail(ArrayList facilityIdList, ArrayList attributes) throws FinderException, NamingException {
		
		try {
			SelectFacility select = new SelectFacility();
			HashMap map = select.getNodeDetail(facilityIdList, attributes);
			return map;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getNodeDetail() -> " + e.getMessage());	
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ノード詳細を設定します。<BR>
	 * 
	 * ファシリティIDに対応するノードに属性attributesを設定します。<BR>
	 * 引数のattributesには、com.clustercontrol.repository.bean.FacilityAttributeConstantで指定される
	 * 属性と設定したいの値のHashMap
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId　ファシリティID
	 * @param attributes 属性名と値のHashMap
	 * @throws FinderException
	 * @throws NamingException
	 */
	public void setNodeDetail(String facilityId, HashMap attributes) throws FinderException, NamingException {
		ModifyFacility modify = new ModifyFacility();
		modify.setNodeDetail(facilityId, attributes);
	}
	
	/**
	 * スコープを新規に追加します。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープの下にpropertyで指定されるスコープを
	 * 追加します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId
	 * @param property
	 * @throws CreateException
	 * @throws NamingException
	 * @throws RemoveException
	 */
	public void addScope(String parentFacilityId, Property property) throws RemoveException, CreateException, NamingException {
		ModifyFacility modify = new ModifyFacility();
		modify.addScope(parentFacilityId, property);
	}
	
	/**
	 * スコープを新規に追加します(表示順指定)。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープの下にpropertyで指定されるスコープを
	 * 追加します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId
	 * @param property
	 * @param sortOrder
	 * @throws CreateException
	 * @throws NamingException
	 * @throws RemoveException
	 */
	public void addScope(String parentFacilityId, Property property, int sortOrder) throws RemoveException, CreateException, NamingException {
		addScope(parentFacilityId,property,sortOrder,true);
	}
	
	/**
	 * スコープを新規に追加します(表示順指定、リポジトリ更新TOPIC未送信選択可能)。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープの下にpropertyで指定されるスコープを
	 * 追加します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId
	 * @param property
	 * @param sortOrder
	 * @throws CreateException
	 * @throws NamingException
	 * @throws RemoveException
	 */
	public void addScope(String parentFacilityId, Property property, int sortOrder, boolean sendFlg) throws RemoveException, CreateException, NamingException {
		ModifyFacility modify = new ModifyFacility();
		modify.addScope(parentFacilityId, property,sortOrder,sendFlg);
	}
	
	/**
	 * スコープの情報を変更します。<BR>
	 * 
	 * 引数propertyで指定した内容でスコープ情報を更新します。<BR>
	 * 引数propertyには、"ファシリティID"、"ファシリティ名"、"説明"（任意）を含める必要があります。
	 * propertyに含まれるファシリティIDに対応するスコープの情報が変更されます。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param property
	 * @throws NamingException
	 * @throws FinderException
	 */
	public void modifyScope(Property property) throws NamingException, FinderException {
		try {
			ModifyFacility modify = new ModifyFacility();
			modify.modifyScope(property);
		} catch (NamingException e) {
			throw e;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.modifyScope() -> " + e.getMessage());	
			throw e;
		}
	}
	
	/**
	 * ノードを新規に追加します。<BR>
	 * 
	 * 引数のpropertyにはノードのプロパティが含まれている必要があります。<BR>
	 * com.clustercontrol.repository.factory.NodePropertyを参照してください。<BR>
	 * またこのメソッドで組み込みスコープである"登録済みノード"スコープにも
	 * 割り当てが行われます。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param property 追加するノード情報のプロパティ
	 * @throws NamingException
	 * @throws CreateException
	 */
	public void addNode(Property property) throws NamingException, CreateException, FinderException {
		addNode(property,true);
	}
	
	/**
	 * ノードを新規に追加します。（リポジトリ更新TOPIC未送信選択可能）<BR>
	 * 
	 * 引数のpropertyにはノードのプロパティが含まれている必要があります。<BR>
	 * com.clustercontrol.repository.factory.NodePropertyを参照してください。<BR>
	 * またこのメソッドで組み込みスコープである"登録済みノード"スコープにも
	 * 割り当てが行われます。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param property 追加するノード情報のプロパティ
	 * @throws NamingException
	 * @throws CreateException
	 */
	public void addNode(Property property, boolean sendFlg) throws NamingException, CreateException, FinderException {
 		Principal pri = m_context.getCallerPrincipal();
 		
 		this.checkFacilityInfo(property);
			
		ModifyFacility modify = new ModifyFacility();
		modify.addNode(property, pri.getName(), sendFlg); //pri.getName()はメソッドの実行ユーザ
		
		//ノード追加されると必ず"登録済みノード"スコープに登録される
		//ファシリティID取得
		ArrayList values = PropertyUtil.getPropertyValue(property, NodeProperty.FACILITY_ID);
		String[] facilityId = new String[1];
		facilityId[0] = (String)values.get(0);
		
		//登録済みノードとしてスコープに割り当てる
		modify.assignNodeScope(FacilityTreeAttributeConstant.REGISTEREFD_SCOPE, facilityId, sendFlg);

	}
	
	/**
	 * ノードを変更します。<BR>
	 * 
	 * 引数のpropertyにはノードのプロパティが含まれている必要があります。<BR>
	 * com.clustercontrol.repository.factory.NodePropertyを参照してください。<BR>
	 * propertyには変更しないデータも含まれている必要あります。（そのまま戻す）
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 2.0.0
	 * @since 1.0.0
	 * 
	 * @param property　変更するノード情報のプロパティ
	 * @throws NamingException
	 * @throws FinderException
	 * @throws CreateException
	 */
	public void modifyNode(Property property) throws NamingException, FinderException, CreateException {
		try {
			Principal pri = m_context.getCallerPrincipal();
			
			this.checkFacilityInfo(property);
			
			ModifyFacility modify = new ModifyFacility();
			modify.modifyNode(property, pri.getName());
			
			//facilitytreeとして登録されている、有効/無効フラグを変更する
			modify.modifyFacilityTreeValid(property);
			
		} catch (NamingException e) {
			throw e;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.modifyNode() -> " + e.getMessage());	
			throw e;
		} catch (CreateException e) {
			throw e;
		}
	}
	
	/**
	 * ノード詳細情報を一括で変更します。<BR>
	 * 
	 * クライアントにあるスコープを指定して変更を行うためのメソッド
	 * 引数のpropertyには変更する属性のみを設定してください。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 2.0.0
	 * @since 1.0.0
	 * 
	 * @param property　変更するノード情報のプロパティ
	 * @param parentFacilityId　スコープのファシリティID
	 * @throws NamingException
	 * @throws FinderException
	 */
	public void modifyNode(Property property, String parentFacilityId) throws NamingException, FinderException {
		try {
			Principal pri = m_context.getCallerPrincipal();
			
	 		this.checkFacilityInfo(property);
			
			ModifyFacility modify = new ModifyFacility();
			modify.modifyAllNode(property, parentFacilityId, pri.getName());
		} catch (NamingException e) {
			throw e;
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.modifyNode() -> " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * ノード情報を削除します。<BR>
	 * 
	 * faciityIDで指定されたノードをリポジトリから削除します。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws UsedFacilityException 
	 */
	public void deleteNode(String facilityId) throws FinderException, RemoveException, NamingException, UsedFacilityException {
		try {
			DeleteFacility delete = new DeleteFacility();
			delete.deleteNode(facilityId);
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.deleteNode() -> " + e.getMessage());
			throw e;
		} catch (RemoveException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		} catch (UsedFacilityException e) {
			throw e;
		}
	}
	
	/**
	 * スコープ情報を削除します。<BR>
	 * 
	 * faciityIDで指定されたスコープをリポジトリから削除します。
	 * 
	  * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @throws FinderException
	 * @throws NamingException
	 * @throws RemoveException
	 * @throws UsedFacilityException 
	 */
	public void deleteScope(String facilityId) throws FinderException, NamingException, RemoveException, UsedFacilityException {
		try {
			DeleteFacility delete = new DeleteFacility();
			delete.deleteScope(facilityId);
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.deleteScope() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		} catch (RemoveException e) {
			throw e;
		} catch (UsedFacilityException e) {
			throw e;
		}
	}
	
	/**
	 * スコープへのノードの割り当てを行います。（リポジトリ更新TOPIC未送信選択可能）<BR>
	 * 
	 * parentFacilityIdで指定されるスコープにfacilityIdsで指定されるノード群を
	 * 割り当てます。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId　ノードを割り当てるスコープ
	 * @param facilityIds 割り当てさせるノード(群)
	 * @throws NamingException
	 * @throws CreateException
	 */
	public void assignNodeScope(String parentFacilityId, String[] facilityIds, boolean sendFlg) throws NamingException, CreateException, FinderException {
		try {
			ModifyFacility modify = new ModifyFacility();
			modify.assignNodeScope(parentFacilityId, facilityIds, sendFlg);
		} catch (NamingException e) {
			throw e;
		} catch (CreateException e) {
			throw e;
		} catch (FinderException e) {
			throw e;
		}
	}
	
	/**
	 * スコープへのノードの割り当てを行います。<BR>
	 * 
	 * parentFacilityIdで指定されるスコープにfacilityIdsで指定されるノード群を
	 * 割り当てます。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId　ノードを割り当てるスコープ
	 * @param facilityIds 割り当てさせるノード(群)
	 * @throws NamingException
	 * @throws CreateException
	 */
	public void assignNodeScope(String parentFacilityId, String[] facilityIds) throws NamingException, CreateException, FinderException {
		assignNodeScope(parentFacilityId,facilityIds,true);
	}
	
	/**
	 * ノードをスコープから削除します。（割り当てを解除します。）<BR>
	 * parentFacilityIdで指定されるスコープからfacilityIdsで指定されるノード群を
	 * 削除（割り当て解除）します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ノードを取り除くスコープ
	 * @param facilityIds 取り除かれるノード（群）
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 */
	public void releaseNodeScope(String parentFacilityId, String[] facilityIds) throws FinderException, NamingException, RemoveException {
		releaseNodeScope(parentFacilityId, facilityIds, true);
	}
	
	
	/**
	 * ノードをスコープから削除します。（割り当てを解除します。リポジトリ更新TOPIC未送信選択可能）<BR>
	 * parentFacilityIdで指定されるスコープからfacilityIdsで指定されるノード群を
	 * 削除（割り当て解除）します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 * 
	 * @version 3.1.0
	 * @since 1.0.0
	 * 
	 * @param parentFacilityId ノードを取り除くスコープ
	 * @param facilityIds 取り除かれるノード（群）
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 */
	public void releaseNodeScope(String parentFacilityId, String[] facilityIds, boolean sendFlg) throws FinderException, NamingException, RemoveException {
		try {
			DeleteFacility delete = new DeleteFacility();
			delete.releaseNodeScope(parentFacilityId, facilityIds,sendFlg);
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.releaseNodeScope() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		} catch (RemoveException e) {
			throw e;
		}
	}
	
	/**
	 * スコープ用プロパティ情報を取得します。<BR>
	 * 
	 * modeを変更することで、戻り値のpropertyを編集できるかが決まります。<BR>
	 * (ADD_MODEすべて可能、MODE_MODIFYファシリティ名、説明のみ可能）
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param mode 変更モード 0:ADD_MODE 1:MODE_MODIFY 2:MODE_SHOW
	 * @param locale クライアントのロケール
	 * @return スコープのプロパティ情報（ファシリティID、ファシリティ名、説明）
	 * @throws FinderException
	 * @throws NamingException
	 */
	public Property getScopeProperty(String facilityId, int mode, Locale locale) throws FinderException, NamingException {
		try {
			ScopeProperty scope = new ScopeProperty();
			return scope.getProperty(facilityId, mode, locale);
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getScopeProperty() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ノードの詳細プロパティを取得します。<BR>
	 * 
	 * faciliyIDで指定されるノードの詳細プロパティを取得します。<BR>
	 * modeを変更することで、戻り値のpropertyを編集できるかが決まります。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param mode 変更モード 0:ADD_MODE 1:MODE_MODIFY 2:MODE_SHOW
	 * @param locale クライアントのロケール
	 * @return ノード情報プロパティ
	 * @throws FinderException
	 * @throws NamingException
	 */
	public Property getNodeProperty(String facilityId, int mode, Locale locale) throws FinderException, NamingException {
		try {
			NodeProperty node = new NodeProperty();
			return node.getProperty(facilityId, mode, locale);
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getScopeProperty() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * 空のノードの詳細プロパティを取得します。<BR>
	 * 
	 * クライアントでノードの追加などを行うときの空propertyを取得します。<BR>
	 * （空のperpertyにも、属性名は設定されるので、localeをマネージャで制御するために
	 * 　マネージャに空のperpertyも取りに来ないといけません。）
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param locale クライアントのロケール
	 * @return ノード情報プロパティ
	 */
	public Property getNodeFilterProperty(Locale locale) {
		NodeFilterProperty node = new NodeFilterProperty();
		return node.getProperty(locale);
	}
	
	/**
	 * ファシリティパスを取得します。<BR>
	 * 
	 * ファシリティパス（スコープをトップから順に">"でつないだ文字列形式）を取得します。<BR>
	 * (例　○○スコープ>××システム>DBサーバ>ノードA)<BR>
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
     *     
     * @ejb.transaction
	 *     type="NotSupported"
	 *     
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId ファシリティID
	 * @param parentFacilityId 上位のファシリティID
	 * @return String ファシリティパス
	 * @throws FinderException
	 * @throws NamingException
	 */
	public String getFacilityPath(String facilityId, String parentFacilityId) throws FinderException, NamingException {
		try {
			SelectFacility select = new SelectFacility();
			return select.getFacilityPath(facilityId, parentFacilityId);
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getFacilityPath() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * ファシリティIDリストを取得します。<BR>
	 * 
	 *  引数のホスト名（ノード名）またはIPアドレスに対応するノードのファシリティIDのリストを
	 *  取得します。<BR>
	 *  戻り値はファシリティID(String)のArrayList
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 * 
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param hostName ホスト名（ノード名）
	 * @param ipAddress　IPアドレス(v4)
	 * @return ファシリティIDの配列
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getFacilityIdList(String hostName, String ipAddress) throws FinderException, NamingException {
		try {
			SelectFacility select = new SelectFacility();
			return select.getFacilityIdList(hostName, ipAddress);
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.getFacilityIdList() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
    /**
     * ファシリティがノードかどうかをチェックします。<BR>
     * 
     * ファシリティIDに対応するものがノードかチェックし、結果をbooleanで返します。
     * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @param facilityId　ファシリティID
	 * @return true：ノード　false:ノードではない（スコープ）
	 * @throws FinderException
	 * @throws NamingException
	 */
	public boolean isNode(String facilityId) throws FinderException, NamingException {
		try {
			SelectFacility select = new SelectFacility();
			return select.isNode(facilityId);
		} catch (FinderException e) {
			m_log.debug("RepositoryControllerBean.isNode() -> " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			throw e;
		}
	}
	
	/**
	 * セパレータ文字列を取得します。<BR>
	 * 
	 * セパレータ文字列はスコープパス表示の際のスコープを区切る文字列
	 * 
	 * @ejb.interface-method
	 * @generated
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 1.0.0
	 * @since 1.0.0
	 * 
	 * @return セパレータ文字列
	 */
	public String getSeparator() {
		return SelectFacility.SEPARATOR;
	}
	
    /**
     * SNMPを利用してノードの情報を取得します。<BR>
     * 
     * 
     * クライアントからSNMPで検出を行った際に呼び出されるメソッドです。<BR>
     * SNMPポーリングにより、ノード詳細プロパティをセットし、クライアントに返す。
     * 戻り値のpropertyにはcom.clustercontrol.repository.factory.NodePropertyの
     * 属性が含まれている。
     * 
     * @ejb.interface-method
     * 
     * @ejb.permission
     * role-name="RepositoryRead"
     * method-intf="Remote"
     *     
     * @ejb.transaction
     *     type="NotSupported"
     * 
     * @version 2.1.2
     * @since 2.1.2
     * 
     * @param pollingData ポーリング対象のIPアドレス、コミュニティ名、バージョン、ポート
     * @param mode 変更モード 0:ADD_MODE 1:MODE_MODIFY 2:MODE_SHOW　
     * @param locale クライアントのロケール
     * @return ノード情報のproperty
     */
    public Property  getNodePropertyBySNMP(Hashtable pollingData,int mode, Locale locale) throws FinderException, NamingException, UnknownHostException {
             return  SearchNodeBySNMP.searchNode(pollingData,mode,locale);
            
    }
    
    /**
     * ファシリティIDより対象ノードのノードプロパティ情報をDataオブジェクトで取得する。<BR>
     * 
     * @ejb.interface-method
     * 
     * @ejb.permission
     * role-name="RepositoryRead"
     * method-intf="Local"
     *     
     * @ejb.transaction
     *     type="NotSupported"
     * 
     * @param facilityId ノードのファシリティID
     * @return ノードのプロパティ情報一式
     * @throws NamingException
     * @throws FinderException
     * 
	 * @version 3.1.0
	 * @since 3.1.0
	 * 
     */
    public FacilityData  getNodeFacilityData(String facilityId) throws NamingException, FinderException{
    	SelectFacility select  = new SelectFacility();
    	return select.getNodeFacilityData(facilityId);   
    }
	
    
	/**
	 * 空のノードプロパティを取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 3.1.0
	 * @since 3.1.0
	 * 
	 * @param locale クライアントのロケール
	 * @return ノード情報プロパティ
	 */
	public Property getNodeProperty(int mode, Locale locale) throws FinderException, NamingException{
		NodeProperty node = new NodeProperty();
		return node.getProperty(mode, locale);
	}
	
	/**
	 * 条件のHashMapに該当するノードのファシリティIDのリストを返却する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 3.1.0
	 * @since 3.1.0
	 * 
	 * @return ArrayList<String>
	 */
	public ArrayList<String> findByCondition(HashMap<String, String> condition) throws FinderException, NamingException{
		SelectFacility selectFacility = new SelectFacility();
		return selectFacility.findByCondition(condition);
	}
	
	/**
	 * 空のスコーププロパティを返却する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 3.1.0
	 * @since 3.1.0
	 * 
	 * @return Property
	 */
	public Property getScopeProperty(int mode, Locale locale){
		ScopeProperty scopeProperty = new ScopeProperty();
		return scopeProperty.getProperty(mode, locale);
	}
	
	/**
	 * FacilityDataオブジェクトを設定した詰め替えたノードプロパティを返却する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryRead"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 * 
	 * @version 3.1.0
	 * @since 3.1.0
	 * 
	 * @return Property
	 */
	public Property getNodeProperty(FacilityData facilityData, Locale locale) throws CreateException,NamingException,FinderException{
		SelectFacility select = new SelectFacility();
		return select.getNodeProperty(facilityData, locale);
	}
	
	/**
	 * リポジトリ更新のTOPICを送信する。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="RepositoryWrite"
     *     method-intf="Remote"
	 *     
     * @ejb.transaction
	 *     type="NotSupported"
	 *     
	 * @version 3.1.0
	 * @since 3.1.0
	 * 
	 */
	public void sendRepositoryUpdateTopic(){
		SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
	}
	
	/**
	 * 
	 * ノード登録・変更時に入力情報が正しいかチェックするためのメソッド。
	 * 不正な場合、EJBExceptionをスローする。
	 * 
	 * チェック内容は、入力必須項目が入力されているかどうか。
	 * 文字数のチェックはしない。
	 * 
	 * @param property
	 * @throws EJBException
	 */
	private void checkFacilityInfo(Property property) throws EJBException {
		
		ArrayList values = null;
		
		//ファシリティID取得
		values = PropertyUtil.getPropertyValue(property, NodeProperty.FACILITY_ID);
		String facilityId = checkString(values.get(0));
		if (facilityId == null) {
			throw new EJBException("facilityId is null");
		}
		
		//ファシリティ名取得
		values = PropertyUtil.getPropertyValue(property, NodeProperty.FACILITY_NAME);
		String facilityName = checkString(values.get(0));
		if (facilityName == null) {
			throw new EJBException("facilityId=" + facilityId + " : facilityName is null");
		}
		
		//プラットフォームを取得
		values = PropertyUtil.getPropertyValue(property, NodeProperty.PLATFORM_FAMILY_NAME);
		String platform = checkString(values.get(0));
		if (platform == null) {
			throw new EJBException("facilityId=" + facilityId + " : platform is null");
		}
		
		//IPアドレスを取得
		values = PropertyUtil.getPropertyValue(property, NodeProperty.IP_ADDRESS_V4);
		String ip4Address = checkString(values.get(0));
		if (ip4Address == null) {
			throw new EJBException("facilityId=" + facilityId + " : ip4Address is null");
		}
		
		//ノード名を取得
		values = PropertyUtil.getPropertyValue(property, NodeProperty.NODE_NAME);
		String nodeName = checkString(values.get(0));
		if (nodeName == null) {
			throw new EJBException("facilityId=" + facilityId + " : nodeName is null");
		}
		
		
		// デバイス情報のバリデーションチェック
		ArrayList deviceList = PropertyUtil.getProperty(property, NodeProperty.DEVICE);
		for(int i = 0; i < deviceList.size(); i++){
			
			Property deviceProperty = (Property)deviceList.get(i);
			
			//デバイス名を取得
			String deviceName = checkString(PropertyUtil.getPropertyValue(deviceProperty, NodeProperty.DEVICE_NAME).get(0));
			//デバイス表示名を取得
			String displayName = checkString(PropertyUtil.getPropertyValue(deviceProperty, NodeProperty.DEVICE_DISPLAY_NAME).get(0));
			//デバイスINDEXを取得
			Integer deviceIndex = checkInteger(PropertyUtil.getPropertyValue(deviceProperty, NodeProperty.DEVICE_INDEX).get(0));
			//デバイスタイプを取得
			String deviceType = checkString(PropertyUtil.getPropertyValue(deviceProperty, NodeProperty.DEVICE_TYPE).get(0));
			
			if ((deviceName == null || deviceName.equals("")) && deviceIndex == null && 
					(deviceType == null || deviceType.equals("")) &&
					(displayName == null || displayName.equals(""))) {
					continue;
			}
			if (deviceName == null) {
				throw new EJBException("facilityId=" + facilityId + ", deviceName is null");
			}
			if (displayName == null) {
				throw new EJBException("facilityId=" + facilityId + ", deviceName=" + deviceName + " : deviceDisplayName is null");
			}
			if (deviceIndex == null) {
				throw new EJBException("facilityId=" + facilityId + ", deviceName=" + deviceName + " : deviceIndex is null");
			}
			if (deviceType == null) {
				throw new EJBException("facilityId=" + facilityId + ", deviceName=" + deviceName + " : deviceType is null");
			}
			
		}
		
		// ファイルシステム情報のバリデーションチェック			
		ArrayList filesystemList = PropertyUtil.getProperty(property, NodeProperty.FILE_SYSTEM);
		for(int i = 0; i < filesystemList.size(); i++){
			
			Property filesystemProperty = (Property)filesystemList.get(i);
			
			//ファイルシステム表示名を取得
			String filesystemDisplayName = checkString(PropertyUtil.getPropertyValue(filesystemProperty, NodeProperty.FILE_SYSTEM_DISPLAY_NAME).get(0));
			//ファイルシステムマウントポイントを取得
			String mountPoint = checkString(PropertyUtil.getPropertyValue(filesystemProperty, NodeProperty.FILE_SYSTEM_MOUNT_POINT).get(0));
			//ファイルシステムINDEXを取得
			Integer filesystemIndex = checkInteger(PropertyUtil.getPropertyValue(filesystemProperty, NodeProperty.FILE_SYSTEM_INDEX).get(0));
			
			if ((filesystemDisplayName == null || filesystemDisplayName.equals("")) &&
					(mountPoint == null || mountPoint.equals("")) && filesystemIndex == null) {
				continue;
			}
			if (filesystemDisplayName == null) {
				throw new EJBException("facilityId=" + facilityId + " : fileSystemDisplayName is null");
			}
			if (mountPoint == null) {
				throw new EJBException("facilityId=" + facilityId + ", fileSystemDisplayName=" + filesystemDisplayName + " : fileSystemMountPoint is null");
			}
			if (filesystemIndex == null) {
				throw new EJBException("facilityId=" + facilityId + ", fileSystemDisplayName=" + filesystemDisplayName + " : fileSystemIndex is null");
			}
			
		}
	}
	/**
	 * Integerのインスタンスかチェックし、違っていればnullを返す
	 * 
	 * @param value
	 * @return
	 */
    private Integer checkInteger(Object value){
		if(value instanceof Integer){
			return (Integer)value;
		}
		else{
			return null;
		}
	}
	
	/**
	 * Stringのインスタンスかチェックし、違っていればnullを返す
	 * 
	 * @param value
	 * @return
	 */
	private String checkString(Object value){
		if(value instanceof String){
			if(((String)value).compareTo("") == 0){
				return null;
			}
			else{
				return (String)value;
			}
		}
		else{
			return null;
		}
	}
	
}