/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.performance.ejb.bmp.CollectorBean;
import com.clustercontrol.performance.ejb.bmp.CollectorDAO;
import com.clustercontrol.performance.ejb.bmp.CollectorPK;


/**
 * 収集設定情報のDBアクセスを行うクラス
 * 
 * EJBのEntutyBeanのCollectorDAOの実装クラス
 * 
 * @version 2.0.0
 * @since 1.0
 */
public class CollectorDAOImpl implements CollectorDAO {
	//ログ出力
	protected static Log m_log = LogFactory.getLog( CollectorDAOImpl.class );
	
	/**
	 * コンストラクター
	 */
	public CollectorDAOImpl() {
		m_log.trace("コンストラクター in CollectorDAO");
	}

	/**
	 * 初期化を行う
	 */
	public void init() {
		
		m_log.trace("init CollectorDAO");
	}
	
	/**
	 * EntityBeanで必要となるloadメソッド
	 * メモリ上のEntityBeanをDB上のデータに更新する
	 * @param pk 収集設定のPK
	 * @param ejb 同期を行うEntityBean
	 */
	public void load(CollectorPK pk, CollectorBean ejb)
	throws EJBException {	
		m_log.debug("load() start : " + pk.getCollectorID());
	
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "SELECT * FROM CC_COLLECTOR_PROFILE WHERE COLLECTORID = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1,pk.collectorID );
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setCollectorID(res.getString("COLLECTORID"));
				ejb.setCollectorType(res.getShort("COLLECTOR_TYPE"));
				ejb.setLabel(res.getString("LABEL"));
				
				Timestamp startTimestamp= res.getTimestamp("START_DATE");    			
				if(startTimestamp != null){
					ejb.setStartDate(new Date(startTimestamp.getTime()));
				} else {
					ejb.setStartDate(null);
				}
				Timestamp stopTimestamp= res.getTimestamp("STOP_DATE");
				if(stopTimestamp != null){
					ejb.setStopDate(new Date(stopTimestamp.getTime()));
				} else {
					ejb.setStopDate(null);
				}
				
				// ファシリティIDを設定
				ejb.setFacilityID(res.getString("TARGET_FACILITYID"));
				
				ejb.setScopePath(res.getString("SCOPE_PATH"));
				ejb.setIntervalSec(res.getInt("INTERVAL_SEC"));
				ejb.setPeriodMin(res.getInt("PERIOD_MIN"));
				ejb.setPresavePeriod(res.getInt("PRESAVE_PERIOD"));
			}else{	
				String msg = "Collector data is not found.";
				m_log.error("load() error : " + pk.getCollectorID() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.getCollectorID() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
//			コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.getCollectorID() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}

		// リアルタイム収集および閾値監視収集の場合は、リポジトリから、
		// 実績収集の場合は、DBからツリー情報を取得し、ScopeTreeオブジェクトを設定する。
		ejb.updateScopeTree();
		
		// 収集項目をDBから取得し設定
		ejb.setCollectorItems(pk.collectorID);
		
		m_log.debug("load() end   : " + pk.getCollectorID());		
	}
	
	/**
	 * EntityBeanで必要となるstoreメソッド
	 * メモリ上のEntityBeandeDB上のデータを更新する
	 * @param ejb 保存するEntityBean
	 */
	public void store(CollectorBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getCollectorID());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "UPDATE CC_COLLECTOR_PROFILE SET " +
			"COLLECTOR_TYPE = ?," +
			"LABEL = ?," +
			"START_DATE = ?," +
			"STOP_DATE = ?," +
			"TARGET_FACILITYID = ?," +
			"SCOPE_PATH = ?," +
			"INTERVAL_SEC = ?," +
			"PERIOD_MIN = ?," +
			"PRESAVE_PERIOD = ? " +
			"WHERE COLLECTORID = ?";
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setShort(1, (short)ejb.getCollectorType());
			stmt.setString(2, ejb.getLabel());
			
			Date startDate = ejb.getStartDate();
			if(startDate != null){
				stmt.setTimestamp(3, new Timestamp(startDate.getTime()));	
			} else {
				stmt.setTimestamp(3, null);				
			}
			
			Date stopDate = ejb.getStopDate();
			if(stopDate != null){
				stmt.setTimestamp(4, new Timestamp(stopDate.getTime()));	
			} else {
				stmt.setTimestamp(4, null);				
			}
			
			stmt.setString(5, ejb.getFacilityID());
			stmt.setString(6, ejb.getScopePath());
			stmt.setInt(7, ejb.getIntervalSec());
			stmt.setInt(8, ejb.getPeriodMin());
			stmt.setInt(9, ejb.getPresavePeriod());
			stmt.setString(10, ejb.getCollectorID());
			
			int row = stmt.executeUpdate();
			
			if (row != 1) {	
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getCollectorID() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getCollectorID() + " SQLException", e);
			throw new EJBException(e.getMessage());
		}finally{
//			コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getCollectorID() + " SQLException", e1);
				throw new EJBException(e1.getMessage());
			}
		}
		
		m_log.debug("store() end   : " + ejb.getCollectorID());
	}	
	
	/**
	 * EntityBeanで必要となるremoveメソッド
	 * DBのデータを削除する
	 * @param pk　削除を行うpkオブジェクト
	 */
	public void remove(CollectorPK pk) throws EJBException {
		m_log.debug("remove()  start: " + pk.getCollectorID());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE  FROM  CC_COLLECTOR_PROFILE WHERE COLLECTORID = ? ";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1,pk.collectorID );
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			
		}
		m_log.debug("remove() end   : " + pk.getCollectorID());
	}
	
	/**
	 * EntityBeanで必要となるcreateメソッド
	 * DBに存在しないEntityBeanを生成する際にDBにインサートする
	 * @param ejb インサートするEntityBeanオブジェクト
	 * @return DB内でレコードを一意識別するpkオブジェクト
	 */
	public CollectorPK create(CollectorBean ejb) throws CreateException,
	EJBException ,DuplicateKeyException{
		
		m_log.debug("create() start : " + ejb.getCollectorID());

		Connection conn = null;
		PreparedStatement stmt = null;
		CollectorPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "INSERT INTO CC_COLLECTOR_PROFILE " +
			"(COLLECTORID, " +
			"COLLECTOR_TYPE, " +
			"LABEL," +
			"START_DATE," +
			"STOP_DATE," +
			"TARGET_FACILITYID," +
			"SCOPE_PATH," +
			"INTERVAL_SEC," +
			"PERIOD_MIN," +
			"PRESAVE_PERIOD)" +
			"values (?,?,?,?,?,?,?,?,?,?);";
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1,ejb.getCollectorID());
			stmt.setShort(2,(short)ejb.getCollectorType());
			stmt.setString(3,ejb.getLabel());
			if(ejb.getStartDate() != null){
				stmt.setTimestamp(4, new Timestamp(ejb.getStartDate().getTime()));
			} else {
				stmt.setTimestamp(4, null);    			
			}
			if(ejb.getStopDate() != null){
				stmt.setTimestamp(5, new Timestamp(ejb.getStopDate().getTime()));
			} else {
				stmt.setTimestamp(5, null);    			
			}
			stmt.setString(6,  ejb.getFacilityID());
			stmt.setString(7,  ejb.getScopePath());
			stmt.setInt(8,  ejb.getIntervalSec());
			stmt.setInt(9, ejb.getPeriodMin());
			stmt.setInt(10, ejb.getPresavePeriod());
			
			int row =	stmt.executeUpdate();
			pk = new CollectorPK(ejb.getCollectorID());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getCollectorID() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getCollectorID() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getCollectorID() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end   : " + ejb.getCollectorID());
		return pk;
	}
	
	/**
	 *  EntityBeanで必要となるfindByPrimaryKeyメソッド
	 * クライアントから要求があったときにそのオブジェクトが既に生成されいるかをチェックする
	 * 存在していれば、一意識別子であるpkオブジェクトを返す。
	 * @param pk 一意識別子のpkオブジェクト
	 * @return pk 一意識別子のpkオブジェクト
	 * @throws FinderException
	 */
	public CollectorPK findByPrimaryKey(CollectorPK pk) throws FinderException {		
		m_log.debug("findbyPK() start : " + pk.getCollectorID());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "SELECT collectorid FROM CC_COLLECTOR_PROFILE WHERE COLLECTORID = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1,pk.collectorID );
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.getCollectorID() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			
			m_log.debug("findbyPK() end : " + pk.getCollectorID());
		}
	}
	
	/**
	 * DBに存在する全ての収集設定を取得する
	 * @return 一意識別子のpkのコレクション
	 */
	public Collection findAll() throws EJBException {
		
		m_log.debug("findAll() start : ");
		
		ArrayList<CollectorPK> ret = new ArrayList<CollectorPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT COLLECTORID FROM CC_COLLECTOR_PROFILE ORDER BY START_DATE";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				CollectorPK pk=  new CollectorPK(res.getString("COLLECTORID"));
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}
	
	/**
	 * 収集種別に該当するEntityBeanを検索するメソッド
	 * @param status 収集種別
	 * @return 一意識別子のpkのコレクション
	 */
	public Collection findByCollectorType(int type) throws EJBException {
		m_log.debug("findCollectorType() start : ");
		ArrayList<CollectorPK> ret = new ArrayList<CollectorPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = 
				"SELECT COLLECTORID FROM CC_COLLECTOR_PROFILE WHERE COLLECTOR_TYPE = ? ORDER BY START_DATE";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			stmt.setShort(1,(short)type);
			res = stmt.executeQuery();
			
			while(res.next()) {
				CollectorPK pk = new CollectorPK(res.getString("COLLECTORID"));	
				ret.add(pk);
			}
		} catch (SQLException e) {
			m_log.error("findByCollectorType() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByCollectorType() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findbyCollectorType() end : ");
		return ret;
	}
	
	/**
	 * 収集期限が無期限の収集を検索する
	 * 
	 * @return 収集期限が無期限である収集のPKのコレクション
	 * @throws EJBException
	 */
	public Collection findByPresave() throws EJBException {
		
		m_log.debug("findByPresave() start : ");
		
		ArrayList<CollectorPK> ret = new ArrayList<CollectorPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "SELECT COLLECTORID FROM CC_COLLECTOR_PROFILE " +
						  "WHERE COLLECTOR_TYPE = 2 " +
						  "AND PERIOD_MIN=-1 " +
						  "AND STOP_DATE is null";
			
			//COLLECTOR_TYPE=2 :実績収集 (0：不明、1：リアルタイム、3：閾値監視)
			//PERIOD_MIN=-1        :期限を指定しない収集である。
			//STOP_DATE is null   :収集中である。
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				CollectorPK pk = new CollectorPK(res.getString("COLLECTORID"));
				m_log.debug("unlimit collector pk : " + pk.toString());
				ret.add(pk);
			}
		} catch (SQLException e) {
			m_log.error("findByCollectorType() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByCollectorType() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findbyCollectorType() end : ");
		return ret;
	}

	/**
	 * 収集種別とファシリティIDに該当するEntityBeanを検索するメソッド
	 * 
	 * @param type 収集種別
	 * @param facilityId ファシリティID
	 * @return 一意識別子のpkのコレクション
	 * 
	 * @see com.clustercontrol.performance.ejb.bmp.CollectorDAO#findByCollectorTypeAndFacilityId(int, java.lang.String)
	 * @see com.clustercontrol.performance.bean.CollectorType
	 */
	public Collection findByCollectorTypeAndFacilityId(int type, String facilityId) throws FinderException {
		m_log.debug("findByCollectorTypeAndFacilityId() start : ");
		ArrayList<CollectorPK> ret = new ArrayList<CollectorPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = 
				"SELECT COLLECTORID FROM CC_COLLECTOR_PROFILE WHERE COLLECTOR_TYPE = ? AND TARGET_FACILITYID = ? ORDER BY START_DATE";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			stmt.setShort(1,(short)type);
			stmt.setString(2, facilityId);
			res = stmt.executeQuery();
			
			while(res.next()) {
				CollectorPK pk = new CollectorPK(res.getString("COLLECTORID"));	
				ret.add(pk);
			}
		} catch (SQLException e) {
			m_log.error("findByCollectorTypeAndFacilityId() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByCollectorTypeAndFacilityId() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByCollectorTypeAndFacilityId() end : ");
		return ret;
	}
}
