/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.dialog;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityInfo;
import com.clustercontrol.bean.FacilityPath;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.ProcessConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.composite.CalendarIdListComposite;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ScopeTreeDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.composite.NotifyInfoComposite;
import com.clustercontrol.syslogng.action.LogManager;
import com.clustercontrol.syslogng.bean.LogFilterInfo;
import com.clustercontrol.util.Messages;

/**
 * syslog-ng[フィルタの作成・変更]ダイアログクラス<BR>
 * 
 * @version 3.0.0
 * @since 1.0.0
 */
public class LogCreateDialog extends CommonDialog {

    /** 入力値を保持するオブジェクト。 */
    private LogFilterInfo inputData = null;

    /** 入力値の正当性を保持するオブジェクト。 */
    private ValidateResult validateResult = null;

    /** 変更対象のフィルタID。 */
    private String logId = null;

    /** 変更対象の順序。 */
    private int order = Integer.MAX_VALUE;

    /** 選択されたスコープのファシリティID。 */
    private String facilityId = null;

    /** 説明。 */
    private Text textDescription = null;

    /** スコープ テキストボックス。 */
    private Text textScope = null;

    /** パターンマッチ表現 テキストボックス。 */
    private Text textPattern = null;

    /** 一致したら処理しない ラジオボタン。 */
    private Button radioNotProcessing = null;

    /** 一致したら処理する ラジオボタン。*/
    private Button radioProcessing = null;
    
    /** カレンダID一覧 コンポジット。 */
    private CalendarIdListComposite calendarId = null;
    
    /** 通知ID一覧 コンポジット。 */
    private NotifyInfoComposite notifyInfo = null;

    /** 重要度 ラベル。 */
    private Label labelPriority = null;
    /** 
     * 重要度 コンボボックス。
     * @see com.clustercontrol.bean.PriorityConstant
     */
    private Combo comboPriority = null;

    /** 監視項目ID ラベル。 */
    private Label labelMonitorId = null;
    /** 監視項目ID テキストボックス。 */
    private Text textMonitorId = null;

    /** メッセージID ラベル。 */
    private Label labelMessageId = null;
    /** メッセージID テキストボックス。 */
    private Text textMessageId = null;

    /** メッセージ ラベル。 */
    private Label labelLogMessage = null;
    /** メッセージ テキストボックス。 */
    private Text textLogMessage = null;
    
    
    /**
     * この設定を有効にする チェックボックス。
     * @see com.clustercontrol.bean.ValidConstant
     */
    private Button confirmValid = null;
    
    /** 作成日時。 */
    private Timestamp regDate = null;

    /** 変更フラグ */
    private boolean modify = false;

    /**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
    public LogCreateDialog(Shell parent) {
        super(parent);
    }

    /**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param logId 変更するフィルタ情報のフィルタID
	 */
    public LogCreateDialog(Shell parent, String logId) {
        super(parent);

        this.logId = logId;
    }

    /**
     * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.syslogng.action.LogManager#get(String)
	 * @see #setInputData(LogFilterInfo)
     */
    protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.syslog-ng.create.modefy"));

        // 変数として利用されるラベル
        Label label = null;
        // 変数として利用されるグリッドデータ
        GridData gridData = null;

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        parent.setLayout(layout);
        
        /*
         * 監視項目ID
         */
        // ラベル
        this.labelMonitorId = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 4;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.labelMonitorId.setLayoutData(gridData);
        this.labelMonitorId.setText(Messages.getString("monitor.id") + " : ");
        // テキスト
        this.textMonitorId = new Text(parent, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = 9;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textMonitorId.setLayoutData(gridData);
        this.textMonitorId.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_64));

        /*
         * 説明
         */
        // ラベル
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 4;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("description") + " : ");
        // テキスト
        this.textDescription = new Text(parent, SWT.BORDER | SWT.LEFT);
        gridData = new GridData();
        gridData.horizontalSpan = 9;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textDescription.setLayoutData(gridData);
        this.textDescription.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_256));
        
        /*
         * 監視条件グループ
         */
        // ラベル
        this.labelMonitorId = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.labelMonitorId.setLayoutData(gridData);
        this.labelMonitorId.setText(Messages.getString("monitor.rule") + " : ");

        // グループ
        Group monitorRuleGroup1 = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        monitorRuleGroup1.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        monitorRuleGroup1.setLayoutData(gridData);
        
        /*
         * カレンダ
         */
		this.calendarId = new CalendarIdListComposite(monitorRuleGroup1, SWT.NONE, true);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		calendarId.setLayoutData(gridData);
		
		/*
         * チェック条件グループ
         */
		//ラベル
        label = new Label(monitorRuleGroup1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("check.rule") + " : ");
        
        // グループ
        Group groupCheckRule = new Group(monitorRuleGroup1, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 15;
        groupCheckRule.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupCheckRule.setLayoutData(gridData);

        /*
         * スコープ
         */
        // ラベル
        label = new Label(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 5;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("scope") + " : ");
        // テキスト
        this.textScope = new Text(groupCheckRule, SWT.BORDER | SWT.LEFT | SWT.READ_ONLY);
        gridData = new GridData();
        gridData.horizontalSpan = 8;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textScope.setLayoutData(gridData);
        // 参照ボタン
        Button button = new Button(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);
        button.setText(Messages.getString("refer"));
        button.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                ScopeTreeDialog dialog = new ScopeTreeDialog(getParentShell(),false,true);
                if (dialog.open() == IDialogConstants.OK_ID) {
                    FacilityTreeItem item = dialog.getSelectItem();
                    FacilityInfo info = item.getData();
                    facilityId = info.getFacilityId();
                    if (info.getType() == FacilityConstant.TYPE_NODE) {
                        textScope.setText(info.getFacilityName());
                    } else {
                        FacilityPath path = new FacilityPath(
                                ClusterControlPlugin.getDefault()
                                        .getSeparator());
                        textScope.setText(path.getPath(item));
                    }
                }
            }
        });

        /*
         * パターンマッチ表現
         */
        // ラベル
        label = new Label(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 5;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("pattern.matching.expression") + " : ");
        // テキスト
        this.textPattern = new Text(groupCheckRule, SWT.BORDER | SWT.LEFT);
        gridData = new GridData();
        gridData.horizontalSpan = 10;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textPattern.setLayoutData(gridData);
        this.textPattern.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_1024));

        /*
         * 出力の条件
         */
        /*
         * 処理しない
         */
        this.radioNotProcessing = new Button(parent, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.radioNotProcessing.setLayoutData(gridData);
        this.radioNotProcessing.setText(Messages
                .getString("don't.process.if.matched"));

        /*
         * 処理する
         */
        this.radioProcessing = new Button(parent, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.radioProcessing.setLayoutData(gridData);
        this.radioProcessing.setText(Messages.getString("process.if.matched"));
        this.radioProcessing.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                setEnabledForOutputCondition(radioProcessing.getSelection());
            }
        });

        /*
         * グループ
         */
        // 大グループ
        Group group1 = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        group1.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        group1.setLayoutData(gridData);
        
        /*
         * 通知ID
         * アプリケーションID
         */
		// 通知情報
		this.notifyInfo = new NotifyInfoComposite(group1, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		notifyInfo.setLayoutData(gridData);

        /*
         * 重要度
         */
        // ラベル
        this.labelPriority = new Label(group1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 5;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.labelPriority.setLayoutData(gridData);
        this.labelPriority.setText(Messages.getString("priority") + " : ");
        // コンボボックス
        this.comboPriority = new Combo(group1, SWT.DROP_DOWN | SWT.READ_ONLY);
        gridData = new GridData();
        gridData.horizontalSpan = 3;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.comboPriority.setLayoutData(gridData);
        this.comboPriority.add(PriorityConstant.STRING_CRITICAL);
        this.comboPriority.add(PriorityConstant.STRING_WARNING);
        this.comboPriority.add(PriorityConstant.STRING_INFO);
        this.comboPriority.add(PriorityConstant.STRING_UNKNOWN);
        this.comboPriority.setText(PriorityConstant.STRING_CRITICAL);
        // 空白
        label = new Label(group1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 7;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);

        /*
         * メッセージID
         */
        // ラベル
        this.labelMessageId = new Label(group1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 5;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.labelMessageId.setLayoutData(gridData);
        this.labelMessageId.setText(Messages.getString("message.id") + " : ");
        // テキスト
        this.textMessageId = new Text(group1, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = 10;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textMessageId.setLayoutData(gridData);
        this.textMessageId.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_64));

        /*
         * ログメッセージ
         */
        // ラベル
        this.labelLogMessage = new Label(group1, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 5;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.labelLogMessage.setLayoutData(gridData);
        this.labelLogMessage.setText(Messages.getString("message") + " : ");
        // テキスト
        this.textLogMessage = new Text(group1, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = 10;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.textLogMessage.setLayoutData(gridData);
        this.textLogMessage.addVerifyListener(
        		new StringVerifyListener(DataRangeConstant.VARCHAR_256));
        
        
        /*
         * 有効／無効
         */
        this.confirmValid = new Button(parent, SWT.CHECK);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.confirmValid.setLayoutData(gridData);
        this.confirmValid
                .setText(Messages.getString("setting.valid.confirmed"));

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 15;
        line.setLayoutData(gridData);

        // サイズを最適化
        // グリッドレイアウトを用いた場合、こうしないと横幅が画面いっぱいになります。
        shell.pack();
        shell.setSize(new Point(550, shell.getSize().y));

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);

        // ログIDが指定されている場合、その情報を初期表示する。
        LogFilterInfo log = LogManager.getInstance().get(this.logId);
        Collection<NotifyRelationInfo> modifyNotify = null;

        if (log == null) {
            log = new LogFilterInfo();
            modify = false;
        }
        else{
        	modify = true;
        	
        	// 通知関連情報のディープコピーを作成
        	if(log.getNotifyId() != null) {
	        	modifyNotify = new ArrayList<NotifyRelationInfo>();
	        	Iterator<NotifyRelationInfo> itr = log.getNotifyId().iterator();
	        	while(itr.hasNext()){
	        		NotifyRelationInfo info = itr.next();
	        		modifyNotify.add(new NotifyRelationInfo(
	        				info.getNotifyGroupId(), 
	        				info.getNotifyId(), 
	        				info.getNotifyType(), 
	        				info.getNotifyFlg()));
	        	}
        	}
        }
        this.setInputData(log, modifyNotify);
    }

    /**
	 * 入力値を保持したフィルタ情報を返します。
	 * 
	 * @return フィルタ情報
	 */
    public LogFilterInfo getInputData() {
        return this.inputData;
    }

    /**
	 * 引数で指定されたフィルタ情報の値を、各項目に設定します。
	 * 
	 * @param log 設定値として用いるフィルタ情報
	 * 
     * @since 1.0.0
	 * 
	 * @see #setEnabledForOutputCondition(boolean)
     */
    protected void setInputData(LogFilterInfo log, Collection<NotifyRelationInfo> modifyNotify) {

        this.inputData = log;

        // 各項目に反映
        this.logId = log.getLogId();
        this.order = log.getOrderNo();
        this.facilityId = log.getFacilityId();

        if (log.getMonitorId() != null) {
            this.textMonitorId.setText(log.getMonitorId());
        }
        if (log.getDescription() != null) {
            this.textDescription.setText(log.getDescription());
        }
        if (log.getCalendarId() != null) {
            this.calendarId.setText(log.getCalendarId());
        }
        if (log.getScope() != null) {
            this.textScope.setText(log.getScope());
        }
        if (log.getPattern() != null) {
            this.textPattern.setText(log.getPattern());
        }
        if (log.getProcessType() == ProcessConstant.TYPE_YES) {
            this.radioProcessing.setSelection(true);
        } else {
            this.radioNotProcessing.setSelection(true);
        }
  
        //通知情報の設定
//        if(log.getNotifyId() != null)
//        	this.notifyInfo.setNotify(log.getNotifyId());
        if(modifyNotify != null && modifyNotify.size() != 0){
        	this.notifyInfo.setNotify(modifyNotify);
        }
        
        //アプリケーション
        if (log.getApplication() != null) {
        	this.notifyInfo.setApplication(log.getApplication());
        }
        
        this.comboPriority.setText(PriorityConstant.typeToString(log.getPriority()));
        if (log.getMessageId() != null) {
            this.textMessageId.setText(log.getMessageId());
        }
        if (log.getMessage() != null) {
            this.textLogMessage.setText(log.getMessage());
        }
        
        this.regDate = log.getRegDate();
        
		// 有効／無効
		if (log.getValid() == ValidConstant.TYPE_VALID) {
			this.confirmValid.setSelection(true);
		}

        // 入力制御
        this.setEnabledForOutputCondition(this.radioProcessing.getSelection());
    }

    /**
	 * 入力値を設定したフィルタ情報を返します。<BR>
	 * 入力値チェックを行い、不正な場合は<code>null</code>を返します。
	 * 
	 * @return フィルタ情報
	 * 
	 * @see #setValidateResult(String, String)
	 */
    protected LogFilterInfo createInputData() {
        LogFilterInfo log = new LogFilterInfo();

        log.setLogId(this.logId);
        log.setOrderNo(this.order);
        log.setFacilityId(this.facilityId);

        if (this.textMonitorId.getText() != null
                && !"".equals((this.textMonitorId.getText()).trim())) {
            log.setMonitorId(this.textMonitorId.getText());
        } else {
            if (this.radioProcessing.getSelection()) {
                this.setValidateResult(Messages.getString("message.hinemos.1"),
                        Messages.getString("message.monitor.1"));
                return null;
            }
        }
        if (this.textDescription.getText() != null
                && !"".equals((this.textDescription.getText()).trim())) {
            log.setDescription(this.textDescription.getText());
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),
                    Messages.getString("message.monitor.26"));
            return null;
        }
        
        // カレンダID
        String calendarId = this.calendarId.getText();
        if (calendarId != null && !"".equals(calendarId.trim())) {
        	log.setCalendarId(calendarId);
        }

        if (this.textScope.getText() != null
                && !"".equals((this.textScope.getText()).trim())) {
            log.setScope(this.textScope.getText());
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),
                    Messages.getString("message.hinemos.3"));
            return null;
        }

        if (this.textPattern.getText() != null
                && !"".equals((this.textPattern.getText()).trim())) {
        	
        	try{
    			Pattern.compile(this.textPattern.getText());
    			log.setPattern(this.textPattern.getText());
    		}
    		catch(PatternSyntaxException e){
    			this.setValidateResult(Messages.getString("message.hinemos.1"),
                        Messages.getString("message.monitor.55"));
                return null;
    		}
            
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),
                    Messages.getString("message.monitor.27"));
            return null;
        }

        if (this.radioProcessing.getSelection()) {
            log.setProcessType(ProcessConstant.TYPE_YES);
        } else {
            log.setProcessType(ProcessConstant.TYPE_NO);
        }
        String priorityText = this.comboPriority.getText();
        log.setPriority(PriorityConstant.stringToType(priorityText));
        
        
        
        // 通知関連情報の設定
        boolean isSetNotifyId = true;
		// 通知グループIDの設定
		log.setNotifyGroupId(NotifyRelationInfo.createNotifyGroupId(
				HinemosModuleConstant.MONITOR_SYSLOGNG, log.getMonitorId()));
		validateResult = notifyInfo.createInputData(log);
		if (this.radioProcessing.getSelection() && validateResult != null) {
			if(validateResult.getID() == null){	// 通知ID警告用出力
				if(!displayQuestion(validateResult)){
					validateResult = null;
					return null;
				}
				else {
					isSetNotifyId = false;
				}
			}
			else{	// アプリケーション未入力チェック
				return null;
			}
		} 
		
		// メッセージIDの設定
        if (this.textMessageId.getText() != null
                && !"".equals((this.textMessageId.getText()).trim())) {
            log.setMessageId(this.textMessageId.getText());
        } else {
            if (this.radioProcessing.getSelection() && isSetNotifyId) {
            	this.setValidateResult(Messages.getString("message.hinemos.1"),
                        Messages.getString("message.monitor.28"));
            	return null;
            }
        }
        
        // メッセージの設定
        if (this.textLogMessage.getText() != null
                && !"".equals((this.textLogMessage.getText()).trim())) {
            log.setMessage(this.textLogMessage.getText());
        } else {
            if (this.radioProcessing.getSelection() && isSetNotifyId) {
            	this.setValidateResult(Messages.getString("message.hinemos.1"),
                        Messages.getString("message.monitor.29"));
            	return null;
            }
        }
        
     
        // 有効
        if (this.confirmValid.getSelection()) {
            log.setValid(ValidConstant.TYPE_VALID);
        } else {
            log.setValid(ValidConstant.TYPE_INVALID);
        }
        
        // 登録日時
        log.setRegDate(this.regDate);
        
        return log;
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    protected ValidateResult validate() {
        // 入力値生成
        this.inputData = this.createInputData();

        if (this.inputData != null) {
            return super.validate();
        } else {
            return validateResult;
        }
    }
    
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    protected boolean action() {
    	boolean result = false;
    	
    	LogFilterInfo info = this.inputData;
    	if(info != null){
            if(!this.modify){
            	// 作成の場合
            	result = LogManager.getInstance().add(info);
            }
            else{
            	// 変更の場合
            	result = LogManager.getInstance().modify(info);
            }
    	}
    	
    	return result;
    }

    /**
     * ＯＫボタンのテキストを返します。
     * 
     * @return ＯＫボタンのテキスト
     */
    protected String getOkButtonText() {
        return Messages.getString("ok");
    }

    /**
     * キャンセルボタンのテキストを返します。
     * 
     * @return キャンセルボタンのテキスト
     */
    protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }

    /**
	 * 処理情報のコンポジットの選択可/不可を設定します。
	 *
	 * @param enable 選択可の場合、<code> true </code>
	 */
    private void setEnabledForOutputCondition(boolean enable) {

    	/*
        this.labelPriority.setEnabled(enable);
        this.labelMessageId.setEnabled(enable);
        this.labelLogMessage.setEnabled(enable);
        */
        this.comboPriority.setEnabled(enable);
        this.notifyInfo.setEnabled(enable);
        this.textMessageId.setEnabled(enable);
        this.textLogMessage.setEnabled(enable);
    }
    
    /**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 */
    private void setValidateResult(String id, String message) {

        this.validateResult = new ValidateResult();
        this.validateResult.setValid(false);
        this.validateResult.setID(id);
        this.validateResult.setMessage(message);
    }
    
}