/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.forward;

import java.text.MessageFormat;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * syslogメッセージを解析するクラス<BR>
 * syslogメッセージを、{@link com.clustercontrol.syslogng.forward.MessageInfo}クラスに変換します。
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class MessageParser {

	/** syslogメッセージの解析用フォーマット。 */
	static private MessageFormat m_msgfmt = new MessageFormat(
			"<{0}>{1,date,MMM dd HH:mm:ss} {2} {3}", Locale.ENGLISH);
	static private MessageFormat m_datefmt = new MessageFormat(
			"{0,date,yyyyMMMddHH:mm:ss}", Locale.ENGLISH);
	
	/** syslogメッセージのインデックス：接頭辞（未使用）。 */
	@SuppressWarnings("unused")
	private final static int INDEX_PRI = 0;

	/** syslogメッセージのインデックス：日付。 */
	private final static int INDEX_DATE = 1;

	/** syslogメッセージのインデックス：ホスト名。 */
	private final static int INDEX_HOST = 2;

	/** syslogメッセージのインデックス：メッセージ。 */
	private final static int INDEX_MSG = 3;
	
	
	/** syslogメッセージ(split)のインデックス：接頭辞＋月。 */
	private final static int INDEX_SPLIT_PRI = 0;
	
	/** syslogメッセージ(split)のインデックス：日。 */
	private final static int INDEX_SPLIT_DAY = 1;
	
	/** syslogメッセージ(split)のインデックス：時刻。 */
	private final static int INDEX_SPLIT_TIME = 2;

	/** ログ出力のインスタンス。 */
	static Log log = LogFactory.getLog(MessageParser.class);

	/**
	 * 引数で指定されたメッセージのsyslogメッセージ情報を取得します。<BR>
	 * 引数で指定されたメッセージを解析し、
	 * 解析した情報を保持するsyslogメッセージ情報を生成し、返します。
	 * 
	 * @param msg syslogメッセージ
	 * @return syslogメッセージ情報
	 */
	static public MessageInfo parse(String msg) {

		//MessageFormatクラスによるメッセージ解析
		Object[] parseArgs = null;
		Object[] parseDate = null;
		Date messageDate = null;
		
		//msgを空白文字でsplitする
		String[] splitMsgs = msg.split("\\s+");
		
		// 接頭辞+月から月のみを切り出す。
		String msgMonth = (splitMsgs[INDEX_SPLIT_PRI].split(">"))[1];
		String msgDay = splitMsgs[INDEX_SPLIT_DAY];
		String msgTime = splitMsgs[INDEX_SPLIT_TIME];
		
		// 日が一桁だった場合、0を付与する
		// 例：1日⇒01日
		if (msgDay.length() == 1) {
			msgDay = "0"+msgDay;
		}
		
		// 現在の時刻を取得
		Calendar nowDate = Calendar.getInstance();
		
		// 現在の年を取得し、日付を再生成する
		Integer nowYear = nowDate.get(Calendar.YEAR);
		String dateString = (nowYear.toString() +
				msgMonth +
				msgDay +
				msgTime);
		
		try {
			// 年を付与した日付のStringクラスをDateクラスにパースし、
			// さらにDate型にキャストする
			parseDate = m_datefmt.parse(dateString);
			messageDate = (Date)parseDate[0];
			
			// ホスト名・メッセージ用をログから切り出す
			parseArgs = m_msgfmt.parse(msg);
			
		} catch (ParseException e) {
			log.error(msg, e);
			return null;
		}
		if (log.isDebugEnabled()) {
			log.debug("parsed msg");
			log.debug(messageDate);
			for (int i = INDEX_HOST; i < parseArgs.length; i++) {
				log.debug(parseArgs[i]);
			}
		}

		//解析結果を　MessageInfo　にセット
		MessageInfo logMessage = new MessageInfo();
		logMessage.setHostName((String) parseArgs[INDEX_HOST]);
		logMessage.setMessage((String) parseArgs[INDEX_MSG]);
		logMessage.setGenerationDate(setYear(messageDate));

		return logMessage;
	}

	/**
	 * 引数で指定された日付に年を付加して設定します。<BR>
	 * syslogメッセージの日付（月日）から、年を決定して日付を返します。
	 * 
	 * @param date 日付文字列
	 * @return	年を含めた日付
	 */
	static private Date setYear(Date date) {

		//現在時の取得
		Calendar nowDate = Calendar.getInstance();
		//戻り値用のカレンダー生成
		Calendar dataDate = Calendar.getInstance();

		//メッセージからparseしたDateを設定
		dataDate.setTime(date);

		int monthNow = nowDate.get(Calendar.MONTH);
		int monthdat = dataDate.get(Calendar.MONTH);
		int year = nowDate.get(Calendar.YEAR);

		//メッセージの月が１２月で、現在が1月の場合、去年のデータとする
		if (monthdat == Calendar.DECEMBER && monthNow == Calendar.JANUARY) {
			year -= 1;
		}
		dataDate.set(Calendar.YEAR, year);

		log.debug(dataDate.getTime().toString());
		return dataDate.getTime();
	}
}
