/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.logagent;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.UnsupportedEncodingException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.logagent.util.LoggerSyslog;
import com.clustercontrol.util.Messages;

/**
 * ログ転送スレッドクラス<BR>
 * 
 * @version 3.0.0
 * @since 2.1.0
 */
public class TransferLogThread extends Thread {

	private final static String UNCHANGED_STATS_PERIOD = "unchanged.stats.period";

	private final static String FILE_MAX_SIZE = "file.max.size";
	
	private final static String LOG_ENCODING = "log.file.encoding";

	public static final String MESSAGE_ID_INFO = "001";

	public static final String MESSAGE_ID_WARNING = "002";

	public static final String MESSAGE_ID_CRITICAL = "003";

	public static final String MESSAGE_ID_UNKNOWN = "004";

	public static final String HINEMOS_LOG_AGENT = "hinemos_log_agent";

	protected TransferLogManager m_transferLogManager;

	/** ログファイル名 */
	protected String m_filePath;

	/** 動作間隔（秒） */
	protected int m_runInterval;

	/** 最初にファイルをチェック */
	protected int m_existenceFlg = YesNoConstant.TYPE_NO;

	/** ファイル変更チェック期間設定（秒） */
	protected int m_unchangedStatsPeriod = 0;

	/** 上限ファイルサイズ設定（byte） */
	protected long m_fileMaxSize = 0L;

	/** ログ転送停止フラグ */
	protected boolean m_stopFlg = false;

	// Syslog転送用ロガー
	protected LoggerSyslog m_syslog = null;

	// ロガー
	static Log log = LogFactory.getLog(TransferLogThread.class);
	
	// ログファイルのエンコーディング
	private String m_logEncoding = "UTF-8";
	
	/**
	 * コンストラクタ
	 * 
	 * @param queue
	 * @param props
	 * @param path
	 *            転送対象ログファイル
	 * @param interval
	 *            動作間隔（秒）
	 * @param flg
	 *            最初にファイルをチェック
	 */
	public TransferLogThread(TransferLogManager transferLogManager,
			Properties props, String path, int interval, int flg) {

		m_transferLogManager = transferLogManager;
		m_filePath = path;
		m_runInterval = interval * 1000;
		m_existenceFlg = flg;

		// ファイル変更チェック期間（秒）
		String sleepInterval = props.getProperty(UNCHANGED_STATS_PERIOD, "5");
		try {
			this.m_unchangedStatsPeriod = Integer.parseInt(sleepInterval) * 1000;
		} catch (NumberFormatException e) {
			log.error("TransferLogManager() : " + UNCHANGED_STATS_PERIOD, e);
		}

		// 上限ファイルサイズ（byte）
		String fileMaxSize = props.getProperty(FILE_MAX_SIZE, "2147483648");
		try {
			this.m_fileMaxSize = Long.parseLong(fileMaxSize);
		} catch (NumberFormatException e) {
			log.error("TransferLogManager() : " + FILE_MAX_SIZE, e);
		}

		// ログファイルのエンコーディングを設定
		m_logEncoding =  props.getProperty(LOG_ENCODING);
		
		// プロパティファイルに設定がない場合
		if(m_logEncoding == null){
			m_logEncoding = System.getProperty("file.encoding"); 
		}
		
		m_syslog = new LoggerSyslog(props);
	}

	/**
	 * スレッドの動作メソッド<BR>
	 * 
	 */
	/*
	 * (非 Javadoc)
	 * 
	 * @see java.lang.Thread#run()
	 */
	@Override
	public void run() {
		log.info("monitor start.  logfile : " + m_filePath 
				+ "  interval : " + m_runInterval + "s"
				+ "  logfile encoding : " + m_logEncoding
				+ "  syslog encoding : " + System.getProperty("file.encoding"));

		long filesize = 0;
		long tmp_filesize = 0;
		long n_unchanged_stats = 0; // ファイルチェック時に、ファイルに変更がなかった回数
		// String carryOver="";
		byte[] cbuf = new byte[1024]; // ファイルからの読み出し分を保持するバッファ
		byte[] carryOverBuf = new byte[0]; // 文字化け対策用に繰り越すバッファ
		byte[] appendedBuf = null; // 繰り越し分にファイルからの読み出し分を追加したバッファ

		File name = new File(this.m_filePath); // 監視対象ログファイル

		// ファイルオープン
		RandomAccessFile fr = openFile(name, true, false);
		if (fr == null) {
			return;
		}

		String logPrefix = HINEMOS_LOG_AGENT + "(" + this.m_filePath + "):";

		while (true) {

			// スリープ
			try {
				Thread.sleep(this.m_runInterval);
			} catch (InterruptedException e) {
			}

			if (this.m_stopFlg) {
				closeFile(fr);
				return;
			}

			try {
				tmp_filesize = fr.length(); // 現在監視しているファイのサイズを取得・・・（１）

				if (filesize == tmp_filesize) {

					// ファイルサイズがm_unchangedStatsPeriod秒間以上変わらなかったら、ファイル切り替わりチェック
					if ((++n_unchanged_stats * this.m_runInterval) >= this.m_unchangedStatsPeriod) {

						// ログローテートされているかチェックする
						if (fr.length() != name.length()) { // 現在監視しているファイルのサイズと本来監視すべきファイルのサイズが異なっている
							if (tmp_filesize == fr.length()) { // 現在監視しているファイルのサイズが（１）で取得した値と同じである
								log.info(this.m_filePath + ":ファイルが切替りました");
								closeFile(fr);

								// ファイルオープン
								fr = openFile(name, false, false);
								if (fr == null) {
									return;
								}

								filesize = 0;
								// carryOver = "";
								carryOverBuf = new byte[0];
							}

						}
						n_unchanged_stats = 0;
					}
					continue;
				}

				n_unchanged_stats = 0;

				if (filesize < tmp_filesize) {
					// デバッグログ
					if (log.isDebugEnabled()) {
						log
								.debug("run() : " + m_filePath + " filesize "
										+ filesize + "    tmp_filesize "
										+ tmp_filesize);
					}

					// 加分読み込み
					int read;
					// StringBuffer sb = new StringBuffer(carryOver);
					StringBuffer sb = new StringBuffer();
					while ((read = fr.read(cbuf)) != -1) {

						// //
						// UTF-8を含むログで文字化けが発生するため修正
						//
						// 最後の改行コードを検索し、最後の改行コード以降は次回に繰越て処理する。
						// carryOverStartには改行コードの次のコードのインデックスが格納される。
						int carryOverStart = read;
						boolean returnCode = false;
						for (int i = read - 1; i >= 0; i--) {
							if (cbuf[i] == '\n' || cbuf[i] == '\r') {
								carryOverStart = i + 1;
								returnCode = true;
								break;
							}
						}

						// デバッグログ
						if (log.isDebugEnabled()) {
							log.debug("run() : " + m_filePath + " read " + read
									+ "    carryOverStart " + carryOverStart);
						}

						// 今回出力処理する分のバッファを作成
						// 前回の繰越分と今回ファイルから読み出したデータのうち
						// 最後の改行までのデータをマージする。
						appendedBuf = new byte[carryOverBuf.length
								+ carryOverStart];
						// for(int i = 0; i<carryOverBuf.length; i++){
						// appendedBuf[i] = carryOverBuf[i];
						// }
						ByteBuffer.wrap(carryOverBuf).get(appendedBuf, 0,
								carryOverBuf.length);

						// int j=0;
						// for(int i = carryOverBuf.length;
						// i<appendedBuf.length; i++){
						// appendedBuf[i] = cbuf[j++];
						// }
						ByteBuffer.wrap(cbuf).get(appendedBuf,
								carryOverBuf.length, carryOverStart);

						// デバッグログ
						if (log.isDebugEnabled()) {
							log
									.debug("run() : " + m_filePath
											+ " appendedBuf size "
											+ appendedBuf.length);
						}

						// 改行コードが含まれない場合
						if (!returnCode) {
							// 今回ファイルから読み込んだものを含めて全て次回へ繰り越す。
							// 出力処理は実施しない。
							log.debug("run() : " + m_filePath
									+ " return code is not exist");
							carryOverBuf = new byte[appendedBuf.length];
							ByteBuffer.wrap(appendedBuf).get(carryOverBuf);
							continue;
						}

						// 繰越データ用バッファを作成
						carryOverBuf = new byte[(read - carryOverStart)];
						try {
							// 最後が改行コード以降にデータがある場合は、次回の処理にまわす
							if (read > carryOverStart) {
								// デバッグログ
								if (log.isDebugEnabled()) {
									log.debug("run() : " + m_filePath
											+ " carryOverBuf size "
											+ carryOverBuf.length);
								}
								ByteBuffer.wrap(cbuf, carryOverStart,
										carryOverBuf.length).get(carryOverBuf);
							}
						} catch (Exception e) {
							// デバッグログ
							log.error("run() : " + e.getMessage(), e);
						}
						// //

						try {
//							sb.append(new String(appendedBuf, 0,
//									appendedBuf.length, System
//											.getProperty("file.encoding")));
							sb.append(new String(appendedBuf, 0,
									appendedBuf.length, m_logEncoding));							
						} catch (UnsupportedEncodingException e) {
							log.error("run() : " + m_filePath + " "
									+ e.getMessage());
						}
					}

					String tmpString = sb.toString();
					String[] result = tmpString.split("\\n");
					// デバッグログ
					if (log.isDebugEnabled()) {
						log.debug("run() : " + m_filePath + " " + sb);
						log.debug("run() : " + m_filePath + " size "
								+ sb.length());
						log.debug("run() : " + m_filePath + " result size "
								+ result.length);
					}

					// 読み込み文字列のサイズが0でない場合は処理する
					if (sb.length() != 0) {
						for (int x = 0; x < result.length; x++) {
							m_syslog.log(logPrefix + result[x]);
						}
					}
					// 改行で終わらない場合の繰越
					// if( tmpString.endsWith("\n") ){
					// m_syslog.log(logPrefix + result[result.length-1]);
					// carryOver = "";
					// }else{
					// carryOver = result[result.length-1];
					// }

					filesize = tmp_filesize;

				} else if (filesize > tmp_filesize) {
					// 切詰
					log.info(this.m_filePath + ":切詰められました");
					fr.seek(0);
					filesize = 0;
					// carryOverBu = "";
					carryOverBuf = new byte[0];
				}

			} catch (IOException e) {
				log.error("run() : " + e.getMessage());
				Object[] args = { this.m_filePath };
				sendMessage(PriorityConstant.TYPE_WARNING, 
						Messages.getString("log.agent"), 
						MESSAGE_ID_WARNING,
						Messages.getString("message.log.agent.4"), 
						Messages.getString("message.log.agent.1", args) + "\n" + e.getMessage());

				// エラーが発生したのでファイルクローズし再オープン

				closeFile(fr);

				// ファイルオープン
				fr = openFile(name, false, true);
				if (fr == null) {
					return;
				}

				filesize = 0;
				// carryOver = "";
				carryOverBuf = new byte[0];
			}
		}
	}

	/**
	 * ログ転送を停止します。<BR>
	 * 
	 */
	public void requestStop() {
		this.m_stopFlg = true;
		this.interrupt();
		
		log.info("monitor stop.   logfile : " + m_filePath 
				+ "  interval : " + m_runInterval + "s"
				+ "  logfile encoding : " + m_logEncoding
				+ "  syslog encoding : " + System.getProperty("file.encoding"));
	}

	/**
	 * 監視管理情報へ通知
	 * 
	 * @param priority
	 *            重要度
	 * @param app
	 *            アプリケーション
	 * @param msgId
	 *            メッセージID
	 * @param msg
	 *            メッセージ
	 * @param msgOrg
	 *            オリジナルメッセージ
	 */
	private void sendMessage(int priority, String app, String msgId,
			String msg, String msgOrg) {

		m_transferLogManager.sendMessage(m_filePath, priority, app, msgId, msg,
				msgOrg);

	}

	/**
	 * 転送対象ログファイルクローズ
	 * 
	 */
	private void closeFile(RandomAccessFile fr) {

		if (fr != null) {
			try {
				fr.close();
			} catch (IOException e) {
				log.debug("run() : " + e.getMessage());
			}
		}
	}

	/**
	 * 転送対象ログファイルオープン
	 * 
	 */
	private RandomAccessFile openFile(File name, boolean init, boolean isOnErr) {

		boolean err = isOnErr;
		RandomAccessFile fr = null;
		// ファイルオープン
		while (true) {

			try {

				fr = new RandomAccessFile(name, "r");

				long filesize = fr.length();
				if (filesize > this.m_fileMaxSize) {
					// ファイルサイズが大きい場合、監視管理へ通知
					Object[] args1 = { this.m_filePath };
					Object[] args2 = { filesize };
					sendMessage(PriorityConstant.TYPE_INFO, 
							Messages.getString("log.agent"), 
							MESSAGE_ID_INFO,
							Messages.getString("message.log.agent.3"), 
							Messages.getString("message.log.agent.1", args1) + ", "	
								+ Messages.getString("message.log.agent.5",args2));
				}

				// ファイルポインタの設定
				if (init) {
					fr.seek(filesize);
				}

				return fr;

			} catch (FileNotFoundException e) {
				if (init && !err) {
					if (this.m_existenceFlg == YesNoConstant.TYPE_YES) {
						// 最初にファイルをチェックする場合、監視管理へ通知
						Object[] args = { this.m_filePath };
						sendMessage(PriorityConstant.TYPE_INFO,
								Messages.getString("log.agent"),
								MESSAGE_ID_INFO, 
								Messages.getString("message.log.agent.2"),
								Messages.getString("message.log.agent.1", args));
					}
					err = true;
				}

			} catch (SecurityException e) {
				if (!err) {
					// 監視管理へ通知
					Object[] args = { this.m_filePath };
					sendMessage(PriorityConstant.TYPE_WARNING, 
							Messages.getString("log.agent"),
							MESSAGE_ID_WARNING, 
							Messages.getString("message.log.agent.4"),
							Messages.getString("message.log.agent.1", args) + "\n" + e.getMessage());
					err = true;
				}
			} catch (IOException e) {
				if (!err) {
					// 監視管理へ通知
					Object[] args = { this.m_filePath };
					sendMessage(PriorityConstant.TYPE_INFO, 
							Messages.getString("log.agent"), 
							MESSAGE_ID_INFO,
							Messages.getString("message.log.agent.4"), 
							Messages.getString("message.log.agent.1", args));
					err = true;
				}

				closeFile(fr);
			}

			// 初期処理は初回のみとする
			// 初回オープンに失敗後、ファイルをオープンすると、ファイルのリードポインタが
			// ファイルのオープン時に設定されてしまうため
			init = false;

			// スリープ
			try {
				Thread.sleep(this.m_runInterval);
			} catch (InterruptedException ie) {
			}

			if (this.m_stopFlg) {
				return null;
			}

		}

	}

	public void setCondition(int chkExistence, int interval) {
		m_existenceFlg = chkExistence;
		m_runInterval = interval * 1000;
	}

	public static void main(String[] args) {

	}

}