/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.snmppoller.bean;

import java.io.Serializable;
import java.net.InetAddress;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.snmppoller.SnmpConfig;

/**
 * ポーリングの設定を保持するクラス
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class PollingConfig implements Serializable {
	protected static Log m_log = LogFactory.getLog( PollingConfig.class ); 
	
	private static final String[] cronExpressionSet = { 
		"*/1 * * * * ? *",
		"*/5 * * * * ? *",
		"*/15 * * * * ? *",
		"*/30 * * * * ? *",
		"0 */1 * * * ? *",
		"0 */5 * * * ? *",
		"0 */15 * * * ? *",
		"0 */30 * * * ? *",
		"0 0 */1 * * ? *"
	};
	
	// 収集設定可能なインターバル
	private static final int[] intervalSet
	= {     1,     5,    15,    30,    60,   300,   900,  1800,  3600 };
	private boolean[] m_intervalFlag
	= { false, false, false, false, false, false, false, false, false };
	
	public static int parseInterval(String cronExpression){
		for(int i=0; i < cronExpressionSet.length; i++){
			if(cronExpressionSet[i].equals(cronExpression)){
				return intervalSet[i];
			}
		}
		
		// 認められていない収集間隔である
		return -1;
	}
	
	public static String parseCronExpression(int interval){
		for(int i=0; i < intervalSet.length; i++){
			if(intervalSet[i] == interval){
				return cronExpressionSet[i];
			}
		}
		
		// 認められていない収集間隔である
		return null;
	}
	
	// SNMPポーリングの設定を行う
	private SnmpConfig m_snmpConfig;
	
	// ノード情報の最終更新日時
	private long m_repositoryModifyTime;
	
	// コンストラクタ
	public PollingConfig(
			InetAddress address, 
			int port,
			int version, 
			String community,
			int retries,
			int timeout,
			long modifyTime){
		m_snmpConfig = new SnmpConfig(address);
		m_snmpConfig.setPort(port);
		m_snmpConfig.setVersion(version);
		m_snmpConfig.setComunity(community);
		m_snmpConfig.setRetries(retries);
		m_snmpConfig.setTimeout(timeout);
		m_repositoryModifyTime = modifyTime;
	}
	
	/**
	 * 指定の収集間隔を追加します。
	 * @param interval
	 * @return 収集間隔を追加することで収集間隔に変更が生じる場合はtrueを返します。
	 */
	public boolean addInterval(int interval){
		// 指定の収集間隔のフラグをたてる
		for(int i=intervalSet.length-1; i >= 0 ; i--){
			if(intervalSet[i] <= interval){
				// 既存の設定を調べる
				if(m_intervalFlag[i]){
					// 既に設定されていたので何もしない
					return false;
				} else {
					// 指定の収集間隔を追加したことで最小収集間隔に変化が生じるか調べる
					int minInterval = getMinInterval();
					if(minInterval == 0 || intervalSet[i] < minInterval){
						// 指定の収集間隔の設定をオンにする
						m_intervalFlag[i] = true;
						// 最小収集間隔に変更が生じるためtrueを返す
						return true;
					} else {
						// 指定の収集間隔の設定をオンにする
						m_intervalFlag[i] = true;
						// 最小収集間隔に変更が生じないためfalseを返す
						return false;
					}
				}
			}
		}
		return false;
	}
	
	/**
	 * 指定のインターバルの設定を削除します。
	 * @param interval
	 */
	public void removeInterval(int interval){
		for(int i=0; i < intervalSet.length; i++){
			if(intervalSet[i] == interval){
				m_intervalFlag[i] = false;
			}
		}
	}
	
	/**
	 * 最小の収集間隔を返します。
	 * 収集間隔がひとつも設定されていない場合は0を返します。
	 * @return 最小の収集間隔
	 */
	public int getMinInterval(){
		for(int i=0; i < m_intervalFlag.length; i++){
			if(m_intervalFlag[i]){
				return intervalSet[i];
			}
		}
		
		// 収集間隔がひとつも設定されていない場合は0を返す。
		return 0;
	}
	
	/**
	 * 指定可能な収集間隔の配列を返します。
	 * @return 指定可能な収集間隔の配列
	 */
	public static int[] getAvailableIntervals(){
		return intervalSet;
	}
	
	/**
	 * SNMPポーリングの設定を返します。
	 * @return SNMPポーリング設定
	 */
	public SnmpConfig getSnmpConfig() {
		return m_snmpConfig;
	}

	/**
	 * ノードの最終更新時刻を返します。
	 * @return 最終更新時刻
	 */
	public long getRepositoryModifyTime() {
		return m_repositoryModifyTime;
	}
}
