/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.port.factory;

import java.util.ArrayList;
import java.util.HashMap;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.jms.JMSException;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.PortRunCountConstant;
import com.clustercontrol.bean.PortRunIntervalConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.monitor.run.bean.MonitorNumericValueInfo;
import com.clustercontrol.monitor.run.factory.RunMonitorNumericValueType;
import com.clustercontrol.port.ejb.entity.MonitorPortInfoLocal;
import com.clustercontrol.port.ejb.entity.MonitorPortInfoPK;
import com.clustercontrol.port.ejb.entity.MonitorPortInfoUtil;
import com.clustercontrol.port.ejb.entity.MonitorProtocolMasterLocal;
import com.clustercontrol.port.ejb.entity.MonitorProtocolMasterUtil;
import com.clustercontrol.port.util.ReachAddressProtocol;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.util.Messages;

/**
 * port監視クラス
 * 
 * @version 2.4.0
 * @since 2.4.0
 */
public class RunMonitorPort extends RunMonitorNumericValueType {

	protected static Log m_log = LogFactory.getLog(RunMonitorPort.class);

	public static final String MESSAGE_ID_INFO = "001"; // 通知

	public static final String MESSAGE_ID_WARNING = "002"; // 警告

	public static final String MESSAGE_ID_CRITICAL = "003"; // 危険

	public static final String MESSAGE_ID_UNKNOWN = "004"; // 不明

	// public static boolean fportEnable;

	/** port監視情報 */
	protected static final ArrayList<String> m_attributeList = new ArrayList<String>();
	static {
		m_attributeList.add(FacilityAttributeConstant.NODENAME);
		m_attributeList.add(FacilityAttributeConstant.IPNETWORKNUMBER);
		m_attributeList.add(FacilityAttributeConstant.IPNETWORKNUMBERV6);
	}

	/** port監視情報 */
	protected MonitorPortInfoLocal m_port = null;

	/** ポート番号 */
	protected int m_portNo;

	/** 試行回数 */
	protected int m_runCount = PortRunCountConstant.TYPE_COUNT_01;

	/** 試行間隔（ミリ秒） */
	protected int m_runInterval = PortRunIntervalConstant.TYPE_SEC_01;

	/** タイムアウト（ミリ秒） */
	protected int m_portTimeout;

	/** サービスID */
	protected String m_serviceId;

	/** サービスプロトコル情報 */
	protected MonitorProtocolMasterLocal m_protocol = null;

	/** メッセージ */
	protected String m_message = null;

	/** 不明メッセージ */
	protected String m_unKnownMessage = null;

	/** オリジナルメッセージ */
	protected String m_messageOrg = null;

	/** 応答時間（ミリ秒） */
	protected long m_response = 0;

	// port実行
	ReachAddressProtocol m_reachability = null;

	/**
	 * 
	 * コンストラクタ
	 * 
	 */
	public RunMonitorPort() throws NamingException, JMSException,
			CreateException {
		super();
	}

	/**
	 * port数を取得
	 * 
	 * @param facilityId
	 *            ファシリティID
	 * @return 値取得に成功した場合、true
	 */
	@Override
	@SuppressWarnings("unchecked")
	public boolean collect(String facilityId) {

		m_nodeDate = m_now.getTime();
		m_message = "";
		m_messageOrg = "";
		m_response = 0;

		if (m_reachability == null) {
			try {
				// m_serviceIdを基にDBよりクラス名を得る
				m_protocol = MonitorProtocolMasterUtil.getLocalHome()
						.findByPrimaryKey(m_serviceId);

				String protocolClassName = "";
				// クラス名の取得
				if (m_protocol.getClassName() != null)
					protocolClassName = m_protocol.getClassName();

				// そのクラスのインスタンスを生成する
				Class cls = Class.forName(protocolClassName);
				m_reachability = (ReachAddressProtocol) cls.newInstance();
				m_reachability.setPortNo(m_portNo);
				m_reachability.setSentCount(m_runCount);
				m_reachability.setSentInterval(m_runInterval);
				m_reachability.setTimeout(m_portTimeout);
			} catch (javax.naming.NamingException e) {
			} catch (javax.ejb.FinderException e) {
			} catch (java.lang.ClassNotFoundException e) {
			} catch (java.lang.InstantiationException e) {
			} catch (IllegalAccessException e) {
			}
		}

		try {
			// ノードの属性取得
			HashMap facilityAttrMap = m_repository.getNodeDetail(facilityId,
					m_attributeList);

			String ipNetworkNumber = (String) facilityAttrMap
					.get(FacilityAttributeConstant.IPNETWORKNUMBER);
			String nodeName = (String) facilityAttrMap
					.get(FacilityAttributeConstant.NODENAME);

			boolean result = m_reachability.isReachable(ipNetworkNumber,
					nodeName);
			m_message = m_reachability.getMessage();
			m_messageOrg = m_reachability.getMessageOrg();
			if (result) {
				m_response = m_reachability.getResponse();
			}
			return result;
		} catch (FinderException e) {
			m_log.debug("run():" + e.getMessage());

			m_message = Messages.getString("message.port.4");
			m_messageOrg = e.getMessage();
			return false;
		} catch (NamingException e) {
			m_log.debug("run():" + e.getMessage());

			m_message = Messages.getString("message.port.4");
			m_messageOrg = e.getMessage();
			return false;
		}
	}

	/*
	 * (非 Javadoc) port監視情報を設定
	 * 
	 * @see com.clustercontrol.monitor.run.factory.OperationNumericValueInfo#setMonitorAdditionInfo()
	 */
	@Override
	public void setCheckInfo() throws FinderException, NamingException {

		// port監視情報を取得
		MonitorPortInfoPK pk = new MonitorPortInfoPK(m_monitorId,
				m_monitorTypeId);
		m_port = MonitorPortInfoUtil.getLocalHome().findByPrimaryKey(pk);

		// port監視情報を設定
		if (m_port.getPortNo() != null)
			m_portNo = m_port.getPortNo().intValue();
		if (m_port.getRunCount() != null)
			m_runCount = m_port.getRunCount().intValue();
		if (m_port.getRunInterval() != null)
			m_runInterval = m_port.getRunInterval().intValue();
		if (m_port.getTimeout() != null)
			m_portTimeout = m_port.getTimeout().intValue();
		if (m_port.getServiceId() != null)
			m_serviceId = m_port.getServiceId();
	}

	/*
	 * (非 Javadoc) 判定結果を取得
	 * 
	 * @see com.clustercontrol.monitor.run.factory.RunMonitorNumericValueType#getCheckResult(boolean)
	 */
	@Override
	public int getCheckResult(boolean ret) {

		int result = -1;
		MonitorNumericValueInfo info = null;

		// 値取得の成功時
		if (ret) {

			// 通知をチェック
			info = (MonitorNumericValueInfo) m_judgementInfoList.get(Integer
					.valueOf(PriorityConstant.TYPE_INFO));
			if (m_response >= info.getThresholdLowerLimit()
					&& m_response <= info.getThresholdUpperLimit()) {
				result = PriorityConstant.TYPE_INFO;
			} else {
				// 警告の範囲チェック
				info = (MonitorNumericValueInfo) m_judgementInfoList
						.get(Integer.valueOf(PriorityConstant.TYPE_WARNING));
				if (m_response >= info.getThresholdLowerLimit()
						&& m_response <= info.getThresholdUpperLimit()) {
					result = PriorityConstant.TYPE_WARNING;
				} else {
					// 危険（通知・警告以外）
					result = PriorityConstant.TYPE_CRITICAL;
				}
			}
		}
		return result;
	}

	/*
	 * (非 Javadoc) ノード用メッセージIDを取得
	 * 
	 * @see com.clustercontrol.monitor.run.factory.OperationMonitor#getMessageId(int)
	 */
	@Override
	public String getMessageId(int id) {

		if (id == PriorityConstant.TYPE_INFO) {
			return MESSAGE_ID_INFO;
		} else if (id == PriorityConstant.TYPE_WARNING) {
			return MESSAGE_ID_WARNING;
		} else if (id == PriorityConstant.TYPE_CRITICAL) {
			return MESSAGE_ID_CRITICAL;
		} else {
			return MESSAGE_ID_UNKNOWN;
		}
	}

	/*
	 * (非 Javadoc) ノード用メッセージを取得
	 * 
	 * @see com.clustercontrol.monitor.run.factory.OperationMonitor#getMessage(int)
	 */
	@Override
	public String getMessage(int id) {
		return m_message;
	}

	/*
	 * (非 Javadoc) ノード用オリジナルメッセージを取得
	 * 
	 * @see com.clustercontrol.monitor.run.factory.OperationMonitor#getMessageOrg(int)
	 */
	@Override
	public String getMessageOrg(int id) {
		return m_messageOrg;
	}

	/*
	 * (非 Javadoc) スコープ用メッセージIDを取得
	 * 
	 * @see com.clustercontrol.monitor.run.factory.RunMonitor#getMessageIdForScope(int)
	 */
	@Override
	public String getMessageIdForScope(int priority) {

		if (priority == PriorityConstant.TYPE_INFO) {
			return MESSAGE_ID_INFO;
		} else if (priority == PriorityConstant.TYPE_WARNING) {
			return MESSAGE_ID_WARNING;
		} else if (priority == PriorityConstant.TYPE_CRITICAL) {
			return MESSAGE_ID_CRITICAL;
		} else {
			return MESSAGE_ID_UNKNOWN;
		}
	}

	/**
	 * トランザクションを開始し、引数で指定された監視情報の監視を実行します。
	 * 
	 * @param monitorTypeId
	 *            監視対象ID
	 * @param monitorId
	 *            監視項目ID
	 * @throws FinderException
	 * @throws CreateException
	 * @throws RemoveException
	 * @throws JMSException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws SystemException
	 * 
	 * @see #runMonitorInfo()
	 */
	public void run(String monitorTypeId, String monitorId)
			throws FinderException, RemoveException, JMSException,
			NamingException, NotSupportedException, HeuristicMixedException,
			HeuristicRollbackException, RollbackException,
			InvalidTransactionException, IllegalStateException,
			SystemException, CreateException {
		super.run(monitorTypeId, monitorId);
	}

}