/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performanceMGR.bean;

import java.util.HashSet;
import java.util.Iterator;
import java.util.TreeSet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * 　Facilityの実装クラス
 * 
 * 種別がスコープであるファシリティの性能値を算出、保持します。
 * 子ファシリティの平均値を性能値として算出します。
 * ScopeTreeクラスの内部で使用することを目的としたクラスです。
 * 
 * @version 1.0
 * @since 1.0
 */
public class Scope extends Facility {
	//	ログ出力
	protected static Log m_log = LogFactory.getLog( Node.class );
	
	private TreeSet m_children = null;
	private HashSet m_nodes = null;	// このスコープの直下にあるノードのリスト
	
	Scope(final String facilityID, final String faclityName){
		super(facilityID, faclityName, Facility.SCOPE);
		m_children = new TreeSet();
		m_nodes = new HashSet();
	}
	
	/**
	 * ファシリティの子供のファシリティを追加する。
	 * @param child 子供のファシリティ
	 */
	public void addChildren(final Facility child){
		m_children.add(child);
		
		// 子要素がノードの場合はノードテーブルにも追加する
		if(child.getType() == Facility.NODE) {
			m_nodes.add(child);
		}
	}
	
	/**
	 * このファシリティ以下に含まれるノードを取得します。
	 * @param nodeSet
	 * @return HashSet
	 */
	public HashSet getNode(HashSet nodeSet){
		Iterator itr = m_children.iterator();
		
		// このスコープ直下のノードを追加する。
		nodeSet.addAll(m_nodes);
		
		// 子スコープのノードを再帰的に取得する。
		while(itr.hasNext()){
			// 子スコープを取得する。
			Facility child = (Facility)itr.next();
			
			// 対象の子スコープがスコープの場合は再帰的に取得する。
			if(child.getType() == Facility.SCOPE){
				((Scope)child).getNode(nodeSet);
			}
		}
		
		return nodeSet;
	}
	
	/**
	 * このファシリティの以下のファシリティに含まれるノードの数を取得します。
	 * @return int このファシリティ以下の全てのファシリティの数
	 */
	public int getNodeCount() {
		int count = 0;
		Iterator itr = m_children.iterator();
		
		// 子ファシリティのノードを再帰的に取得する。
		while(itr.hasNext()){
			// 子ファシリティを取得する。
			Facility child = (Facility)itr.next();
			
			count += child.getNodeCount();
		}
		
		return count;
	}
	
	/**
	 * 性能値を取得する
	 * @return 性能値 
	 */
	public double calcValue(final String itemCode, final int deviceIndex, final String deviceName){
		double total = 0;
		int counter = 0;
		
		Iterator itr = m_children.iterator();
		
		while(itr.hasNext()){
			// 子ファシリティを取得する
			Facility child = (Facility)itr.next();
			
			// 子ファシリティの子孫にノードが含まれる場合はそのファシリティを計算対象とする。
			if(child.getNodeCount() != 0) {
				double childValue = child.calcValue(itemCode, deviceIndex, deviceName);

				// 最近の算出結果が計算不能の場合は、前回値を用いる
				if(Double.isNaN(childValue)){
					childValue = 
						child.getCalcValueBuffer(new CollectorItemPK(itemCode, deviceIndex));
				}
				
				// 性能値の算出ができない場合(性能値算出式の分母が0になるような場合)は、
				// childValue の値は、NaNとなる。
				// ここでは、性能値が正常に算出できているノードの性能値のみを用いて
				// スコープ代表値の算出を行う。
				// 平均を求める分母となるノード数は、性能値を算出できたノードの数となる。
				if(!Double.isNaN(childValue)){
					total += childValue;
					counter++;
				}
			}
		}
		
		double returnValue;
		
		// スコープ代表値をノードの平均で求める。
		// 分母となるスコープ数が0の場合は、平均値を計算できないため、スコープ代表値をNaNとする。
		if(counter != 0){
			returnValue = (double)total / counter;
		} else {
			returnValue = Double.NaN;
		}
		
		// 性能値が算出できた場合はバッファに保存する
		if(!Double.isNaN(returnValue)){
			setCalcValueBuffer(new CollectorItemPK(itemCode, deviceIndex), returnValue);
		}
		
		m_log.debug("calcValue() : " + m_facilityID + " " + 
				itemCode + " " + deviceIndex + "  " + returnValue);
		
		return returnValue;
	}
	
	/**
	 * このファシリティの子供のファシリティのファシリティIDのリストを返します。
	 * 
	 * @return　子供のファシリティのリスト
	 */
	public String[] getChildrenID(){
		String[] facilityIDs = new String[m_children.size()];
		
		Iterator itr = m_children.iterator();
		
		for(int i=0; itr.hasNext(); i++){
			facilityIDs[i] = ((Facility)itr.next()).getFacilityID();
		}
		return facilityIDs;
	}
}
