/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.composite;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;

import com.clustercontrol.performance.bean.GraphProperty;
import com.clustercontrol.performance.util.CollectorItemCodeFactory;
import com.clustercontrol.performance.util.Messages;
import com.clustercontrol.performanceMGR.bean.CollectorItemInfo;

/**
 * 実績グラフを表示するための収集項目を設定するコンポジットクラス
 * 
 * @version 1.0
 * @since 1.0
 *  
 */
public class RecordGraphSetItemComposite extends Composite {

    private GridLayout gridLayoutButton = null;

    private Group group = null;

    private Combo combo = null;

    private Button radioButton1;

    private Button radioButton2;

    private Button radioButton3;

    private Button radioButton4;

    // 収集項目の情報を保持
    private String categoryCode;

    private List itemList;

    private ArrayList availableItems; //指定のカテゴリに含まれる（選択可能な）収集項目のリスト

    private int selectItemID;

    private int graphType;

    /**
     * コンストラクター
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     */
    public RecordGraphSetItemComposite(Composite parent, int style) {
        super(parent, style);

        availableItems = new ArrayList();

        initialize();
    }

    /**
     * タイトルを設定します。
     * 
     * @param title グラフのタイトル
     */
    public void setTitle(String title) {
        group.setText(title);
    }

    /**
     * 実績グラフの描画で選択可能な収集項目の情報を設定します。
     *  
     * @param categoryCode 収集項目カテゴリのコード
     * @param itemList 収集項目
     * @param selectItemID 選択した項目のID
     * @param graphType グラフ種別
     */
    public void setData(String categoryCode, List itemList, int selectItemID,
            int graphType) {
        this.categoryCode = categoryCode;
        this.itemList = itemList;
        this.selectItemID = selectItemID;
        this.graphType = graphType;

        // コンボボックスにデータを設定
        setComboData();

        // ラジオボタンを設定
        setButtonSelection();
    }

    /**
     * This method initializes group
     *  
     */
    private void createGroup() {
        group = new Group(this, SWT.NONE);

        //GridDataの設定
        gridLayoutButton = new GridLayout();
        gridLayoutButton.numColumns = 3; //3列
        group.setLayout(gridLayoutButton);

        createCombo();

        GridData gridDataButton;

        //		非表示ボタン
        radioButton1 = new Button(group, SWT.RADIO);
        radioButton1.setText(Messages.getString("HIDDEN"));
        gridDataButton = new GridData(GridData.VERTICAL_ALIGN_BEGINNING);
        gridDataButton.horizontalSpan = 3; //３列分を占める
        radioButton1.setLayoutData(gridDataButton);

        //		スコープ代表値ボタン
        radioButton2 = new Button(group, SWT.RADIO);
        radioButton2.setText(Messages.getString("SCOPE_REPRESENTING_VALUE"));
        gridDataButton = new GridData(GridData.VERTICAL_ALIGN_BEGINNING);
        radioButton2.setLayoutData(gridDataButton);

        //		スコープ代表値詳細ボタン
        radioButton3 = new Button(group, SWT.RADIO);
        radioButton3.setText(Messages
                .getString("DETAIL_SCOPE_REPRESENTING_VALUE"));
        gridDataButton = new GridData(GridData.VERTICAL_ALIGN_BEGINNING);
        radioButton3.setLayoutData(gridDataButton);

        //		サブスコープ代表値ボタン
        radioButton4 = new Button(group, SWT.RADIO);
        radioButton4.setText(Messages.getString("SUB_SCOPE_REP_VAL"));
        gridDataButton = new GridData(GridData.VERTICAL_ALIGN_BEGINNING);
        radioButton4.setLayoutData(gridDataButton);

    }

    /**
     * This method initializes combo
     *  
     */
    private void createCombo() {
        combo = new Combo(group, SWT.NONE | SWT.READ_ONLY);

        //GridDataの設定
        GridData gridDataCombo = new GridData(GridData.VERTICAL_ALIGN_BEGINNING);
        gridDataCombo.horizontalSpan = 3; //3列分を占める
        combo.setLayoutData(gridDataCombo);

        combo.addSelectionListener(new SelectionListener() {
            public void widgetDefaultSelected(SelectionEvent e) {
            }

            public void widgetSelected(SelectionEvent e) {
                // 詳細表示可能か否かをチェック
                checkItemList(getSelectItem());
            }
        }

        );

        combo.setBounds(new org.eclipse.swt.graphics.Rectangle(5, 15, 200, 20));
    }

    /**
     * 詳細表示可能か否かを調べ、選択されている収集項目が詳細表示できない項目の場合は、 ラジオボタンの”詳細グラフ”を選択不可とする。
     */
    private void checkItemList(CollectorItemInfo itemInfo) {
        // 詳細表示可能か否かをチェック
        if (CollectorItemCodeFactory.checkItemList(itemInfo, availableItems)) {
            setDetailButtonEnabled(true);
        } else {
            // 詳細表示が選択されていた場合はスコープ代表値に変更
            if (getGraphType() == GraphProperty.TYPE2) {
                graphType = GraphProperty.TYPE1;
                setButtonSelection();
            }
            setDetailButtonEnabled(false);
        }
    }

    /**
     * スコープ代表値詳細ラジオボタンの有効無効を設定します。
     * 
     * @param enabled 有効とする場合は、true
     */
    public void setDetailButtonEnabled(boolean enabled) {
        radioButton3.setEnabled(enabled);
    }

    /**
     * サブスコープ代表値ラジオボタンの有効無効を設定します。
     * 
     * @param enabled 有効とする場合は、true
     */
    public void setSubScopeButtonEnabled(boolean enabled) {
        radioButton4.setEnabled(enabled);
    }

    /**
     * コンボボックスにデータを設定します。
     */
    private void setComboData() {
        // 情報を設定
        Iterator itr = itemList.iterator();

        int i = 0;
        int selectIndex = 0;
        while (itr.hasNext()) {
            CollectorItemInfo itemInfo = (CollectorItemInfo) itr.next();

            // 指定のカテゴリであるかチェック
            if (CollectorItemCodeFactory.getCategoryCode(
                    itemInfo.getCollectorItemCode()).equals(this.categoryCode)) {
                // コンボボックスに収集項目名を追加
                combo.add(CollectorItemCodeFactory.getFullItemName(itemInfo
                        .getCollectorItemCode(), itemInfo.getDeviceName()));
                // 選択可能な収集項目リストに収集項目情報を追加
                availableItems.add(itemInfo);

                // 現在グラフ表示されているものが選択された状態にする
                if (itemInfo.getCollectorItemID() == this.selectItemID) {
                    selectIndex = i;
                }
                i++; // インクリメント
            }
        }

        combo.select(selectIndex);

        // 詳細表示可能か否かをチェック
        checkItemList((CollectorItemInfo) this.availableItems.get(selectIndex));
    }

    /**
     * ラジオボタンを設定します。
     */
    private void setButtonSelection() {
        // 選択されているグラフ種別に対応するボタンが選択された状態にする
        switch (graphType) {
        case GraphProperty.INVISIBLE:
            radioButton1.setSelection(true);
            radioButton2.setSelection(false);
            radioButton3.setSelection(false);
            radioButton4.setSelection(false);
            break;
        case GraphProperty.TYPE1:
            radioButton1.setSelection(false);
            radioButton2.setSelection(true);
            radioButton3.setSelection(false);
            radioButton4.setSelection(false);
            break;
        case GraphProperty.TYPE2:
            radioButton1.setSelection(false);
            radioButton2.setSelection(false);
            radioButton3.setSelection(true);
            radioButton4.setSelection(false);
            break;
        case GraphProperty.TYPE3:
            radioButton1.setSelection(false);
            radioButton2.setSelection(false);
            radioButton3.setSelection(false);
            radioButton4.setSelection(true);
            break;
        default:
            radioButton1.setSelection(true);
            radioButton2.setSelection(false);
            radioButton3.setSelection(false);
            radioButton4.setSelection(false);
        }
    }

    private void initialize() {
        createGroup();
        this.setLayout(new FillLayout());
    }

    /**
     * グラフ種別を取得します。
     * 
     * @return グラフ種別
     */
    public int getGraphType() {
        if (radioButton1.getSelection()) {
            this.graphType = GraphProperty.INVISIBLE;
        } else if (radioButton2.getSelection()) {
            this.graphType = GraphProperty.TYPE1;
        } else if (radioButton3.getSelection()) {
            this.graphType = GraphProperty.TYPE2;
        } else if (radioButton4.getSelection()) {
            this.graphType = GraphProperty.TYPE3;
        }

        return this.graphType;
    }

    /**
     * 選択されている収集項目を戻します。
     * 
     * @return 選択されている収集項目
     */
    public CollectorItemInfo getSelectItem() {
        return (CollectorItemInfo) this.availableItems.get(this.combo
                .getSelectionIndex());
    }

    /**
     * グラフ種別を設定します。
     * 
     * @param graphType　グラフ種別
     */
    public void setGraphType(int graphType) {
        this.graphType = graphType;
        setButtonSelection();
    }
}