/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.notify.bean.NotifyEventInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 通知情報を検索するクラス<BR>
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class SelectNotify {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( SelectNotify.class );
	
	/**
	 * 通知情報を返します。
	 * 
	 * @param notifyId 取得対象の通知ID
	 * @return 通知情報
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 * @see com.clustercontrol.notify.ejb.entity.NotifyEventInfoBean
	 */
	public NotifyInfo getNotify(String notifyId) throws FinderException, NamingException {

		NotifyInfo bean = null;
		try 
		{
			// 通知情報を取得
			NotifyInfoLocal notify = NotifyInfoUtil.getLocalHome().findByPrimaryKey(notifyId);
			
			ArrayList<NotifyEventInfo> notifyEvents = new ArrayList<NotifyEventInfo>();
			Collection ct = notify.getNotifyEventInfo();
			
            Iterator itr = ct.iterator();
            NotifyEventInfoLocal local = null;
            while(itr.hasNext()){
                local = (NotifyEventInfoLocal)itr.next();
                NotifyEventInfo event = new NotifyEventInfo(
                		notifyId,
						local.getPriority(),
						local.getEventNormalFlg(),
						local.getEventNormalState(),
						local.getEventInhibitionFlg(),
						local.getEventInhibitionState(),
						local.getMailFlg(),
						local.getMailAddress());
                notifyEvents.add(event);
            }
			
			bean = new NotifyInfo(
					notify.getNotifyId(),
					notify.getDescription(),
					notify.getEventFlg(),
					notify.getInhibitionFlg(),
					notify.getInhibitionFrequency(),
					notify.getInhibitionPeriod(),
					notify.getRegDate(),
					notify.getStatusFlg(),
					notify.getStatusInvalidFlg(),
					notify.getStatusUpdatePriority(),
					notify.getStatusValidPeriod(),
					notify.getUpdateDate(),
					notify.getMailTemplateId(),
					notifyEvents
			);
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { notifyId };
            apllog.put("SYS", "004", args);
            m_log.debug("getNotify():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            String[] args = { notifyId };
            apllog.put("SYS", "004", args);
            m_log.debug("getNotify():" + e.getMessage());
			throw e;
		}
		return bean;
	}
	
	/**
	 * 通知ID一覧を返します。
	 * <p>
	 * 通知IDの昇順に並んだ通知ID一覧を返します。
	 * 
	 * @return 通知ID一覧
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 */
	public ArrayList getNotifyIdList() throws FinderException, NamingException {

		ArrayList<String> list = null;
		try 
		{
			// 通知情報一覧を取得
			Collection ct = NotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();
			
			Iterator itr = ct.iterator();
			while(itr.hasNext())
			{
				if(list == null){
					list = new ArrayList<String>();
				}
				
				NotifyInfoLocal notify = (NotifyInfoLocal)itr.next();
				list.add(notify.getNotifyId());
			}
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "005");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "005");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		}
		return list;
	}

	/**
	 * 通知情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>通知IDの昇順に並んだ全ての通知情報を取得します。</li>
	 * <li>１通知情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.NotifyTabelDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１通知情報を保持するリストを、通知情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>通知情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 通知情報1 {カラム1の値, カラム2の値, … }, 通知情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.bean.NotifyTabelDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList getNotifyList() throws CreateException, FinderException, NamingException {

		ArrayList list = null;
		try 
		{
			// 通知情報一覧を取得
			Collection ct = NotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();
			
			// 2次元配列に変換
			list = this.collectionToArray(ct);
			
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * 通知情報マップを返します。
	 * <p>
	 * <ol>
	 * <li>全ての通知情報を取得します。</li>
	 * <li>通知情報マップに、通知情報をセットします。</li>
	 *  <dl>
	 *   <dt>通知情報マップ</dt>
	 *   <dd>キー：通知ID</dd>
	 *   <dd>値　：通知情報</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報マップ
	 * @throws FinderException
	 * @throws NamingException
	 */
	public HashMap<String, NotifyInfo> getNotifyMap() throws FinderException, NamingException {

		HashMap<String, NotifyInfo> map = new HashMap<String, NotifyInfo>();
		try 
		{
			// 通知情報一覧を取得
			Collection ct = NotifyInfoUtil.getLocalHome().findAll();
			
			Iterator itr = ct.iterator();
			
			NotifyInfoLocal notify = null;
			ArrayList<NotifyEventInfo> notifyEventsList = null;
			Collection ctEvents = null; 
			Iterator itrEvents = null;
			NotifyEventInfoLocal local = null;
			while(itr.hasNext()){
				
				notify = (NotifyInfoLocal)itr.next();
				
				ctEvents = notify.getNotifyEventInfo();
				itrEvents = ctEvents.iterator();
		        local = null;
		        
		        notifyEventsList = new ArrayList<NotifyEventInfo>();
		        while(itrEvents.hasNext()){
		            local = (NotifyEventInfoLocal)itrEvents.next();
		            NotifyEventInfo event = new NotifyEventInfo(
		            		local.getNotifyId(),
							local.getPriority(),
							local.getEventNormalFlg(),
							local.getEventNormalState(),
							local.getEventInhibitionFlg(),
							local.getEventInhibitionState(),
							local.getMailFlg(),
							local.getMailAddress());
		            notifyEventsList.add(event);
		        }
				
		        NotifyInfo bean = new NotifyInfo(
						notify.getNotifyId(),
						notify.getDescription(),
						notify.getEventFlg(),
						notify.getInhibitionFlg(),
						notify.getInhibitionFrequency(),
						notify.getInhibitionPeriod(),
						notify.getRegDate(),
						notify.getStatusFlg(),
						notify.getStatusInvalidFlg(),
						notify.getStatusUpdatePriority(),
						notify.getStatusValidPeriod(),
						notify.getUpdateDate(),
						notify.getMailTemplateId(),
						notifyEventsList
				);
		        map.put(notify.getNotifyId(), bean);
			}
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyMap():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyMap():" + e.getMessage());
			throw e;
		}
		return map;
	}

	
	/**
	 * DBより取得した通知情報のObjectの2次元配列を取得します。<BR>
	 * <p>
	 * <ol>
	 * <li>通知情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.NotifyTabelDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>1つの通知情報を保持するリストを、通知情報一覧を保持するリスト（{@link ArrayList}）に格納します。
	 * <p>
	 *  <dl>
	 *  <dt>通知情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 通知情報1 {カラム1の値, カラム2の値, … }, 通知情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @param ct 通知情報のコレクション
	 * @return 通知情報一覧（Objectの2次元配列）
	 * 
	 * @see com.clustercontrol.notify.bean.NotifyTabelDefine
	 */
	@SuppressWarnings("unchecked")
	private ArrayList collectionToArray(Collection ct) throws CreateException, FinderException, NamingException{
		
		ArrayList list = new ArrayList();
		Iterator itr = ct.iterator();
		while(itr.hasNext())
		{
			NotifyInfoLocal notify = (NotifyInfoLocal)itr.next();
			
			ArrayList info = new ArrayList();
			info.add(notify.getNotifyId());
			info.add(notify.getDescription());
			info.add(new Boolean(ValidConstant.typeToBoolean(notify.getStatusFlg().intValue())));
			info.add(new Boolean(ValidConstant.typeToBoolean(notify.getEventFlg().intValue())));
			info.add(notify.getUpdateDate() == null ? null:new Date(notify.getUpdateDate().getTime()));
			list.add(info);
		}
		return list;
	}
}
