/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.util;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.jface.dialogs.MessageDialog;

import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.bean.JudgmentObjectConstant;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobObjectInfo;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobWaitRuleInfo;
import com.clustercontrol.util.Messages;

/**
 * ジョブユーティリティクラス
 * ジョブツリーアイテムに関するユーティリティを格納するクラスです。
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class JobUtil {
	
	/** コピーしたジョブIDの接頭語 */
	private static final String COPY_OF = "Copy Of?";
	
	/**
	 * 引数で指定されたジョブツリーアイテムのコピーを作成する
	 * 
	 * @param original コピー元ジョブツリーアイテム
	 * @return コピーとして作成されたジョブツリーアイテム
	 */
	public static JobTreeItem copy(JobTreeItem original) {
		JobTreeItem clone = null;
		if(original != null){
			clone = (JobTreeItem)original.clone();
		}
		
		return clone;
	}
	
	/**
	 * @param original コピー元ジョブツリーアイテム
	 * @param top コピー元ジョブツリーアイテム
	 * @return コピーとして作成されたジョブツリーアイテム
	 */
	public static JobTreeItem copy(JobTreeItem original, JobTreeItem top) {
		JobTreeItem clone = copy(original);
		
		//待ち条件を削除する
		deleteWaitRule(clone);
		
		//ジョブIDの変更
		changeJobId(clone, top, clone);
		
		return clone;
	}
	
	/**
	 * ジョブIDを一括変更する<BR>
	 * ジョブツリーアイテムのツリー階層の全てに対して一括変更する。<BR>
	 * topとcloneに指定されたジョブツリーアイテムを、ジョブIDの重複チェック対象とする。<BR>
	 * ジョブIDが重複した場合、コピーしたジョブIDの接頭語とカウンタを付加してジョブIDを決定する。
	 * 
	 * @param item ジョブID変更対象のジョブツリーアイテム
	 * @param top ジョブIDの重複チェック対象のジョブツリーアイテム
	 * @param clone ジョブIDの重複チェック対象のジョブツリーアイテム
	 */
	protected static void changeJobId(JobTreeItem item, JobTreeItem top, JobTreeItem clone) {
		if(item == null || top == null)
			return;
		
		//ジョブIDを変更
		JobInfo info = item.getData();
		if(info != null && info instanceof JobInfo){
			int count = 1;
			StringBuffer jobId = new StringBuffer();
			jobId.append(info.getId());
			while(true){              
				if(!findJobId(jobId.toString(), top) && !findJobId(jobId.toString(), clone)){
					break;
				}
				jobId.delete(0, jobId.length());
				if(count == 1){
					jobId.append(COPY_OF.replace("?", " "));
				}
				else{
					jobId.append(COPY_OF.replace("?", " (" + count + ") "));
				}
				jobId.append(info.getId());
				count++;
			}
			info.setId(jobId.toString());
		}
		
		//子JobTreeItemを取得
		JobTreeItem[] childrens = item.getChildren();
		for(int i = 0; i < childrens.length; i++){
			changeJobId(childrens[i], top, clone);
		}
	}
	
	/**
	 * ジョブツリーアイテムからジョブ待ち条件情報を削除する<BR>
	 * ジョブツリーアイテムのツリー階層の全てが削除対象
	 * 
	 * @param item ジョブ待ち条件情報を削除するジョブツリーアイテム
	 */
	protected static void deleteWaitRule(JobTreeItem item) {
		if(item == null)
			return;
		
		JobInfo info = item.getData();
		if(info != null && info instanceof JobInfo){
			//待ち条件を削除する
			JobWaitRuleInfo waitRule = info.getWaitRule();
			if(waitRule != null && waitRule instanceof JobWaitRuleInfo){
				if(waitRule.getObject() != null && waitRule.getObject() instanceof ArrayList){
					waitRule.setObject(new ArrayList());
				}
			}
		}
		
		//子JobTreeItemを取得
		JobTreeItem[] childrens = item.getChildren();
		for(int i = 0; i < childrens.length; i++){
			deleteWaitRule(childrens[i]);
		}
	}
	
	/**
	 * ジョブツリーアイテムからジョブIDが一致するインスタンスの有無を返す<BR>
	 * ジョブツリーアイテムのツリー階層の全てが検索対象
	 * 
	 * @param jobId ジョブID
	 * @param item ジョブツリーアイテム
	 * @return ジョブIDが一致するジョブツリーアイテムがあればtrue、なければfalse。
	 */
	public static boolean findJobId(String jobId, JobTreeItem item) {
		boolean find = false;
		
		//ジョブIDをチェック
		JobInfo info = item.getData();
		if(info != null && info instanceof JobInfo){
			if(jobId.compareTo(info.getId()) == 0){
				find = true;
				return find;
			}
		}
		
		//子JobTreeItemを取得
		JobTreeItem[] childrens = item.getChildren();
		for(int i = 0; i < childrens.length; i++){
			find = findJobId(jobId, childrens[i]);
			if(find){
				break;
			}
		}
		
		return find;
	}
	
	/**
	 * ジョブツリーアイテムの最上位のインスタンスを取得する
	 * 
	 * @param item ジョブツリーアイテム
	 * @return 最上位のジョブツリーアイテム
	 */
	public static JobTreeItem getTopJobTreeItem(JobTreeItem item) {
		if(item == null)
			return null;
		
		while (item.getParent() != null) {
			if(item.getParent().getData().getType() == JobConstant.TYPE_COMPOSITE){
				item = item.getParent();
				break;
			}
			else{
				item = item.getParent();
			}
		}
		
		return item;
	}
	
	/**
	 * ジョブツリーアイテムのジョブ待ち条件情報をチェックする
	 * 
	 * @param item ジョブ待ち条件情報をチェックするジョブツリーアイテム
	 */
	public static boolean checkWaitRule(JobTreeItem item) {
		boolean check = true;
		
		if(item == null)
			return check;
		
		if(item.getData() != null && item.getData() instanceof JobInfo){
			//ジョブID取得
			String jobId = item.getData().getId();
			
			//待ち条件情報を取得する
			JobWaitRuleInfo waitRule = item.getData().getWaitRule();
			if(waitRule != null && waitRule instanceof JobWaitRuleInfo && 
					waitRule.getObject() != null && waitRule.getObject().size() > 0){
				
				Iterator itr = waitRule.getObject().iterator();
				while(itr.hasNext()) {
					//判定対象を取得
					JobObjectInfo objectInfo = (JobObjectInfo)itr.next();
					if(objectInfo.getType() != JudgmentObjectConstant.TYPE_TIME){
						//判定対象のジョブIDが同一階層に存在するかチェック
						boolean find = false;
						String targetJobId = objectInfo.getJobId();
						JobTreeItem[] childrens = item.getParent().getChildren();
						for(int i = 0; i < childrens.length; i++){
							//ジョブIDをチェック
							JobInfo childInfo = childrens[i].getData();
							if(childInfo != null && childInfo instanceof JobInfo && 
									!jobId.equals(childInfo.getId())){
								if(targetJobId.compareTo(childInfo.getId()) == 0){
									find = true;
									break;
								}
							}
						}
						if(!find){
							String args[] = {jobId, targetJobId};
							MessageDialog.openWarning(
									null, 
									Messages.getString("message.hinemos.1"), 
									Messages.getString("message.job.59", args));
							check = false;
							return check;
						}
					}
				}
			}
		}
		
		//子JobTreeItemを取得
		JobTreeItem[] childrens = item.getChildren();
		for(int i = 0; i < childrens.length; i++){
			check = checkWaitRule(childrens[i]);
			if(!check){
				break;
			}
		}
		
		return check;
	}
}