/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.composite;

import java.util.ArrayList;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;

import com.clustercontrol.notify.action.GetNotify;
import com.clustercontrol.notify.dialog.NotifyCreateDialog;
import com.clustercontrol.util.Messages;


/**
 * 通知ID一覧コンポジットクラス<BR>
 * <p>
 * <dl>
 *  <dt>コンポジット</dt>
 *  <dd>「通知ID」 ラベル</dd>
 *  <dd>「通知ID一覧」 コンボボックス</dd>
 *  <dd>「参照」 ボタン</dd>
 * </dl>
 * 
 * @version 2.1.0
 * @since 2.0.0
 */
public class NotifyIdListComposite extends Composite {

	/** 通知ID ラベル文字列。 */
	private String m_text = null;

    /** 通知ID ラベル。 */
	private Label m_labelNotifyId = null;
	
    /** 通知ID一覧 コンボボックス。 */
    private Combo m_comboNotifyId = null;
    
    /** 参照 ボタン。 */
    private Button m_buttonRefer = null;


    /**
     * インスタンスを返します。
	 * <p>
	 * 初期処理を呼び出し、コンポジットを配置します。
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * @param labelFlg 通知IDラベル表示フラグ（表示する場合、<code> true </code>）。
     * 
     * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize(Composite, boolean)
	 */
    public NotifyIdListComposite(Composite parent, int style, boolean labelFlg) {
        super(parent, style);
        m_text = Messages.getString("notify.id");
        
        this.initialize(parent, labelFlg);
    }
    
    /**
     * インスタンスを返します。
     * <p>
	 * 初期処理を呼び出し、コンポジットを配置します。
	 * 通知IDラベルの文字列を指定します。
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * @param text 通知ID ラベル文字列
     * 
     * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize(Composite, boolean)
     */
    public NotifyIdListComposite(Composite parent, int style, String text) {
        super(parent, style);
        m_text = text;
        
        this.initialize(parent, true);
    }
    

    /**
	 * コンポジットを配置します。
	 * 
	 * @see #update()
	 */
    private void initialize(Composite parent, boolean labelFlg) {
    	
        // 変数として利用されるグリッドデータ
        GridData gridData = null;
        
        GridLayout layout = new GridLayout(1, true);
        layout.marginHeight = 0;
        layout.marginWidth = 0;
        if(labelFlg){
        	layout.numColumns = 15;	
        }
        else{
        	layout.numColumns = 10;
        }
        this.setLayout(layout);
        
        /*
         * 通知ID
         */
        
        if(labelFlg){
            // ラベル
            this.m_labelNotifyId = new Label(this, SWT.NONE);
            gridData = new GridData();
            gridData.horizontalSpan = 5;
            gridData.horizontalAlignment = GridData.FILL;
            gridData.grabExcessHorizontalSpace = true;
            this.m_labelNotifyId.setLayoutData(gridData);
            this.m_labelNotifyId.setText(m_text + " : ");
        }
        
        // コンボボックス
        this.m_comboNotifyId = new Combo(this, SWT.DROP_DOWN | SWT.READ_ONLY);
        gridData = new GridData();
        gridData.horizontalSpan = 8;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_comboNotifyId.setLayoutData(gridData);
        
        // 参照ボタン
        this.m_buttonRefer = new Button(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_buttonRefer.setLayoutData(gridData);
        this.m_buttonRefer.setText(Messages.getString("refer"));
        this.m_buttonRefer.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
            	
            	String notifyId = m_comboNotifyId.getText();
            	if(notifyId != null && !"".equals(notifyId.trim())){
                    // シェルを取得
                    Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
                    
                    // 通知登録/変更画面を開く
            		NotifyCreateDialog dialog = new NotifyCreateDialog(shell, notifyId, true);
                    dialog.open();
            	}
            }
        });
        
        this.update();
    }

    /**
     * コンポジットを更新します。<BR>
     * 通知ID一覧情報を取得し、通知ID一覧コンポジットにセットします。
	 * 
	 * @see com.clustercontrol.notify.action.GetNotify#getNotifyIdList()
     */
    public void update() {
    	
    	// 空欄
    	this.m_comboNotifyId.add("");
        
    	// データ取得
        ArrayList list = new GetNotify().getNotifyIdList();
        if(list != null){
        	// 通知IDリスト
        	for(int index=0; index<list.size(); index++){
        		this.m_comboNotifyId.add((String)list.get(index));
        	}
        }
    }

	/* (非 Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		this.m_comboNotifyId.setEnabled(enabled);
		this.m_buttonRefer.setEnabled(enabled);
	}

	/**
	 * 通知IDを返します。
	 * 
	 * @return 通知ID
	 * 
	 * @see org.eclipse.swt.widgets.Combo#getText()
	 */
	public String getText() {
		return this.m_comboNotifyId.getText();
	}

	/**
	 * 通知IDを設定します。
	 * 
	 * @param string 通知ID
	 * 
	 * @see org.eclipse.swt.widgets.Combo#setText(java.lang.String)
	 */
	public void setText(String string) {
		this.m_comboNotifyId.setText(string);
	}
}