/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.monitor.run.composite;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.jobmanagement.common.composite.JobIdSelectionListener;
import com.clustercontrol.monitor.run.bean.MonitorStringValueInfo;
import com.clustercontrol.monitor.run.bean.TruthConstant;
import com.clustercontrol.util.Messages;

/**
 * 文字列監視の判定情報（ジョブ実行）コンポジットクラス<BR>
 * <p>
 * <dl>
 *  <dt>コンポジット</dt>
 *  <dd>「ジョブID」 テキストボックス</dd>
 *  <dd>「参照」 ボタン</dd>
 *  <dd>「通知抑制と連動」 チェックボックス</dd>
 *  <dd>「呼出失敗時」 コンボボックス</dd>
 * </dl>
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class StringRunJobComposite extends Composite {
	
	/** カラム数（ジョブID）。 */
	public static final int WIDTH_JOB_ID	 = 4;
	
	/** カラム数（参照ボタン）。 */
	public static final int WIDTH_REF_BTN = 2;
	
	/** カラム数（通知抑制と連動）。 */
	public static final int WIDTH_INHIBITION_FLG = 3;
	
	/** カラム数（呼出失敗時）。 */
	public static final int WIDTH_FAILURE_PRIORITY = 2;
	
	/** ジョブID テキストボックス。 */
	private Text m_textJobId = null;
	
	/** 参照 ボタン。 */
	private Button m_buttonRefer = null;
	
	/** 通知抑制と連動 チェックボックス。 */ 
	private Button m_checkInhibitionFlg = null;
	
	/** 呼出失敗時の重要度 コンボボックス。 */ 
	private Combo m_comboFailurePriority = null;
	
	
	/**
	 * インスタンスを返します。
	 * <p>
	 * 初期処理を呼び出し、コンポジットを配置します。
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public StringRunJobComposite(Composite parent, int style) {
		super(parent, style);
		
		this.initialize();
	}
	
	/**
	 * コンポジットを配置します。
	 */
	private void initialize() {
		
		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;
		
		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.numColumns = 15;	
		this.setLayout(layout);
		
		// ジョブID
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_JOB_ID + WIDTH_REF_BTN;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("job.id"));
		
		// 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 1;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
		
		// 通知抑制と連動
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_INHIBITION_FLG;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("couple.with.notify.inhibition"));
		
		// 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 1;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
		
		// 呼出失敗時
		label = new Label(this, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_FAILURE_PRIORITY + 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("failure.call.value"));
		
		// 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 1;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
		
		/*
         * 実行ジョブ
         */
		this.m_textJobId = this.getTextJobId(this, WIDTH_JOB_ID);
		this.m_buttonRefer = this.getButtonRefer(this, WIDTH_REF_BTN);
		this.m_buttonRefer.addSelectionListener(new JobIdSelectionListener(m_textJobId));
		
		// 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 1;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
		
		this.m_checkInhibitionFlg = this.getCheckBox(this, WIDTH_INHIBITION_FLG);
		
		// 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 1;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
		this.m_comboFailurePriority = this.getComboPriority(this, WIDTH_FAILURE_PRIORITY);
		
		// 空白
        label = new Label(this, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 2;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
	}
	
	/**
	 * 引数で指定された判定情報の値を、各項目に設定します。
	 * 
	 * @param info 設定値として用いる判定情報
	 */
	public void setInputData(MonitorStringValueInfo info) {
		
		if(info != null){
			if(info.getJobId() != null){
				this.m_textJobId.setText(info.getJobId());
			}
			this.m_checkInhibitionFlg.setSelection(YesNoConstant.typeToBoolean(info.getJobInhibitionFlg()));
			this.m_comboFailurePriority.setText(PriorityConstant.typeToString(info.getJobFailurePriority()));
		}
	}
	
	/**
	 * 引数で指定された監視情報に、入力値を設定します。
	 * <p>
	 * 入力値チェックを行い、不正な場合は認証結果を返します。
	 * 不正ではない場合は、<code>null</code>を返します。
	 * 
	 * @param info 入力値を設定する監視情報
	 * @return 検証結果
	 * 
	 * @see #setMonitorStringValueInfo(MonitorStringValueInfo, Text, Button, Combo)
	 */
	public ValidateResult createInputData(MonitorStringValueInfo info) {
		
		ValidateResult validateResult = setMonitorStringValueInfo(info, this.m_textJobId, this.m_checkInhibitionFlg, this.m_comboFailurePriority, TruthConstant.TYPE_TRUE);
		
		return validateResult;
	}
	
	/**
	 * 引数で指定された判定情報に、ジョブID，通知抑制と連動 及び 呼出失敗時を設定します。
	 * <p>
     * 入力値チェックを行い、不正な場合は認証結果を返します。
	 * 不正ではない場合は、<code>null</code>を返します。
	 * 
	 * @param info 判定情報
	 * @param textJobId ジョブIDテキストボックス
	 * @param checkInhibitionFlg 通知抑制と連動チェックボックス
	 * @param comboFailurePriority 呼出失敗時コンボボックス
	 * @return 検証結果
	 * 
	 * @see com.clustercontrol.monitor.run.bean.MonitorStringValueInfo
	 */
	@SuppressWarnings("unchecked")
	public ValidateResult setMonitorStringValueInfo(MonitorStringValueInfo info, Text textJobId, Button checkInhibitionFlg, Combo comboFailurePriority, int truthValue) {
		
		String jobId = textJobId.getText();
		Integer jobInhibitionFlg = Integer.valueOf(YesNoConstant.booleanToType(checkInhibitionFlg.getSelection()));
		Integer jobFailurePriority = PriorityConstant.stringToType(comboFailurePriority.getText());
		
		if(info == null){
			info = new MonitorStringValueInfo();
		}
		
		// ジョブID
		if (jobId != null && !"".equals(jobId.trim())) {
			info.setJobId(jobId);	
		}
		else{
			if(info.getJobRun() == YesNoConstant.TYPE_YES){
				String[] args = {""};
				return setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.25", args));
			}
		}
		// 通知抑制と連動
		info.setJobInhibitionFlg(jobInhibitionFlg);
		// 呼出失敗時
		info.setJobFailurePriority(jobFailurePriority);
		
		return null;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		
		setEnabled(enabled, m_textJobId, m_buttonRefer, m_checkInhibitionFlg, m_comboFailurePriority);
	}
	
	/**
	 * 引数で指定されたコンポジットの選択可/不可を設定します。
	 * 
	 * @param enabled 選択可の場合、</code> true </code>
	 * @param textJobId 設定対象のジョブIDテキストボックス
	 * @param buttonRefer 設定対象の参照ボタン
	 * @param checkInhibitionFlg 設定対象の通知抑制と連動チェックボックス
	 * @param comboFailurePriority	設定対象の呼出失敗時コンボボックス
	 */
	public void setEnabled(boolean enabled,
			Text textJobId,
			Button buttonRefer,
			Button checkInhibitionFlg,
			Combo comboFailurePriority) {
		
		textJobId.setEnabled(enabled);
		buttonRefer.setEnabled(enabled);
		checkInhibitionFlg.setEnabled(enabled);
		comboFailurePriority.setEnabled(enabled);
	}
	
	/**
	 * 重要度のコンボボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param horizontalSpan コンボボックスのカラム数
	 * @return コンボボックス
	 */
	protected Combo getComboPriority(Composite parent, int horizontalSpan) {
		
		Combo combo = new Combo(this, SWT.DROP_DOWN | SWT.READ_ONLY);
		GridData gridData = new GridData();
		gridData.horizontalSpan = horizontalSpan;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		combo.setLayoutData(gridData);
		combo.add(PriorityConstant.STRING_CRITICAL);
		combo.add(PriorityConstant.STRING_WARNING);
		combo.add(PriorityConstant.STRING_INFO);
		combo.add(PriorityConstant.STRING_UNKNOWN);
		combo.setText(PriorityConstant.STRING_UNKNOWN);
		
		return combo;
	}
	
	/**
	 * チェックボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param horizontalSpan チェックボックスのカラム数
	 * @return チェックボックス
	 */
	protected Button getCheckBox(Composite parent, int horizontalSpan) {
		
		// チェックボックス
		Button button = new Button(parent, SWT.CHECK);
		GridData gridData = new GridData();
		gridData.horizontalSpan = horizontalSpan;
		gridData.horizontalAlignment = GridData.CENTER;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);
		
		return button;
	}
	
	/**
	 * ジョブIDのテキストボックスを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param horizontalSpan テキストボックスのカラム数
	 * @return テキストボックス
	 */
	protected Text getTextJobId(Composite parent, int horizontalSpan) {
		
		Text text = new Text(this, SWT.BORDER | SWT.LEFT | SWT.READ_ONLY);
		GridData gridData = new GridData();
		gridData.horizontalSpan = horizontalSpan;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		text.setLayoutData(gridData);
		
		return text;
	}
	
	/**
	 * 参照ボタンを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param horizontalSpan ボタンのカラム数
	 * @return ボタン
	 */
	protected Button getButtonRefer(Composite parent, int horizontalSpan) {
		
		Button button = new Button(this, SWT.NONE);
		GridData gridData = new GridData();
		gridData.horizontalSpan = horizontalSpan;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);
		button.setText(Messages.getString("refer"));
		
		return button;
	}
	
	
	/**
	 * 無効な入力値の情報を設定します。
	 * 
	 * @param id ID
	 * @param message メッセージ
	 * @return 認証結果
	 */
	protected ValidateResult setValidateResult(String id, String message) {
		
		ValidateResult validateResult = new ValidateResult();
		validateResult.setValid(false);
		validateResult.setID(id);
		validateResult.setMessage(message);
		
		return validateResult;
	}
}