/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.agent;

import java.util.Properties;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.naming.InitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.util.InternalLogger;
import com.clustercontrol.jobmanagement.bean.QueueConstant;
import com.clustercontrol.jobmanagement.message.RunResultInfo;

/**
 * ジョブ実行結果（チェック、開始含む）をQueue送信するクラス<BR>
 *
 * エージェントからの戻りメッセージはこのメソッドを用いて
 * マネージャに返送されます。
 *
 * @version $Revision: 3402 $
 * @since 1.0.0
 */
public class SendQueue implements ExceptionListener {

	private static final String QUEUE_CON_FACTORY = "ConnectionFactory";

	private static final String QUEUE_USER_NAME = "queue.user.name";

	private static final String QUEUE_USER_PASSWORD = "queue.user.password";
	
	private static final long SEND_TIMEOUT = 60 * 1000l;

	private Properties m_props;

	private QueueConnection m_con;

	private Queue m_queue;

	private QueueSession m_session;

	private QueueConnectionFactory m_factory;

	private boolean m_isErr = false;
	
	private long m_interval = 10000;

	//ロガー
	static private Log log = LogFactory.getLog(SendQueue.class);
	
	// reInitialメソッドが複数スレッドから呼ばれないよにするロック用オブジェクト
	private final Object reInitialLock = new Object();

	// Queue送信処理が複数スレッドから呼ばれないよにするためのロック用オブジェクト
	private final Object sendQueueLock = new Object();
	
	// 内部エラー出力抑制フラグ
	// エージェント → マネージャ間通信に失敗した場合にエラー出力を行うが、
	// 継続して通信できていない状態の場合は出力を制御する
	volatile private boolean internalErroInhibitflag = false;
	
	/**
	 * コンストラクタ
	 * @param props プロパティファイル情報
	 */
	public SendQueue(Properties props) {
		super();
		m_props = props;
		
		// SendQueue再接続処理実行間隔取得
		String interval = m_props.getProperty("sendqueue.reconnection.interval");
		if (interval != null) {
			try {
				// プロパティファイルには秒で記述されているため、1000をかける
				m_interval = Integer.parseInt(interval) * 1000;
			} catch (NumberFormatException e) {
				log.error("sendqueue.reconnection.interval",e);
			}
		}
		
		initial();
	}

	/**
	 *  再接続の処理を行います。<BR>
	 */
	private boolean reInitial() {
		synchronized (reInitialLock) {
			boolean ret = false;

			log.info("再接続処理!");

			terminate();

			if (initial()) {

				ret = true;

				log.info("再接続処理:成功!");

				//エラーフラグ解除
				m_isErr = false;


			} else {
				log.info("再接続処理:失敗!");
			}

			return ret;
		}
	}
	/**
	 * マネージャへの接続の終了処理を行います。<BR>
	 *  
	 */
	public void terminate() {
		
		try {
			if (m_session != null)
				m_session.close();
		} catch (JMSException e) {
		}

		try {
			if (m_con != null)
				m_con.close();
		} catch (JMSException e1) {
		}
	}

    /**
     * メッセージを送信します。<BR>
     * 
	 * マネージャからの実行に対する応答メッセージを送信します。<BR>
	 * 処理失敗は1回再試行します。
	 * @param msg
	 */
	synchronized public boolean put(RunResultInfo info) {
		log.debug("put() start  : "  + info.getCommandType() + ", " + info.getSessionId()+ ", " + info.getJobId());
			
		ExecutorService es = null;
		Future<?> task = null;
		try {
			// Executorオブジェクトの生成
			SenderThreadFactory threadFactory = new SenderThreadFactory(info.getSessionId());
			es = Executors.newSingleThreadExecutor(threadFactory);
				
			// タスクの実行
			log.debug("put() submit : "  + info.getCommandType() + ", " + info.getSessionId()+ ", " + info.getJobId());
			task = es.submit(new Sender(info));
			task.get(SEND_TIMEOUT, TimeUnit.MILLISECONDS);
			return true;
		} catch (Exception e) {
			if(internalErroInhibitflag == true){
				// 抑制されている場合は簡略なメッセージとする
				log.error("Failed to connect to MGR");
			} else {
				log.error("Failed to connect to MGR", e);
			}
			
			// 抑制されていない場合に出力する
			if(internalErroInhibitflag == false){
				// syslogにログ出力
				InternalLogger.error("hinemos_jobagent: Failed to connect to MGR");
				// エラーログ出力抑制フラグを設定する
				internalErroInhibitflag = true;
			}
		} finally {
			// Senderのcall()メソッドが終了した後でExecutorServiceのshutdown()メソッドを実行するように待ち合わせる
    		synchronized (sendQueueLock) {
    			log.debug("put() end    : "  + info.getCommandType() + ", " + info.getSessionId()+ ", " + info.getJobId());
    			if(es != null){
    				es.shutdown();
    			}
    		}
		}
			
		return false;
	}

	/**
	 * マネージャとの接続を初期化します。<BR>
	 * @return　true:成功　false:失敗
	 */
    private boolean initial() {

		log.info("EJB接続初期化");

		InitialContext con = null;

		try {
			//InitialContextの生成
			con = new InitialContext(m_props);

			//コネクションファクトリ生成
			m_factory = (QueueConnectionFactory) con.lookup(QUEUE_CON_FACTORY);

			//コネクション生成
			if (m_props.getProperty(QUEUE_USER_NAME) != null) {
				//ユーザ認証
				m_con = m_factory.createQueueConnection(m_props
						.getProperty(QUEUE_USER_NAME), m_props
						.getProperty(QUEUE_USER_PASSWORD));
			} else {
				//ユーザ認証なし
				m_con = m_factory.createQueueConnection();
			}

			
			//セッション生成
			m_session = m_con.createQueueSession(false,
					Session.AUTO_ACKNOWLEDGE);
			
			//エラーハンドラセット
	        m_con.setExceptionListener(this);
	        
			//コネクション開始
			m_con.start();

			//メッセージQueue取得
			m_queue = (Queue) con.lookup(QueueConstant.QUEUE_NAME_STATUS);

			
			

		} catch (Exception e) {
			log.error("Init", e);
			m_isErr = true;
			return false;
		} finally {
			try {
				if (con != null)
					con.close();
			} catch (Exception e1) {
			}
		}
		// エラー出力抑制フラグをクリア
		internalErroInhibitflag = false;
		
		return true;

	}
    
    /* 通信エラーハンドラ
     * (non-Javadoc)
     * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
     */
    public void onException(JMSException arg0) {

    	log.error(arg0);
    	
    	m_isErr = true;
        
        terminate();
    }
    
    /**
     * Queueメッセージ送信処理を実行するタスク用クラス
     */
    private class Sender implements Callable<Boolean> {
    	private RunResultInfo m_info;
    	
    	public Sender(RunResultInfo info){
    		m_info = info;
    	}
    	
    	public Boolean call() throws Exception {
    		synchronized (sendQueueLock) {
    			if (m_isErr) {
    				reInitial();
    			}
    			
    			QueueSender sender = null;
    			while(true) {
    				try {
    					//Sender作成
    					sender = m_session.createSender(m_queue);
    					//メッセージ作成
    					ObjectMessage mess = m_session.createObjectMessage(m_info);
    					//送信
    					log.info("オブジェクトメッセージ送信 : SessionID=" + m_info.getSessionId()
    							+ ", JobID=" + m_info.getJobId()
    							+ ", CommandType=" + m_info.getCommandType());
    					sender.send(mess);
    					
    					return true;
    					
    				} catch (Exception e) {
    					log.error(e);
    					log.info("オブジェクトメッセージ送信リトライ");
    					
    					try {
//    						wait(m_interval);
    						Thread.sleep(m_interval);
    					} catch (InterruptedException e1) {
    					}
    					
    					reInitial();
    					
    				} finally {
    					try {
    						if (sender != null)
    							sender.close();
    					} catch (Exception e1) {
    					}
    				}
    			}
    		}
    	}
    }
    
    /**
     * Queueメッセージ送信処理を実行するタスク用のThreadFactory
     */
	private class SenderThreadFactory implements ThreadFactory {
		private final String m_threadName;
		
		public SenderThreadFactory(String threadName){
			m_threadName = threadName;
		}
		
		public Thread newThread(Runnable r) {
			return new Thread(r, "Sender-" + m_threadName);
		}
	}
}
