/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.agent;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.util.EjbConnectionManager;
import com.clustercontrol.repository.ejb.session.RepositoryController;

/**
 * エージェントメインクラス<BR>
 * 
 * 管理対象ノードで起動する際に呼び出されるクラスです。
 * 
 * @version $Revision: 680 $
 * @since 1.0.0
 */
public class Agent {


	private Properties m_props;

	private EjbConnectionManager m_ejbConnectionManager;

	private ReceiveTopic m_receiveTopic;

	private SendQueue m_sendQueue;
	
	private UpdateRepositoryInfoReceiveTopic m_updateRepository;

	/** ファシリティID更新用タイマー * */
	protected Timer m_timer = new Timer(true);

	private long m_cacheInterval = 600000;
	
	//ロガー
	static private Log log = LogFactory.getLog(Agent.class);

	
	
	/**
	 * メイン処理
	 * 
	 * @param args プロパティファイル名
	 */
	public static void main(String[] args) throws Exception{

		Agent agent = new Agent(args[0]);

		//エージェント処理開始
		agent.exec();

		
		//終了待ち
		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		while (true) {
			try {
				String line = br.readLine();
				if (line.equals("quit") ) {
					break;
				}
			} catch (Exception e) {
			}
			synchronized (agent) {
			    agent.wait();
            } 
		}
		

		//------------
		//-- 終了処理
		//------------
		
		agent.terminate();
		
		
	}

	/**
	 * コンストラクタ
	 */
	public Agent(String propFileName) throws Exception{

		//------------
		//-- 初期処理
		//------------

		//プロパティファイル読み込み
		m_props = new Properties();
		try {
			m_props.load(new FileInputStream(propFileName));
		} catch (Exception e) {
			log.error(e);
			throw e;
		}

		//ファイシリティキャッシュ更新インターバル
		String casheInterval = m_props.getProperty("facility.update.interval");
		if (casheInterval != null) {
			try {
				m_cacheInterval = Long.parseLong(casheInterval);
			} catch (NumberFormatException e) {
				log.error("facility.update.interval",e);
			}
		}
		
		m_ejbConnectionManager = new EjbConnectionManager(m_props);
		m_sendQueue = new SendQueue(m_props);

		//終了呼び出し設定
    	Runtime.getRuntime().addShutdownHook(new Thread() {
    	    public void run() { terminate(); }
    	});



	}

	
	/**
	 * エージェント処理実行実行します。<BR>
	 * 
	 * メインメソッドから呼び出されるメソッドで
	 * トピックへの接続を行います。
	 */
	public void exec() {
		//-----------------
		//-- トピック接続
		//-----------------
		if (log.isDebugEnabled())
			log.debug("トピック接続");

		//自局のファシリティID取得

		m_receiveTopic = new ReceiveTopic(getFacility(), m_sendQueue, m_props);
		
		m_updateRepository = new UpdateRepositoryInfoReceiveTopic(this, m_props);

		//キャッシュ更新タイマー開始
		m_timer.schedule(new ReflashFilterTask(), m_cacheInterval,
				m_cacheInterval);

		
		
	}
	
	/**
	 * ファシリティIDを取得します。<BR>
	 * 自局のIPアドレスと、ホスト名でリポジトリからファシリティIDを取得します<BR>
	 * @return　ファシリティIDのCollection
	 * @since 1.0.0
	 */
	public Collection getFacility() {

		//RepositoryControllerを取得

		HashMap<String, String> map = new HashMap<String, String>();

		RepositoryController repository = null;
		try {
			repository = m_ejbConnectionManager.getRepositoryController();
		} catch (Exception e1) {
			return map.values();
		}

		try {
			//ネットワーク情報取得
			Enumeration networkInterfaces = NetworkInterface
					.getNetworkInterfaces();
			if (null != networkInterfaces) {

				while (networkInterfaces.hasMoreElements()) {

					NetworkInterface ni = (NetworkInterface) networkInterfaces
							.nextElement();

					if (log.isDebugEnabled()) {
						try {
							log.debug("ネットワーク検出:\t"
									+ new String(ni.getDisplayName().getBytes(
											"iso-8859-1")));
						} catch (UnsupportedEncodingException e) {
							log.debug("ネットワーク検出:\t", e);
						}
					}

					Enumeration inetAddresses = ni.getInetAddresses();
					while (inetAddresses.hasMoreElements()) {

						InetAddress in4 = (InetAddress) inetAddresses.nextElement();

						if (log.isDebugEnabled())
							log.debug("IP/HOST検出:\t" + in4.getHostAddress()
									+ "/" + in4.getHostName());

						//ローカルホスト以外の時場合
						if (in4.getHostAddress().compareTo("127.0.0.1") != 0) {

							//IPとホスト名で、ファイシリティIDを特定
							ArrayList facilityIdList = repository.getFacilityIdList(in4.getHostName(), in4.getHostAddress());

							if (log.isDebugEnabled()) {
								if (facilityIdList.size() == 0) {
									log.debug("対応ファシリティなし");
								}
							}

							for (Iterator iter = facilityIdList.iterator(); iter.hasNext();) {
								String facilityId = (String) iter.next();

								if (log.isDebugEnabled())
									log.debug("対応ファシリティ:" + facilityId);

								//ファシリティの対象に追加
								if (map.containsKey(facilityId) == false) {
									map.put(facilityId, facilityId);
								}
							}
						}
					}

				}
			}

		} catch (Exception e) {
			log.debug("ファシリティ情報取得失敗");
			log.error(e);
			
		}

		return map.values();

	}

	/**
	 * 終了処理を行います。<BR>
	 */
	public void terminate() {

		m_receiveTopic.terminate();
		
		m_updateRepository.terminate();

		m_sendQueue.terminate();

		m_timer.cancel();

	}
	
	/**
	 * ファイシリティID再取得タイマータスク<BR>
	 * 
	 * ファシリティIDは一定のタイミングで再取されます。
	 * このタイマーを管理します。
	 */
	protected class ReflashFilterTask extends TimerTask {

		/**
		 * キャッシュ更新
		 */
		public void run() {

			setFacility();

		}

	}

	/**
	 * キャッシュを更新します。<BR>
	 * 
	 * マネージャからファシリティ情報を取得して、
	 * キャッシュにセットします。
	 * 
	 */
	public void setFacility() {

		m_receiveTopic.setFacilityIdList(getFacility());

	}
}
