/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.syslogng.forward;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.NoSuchElementException;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.OutputNotifyGroupInfo;
import com.clustercontrol.bean.ProcessConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.syslogng.bean.LogFilterInfo;

/**
 * メッセージを管理するクラス<BR>
 * メッセージを受け取り、フィルタリング処理を行い、 メッセージを送信します。
 *  
 * @version 3.0.0
 * @since 1.0.0
 */
public class LogManager extends Thread {
	
	/** ファイルのパス。 */
	private final static String TXT_FILE_PATH = "msg.file.txt.path";
	
	/** ファイルの拡張子。 */
	private final static String TXT_FILE_SUFFIX = "msg.file.txt.syffix";

	/** ディレクトリ下に作成するファイルの上限 */
	private final static String MAX_FILE_COUNT = "msg.max.file.count";
	
	/** 多重化ID。 */
	private final static String MULTI_ID = "multi.id";
	
	/** オリジナルメッセージ判定用文字列 */
	private static String ORG_MSG = "${ORIGINAL_MSG}";
	
	/** ファイル入出力のインスタンス。 */
	private FileUtil m_fileUtil;
	
	/** EJBアクセスのインスタンス。 */
	private EJBContoroller m_ejbControl;
	
	/** トピック受信のインスタンス。 */
	private UpdateRepositoryInfoReceiveTopic m_updateRepository;
	
	/** メッセージキュー。 */
	private LinkedList<Object> m_msgList = new LinkedList<Object>();
	
	/**
	 * フィルタ情報一覧。
	 * @see com.clustercontrol.syslogng.forward.LogFilterRepositoryInfo
	 */
	private ArrayList m_filterList = null;
	
	/** 待機状態フラグ。 */
	private boolean m_waiting = false;
	
	/** メッセージ処理エラーフラグ。 */
	private boolean m_processErr = true;
	
	/** 多重化ID。 */
	private String m_multiId = null;
	
	/** ログ出力のインスタンス。 */
	private Log log = LogFactory.getLog(this.getClass());
	
	/** オリジナルメッセージ出力文字数 */
	private int org_msg_length = 256;
	
	/**
	 * コンストラクタ。
	 * 
	 * @param props プロパティファイル情報
	 * 
	 * @see com.clustercontrol.syslogng.forward.FileUtil#FileUtil(String, String)
	 * @see com.clustercontrol.syslogng.forward.EJBContoroller#EJBContoroler(LogManager, Properties)
	 * @see com.clustercontrol.syslogng.forward.UpdateRepositoryInfoReceiveTopic#UpdateRepositoryInfoReceiveTopic(EJBContoroller, Properties)
	 */
	public LogManager(Properties props) {
		setName("LogManager");
		
		String filePath = props.getProperty(TXT_FILE_PATH, ".");
		String msgSuffix = props.getProperty(TXT_FILE_SUFFIX, ".txt");
		int maxFileCount;
		try {
			maxFileCount = Integer.parseInt(props.getProperty(MAX_FILE_COUNT, ".txt"));
		} catch (Exception e){
			maxFileCount = -1;
		}
		
		m_fileUtil = new FileUtil(filePath, msgSuffix, maxFileCount);
		
		m_ejbControl = new EJBContoroller(this, props);
		
		m_updateRepository = new UpdateRepositoryInfoReceiveTopic(this, m_ejbControl, props);
		
		m_multiId = props.getProperty(MULTI_ID, null);
		
	}
	
	/**
	 * メッセージ受信処理を開始します。<BR>
	 * 
	 * @see java.lang.Runnable#run()
	 * @see #getMsg()
	 * @see #proccess(String)
	 */
	public void run() {
		
		log.debug("Thread Start!!");
		
		while (true) {
			
			Object msg = null;
			
			//処理要求待ち
			synchronized (this) {
				
				msg = getMsg();
				if (msg == null) {
					//メッセージがないので処理待ち
					log.debug(getName() + ": Waiting for processing...");
					m_waiting = true;
					try {
						wait();
					} catch (InterruptedException e) {
						break;
					}
					
					m_waiting = false;
					continue;
				}
				
			}
			
			
			//メッセージを受信
			if (msg instanceof String) {
				log.debug("proccess msg:" + msg);
				//キャッシュが取得できないときはファイルへ出力
				if (m_filterList == null) {
					log.debug("file output msg:" + msg);
					m_fileUtil.write((String) msg);
					m_processErr = true;
					continue;
				}
				
				//メッセージを処理
				proccess((String) msg);
				
				//フィルタ条件受信
			} else if (msg instanceof ArrayList) {
				
				log.debug("フィルター受信:");
				
				//フィルタ条件格納
				m_filterList = (ArrayList) msg;
				
				//エラー状態の場合、
				if(m_processErr){
					//ファイルに出力されているメッセージをまず処理する。
					processRetry();
					m_processErr = false;
				}
			}
			
		}
		
		//終了処理
		terminate();
		
		log.debug("Thread End!!");
		
	}
	
	/**
	 * メッセージを受信します。<BR>
	 * syslogメッセージ および フィルタ情報を受信します。スレッドを再開します。
	 * 
	 * @param obj メッセージ　
	 */
	public void add(Object obj) {
		
		boolean notifyFlg = false;
		
		synchronized (m_msgList) {
			if (m_msgList.size() == 0) {
				notifyFlg = true;
			}
			
			
			//メッセージを処理
			if (obj instanceof String) {
				//メッセージ
				m_msgList.add(obj);
			} else if (obj instanceof ArrayList) {
				//フィルター条件(Queの先頭に追加)
				m_msgList.add(0, obj);
				
			}
			
		}
		
		if ( notifyFlg && isWaiting() ) {
			//スレッドに処理開始指示
			synchronized (this) {
				this.notify();
			}
		}
		
	}
	
	/**
	 * メッセージをファイルに出力します。<BR>
	 * 
	 * @param msg メッセージ　
	 */
	public void addMsg(String msg) {
		log.debug("file output msg:" + msg);
		m_fileUtil.write((String) msg);
		m_processErr = true;
	}
	
	/**
	 * メッセージを返します。<BR>
	 * メッセージ用のキューから、先頭の１メッセージを取得します。
	 * メッセージがない場合は、<code> null </code>を返します。
	 * 
	 * @return メッセージ
	 */
	protected Object getMsg() {
		synchronized (m_msgList) {
			try {
				return m_msgList.removeFirst();
				
			} catch (NoSuchElementException e) {
				return null;
			}
			
		}
	}
	
	/**
	 * 待機状態フラグを取得します。<BR>
	 * 
	 * @return 待機状態フラグ
	 */
	synchronized public boolean isWaiting() {
		return m_waiting;
	}
	
	/**
	 * syslogメッセージを解析し出力します。<BR>
	 * <p>
	 * <ol>
	 *  <li>syslogメッセージをパースします。</li>
	 *  <li>フィルタ情報を基に、順にフィルタリングを行います。</li>
	 *  <li>フィルタ情報に一致した場合、ログ出力用メッセージを生成し送信します。</li>
	 * </ol>
	 * 
	 * @param msg syslogメッセージ
	 * 
	 * @see com.clustercontrol.syslogng.forward.MessageParser#parse(String)
	 * @see #makeMessage(String, MessageInfo, LogFilterRepositoryInfo, String)
	 * @see com.clustercontrol.syslogng.forward.EJBContoroller#isRun(String, LogOutputNotifyInfo)
	 * @see com.clustercontrol.syslogng.forward.LogFilterRepositoryInfo#contains(String)
	 * @see com.clustercontrol.syslogng.forward.EJBContoroller#sendMsg(LogOutputNotifyInfo)
	 */
	protected void proccess(String msg) {
		
		//メッセージをパース
		MessageInfo logmsg = MessageParser.parse(msg);
		if (logmsg == null) {
			return;
		}
		//フィルタ条件のリストで順番にフィルタリング
		for (Iterator iter = m_filterList.iterator(); iter.hasNext();) {
			
			//
			//ファシリティIDが対象ノードかどうかをチェック
			//
			LogFilterRepositoryInfo filterRepInfo = (LogFilterRepositoryInfo) iter
			.next();
			String facilityID = filterRepInfo.getFilter().getFacilityId();
			
			if(facilityID.equals(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE)){
				//未登録ノードの場合には、第２引数をfalseとしてcontaionsを呼ぶ
				facilityID = filterRepInfo.contains(logmsg.getHostName(),false);
			}else{
				//通常の場合には、第２引数をtrueとしてcontailnsを呼ぶ
				facilityID = filterRepInfo.contains(logmsg.getHostName(),true);
			}
			if (facilityID == null) {
				continue;
			}
			
			//
			//メッセージ条件でマッチング
			//
			LogFilterInfo filterInfo = filterRepInfo.getFilter();
			String pattern = filterInfo.getPattern();
			
			try {
				if (logmsg.getMessage().matches(pattern)) {
					//マッチした場合、
					//処理しないならば、処理終了
					if (filterInfo.getProcessType() == ProcessConstant.TYPE_YES) {
						
						log.debug("Process:" + filterInfo.getDescription() + ":"
								+ msg);
						
							// メッセージ作成（未登録ノードの時は、送信元ホスト名をスコープ名にする。）
							OutputNotifyGroupInfo logOutput  = makeMessage(msg, logmsg,
									filterRepInfo, facilityID,logmsg.getHostName());

						// 稼動日チェック
						boolean runFlg = false;
						if(filterInfo.getCalendarId() == null){
							// カレンダ指定なし
							runFlg = true;
						}
						else{
							// 稼動日かどうか
							runFlg = m_ejbControl.isRun(filterInfo.getCalendarId(), logOutput);
						}
						
						//
						//メッセージ送信処理
						//
						if(runFlg){
							
							//メッセージ送信
							m_ejbControl.sendMsg(logOutput);
						}
						
					} else {
						
						log.debug("Non Process:" + filterInfo.getDescription()
								+ ":" + msg);
					}
					
					return;
				}
			} catch (Exception e) {
				log.error("フィルタ条件が無効;"+filterInfo.getDescription(),e);
			}
		}
		
		log.debug("該当フィルターなし:" + msg);
		return;
		
	}
	
	/**
	 * 引数で指定された情報より、ログ出力メッセージを生成し返します。
	 * 
	 * @param msg メッセージ
	 * @param logmsg syslogメッセージ情報
	 * @param filterRepInfo フィルタとリポジトリ情報
	 * @param facilityID ファシリティID
	 * @return ログ出力メッセージ
	 * 
	 * @since 1.0.0
	 */
	private OutputNotifyGroupInfo makeMessage(String msg, MessageInfo logmsg,
			LogFilterRepositoryInfo filterRepInfo, String facilityID,String NodeName) {
		
		
		log.debug("Make ObjectMsg");
		
		
		LogFilterInfo filterInfo = filterRepInfo.getFilter();
		
//		LogOutputInfo logOutput = new LogOutputInfo();
		OutputNotifyGroupInfo output = new OutputNotifyGroupInfo();
		
		output.setPluginId(HinemosModuleConstant.MONITOR_SYSLOGNG);
		output.setMonitorId(filterInfo.getMonitorId());
		output.setFacilityId(facilityID);
		
		if(facilityID.equals(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE)){
			//未登録ノードの場合には送信元のホスト名を表示する。
			output.setScopeText(NodeName);
		}else{
			output.setScopeText(filterRepInfo.getNodeName(facilityID));
		}
		
		output.setApplication(filterInfo.getApplication());
		output.setMessageId(filterInfo.getMessageId());
		
		// メッセージに${ORIGINAL_MSG}のみが入力されている場合は、
		// オリジナルメッセージ(文字数上限：org_msg_length)をメッセージに出力する		
		if(ORG_MSG.equals(filterInfo.getMessage())) {
			// オリジナルメッセージの文字数が上限を超えていない場合
			if (logmsg.getMessage().length() < org_msg_length + 1) {
				output.setMessage(logmsg.getMessage());
			}
			// オリジナルメッセージのも十巣が上限を超えている場合
			else {
				// 先頭からorg_msg_length分文字を出力させる
				output.setMessage(logmsg.getMessage().substring(0,org_msg_length));	
			}
		}
		else {
			output.setMessage(filterInfo.getMessage());
		}
		
		output.setMessageOrg(msg);
		output.setPriority(filterInfo.getPriority());
		output.setGenerationDate(logmsg.getGenerationDate());
		
		output.setNotifyGroupId(filterInfo.getNotifyGroupId());
		if(m_multiId != null && !"".equals(m_multiId)){
			output.setMultiId(m_multiId);	
		}
		
//		logOutput.setConfirmFlg( ConfirmConstant.booleanToType(filterInfo.isConfirmFlg()) );
/*		if(filterInfo.getJobRun() == YesNoConstant.TYPE_YES){
			LogOutputJobRunInfo jobRunInfo = new LogOutputJobRunInfo();
			jobRunInfo.setJobRun(filterInfo.getJobRun());
			jobRunInfo.setJobId(filterInfo.getJobId());
			jobRunInfo.setJobInhibitionFlg(filterInfo.getJobInhibitionFlg());
			jobRunInfo.setJobFailurePriority(filterInfo.getJobFailurePriority());
			logOutput.setJobRun(jobRunInfo);
		}
	*/	
//		logOutput.setEventLogFlg(filterInfo.isNotifyFlg());
//		logOutput.setStatusInfoFlg(false);
		
//		int exflg =  filterInfo.getExclusionFlg();
//		switch (exflg) {
//		case ExclusionConstant.TYPE_FREQUENCY:
//		
//		logOutput.setExcludePeriod(0);
//		logOutput.setExcludeNumber(filterInfo.getExclusionFrequency());
//		break;
//		case ExclusionConstant.TYPE_PERIOD:
//		
//		logOutput.setExcludePeriod(filterInfo.getExclusionPeriod());
//		logOutput.setExcludeNumber(0);
//		break;
//		
//		default:
//		logOutput.setExcludePeriod(0);
//		logOutput.setExcludeNumber(0);
//		break;
//		}
		
//		if (filterInfo.isMailFlg()) {
//		String address = filterInfo.getMailAddress();
//		StringTokenizer t = new StringTokenizer(address, ";");
//		ArrayList addressList = new ArrayList();
//		while (t.hasMoreTokens()) {
//		addressList.add(t.nextToken());
//		
//		}
//		logOutput.setAddress((String[]) addressList.toArray(new String[0]));
//		}
		return output;
	}
	
	/**
	 * メッセージを再解析／出力します。
	 * <p>
	 * <ol>
	 *  <li>メッセージファイル一覧を取得します。</li>
	 *  <li>メッセージファイルを読み込みます。</li>
	 *  <li>読み込んだメッセージを解析し出力します。</li>
	 * </ol>
	 *
	 * @see com.clustercontrol.syslogng.forward.FileUtil#getFileList()
	 * @see com.clustercontrol.syslogng.forward.FileUtil#readTxtFile(File)
	 * @see #proccess(String)
	 */
	synchronized private void processRetry() {
		
		log.info("メッセージ再処理!");
		
		//ファイル一覧取得
		Collection col = m_fileUtil.getFileList();
		
		for (Iterator iter = col.iterator(); iter.hasNext();) {
			File file = (File) iter.next();
			//ファイルを読み込み
			String msg = m_fileUtil.readTxtFile(file);
			
			//送信
			proccess(msg);
			//送信したファイルを削除
			file.delete();
		}
		
	}
	/**
	 * サーバ接続の終了処理を行います。
	 * 
	 * @see com.clustercontrol.syslogng.forward.EJBContoroller#terminate()
	 * @see com.clustercontrol.syslogng.forward.UpdateRepositoryInfoReceiveTopic#terminate()
	 */
	private void terminate() {
		
		m_ejbControl.terminate();
		
		m_updateRepository.terminate();
	}
	
	/**
	 * 全てのTopicとQueueの再接続処理を行う
	 * 
	 */
	public void reConnection() {
		
		log.info("SyslogForward reconnect start");
		
		m_ejbControl.reconnect();
		
		m_updateRepository.reconnect();
		
		log.info("SyslogForward reconnect end");
	}
	
	
}
