/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/


package com.clustercontrol.performance.bean;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import javax.ejb.EJBException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityInfo;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoData;
import com.clustercontrol.performance.util.Repository;
import com.clustercontrol.poller.NotInitializedException;

/**
 * 収集対象スコープツリーの各スコープ，ノードの関係を保持し、
 * 指定のファシリティ（スコープもしくはノード）の性能値を算出して返す機能を提供するクラス
 * 
 * @version 1.0
 * @since 1.0
 */
public class ScopeTree implements Serializable {
	private static final long serialVersionUID = 3172674418733261184L;

	//	ログ出力
	protected static Log m_log = LogFactory.getLog(ScopeTree.class);
	
//	//** ルートとなるファシリティのID */
//	private static String rootScopeID = "root";
//	//**ルートとなるファシリティの名称　*/
//	private static String rootScopeName = "root";
	
	private Hashtable<String, Scope> m_scopeTable;  // スコープ情報を保持するテーブル
	private Hashtable<String, Node> m_nodeTable;   // ノード情報を保持するテーブル
	
	private long m_lastPollingTime;			// 最終ポーリング時刻
	private long m_lastCollectTime = 0;			// 最終収集日時を保持
	
	private int m_intarvalSec = 0;
	
	// MIB情報を取得した際に正しく情報が更新されているかのフラグ
	// ポーリングに失敗し情報が更新されていない場合はfalseとなる
	volatile private boolean updateFlag = false;

	/**
	 * コンストラクター
	 * 
	 * @param treeItem FacilityTreeItemの階層オブジェクト
	 */
	public ScopeTree(
			FacilityTreeItem treeItem, 
			List<CollectorDeviceInfoData> deviceDataList, 
			HashMap<String, String> platformMap,
			int intervalSec){
		m_log.debug("ScopeTree コンストラクタ");
		
		m_scopeTable = new Hashtable<String, Scope>();
		m_nodeTable = new Hashtable<String, Node>();
		
		m_intarvalSec = intervalSec;
		
		this.setScopeTree(treeItem, deviceDataList, platformMap);
	}
	
	/**
	 * ファシリティツリーにスコープを追加します。
	 * 
	 * @param facilityID　ファシリティID
	 * @param facilityName ファシリティ名
	 * @param parents 親のファシリティID
	 */
	public void addScope(final String facilityID, final String facilityName, final String parent) {
		Scope scope = new Scope(facilityID, facilityName);
		
		// 親ファシリティを取得
		Scope parentScope = (Scope)m_scopeTable.get(parent);
			
		if (parentScope == null) {
		} else {
			// 新規追加ファシリティに親を登録する。
			scope.addParents(parentScope);
				
			// 親ファシリティに登録する。
			parentScope.addChildren(scope);		
		}
		
		// ファシリティテーブルにファシリティを追加
		m_scopeTable.put(facilityID, scope);
	}
	
	/**
	 * ファシリティツリーにノードを追加します。
	 * 
	 * @param facilityID　ファシリティID
	 * @param facilityName ファシリティ名
	 * @param parents 親のファシリティID
	 */
	public void addNode(
			final String facilityId, 
			final String facilityName, 
			final List<CollectorDeviceInfoData> deviceDataList, 
			String parent,
			HashMap<String, String> platformMap) {
		m_log.debug("addNode facilityId : " + facilityId +  ", " + facilityName);
		
		// 既に同じIDのノードが登録されているかどうかを調べる
		Node node = (Node)m_nodeTable.get(facilityId);
		String platformId = null;
		if(node == null){
			if(platformMap == null) {
				platformId = Repository.getPlatformId(facilityId);
				m_log.debug("addNode facilityId : " + facilityId +  ", " + platformId);
			}
			else {
				platformId = platformMap.get(facilityId);
			}
			
			node = new Node(facilityId, facilityName, platformId, deviceDataList);
		}
		
		// 親ファシリティを取得
		Scope parentScope = m_scopeTable.get(parent);	
		if (parentScope == null) {
			// エラー処理
		} else {
			// 新規追加ファシリティに親を登録する。
			node.addParents(parentScope);
			
			// 親ファシリティにノードを登録する。
			parentScope.addChildren(node);		
		}
		
		// ノードテーブルにノードを追加
		m_nodeTable.put(facilityId, node);
	}
	
	/**
	 * 指定のファシリティIDのファシリティもしくはノードが存在するか否かを判定します。
	 * 
	 * @param facilityID
	 * @return　true 存在する　false 存在しない
	 */
	public boolean contains(String facilityID){
		Scope scope = (Scope)m_scopeTable.get(facilityID);	
		Node node = (Node)m_nodeTable.get(facilityID);
		if(scope == null && node == null){
			return false;
		} else {
			return true;
		}
	}
	
	/**
	 * 指定ファシリティに含まれるノードのファシリティIDのリストを返します。
	 * 
	 * @param facilityID ファシリティID
	 * @return ノードのファシリティIDのリスト
	 */
	public String[] getNodeIDList(String facilityID){
		Scope scope = (Scope)m_scopeTable.get(facilityID);	
		
		String[] nodeListFaciliyID = null;
		
		HashSet<Node> nodeListSet = new HashSet<Node>();
		if (scope != null){
			// 指定ファシリティがスコープであった場合
			scope.getNode(nodeListSet);
			Node[] nodeList = new Node[nodeListSet.size()];
			nodeListSet.toArray(nodeList);
			
			nodeListFaciliyID = new String[nodeList.length];
			for(int i=0; i<nodeList.length; i++){
				nodeListFaciliyID[i] = nodeList[i].getFacilityId();
			}
		} else {
			// 指定ファシリティがノードであった場合
			Node node = ((Node)m_nodeTable.get(facilityID));
			
			if(node != null){
				nodeListFaciliyID = new String[1];
				nodeListFaciliyID[0] = node.getFacilityId();
			} else {
				// 無効なノードを参照しようとした
				m_log.debug("Facility not found : " + facilityID);
				nodeListFaciliyID = new String[0];
			}
		}
		
		return nodeListFaciliyID;
	}
	
	/**
	 * 指定ファシリティのサブファシリティのファシリティIDのリストを返します。
	 * 
	 * @param facilityID
	 * @return サブファシリティのファシリティIDのリスト
	 */
	public String[] getSubScopeIDList(String facilityID){
		Scope scope = (Scope)m_scopeTable.get(facilityID);
		
		if(scope == null){
			// エラー処理
			String message= "Facility not found : " + facilityID;
			throw new EJBException(message);
		}	
		
		return scope.getChildrenID();
	}
	
	/**
	 * 指定ファシリティ以下に含まれるノードにMIB値を設定する。
	 * 
	 * @param facilityId ファシリティID
	 * @param oids 収集対象のOIDの配列
	 * @param interval 収集間隔
	 */
	synchronized public long fetchMibValue(final String collectorId, final String facilityId)
	 throws NotInitializedException {
		// 指定ファシリティの下に含まれる全てのノードのファシリティIDを取得
		String[] nodeFid = getNodeIDList(facilityId);

		// 最終ポーリング時刻を保持
		long lastPollingTime = m_lastPollingTime;
		
		for(int i=0; i<nodeFid.length; i++){
			Node node = ((Node)m_nodeTable.get(nodeFid[i]));
			
			if(node != null){
				// MIBの値を取得し収集日時を設定する
				long lastCollectTime = node.fetchMibValue(collectorId);
				
				// 全てのノードで時刻を一致させるために最後に収集されたノードの値の収集時刻とする
				m_lastPollingTime = Math.max(m_lastPollingTime, lastCollectTime);
				
				// デバッグログ
				if(m_log.isDebugEnabled()){
					m_log.debug("FacilityID : " + nodeFid[i]
							+ "  PreviousCollectTime : " + new Date(lastPollingTime)
							+ "  LastCollectTime : " + new Date(m_lastPollingTime));
				}
				
			} else {
				// エラー処理
				String message= "Facility not found : " + nodeFid[i];
				throw new EJBException(message);
			}
		}
		
		// 前回収集時刻と今回の収集時刻が異なればフラグをtrueとする
		if(m_lastPollingTime > lastPollingTime) {
			updateFlag = true;
		} else {
			updateFlag = false;
		}
		
		// ポーリングの結果が正しく取得できているか否かで取得日時を変える
		// 前回と同じ取得時刻であるとduplicate keyが発生しDBに格納できない
		// 最終ポーリング時刻が更新されている場合は、その時刻を収集時刻とする
		if(updateFlag && (m_lastCollectTime < m_lastPollingTime)){  // fetchMibValue()実行の結果、最終ポーリング時刻が更新された場合
			updateFlag = false;  // フラグをクリア
			// 最終収集時刻を最終ポーリング時刻とする
			m_lastCollectTime = m_lastPollingTime;
		} else {
			// 現時刻を取得
			long currentTime = System.currentTimeMillis();
			
			// 収集間隔で定期的に実行されることを前提として、予定収集時刻を算出する
			// 現時刻の直近の収集予定時刻を求める
			// 例） 収集間隔10秒で設定されている場合で、10:00:34に、ここに到達した場合、
			//    10:00:30 となる
			long collectTime = (currentTime / m_intarvalSec) * m_intarvalSec;

			// 最終収集時刻より、予定収集時刻が大きい場合は、それを収集時刻として設定
			if(m_lastCollectTime < collectTime){
				m_lastCollectTime = collectTime;
			} else {
				// 予定収集時刻を最終収集時刻として設定すると、
				// 時間が遡ってしまうため、最終収集時刻と現時刻のうち大きい方を取得
				// 最終収集時刻と同じにならないように1（ミリ秒）を足しこむ
				m_lastCollectTime = Math.max(m_lastCollectTime, currentTime) + 1;
			}
		}
		
		return m_lastCollectTime;
	}
	
	/**
	 * 対象ファシリティの性能値を算出して返す。
	 * 
	 * @param facilityID ファシリティID
	 * @param itemCode　収集項目コード 　
	 * @param deviceIndex　デバイスのインデックス 　
	 * @return 計算された性能データ
	 */
	synchronized public CollectedDataInfo getValue(
			final String facilityId, 
			final CollectorItemInfo cii){
		double value = Double.NaN;
		
		Date d = new Date(m_lastCollectTime);
		
		// スコープの場合
		Facility facility = (Facility)m_scopeTable.get(facilityId);
		
		// ノードの場合
		if(facility == null){
			facility = (Facility)m_nodeTable.get(facilityId);
		}
		
		if(facility == null){
			// 無効なノードを参照しようとした
			m_log.debug("Facility not found : " + facility);
			return new CollectedDataInfo(facilityId, cii.getItemCode(), cii.getDeviceName(), cii.getCollectMethod(), d, value);
		} else {
			value = facility.calcValue(cii);
		}
		
		// デバッグログ
		if(m_log.isDebugEnabled()){
			m_log.debug("FacilityID : " + facilityId
					+ "  Code : " + cii.getItemCode()
					+ "  Device : " + cii.getDeviceName()
					+ "  Date : " + d
					+ "  updateFlg : " + updateFlag);
		}
		
		return new CollectedDataInfo(facilityId, cii.getItemCode(), cii.getDeviceName(), cii.getCollectMethod(), d, value);
	}

	/**
	 * リアルタイム収集時に値を算出できなかった場合に、前回の計算時の計算値を返す。
	 * 
	 * @param facilityID ファシリティID
	 * @param itemCode　収集項目コード 　
	 * @param deviceIndex　デバイスのインデックス 　
	 * @return 計算された性能データ
	 */
	synchronized public CollectedDataInfo getTempValue(
			final String facilityId, 
//			final String itemCode, 
//			final int deviceIndex,
//			final String deviceName
			CollectorItemInfo cii){
		double value = Double.NaN;
		
		Facility facility = (Facility)m_scopeTable.get(facilityId);
		
		if(facility == null){
			facility = (Facility)m_nodeTable.get(facilityId);
		}
		
		if (facility == null){
			// エラー処理
			// 登録されていないファシリティを参照しようとした
			String message= "Facility not found : " + facilityId;
			throw new EJBException(message);
		} else {
			value = facility.getCalcValueBuffer(cii);
		}
		
		CollectedDataInfo ret;
		if(m_lastCollectTime == 0){
			// 今まで一度も収集されていない場合(前回の収集)
			ret = new CollectedDataInfo(facilityId, cii.getItemCode(), cii.getDeviceName(), cii.getCollectMethod(), new Date(), Double.NaN);
//			ret.setDate(new Date());  // 現時刻を設定
//			ret.setValue(Double.NaN); // 性能値としてNaNを設定
		} else {
			ret = new CollectedDataInfo(facilityId, cii.getItemCode(), cii.getDeviceName(), cii.getCollectMethod(), new Date(m_lastCollectTime), value);
//			ret.setDate(new Date(m_lastCollectTime));  // 収集日時を設定
//			ret.setValue(value);  // 性能値を設定
		}
		
		return ret;
	}

	/**
	 * 指定のファシリティを返します。
     *
	 * @param facilityID ファシリティID(スコープもしくはノード)
	 * @return 指定のファシリティ（スコープもしくはノード）を表現するインスタンス
	 */
	public Facility getFacility(final String facilityID){
		Scope scope = (Scope)m_scopeTable.get(facilityID);	
		Node node = (Node)m_nodeTable.get(facilityID);
		
		if (scope != null && node == null){
			return scope;		
		} else if (scope == null && node != null){
			return node;
		} else {
			// エラー処理
			String message= "Facility not found : " + facilityID;
			throw new EJBException(message);
		}
	}
	
	/**
	 * ファシリティツリーの内容を設定します。
	 * 
	 * @param treeItem FacilityTreeItemの階層オブジェクト
	 */
	private void setScopeTree(FacilityTreeItem treeItem, List<CollectorDeviceInfoData> deviceDataList, HashMap<String, String> platformMap) {
		FacilityInfo info = treeItem.getData();
		
		// ファシリティがスコープの場合
		if(info.getType() == FacilityConstant.TYPE_SCOPE){
			this.addScope(info.getFacilityId(), 
					info.getFacilityName(), 
					treeItem.getParent().getData().getFacilityId()
					);
		} else
		// ファシリティがノードでかつ管理対象の場合の場合
		if(info.getType() == FacilityConstant.TYPE_NODE && info.isValid()){
			Iterator<CollectorDeviceInfoData> itr = deviceDataList.iterator();
			
			// 指定のノードのデバイス情報だけを抽出
			ArrayList<CollectorDeviceInfoData> nodeDeviceList = new ArrayList<CollectorDeviceInfoData>();
			while(itr.hasNext()){
				CollectorDeviceInfoData deviceData = itr.next();
				if(info.getFacilityId().equals(deviceData.getFacilityId())){
					nodeDeviceList.add(deviceData);
				}
			}
			
			this.addNode(
					info.getFacilityId(),
					info.getFacilityName(),
					nodeDeviceList,
					treeItem.getParent().getData().getFacilityId(),
					platformMap
					);
		}
		
		// 子ファシリティを求める
		FacilityTreeItem[] tmpItem = treeItem.getChildren();
		
		if(tmpItem.length != 0){  // 子ファシリティに対して再帰的に設定
			for(int i = 0; i < tmpItem.length; i++){
				setScopeTree(tmpItem[i], deviceDataList, platformMap);
			}
		}
	}

	/**
	 * 指定のファシリティIDのオブジェクトがスコープか否かを判定します
	 * @param facilityID ファシリティID
	 * @return 指定のファシリティIDのオブジェクトがスコープの場合は true を返します
	 */
	public boolean isScope(final String facilityID){
		return m_scopeTable.get(facilityID) != null;
	}
	
	/**
	 * 指定のファシリティIDのオブジェクトがノードか否かを判定します
	 * @param facilityID ファシリティID
	 * @return 指定のファシリティIDのオブジェクトがノードの場合は true を返します
	 */
	public boolean isNode(final String facilityID){
		return m_nodeTable.get(facilityID) != null;
	}
	
	/**
	 * 登録されている全てのスコープを取得します。
	 * @return スコープのファシリティIDの配列
	 */
	public String[] getAllFacilityIdList(){
		// 返却値を格納する配列
		String[] ret = new String[m_nodeTable.size() + m_scopeTable.size()];

		int i=0;
		
		// スコープのファシリティIDを取得
		Iterator<Scope> sItr = m_scopeTable.values().iterator();
		while(sItr.hasNext()){
			Scope scope = sItr.next();
			ret[i] = scope.getFacilityId();
			i++;
		}

		// ノードのファシリティIDを取得
		Iterator<Node> nItr = m_nodeTable.values().iterator();
		while(nItr.hasNext()){
			Node node = nItr.next();
			ret[i] = node.getFacilityId();
			i++;
		}
		
		return ret;
	}

	/**
	 *  現在ポーラからこのオブジェクトに取り込んだMIB値の中で
	 *  最後に収集したものの時刻を取得します。
	 *  
	 * @return 最終収集時刻
	 */
	public long getLastCollectTime() {
		return m_lastCollectTime;
	}
}
