/*
                                                                                                
Copyright (C) 2008 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
*/

package com.clustercontrol.poller.snmp;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * SNMPポーリングのための情報を保持するクラス
 */
public class SnmpPollingConfig {
	private static Log m_log = LogFactory.getLog( SnmpPollingConfig.class );
	private Object modifyLock = new Object();
	
	/**
	 * SNMPポーリングのためのSNMPパラメータ設定を保持する
	 */
	private SnmpConfig m_snmpConfig;
	
	/**
	 * ポーリングしたOID群の結果のインデックスが一致しているか否かをチェックするか否かを指定するフラグ
	 */
	private boolean m_indexCheckFlg;
	
	/**
	 * Key   : 収集名
	 * Value : 収集間隔とOIDリストの設定
	 */
	private ConcurrentHashMap<String ,IntervalAndOids> m_targetOidMap;

	/**
	 * m_targetOidMapが更新されるたびに更新の必要がある
	 * 
	 * Key   : OID
	 * Value : OID毎のポーリング実行間隔設定
	 */
	private ConcurrentHashMap<String, IntervalAndNextTime> m_scheduleMap;

	/**
	 * 収集間隔毎にデータテーブルへの次回書き込み時刻を保持する
	 * m_targetOidMapが更新されるたびに更新の必要がある
	 * 
	 * Key   : 収集間隔（秒単位）
	 * Value : 次回書き込み時刻
	 */
	private ConcurrentHashMap<Integer, Long> m_refreshTableTimes;
	
	/**
	 * 最小の収集間隔（秒単位）
	 */
	private int m_minPollingInterval = -1;
	
	/**
	 * 指定のSNMPパラメータ設定でインスタンスを生成する
	 * 
	 * @param config SNMPパラメータ設定
	 */
	public SnmpPollingConfig(SnmpConfig config, boolean indexCheckFlg){
		m_snmpConfig = config;
		m_targetOidMap = new ConcurrentHashMap<String ,IntervalAndOids>();
		m_scheduleMap = new ConcurrentHashMap<String, IntervalAndNextTime>();
		m_refreshTableTimes = new ConcurrentHashMap<Integer, Long>();
		m_indexCheckFlg = indexCheckFlg;
	}
	
	/**
	 * 収集対象のOIDを登録する
	 * @param collectorName 収集名
	 * @param interval 収集間隔
	 * @param oids OIDのリスト
	 */
	public void addPollingOids(String collectorName, int interval, List<String> oids){
		synchronized (modifyLock) {
			// 収集間隔とOIDリストの設定を収集名をキーにしてマップに登録する
			m_targetOidMap.put(collectorName, new IntervalAndOids(interval, oids));
		}
		
		// OID単位の収集間隔を更新する
		rescheduling();
	}

	/**
	 * 指定の収集名で登録されている収集対象OIDを収集対象から除く
	 * @param collectorName 収集名
	 * @return 最小収集間隔が更新された場合はtrueを返す
	 */
	public boolean removePollingOids(String collectorName){
		synchronized (modifyLock) {
			// 指定の収集名の設定を削除する
			m_targetOidMap.remove(collectorName);
		}
		
		// 収集間隔を更新する
		return rescheduling();
	}
	
	/**
	 * 収集名で登録されている全ての収集対象OIDを収集対象から除く
	 * @param collectorName 収集名
	 * @return 最小収集間隔が更新された場合はtrueを返す
	 */
	public boolean removePollingAllOids(){
		synchronized (modifyLock) {
			// 指定の収集名の設定を削除する
			m_targetOidMap.clear();
		}
		
		// 収集間隔を更新する
		return rescheduling();
	}
	

	/**
	 * OID単位の収集間隔を更新し、
	 * 最小の収集間隔（秒単位）を算出します（m_minPollingIntervalに設定）。
	 * 新規に追加された収集間隔がある場合、収集間隔毎のデータテーブルへの次回書き込み時刻を設定します。
	 * @return 最小収集間隔が更新された場合はtrueを返す
	 */
	private boolean rescheduling(){
		synchronized(modifyLock){
			// 最小収集間隔が更新されたことを示すフラグ
			boolean modifyFlg = false;

			// OID単位の収集間隔を更新します
			m_scheduleMap.clear();

			// 一旦最小収集間隔をリセットする
			m_minPollingInterval = Integer.MAX_VALUE;

			// 収集名ごとに登録されている全ての収集対象OIDに対して処理するように
			// キーである収集名のセットでループをまわす
			Iterator<String> itr = m_targetOidMap.keySet().iterator();
			while(itr.hasNext()){
				String collectorName = itr.next();
				IntervalAndOids setting = m_targetOidMap.get(collectorName);

				// 収集ＩＤ毎の収集対象OIDのリストを取得する
				List<String> oidList = setting.getOidList();

				// 収集ＩＤ毎の収集間隔を取得する
				int interval = setting.getPollingInterval();

				// 最小収集間隔を設定する
				if(m_minPollingInterval > interval){
					m_log.debug("Set Min Polling Intreval : " + interval);
					m_minPollingInterval = interval;
					modifyFlg = true;
				}

				// 各OIDをキーにしてOID毎の収集間隔を設定する
				Iterator<String> oidItr = oidList.iterator();
				while(oidItr.hasNext()){
					String oid = oidItr.next();

					// 既に設定されている収集間隔がある場合は、その値と比較し、
					// 小さい場合に、更新する。
					if(m_scheduleMap.get(oid) != null){
						if(m_scheduleMap.get(oid).getPollingInterval() > interval){
							m_scheduleMap.get(oid).setPollingInterval(interval);
						}
					} else {
						m_scheduleMap.put(oid, new IntervalAndNextTime(interval));
					}
				}

				// 新規に追加された収集間隔がある場合は、
				// 収集間隔毎にテーブルへの次回書き込み時刻を設定する
				if(m_refreshTableTimes.get(interval) == null){
					m_refreshTableTimes.put(interval, 0L);
				}
			}

			// 最小収集間隔が設定されていない場合は、-lとする
			if(m_minPollingInterval == Integer.MAX_VALUE){
				m_minPollingInterval = -1;
				modifyFlg = true;
			}

			return modifyFlg;
		}
	}

	
	public List<String> getCurrentTargetOids(long currentTime){
		List<String> currentTarget = new ArrayList<String>();
		
		// 各OIDをキーにしてOID毎の次回収集時刻を取得する
		Iterator<String> itr = m_scheduleMap.keySet().iterator();
		while(itr.hasNext()){
			String oid = itr.next();
			
			if(m_scheduleMap.get(oid) != null){
				long nextPollingTime = m_scheduleMap.get(oid).getNextPollingTime();
				int interval = m_scheduleMap.get(oid).getPollingInterval();
				
				if(nextPollingTime <= currentTime){
					currentTarget.add(oid);
					
					// 次回ポーリング時刻を設定する
					long intervalMillis = interval * 1000L;
					
					// 処理遅延が発生した場合を考慮し、現時刻よりも先の時刻を設定するように算出
					long count = (currentTime - nextPollingTime) / intervalMillis;
					
					nextPollingTime = nextPollingTime + intervalMillis * (count + 1);
					
					m_scheduleMap.get(oid).setNextPollingTime(nextPollingTime);
				}
			} 
		}
		
		return currentTarget;
	}
	
	/**
	 * 与えられた時刻で更新すべき収集間隔のリストを返します。
	 * 
	 * 例）
	 *   収集間隔、1分、5分で収集されている場合
	 *   前回収集時刻が、10:00:00 の場合で、currentTimeとして、10:01:08が与えられると、
	 *   収集間隔 1分 は、この時刻で更新すべき
	 *   収集間隔 5分 は、10:05:00までは更新すべきでない
	 *   よって、
	 *   返る収集間隔は、1分（=60）となります。
	 * 
	 * @param currentTime 現在時刻（チェック対象時刻）
	 * @return 収集間隔のリスト
	 */
	public List<Integer> getCurrentRefreshIntervals(long currentTime){
		m_log.debug("current time : " + currentTime);
		
		List<Integer> currentTarget = new ArrayList<Integer>();
		
		// 収集間隔をキーにして次回書き込み時刻を取得する
		Iterator<Integer> itr = m_refreshTableTimes.keySet().iterator();
		while(itr.hasNext()){
			int interval = itr.next();
			
			long nextTime = m_refreshTableTimes.get(interval);
			if(nextTime <= currentTime){
				currentTarget.add(interval);
				
				// 次回書き込み時刻を設定する
				long intervalMillis = interval * 1000L;
				
				// 処理遅延が発生した場合を考慮し、現時刻よりも先の時刻を設定するように算出
				long count = (currentTime - nextTime) / intervalMillis;
				
				nextTime = nextTime + intervalMillis * (count + 1);
				
				m_log.debug("put nexttime : " + interval + ", " + nextTime);

				m_refreshTableTimes.put(interval, nextTime);
			}
		}
		
		return currentTarget;
	}
	
	public Set<String> getCollectorNames(){
		return m_targetOidMap.keySet();
	}
	
	/**
	 * SNMPポーリングに必要な設定値を返します。
	 * @return SNMPパラメータ設定
	 */
	public SnmpConfig getSnmpConfig() {
		return m_snmpConfig;
	}
	
	/**
	 * SNMPポーリングに必要な設定値を設定します。
	 * @param config SNMPパラメータ設定
	 */
	public void setSnmpConfig(SnmpConfig config) {
		m_snmpConfig = config;
	}

	/**
	 * 設定されている収集名のうち収集間隔が最小のものの値を返します。
	 * @return 最小収集間隔
	 */
	public int getMinPollingInterval() {
		return m_minPollingInterval;
	}
	
	/**
	 * 指定の収集名の収集が設定されていることを確認する
	 * @param collectorName 収集名
	 * @return 設定されている場合はtrue
	 */
	public boolean containsCollectorName(String collectorName){
		synchronized (modifyLock) {
			return m_targetOidMap.containsKey(collectorName);
		}
	}

	public boolean isIndexCheckFlg() {
		return m_indexCheckFlg;
	}
	
	/**
	 * 管理している収集名ごとの情報を文字列のリストで出力します。
	 * @return 管理している収集名ごとの収集期間、最終収集時刻情報
	 */
	public String getDebugInfo(){
		String debugStr = "";
		
		Iterator<String> itr = m_targetOidMap.keySet().iterator();
		
		while(itr.hasNext()){
			String collectorName = itr.next();
			List<String> oidList = m_targetOidMap.get(collectorName).m_oidList;
			int interval = m_targetOidMap.get(collectorName).getPollingInterval();
			
			String str = "\t" + collectorName + "  interval : " + interval + "\n";
			Iterator<String> oids = oidList.iterator();
			while(oids.hasNext()){
				String oid = oids.next();
				IntervalAndNextTime ian = m_scheduleMap.get(oid);
				str = str + "\t" + oid
				+ "\t" + ian.getPollingInterval()
				+ "\t" + new Date(ian.getNextPollingTime()) + "\n";
			}
			
			debugStr = debugStr + str;
		}
		
		return debugStr;
	}
	
	private class IntervalAndOids {
		private int m_pollingInterval;
		private List<String> m_oidList;
		
		public IntervalAndOids(int pollingInterval, List<String> oidList){
			m_pollingInterval = pollingInterval;
			m_oidList = oidList;
		}
		
		public int getPollingInterval(){
			return m_pollingInterval;
		}
		
		public List<String> getOidList(){
			return m_oidList;
		}
	}
	
	private class IntervalAndNextTime {
		private int m_pollingInterval;
		private long m_nextPollingTime;
		
		public IntervalAndNextTime(int pollingInterval){
			m_pollingInterval = pollingInterval;
			m_nextPollingTime = 0L;
		}

		public int getPollingInterval(){
			return m_pollingInterval;
		}
		
		public void setPollingInterval(int interval){
			m_pollingInterval = interval;
		}
		
		public long getNextPollingTime(){
			return m_nextPollingTime;
		}
		
		public void setNextPollingTime(long nextPollingTime){
			m_nextPollingTime = nextPollingTime;
		}
	}
}
