/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.factory;

import java.util.Collection;
import java.util.Iterator;

import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.commons.ejb.UsedFacilityException;
import com.clustercontrol.repository.bean.TopicConstant;
import com.clustercontrol.repository.ejb.entity.DeviceInfoLocal;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityTreeLocal;
import com.clustercontrol.repository.ejb.entity.FacilityTreeUtil;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.repository.ejb.entity.FileSystemInfoLocal;
import com.clustercontrol.repository.message.UpdateRepositoryInfo;
import com.clustercontrol.repository.util.SendTopic;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ファシリティを削除するクラス<BR>
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class DeleteFacility {
    protected static Log m_log = LogFactory.getLog( DeleteFacility.class );
    
	/**
	 * ノード削除
	 * 
	 * @param facilityId
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws UsedFacilityException 
	 */
	public void deleteNode(String facilityId) throws FinderException, RemoveException, NamingException, UsedFacilityException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.REPOSITORY);
    	
    	//ファシリティIDが他の機能で使用されているかチェック
    	new CheckFacility().isUseFacility(facilityId);
    	
    	try {
			//ファシリティIDのFacilityを取得
			FacilityLocal facility = FacilityUtil.getLocalHome().findByFacilityId(facilityId);

			//ファシリティIDでFacilityTreeを取得
			Collection collection = facility.getFacilityTree();
			if(collection != null){
				Iterator itr = collection.iterator();
				while(itr.hasNext()){
					FacilityTreeLocal facilityTree = (FacilityTreeLocal)itr.next();
					//FacilityTreeを削除
					removeFacilityTree(facilityTree);
				}
			}
			//Facilityを削除
			removeFacility(facility);
			
			try {
				//Topicに送信
				SendTopic send = new SendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
				send.put(new UpdateRepositoryInfo());
			} catch (Exception e) {
				m_log.debug("addScope() : 更新情報送信エラー : " + e.getMessage());
			}
		} catch (EJBException e) {
            AplLogger apllog = new AplLogger("REP", "rep");
            String[] args = {facilityId};
            apllog.put("SYS", "010", args);
            
			m_log.debug("deleteNode() : " + e.getMessage());
			throw e;
		} catch (RemoveException e) {
            AplLogger apllog = new AplLogger("REP", "rep");
            String[] args = {facilityId};
            apllog.put("SYS", "010", args);
            
            m_log.debug("deleteNode() : " + e.getMessage());
			throw e;
		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("REP", "rep");
            String[] args = {facilityId};
            apllog.put("SYS", "010", args);
            
            m_log.debug("deleteNode() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * スコープ削除
	 * 
	 * @param facilityId
	 * @throws FinderException
	 * @throws NamingException
	 * @throws RemoveException
	 * @throws UsedFacilityException 
	 */
	public void deleteScope(String facilityId) throws FinderException, NamingException, RemoveException, UsedFacilityException  {
    	//アクセスロック
    	AccessLock.lock(AccessLock.REPOSITORY);
    	
    	//ファシリティIDが他の機能で使用されているかチェック
    	new CheckFacility().isUseFacility(facilityId);
    	
    	try {
			//ファシリティIDのFacilityを取得
			FacilityLocal facility = FacilityUtil.getLocalHome().findByFacilityId(facilityId);
			
			//ファシリティIDでFacilityTreeを取得
			Collection collection = facility.getFacilityTree();
			if(collection != null){
				Iterator itr = collection.iterator();
				while(itr.hasNext()){
					FacilityTreeLocal facilityTree = (FacilityTreeLocal)itr.next();
					//FacilityTreeを削除
					removeFacilityTree(facilityTree);
				}
			}
			
			try {
				//Topicに送信
				SendTopic send = new SendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
				send.put(new UpdateRepositoryInfo());
			} catch (Exception e) {
				m_log.debug("addScope() : 更新情報送信エラー : " + e.getMessage());
			}
		} catch (FinderException e) {
            AplLogger apllog = new AplLogger("REP", "rep");
            String[] args = {facilityId};
            apllog.put("SYS", "013", args);
            
            m_log.debug("deleteScope() : " + e.getMessage());
			throw e;
		} catch (EJBException e) {
            AplLogger apllog = new AplLogger("REP", "rep");
            String[] args = {facilityId};
            apllog.put("SYS", "013", args);
            
            m_log.debug("deleteScope() : " + e.getMessage());
			throw e;
		} catch (RemoveException e) {
            AplLogger apllog = new AplLogger("REP", "rep");
            String[] args = {facilityId};
            apllog.put("SYS", "013", args);
            
            m_log.debug("deleteScope() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * スコープへのノードの割当て解除
	 * 
	 * @param parentFacilityId
	 * @param facilityIds
	 * @throws FinderException
	 * @throws NamingException
	 * @throws RemoveException
	 */
	public void releaseNodeScope(String parentFacilityId, String[] facilityIds) throws FinderException, NamingException, RemoveException {
    	//アクセスロック
    	AccessLock.lock(AccessLock.REPOSITORY);
    	
    	String facilityId = null;
	    try {
			//ファシリティID直下のノードのFacilityTreeを取得
			Collection collection = FacilityTreeUtil.getLocalHome().findOneLevelNode(parentFacilityId);
			if(collection != null){
				Iterator itr = collection.iterator();
				while(itr.hasNext()){
					//ノードのFacilityTreeが削除するファシリティIDと一致したら削除
					FacilityTreeLocal facilityTree = (FacilityTreeLocal)itr.next();
					for(int i = 0; i < facilityIds.length; i++){
						String nodeFacilityId = facilityTree.getFacilityId();
						if(nodeFacilityId.compareTo(facilityIds[i]) == 0){
						    facilityId = facilityIds[i];
							removeFacilityTree(facilityTree);
							break;
						}
					}
				}
			}
			
			try {
				//Topicに送信
				SendTopic send = new SendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
				send.put(new UpdateRepositoryInfo());
			} catch (Exception e) {
				m_log.debug("addScope() : 更新情報送信エラー : " + e.getMessage());
			}
		} catch (EJBException e) {
            AplLogger apllog = new AplLogger("REP", "rep");
            String[] args = {facilityId, parentFacilityId};
            apllog.put("SYS", "016", args);
            
            m_log.debug("releaseNodeScope() : " + e.getMessage());
			throw e;
		} catch (NamingException e) {
            AplLogger apllog = new AplLogger("REP", "rep");
            String[] args = {facilityId, parentFacilityId};
            apllog.put("SYS", "016", args);
            
            m_log.debug("releaseNodeScope() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * Facility削除処理
	 * 
	 * @param facility
	 * @throws RemoveException
	 * @throws NamingException 
	 */
	protected void removeFacility(FacilityLocal facility) throws RemoveException, NamingException {
		//子DeviceInfoを取得
		Collection collectionDevice = null;
		//子FileSystemInfoを取得
		Collection collectionFileSystem = null;
		
		try {
			collectionDevice = facility.getDeviceInfo();
			collectionFileSystem = facility.getFileSystemInfo();
		} catch (FinderException e) {
		    m_log.debug("removeFacility() : " + e.getMessage());
		}
		
		try {
			//子DeviceInfoを削除する
			if(collectionDevice != null && collectionDevice.size() > 0){
				Iterator itr = collectionDevice.iterator();
				while(itr.hasNext()){
					DeviceInfoLocal device = (DeviceInfoLocal)itr.next();
					device.remove();
				}
			}
		
			//子FileSystemInfoを削除する
			if(collectionFileSystem != null && collectionFileSystem.size() > 0){
				Iterator itr = collectionFileSystem.iterator();
				while(itr.hasNext()){
					FileSystemInfoLocal fileSystem = (FileSystemInfoLocal)itr.next();
					fileSystem.remove();
				}
			}
			
			//自Facilityを削除
			facility.remove();
			
		} catch (RemoveException e) {
		    m_log.debug("removeFacility() : " + e.getMessage());
			throw e;
		}
	}
	
	/**
	 * FacilityTree削除処理
	 * 
	 * @param facilityTree
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws FinderException
	 */
	protected void removeFacilityTree(FacilityTreeLocal facilityTree) throws RemoveException, NamingException, FinderException {
		
		//子を取得
		Collection collection = null;
		try {
			collection = facilityTree.getChildren();
		} catch (FinderException e) {
		    m_log.debug("removeFacilityTree() : " + e.getMessage());
		}
		
		//子FacilityTreeを削除する
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				FacilityTreeLocal children = (FacilityTreeLocal)itr.next();
				removeFacilityTree(children);
			}
		}
		
		try {
			//自FacilityTreeがスコープならば、Facilityを削除する
			FacilityLocal facility = facilityTree.getFacility();
			if(facility.getType() == FacilityConstant.TYPE_SCOPE){
				removeFacility(facility);
			}

			//自FacilityTreeを削除する
			FacilityTreeUtil.getLocalHome().remove(facilityTree.getPrimaryKey());
			
		} catch (RemoveException e) {
		    m_log.debug("removeFacilityTree() : " + e.getMessage());
			throw e;
		} catch (FinderException e) {
		    m_log.debug("removeFacilityTree() : " + e.getMessage());
			throw e;
		}
	}
}
