/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.factory;

import java.rmi.RemoteException;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;

import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.jobs.ee.ejb.EJBInvokerJob;

import com.clustercontrol.accesscontrol.factory.AccessLock;
import com.clustercontrol.bean.Schedule;
import com.clustercontrol.bean.ScheduleConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.jobmanagement.bean.JobTriggerInfo;
import com.clustercontrol.jobmanagement.bean.JobTriggerTypeConstant;
import com.clustercontrol.jobmanagement.bean.QuartzConstant;
import com.clustercontrol.jobmanagement.bean.ScheduleInfo;
import com.clustercontrol.quartzmanager.ejb.session.QuartzManager;
import com.clustercontrol.quartzmanager.util.QuartzUtil;
import com.sun.jmx.trace.Trace;

/**
 * スケジュール情報を操作するクラスです。
 *
 * @version 2.4.0
 * @since 1.0.0
 */
public class ModifySchedule {
	/** ログ出力のインスタンス */
    protected static Log m_log = LogFactory.getLog( ModifySchedule.class );
    
	/**
	 * スケジュール情報を基にQuartzにジョブを登録します。<BR>
	 * Quartzからは、{@link com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#scheduleRunJob(String, String, JobTriggerInfo)} が呼び出されます。
	 * 
     * @param info スケジュール情報
     * @param user ユーザID
     * @throws ParseException
     * @throws NamingException
     * @throws SchedulerException
     * 
     * @see org.quartz.JobDetail
     * @see com.clustercontrol.jobmanagement.bean.QuartzConstant
     * @see com.clustercontrol.jobmanagement.bean.JobTriggerInfo
     * @see com.clustercontrol.jobmanagement.util.QuartzUtil#getQuartzManager()
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#addSchedule(org.quartz.JobDetail, org.quartz.Trigger)
     * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#getCronString(Schedule)
     */
    public void addSchedule(ScheduleInfo info, String user) throws ParseException, NamingException, SchedulerException {
        m_log.debug("addSchedule() : id=" + info.getId() + ", jobId=" + info.getJobId());
    	
        //アクセスロック
        AccessLock.lock(AccessLock.JOB);
        
        //JobDetail作成
	    JobDetail job = new JobDetail(
	            info.getId(), 
	            QuartzConstant.GROUP_NAME, 
	            EJBInvokerJob.class);
	    
	    //ジョブ名を設定
	    job.setDescription(info.getName());
	    
	    //ジョブ完了時に削除されないようにする。
	    job.setDurability(true);
	    //ジョブ実行失敗時に再実行するようにする。
	    job.setRequestsRecovery(true);
	    
	    //JobDetailに呼び出すクラスとメソッドを設定
	    job.getJobDataMap().put(
	            EJBInvokerJob.EJB_JNDI_NAME_KEY, 
	            QuartzConstant.JNDI_NAME);
	    job.getJobDataMap().put(
	            EJBInvokerJob.EJB_METHOD_KEY, 
	            QuartzConstant.METHOD_NAME);
	    String serverName = System.getProperty("jboss.server.name");
	    if(serverName.equals("all")){
	        job.getJobDataMap().put(
	                EJBInvokerJob.PROVIDER_URL, 
	                "jnp://localhost:1100");
	    }
	    
	    //実行契機情報の作成
	    JobTriggerInfo triggerInfo = new JobTriggerInfo();
	    triggerInfo.setTrigger_type(JobTriggerTypeConstant.TYPE_SCHEDULE);
	    triggerInfo.setTrigger_info(info.getName()+"("+info.getId()+")");
        
	    //JobDetailに呼び出すメソッドの引数を設定
	    Object[] jdArgs = new Object[3];
	    Class[] jdArgsType = new Class[3];
	    //ジョブIDを設定
	    jdArgs[0] = info.getJobId();
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
	    jdArgsType[0] = String.class;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
	    //カレンダIDを設定
	    jdArgs[1] = info.getCalendarId();
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
	    jdArgsType[1] = String.class;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
	    //実行契機情報を設定
	    jdArgs[2] = triggerInfo;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
	    jdArgsType[2] = JobTriggerInfo.class;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
	    
	    //JobDetailに有効/無効を設定
	    if(info.getValid() == ValidConstant.TYPE_VALID){
	        job.getJobDataMap().put(QuartzConstant.VALID_KEY, new Integer(ValidConstant.TYPE_VALID));
	    }
	    else{
	        job.getJobDataMap().put(QuartzConstant.VALID_KEY, new Integer(ValidConstant.TYPE_INVALID));
	    }
	    
	    //JobDetailにジョブ名、登録日時、更新日時を設定
	    Date now = new Date();
	    job.getJobDataMap().put(QuartzConstant.JOB_NAME_KEY, info.getJobName());
	    job.getJobDataMap().put(QuartzConstant.CREATE_USER_KEY, user);
	    job.getJobDataMap().put(QuartzConstant.CREATE_DATE_KEY, now);
	    job.getJobDataMap().put(QuartzConstant.UPDATE_USER_KEY, user);
	    job.getJobDataMap().put(QuartzConstant.UPDATE_DATE_KEY, now);
	    job.getJobDataMap().put(QuartzConstant.SCHEDULE_KEY, info.getSchedule());
	    
	    //CronTriggerを作成
	    CronTrigger cronTrigger = new CronTrigger(info.getId(), QuartzConstant.GROUP_NAME);
	    
        //起動失敗した場合は、次の起動予定時刻をセットするように設定
        cronTrigger.setMisfireInstruction(CronTrigger.MISFIRE_INSTRUCTION_DO_NOTHING);

        //スケジュールを設定
        cronTrigger.setCronExpression(getCronString(info.getSchedule()));
        
        //QuartzのSchedulerをルックアップ
        QuartzManager manager = QuartzUtil.getQuartzManager();
		
        //Schedulerにジョブを登録する
		try {
			manager.addSchedule(job, cronTrigger);
		} catch (RemoteException e) {
		}
        
        //無効の場合、ジョブ登録後にポーズ
	    if(info.getValid() == ValidConstant.TYPE_INVALID){
	    	try {
				manager.pauseSchedule(job.getName(), QuartzConstant.GROUP_NAME);
			} catch (RemoteException e) {
			}
	    }
    }
    
	/**
	 * スケジュール情報を基にQuartzに登録したジョブを変更します。<BR>
	 * Quartzからは、{@link com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#scheduleRunJob(String, String,JobTriggerInfo)} が呼び出されます。
	 * 
     * @param info スケジュール情報
     * @param user ユーザID
     * @throws NamingException
     * @throws SchedulerException
     * @throws ParseException
     * 
     * @see org.quartz.JobDetail
     * @see com.clustercontrol.jobmanagement.bean.QuartzConstant
     * @see com.clustercontrol.jobmanagement.bean.JobTriggerInfo
     * @see com.clustercontrol.jobmanagement.util.QuartzUtil#getQuartzManager()
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#getJobDetail(java.lang.String, java.lang.String)
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#addSchedule(org.quartz.JobDetail, org.quartz.Trigger)
     * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#getCronString(Schedule)
     * @see com.clustercontrol.jobmanagement.factory.ModifySchedule#deleteSchedule(String)
     */
    public void modifySchedule(ScheduleInfo info, String user) throws NamingException, SchedulerException, ParseException {
        m_log.debug("modifySchedule() : id=" + info.getId() + ", jobId=" + info.getJobId());
        
    	//アクセスロック
        AccessLock.lock(AccessLock.JOB);
        
        //QuartzのSchedulerをルックアップ
        QuartzManager manager = QuartzUtil.getQuartzManager();
		
        //JobDetail取得
        JobDetail job = null;
		try {
			job = manager.getJobDetail(info.getId(), QuartzConstant.GROUP_NAME);
		} catch (RemoteException e) {
		}

	    //ジョブ名を設定
	    job.setDescription(info.getName());
	    
	    //ジョブ完了時に削除されないようにする。
	    job.setDurability(true);
	    //ジョブ実行失敗時に再実行するようにする。
	    job.setRequestsRecovery(true);
	    
	    //実行契機情報の作成
	    JobTriggerInfo triggerInfo = new JobTriggerInfo();
	    triggerInfo.setTrigger_type(JobTriggerTypeConstant.TYPE_SCHEDULE);
	    triggerInfo.setTrigger_info(info.getName()+"("+info.getId()+")");
	    
	    //JobDetailに呼び出すメソッドの引数を設定
	    Object[] jdArgs = new Object[3];
	    Class[] jdArgsType = new Class[3];
	    //ジョブIDを設定
	    jdArgs[0] = info.getJobId();
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
	    jdArgsType[0] = String.class;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
	    //カレンダIDを設定
	    jdArgs[1] = info.getCalendarId();
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
	    jdArgsType[1] = String.class;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
	    //実行契機情報を設定
	    jdArgs[2] = triggerInfo;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARGS_KEY, jdArgs);
	    jdArgsType[2] = JobTriggerInfo.class;
	    job.getJobDataMap().put(EJBInvokerJob.EJB_ARG_TYPES_KEY, jdArgsType);
	    
	    //JobDetailに有効/無効を設定
	    if(info.getValid() == ValidConstant.TYPE_VALID){
	        job.getJobDataMap().put(QuartzConstant.VALID_KEY, new Integer(ValidConstant.TYPE_VALID));
	    }
	    else{
	        job.getJobDataMap().put(QuartzConstant.VALID_KEY, new Integer(ValidConstant.TYPE_INVALID));
	    }
	    
	    //JobDetailに登録日時、更新日時を設定
	    job.getJobDataMap().put(QuartzConstant.JOB_NAME_KEY, info.getJobName());
	    job.getJobDataMap().put(QuartzConstant.CREATE_USER_KEY, info.getCreateUser());
	    job.getJobDataMap().put(QuartzConstant.CREATE_DATE_KEY, info.getCreateTime());
	    job.getJobDataMap().put(QuartzConstant.UPDATE_USER_KEY, user);
	    job.getJobDataMap().put(QuartzConstant.UPDATE_DATE_KEY, new Date());
	    job.getJobDataMap().put(QuartzConstant.SCHEDULE_KEY, info.getSchedule());
	    
	    //CronTriggerを作成
	    CronTrigger cronTrigger = new CronTrigger(info.getId(), QuartzConstant.GROUP_NAME);

	    // 実行に失敗した場合は次のfiretimeをセットする。
	    cronTrigger.setMisfireInstruction(CronTrigger.MISFIRE_INSTRUCTION_DO_NOTHING);
	    
        //スケジュールを設定
        cronTrigger.setCronExpression(getCronString(info.getSchedule()));
        
        //
        deleteSchedule(info.getId());
        
        //Schedulerにジョブを登録する
        try {
			manager.addSchedule(job, cronTrigger);
		} catch (RemoteException e) {
		}
        
        //無効の場合、ジョブ登録後にポーズ
	    if(info.getValid() == ValidConstant.TYPE_INVALID){
	    	try {
				manager.pauseSchedule(job.getName(), job.getGroup());
			} catch (RemoteException e) {
			}
	    }
    }
    
	/**
	 * スケジュール情報を基にQuartzに登録したジョブを削除します。
	 * 
     * @param scheduleId スケジュールID
     * @throws NamingException
     * @throws SchedulerException
     * 
     * @see com.clustercontrol.jobmanagement.bean.QuartzConstant
     * @see com.clustercontrol.jobmanagement.util.QuartzUtil#getQuartzManager()
     * @see com.clustercontrol.quartzmanager.ejb.session.QuartzManager#deleteSchedule(java.lang.String, java.lang.String)
     */
    public void deleteSchedule(String scheduleId) throws NamingException, SchedulerException {
        m_log.debug("deleteSchedule() : id=" + scheduleId);
        
    	//アクセスロック
        AccessLock.lock(AccessLock.JOB);
        
        //QuartzのSchedulerをルックアップ
        QuartzManager manager = QuartzUtil.getQuartzManager();
		
        //ジョブ削除
		try {
			manager.deleteSchedule(scheduleId, QuartzConstant.GROUP_NAME);
		} catch (RemoteException e) {
		}
    }
    
	/**
	 * スケジュール情報を基にcron文字列を作成します。
	 * 
     * @param scheduleId スケジュールID
     * @throws NamingException
     * @throws SchedulerException
     * 
     * @see com.clustercontrol.bean.ScheduleConstant
     */
    protected String getCronString(Schedule schedule){
        m_log.debug("getCronString()");
        
        //cron形式でスケジュールを作成する
        StringBuffer cron = new StringBuffer();
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(schedule.getDate());
        if(schedule.getType() == ScheduleConstant.TYPE_EVERY_YEAR){
            cron.append(calendar.get(Calendar.SECOND));
            cron.append(" ");
            cron.append(calendar.get(Calendar.MINUTE));
            cron.append(" ");
            cron.append(calendar.get(Calendar.HOUR_OF_DAY));
            cron.append(" ");
            cron.append(calendar.get(Calendar.DAY_OF_MONTH));
            cron.append(" ");
            cron.append(calendar.get(Calendar.MONTH) + 1);
            cron.append(" ? *");
        }
        else if(schedule.getType() == ScheduleConstant.TYPE_EVERY_MONTH){
            cron.append(calendar.get(Calendar.SECOND));
            cron.append(" ");
            cron.append(calendar.get(Calendar.MINUTE));
            cron.append(" ");
            cron.append(calendar.get(Calendar.HOUR_OF_DAY));
            cron.append(" ");
            cron.append(calendar.get(Calendar.DAY_OF_MONTH));
            cron.append(" * ? *");
        }
        else if(schedule.getType() == ScheduleConstant.TYPE_EVERY_DAY){
            cron.append(calendar.get(Calendar.SECOND));
            cron.append(" ");
            cron.append(calendar.get(Calendar.MINUTE));
            cron.append(" ");
            cron.append(calendar.get(Calendar.HOUR_OF_DAY));
            cron.append(" * * ? *");
        }
        else if(schedule.getType() == ScheduleConstant.TYPE_EVERY_HOUR){
            cron.append(calendar.get(Calendar.SECOND));
            cron.append(" ");
            cron.append(calendar.get(Calendar.MINUTE));
            cron.append(" * * * ? *");
        }
        else if(schedule.getType() == ScheduleConstant.TYPE_EVERY_WEEK){
            cron.append(calendar.get(Calendar.SECOND));
            cron.append(" ");
            cron.append(calendar.get(Calendar.MINUTE));
            cron.append(" ");
            cron.append(calendar.get(Calendar.HOUR_OF_DAY));
            cron.append(" ? * ");
            cron.append(calendar.get(Calendar.DAY_OF_WEEK));
            cron.append(" *");
        }
        else{
            cron.append(calendar.get(Calendar.SECOND));
            cron.append(" ");
            cron.append(calendar.get(Calendar.MINUTE));
            cron.append(" ");
            cron.append(calendar.get(Calendar.HOUR_OF_DAY));
            cron.append(" ");
            cron.append(calendar.get(Calendar.DAY_OF_MONTH));
            cron.append(" ");
            cron.append(calendar.get(Calendar.MONTH) + 1);
            cron.append(" ? ");
            cron.append(calendar.get(Calendar.YEAR));
        }
        return cron.toString();
    }
}
