/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.troubledetection.output;

import java.sql.Timestamp;
import java.util.Calendar;
import java.util.HashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.EventConfirmConstant;
import com.clustercontrol.bean.ExclusionConstant;
import com.clustercontrol.monitor.message.LogOutputInfo;
import com.clustercontrol.troubledetection.message.LogOutputControlInfo;

/**
 * 出力抑制クラス
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class OutputControl {
	private static Log log = LogFactory.getLog( OutputControl.class );
	
	private static HashMap<String, LogOutputControlInfo> controlBuffer = new HashMap<String, LogOutputControlInfo>();
	
	private Timestamp outputDate = null;
	
	/** イベントログ情報生成フラグ */
	private boolean insertFlg;
	
	/** イベントログ情報更新フラグ */
	private boolean updateFlg;
	
	/**
	 * イベントログ情報出力処理
	 * 
	 * @param logOutput
	 * @param outputDate
	 * @return
	 */
	public void control(LogOutputInfo logOutput, Timestamp outputDate) {
		log.debug("control() start : MonitorId=" + logOutput.getMonitorId() + ", MessageId" + logOutput.getMessageId());
		
		this.outputDate = outputDate;
		this.insertFlg = false;
		this.updateFlg = false;

		if(logOutput.isEventLogFlg())
		{
			// 抑制する場合
			if(logOutput.getExcludeFlg() == ExclusionConstant.TYPE_FREQUENCY || 
					logOutput.getExcludeFlg() == ExclusionConstant.TYPE_PERIOD || 
					logOutput.getExcludeFlg() == ExclusionConstant.TYPE_PRIORITY){
				
				// 情報発生日時から見て最新の抑制されていないイベントログ情報の検索
				LogOutputControlInfo controlInfo = controlBuffer.get(logOutput.getMonitorId());
				
				// すでにイベントログ情報が存在する場合
				if(controlInfo != null)
				{
					// 抑制対象の場合、イベントログ情報を更新
					if(isInhibited(logOutput, controlInfo)){
						updateFlg = true;
					}
					else{
						insertFlg = true;
					}
				}
				// イベントログ情報が存在しない場合
				else
				{
					// イベントログ情報の生成
					insertFlg = true;
				}
			}
			// 抑制しない場合
			else{
				// イベントログ情報の生成
				insertFlg = true;
			}
			
			// イベントログ情報の生成
			if(insertFlg)
			{
				LogOutputControlInfo controlInfo = new LogOutputControlInfo();
				controlInfo.setLogOutputInfo(logOutput);
				controlInfo.setDuplicationCount(0);
				controlBuffer.put(logOutput.getMonitorId(), controlInfo);
				
				log.debug("control() insert : MonitorId=" + logOutput.getMonitorId() + ", MessageId" + logOutput.getMessageId());
			}
			
			// エンティティ情報の更新（抑制対象）
			if(updateFlg)
			{
				// 元のエンティティ情報を更新
				LogOutputControlInfo controlInfo = controlBuffer.get(logOutput.getMonitorId());
				controlInfo.setDuplicationCount(controlInfo.getDuplicationCount() + 1);
				controlBuffer.put(logOutput.getMonitorId(), controlInfo);
				
				log.debug("control() update : MonitorId=" + logOutput.getMonitorId() + ", MessageId" + logOutput.getMessageId());
				
				if(logOutput.getExcludeConfirmFlg() == EventConfirmConstant.TYPE_CONFIRMED ||
						logOutput.getExcludeConfirmFlg() == EventConfirmConstant.TYPE_UNCONFIRMED){
					insertFlg = true;
					
					log.debug("control() insert : MonitorId=" + logOutput.getMonitorId() + ", MessageId" + logOutput.getMessageId());
				}
			}
		}
		
		log.debug("control() end : MonitorId=" + logOutput.getMonitorId() + ", MessageId" + logOutput.getMessageId());
	}

	/**
	 * 抑制対象判定処理
	 * 
	 * @param logOutput ログ情報
	 * @param eventLog イベントログ情報
	 * @return　抑制対象の場合、true
	 */
	public boolean isInhibited(LogOutputInfo logOutput, LogOutputControlInfo controlInfo){
		
		// 抑制回数の範囲内か
		if(logOutput.getExcludeFlg() == ExclusionConstant.TYPE_FREQUENCY){
			if(controlInfo.getDuplicationCount() < logOutput.getExcludeNumber()){
				return true;
			}
		}
		// 抑制期間の範囲内か
		else if(logOutput.getExcludeFlg() == ExclusionConstant.TYPE_PERIOD){
			
			Calendar cal = Calendar.getInstance();
			cal.setTime(controlInfo.getLogOutputInfo().getGenerationDate());
			cal.add(Calendar.MINUTE, logOutput.getExcludePeriod());
			
			if(cal.getTime().compareTo(logOutput.getGenerationDate()) > 0){
				return true;
			}
		}
		//抑制する重要度か
		else if(logOutput.getExcludeFlg() == ExclusionConstant.TYPE_PRIORITY){
			if(controlInfo.getLogOutputInfo().getPriority() == logOutput.getPriority()){
				return true;
			}
		}
		return false;
	}
	
	
	/**
	 * @return m_outputDate を戻します。
	 */
	public Timestamp getOutputDate() {
		return this.outputDate;
	}
	
	/**
	 * @param outputDate を設定します。
	 */
	public void setOutputDate(Timestamp outputDate) {
		this.outputDate = outputDate;
	}
	
	/**
	 * @return insertFlg を戻します。
	 */
	public boolean isInsertFlg() {
		return insertFlg;
	}
	
	/**
	 * @return updateFlg を戻します。
	 */
	public boolean isUpdateFlg() {
		return updateFlg;
	}
}
