/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.snmptrap.factory;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.repository.message.UpdateRepositoryInfo;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapInfo;
import com.clustercontrol.snmptrap.bean.QueueConstant;
import com.clustercontrol.snmptrap.bean.SnmpTrapFacilityInfo;
import com.clustercontrol.snmptrap.bean.SnmpTrapOidInfo;
import com.clustercontrol.snmptrap.ejb.session.MonitorSnmpTrapRunManagementLocal;
import com.clustercontrol.snmptrap.ejb.session.MonitorSnmpTrapRunManagementUtil;
import com.clustercontrol.snmptrap.message.SnmpTrapMessageInfo;
import com.clustercontrol.snmptrap.message.UpdateSnmpTrapInfo;
import com.clustercontrol.snmptrap.util.ReceiveRepositoryTopic;
import com.clustercontrol.snmptrap.util.ReceiveSnmpTrapTopic;

/**
 * SNMPトラップ受信を管理するクラス<BR>
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class TrapSnmpManager {
	
	protected static Log m_log = LogFactory.getLog( TrapSnmpManager.class );
	
	/** プラグインID */
	static public final String PLUGIN_ID = "SNMPTRAP";
	
	/** メッセージID */
	public static final String MESSAGE_ID_INFO = "001";
	public static final String MESSAGE_ID_WARNING = "002";
	public static final String MESSAGE_ID_CRITICAL = "003";
	public static final String MESSAGE_ID_UNKNOWN = "004";
	
	/** オリジナルメッセージ */
	public static final String REGEX_PREFIX = "%parm\\[#";
	public static final String REGEX_SUFFIX = "\\]%";
	
	/** ノード属性情報 */
	protected static final ArrayList<String> m_attributeList = new ArrayList<String>();
	static{
		m_attributeList.add(FacilityAttributeConstant.IPNETWORKNUMBER);
		m_attributeList.add(FacilityAttributeConstant.IPNETWORKNUMBERV6);
		m_attributeList.add(FacilityAttributeConstant.NODENAME);
	}
	
	/** IPアドレス情報キャッシュ */
	// Key：IPアドレス、Value：ファイシリティIDリスト
	protected Map<String, ArrayList<String>> m_ipAddressMap = new ConcurrentHashMap<String, ArrayList<String>>();
	
	/** ホスト名情報キャッシュ */
	// Key：ホスト名、Value：ファイシリティIDリスト
	protected Map<String, ArrayList<String>> m_hostNameMap = new ConcurrentHashMap<String, ArrayList<String>>();
	
	/** SNMPTRAP監視情報キャッシュ */
	// Key：監視項目ID、Value：SNMPTRAP監視情報
	protected Map<String, SnmpTrapFacilityInfo> m_snmpTrapInfoMap = null;
	
	protected TrapSnmp m_trapSnmp = null;
	
	protected ReceiveRepositoryTopic m_receiveRepositoryTopic = null;
	
	protected ReceiveSnmpTrapTopic m_receiveSnmpTrapTopic = null;
	
	protected SendQueue m_sendQueue = null;
	
	/** リポジトリ情報 */
	protected RepositoryControllerLocal m_repository;
	
	/**
	 * 開始処理
	 */
	public void exec() {
		
		try {
			m_sendQueue = new SendQueue(QueueConstant.QUEUE_NAME_NOTIFY);
			m_repository = RepositoryControllerUtil.getLocalHome().create();
			
			// キャッシュ取得
			
			updateCache();
			
			// 受信トピック生成(SNMPTrap監視設定、リポジトリ情報の変更を
			// お知らせする用
			m_receiveRepositoryTopic = new ReceiveRepositoryTopic(this);
			m_receiveSnmpTrapTopic = new ReceiveSnmpTrapTopic(this);
			
			// SNMPトラップ受信 開始
			m_trapSnmp = new TrapSnmp(this);
			m_trapSnmp.exec();
			
		} catch (NamingException e) {
			e.printStackTrace();
		} catch (JMSException e) {
			e.printStackTrace();
		} catch (CreateException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * 終了処理
	 */
	public void terminate() {
		
		if(m_trapSnmp != null)
			m_trapSnmp.terminate();
		
		if(m_receiveRepositoryTopic != null)
			m_receiveRepositoryTopic.terminate();
		
		if(m_receiveSnmpTrapTopic != null)
			m_receiveSnmpTrapTopic.terminate();
	}
	
	/**
	 * SNMPTRAP監視対象かチェックし、対象の場合は通知
	 *  snmpReceivedTrap（トラップ到着）から呼ばれるメソッド
	 * 
	 * @param trapDate SNMPトラップ受信日時
	 * @param communityName コミュニティ名
	 * @param trapOid OID
	 * @param genericId Generic ID
	 * @param specificId Specific ID
	 * @param varBindValue varBindValue
	 * @param facilityIdList ファシリティID一覧
	 * @return 
	 */
	public boolean find(
			Date trapDate,
			String communityName,
			String trapOid, 
			int genericId, 
			int specificId, 
			String[] varBind,
			ArrayList<String> facilityIdList,
			String agentAdder) {
		
		m_log.debug("find() : start" );
		m_log.debug("find() : received trapOID : " + trapOid);
		m_log.debug("find() : received generic : " + Integer.toString(genericId) );
		m_log.debug("find() : received specific: " + Integer.toString(specificId) );
		
		if(m_snmpTrapInfoMap == null)
			return true;
		
		Set set = m_snmpTrapInfoMap.keySet(); //setはMonitorIdの集合
		
		for (Iterator iter = set.iterator(); iter.hasNext();) {
			String monitorId = (String) iter.next();
			
			SnmpTrapFacilityInfo info = m_snmpTrapInfoMap.get(monitorId);
			if(info == null)
				continue;
			
			// コミュニティ名が同一か
			
			String communityNameCache = info.getCommunityName();
			/*m_log.error("find() : communityNameCache " + communityNameCache);
			 m_log.error("find() : communityName " + communityName);*/
			if(communityNameCache.equals(communityName)){
				
				// ファシリティIDが含まれているか
				// ファシリティIDが含まれているか
				ArrayList<String> facilityIdListCache ;
				
				
				if(info.getFacilityId().equals(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE)){
				//未登録ノードの場合には、UNREGISTEREFD_SCOPEだけが検出対象スコープ	
					 facilityIdListCache =  new ArrayList<String>();
					 facilityIdListCache.add(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE);	
					
					 /*
					  *UNERGISTEREDがスコープに設定されている場合には、info.getFacilityIdList();
					  *すると、登録済みノードがすべて取れてくるので、使わない。
					  */
				}else{
					/*それ以外の場合には、ノード一覧を取得*/
					 facilityIdListCache = info.getFacilityIdList();
				}
				
				ArrayList<String> targetFacilityIdList = new ArrayList<String>(facilityIdList);
				targetFacilityIdList.retainAll(facilityIdListCache);
				//m_log.error("monitorId : " + info.getMonitorId());
				//m_log.error("find() : targetFacilityIdList.size() " + targetFacilityIdList.size());
				if(targetFacilityIdList.size() > 0){
					
					if((info.getCheckMode() == MonitorSnmpTrapInfo.ALL_OID) ||
							(info.getCheckMode() == MonitorSnmpTrapInfo.UNREGISTERED_OID)){
						m_log.debug( "find() : search from master db" );
						//m_log.trace(" MonitorSnmpTrapInfo.ALL_OID or MonitorSnmpTrapInfo.UNREGISTERED_OID ");
						//m_log.trace("info.getCheckMode() :" + info.getCheckMode() );
						//すべてのを監視する場合と未登録OIDを監視する場合には、
						//OIDのチェックを行わず、Queueに投げ込む。
						for (int j = 0; j < targetFacilityIdList.size(); j++) {
							String facilityId = targetFacilityIdList.get(j);
							SnmpTrapMessageInfo message = new SnmpTrapMessageInfo(
									monitorId,
									facilityId,
									trapDate,
									trapOid,
									genericId,
									specificId,
									varBind,
									agentAdder
							);
							
							// SNMPトラップ情報通知
							try {
								m_log.debug( "find() : send message ("+trapOid+":"+genericId+":"+specificId +") of "+facilityId  );
								m_sendQueue.put(message);
							} catch (JMSException e) {
								m_log.error("find() : 通知失敗。" + e.getMessage());
								return false;
							}
						}
					}else{
						// 一致するOID情報が存在するか
						m_log.debug( "find() : search from each monitor" );
						ArrayList<SnmpTrapOidInfo> oidListCache = info.getOidList();
										
						// 監視項目中にマッチするものがあったかどうかのフラグ
						int matched = 0;
						for (int index = 0; index < oidListCache.size(); index++) {
							SnmpTrapOidInfo oidInfo = oidListCache.get(index);
							if(	((trapOid.equals(oidInfo.getTrapOid()))
									&&(genericId == oidInfo.getGenericId())
									&&(specificId == oidInfo.getSpecificId()))){
								//m_log.error("oidInfo : " + oidInfo.getGenericId() + " " + oidInfo.getTrapOid() + " " + oidInfo.getSpecificId());
								//同一のIPアドレスの複数のファシリティIDが存在する場合、全て通知する。

								for (int j = 0; j < targetFacilityIdList.size(); j++) {
									String facilityId = targetFacilityIdList.get(j);
									SnmpTrapMessageInfo message = new SnmpTrapMessageInfo(
											monitorId,
											facilityId,
											trapDate,
											trapOid,
											genericId,
											specificId,
											varBind,
											agentAdder
									);
									// SNMPトラップ情報通知
									try {
										m_log.debug( "find() : send message ("+trapOid+":"+genericId+":"+specificId +")of "+facilityId );
										m_sendQueue.put(message);
									} catch (JMSException e) {
										m_log.error("find() : 通知失敗。" + e.getMessage());
										return false;
									}
								}
								// added by takeshi mutou
								matched++;
								break;
							}
						}
						// 企業番号付の一般トラップが1件もマッチしなかった場合に再実行し、企業番号無しの一般トラップにマッチさせる。
						if((matched==0)&&(genericId!=TrapSnmp.ENTERPRISE_GENERIC_ID)){
							m_log.debug( "find() : generic trap : id "+genericId );
							
							for (int index = 0; index < oidListCache.size(); index++){
								SnmpTrapOidInfo oidInfo = oidListCache.get(index);
								if( oidInfo.getTrapOid().equals(TrapSnmp.GENERIC_TRAPS.get(genericId).toString())){
									for (int j = 0; j < targetFacilityIdList.size(); j++) {
										String facilityId = targetFacilityIdList.get(j);
										SnmpTrapMessageInfo message = new SnmpTrapMessageInfo(
												monitorId,
												facilityId,
												trapDate,
												trapOid,
												// m_trapSnmp.GENERIC_TRAPS.get(genericId).toString(),
												genericId,
												specificId,
												varBind,
												agentAdder
										);
										// SNMPトラップ情報通知
										try {
											m_sendQueue.put(message);
											m_log.debug( "find() : send message ("+trapOid+":"+genericId+":"+specificId +")of "+facilityId );
										} catch (JMSException e) {
											m_log.error("find() : 通知失敗。" + e.getMessage());
											return false;
										}
									}
									break;
								}
							}
						}
					}
				}
			}
		}
		m_log.debug("find() : end" );
		return true;
	}
	
	/**
	 * キャッシュ情報更新
	 * 
	 * @return
	 */
	public void updateCache() {
		
		
		// SNMPTRAP監視情報更新
		updateSnmpTrapInfo();
		
		// リポジトリ情報更新
		updateRepositoryInfo();
	}
	
	/**
	 * キャッシュ情報更新
	 * 
	 * @param updateInfo SNMPTRAP情報更新情報
	 * @return
	 */
	public boolean updateCache(UpdateSnmpTrapInfo updateInfo) {
		
		// SNMPTRAP監視情報更新
		String monitorId = updateInfo.getMonitorId();
		int type = updateInfo.getType();
		
		boolean updateRepositoryFlg = true;
		
		try {
			// 登録時
			if(type == UpdateSnmpTrapInfo.TYPE_ADD){ 
				
				MonitorSnmpTrapRunManagementLocal management = MonitorSnmpTrapRunManagementUtil.getLocalHome().create();
				SnmpTrapFacilityInfo info = management.getSnmpTrapInfo(monitorId);
				
				// 有効な場合
				if(info != null){
					m_snmpTrapInfoMap.put(monitorId, info);
				}
				else{
					// リポジトリキャッシュ更新なし
					updateRepositoryFlg = false;
				}
			}
			// 更新時
			else if(type == UpdateSnmpTrapInfo.TYPE_UPDATE){
				
				MonitorSnmpTrapRunManagementLocal management = MonitorSnmpTrapRunManagementUtil.getLocalHome().create();
				SnmpTrapFacilityInfo info = management.getSnmpTrapInfo(monitorId);
				
				// 有効な場合
				if(info != null){
					// ファシリティIDが変更されているか
					SnmpTrapFacilityInfo infoCache = m_snmpTrapInfoMap.get(monitorId);
					if(infoCache != null){
						String facilityIdCache = infoCache.getFacilityId();
						if(facilityIdCache.equals(info.getFacilityId())){
							// リポジトリキャッシュ更新なし
							updateRepositoryFlg = false;
						}
						m_snmpTrapInfoMap.remove(monitorId);
					}
					m_snmpTrapInfoMap.put(monitorId, info);
				}
				else{
					m_snmpTrapInfoMap.remove(monitorId);
				}
			}
			// 削除時
			else if(type == UpdateSnmpTrapInfo.TYPE_DELETE){
				m_snmpTrapInfoMap.remove(monitorId);
			}
			
			if(updateRepositoryFlg){
				// リポジトリ情報更新
				updateRepositoryInfo();
			}
			return true;
			
		} catch (CreateException e) {
			m_log.error("updateCache(UpdateSnmpTrapInfo) : SNMPTRAP監視情報更新失敗。" + e.getMessage());
		} catch (FinderException e) {
			m_log.error("updateCache(UpdateSnmpTrapInfo) : SNMPTRAP監視情報更新失敗。" + e.getMessage());
		} catch (NamingException e) {
			m_log.error("updateCache(UpdateSnmpTrapInfo) : SNMPTRAP監視情報更新失敗。" + e.getMessage());
		}
		
		m_snmpTrapInfoMap = new ConcurrentHashMap<String, SnmpTrapFacilityInfo>();
		return false;
	}
	
	/**
	 * キャッシュ情報更新
	 * 
	 * @param updateInfo リポジトリ情報更新情報
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public boolean updateCache(UpdateRepositoryInfo updateInfo) {
		
		ArrayList<String> facilityIdList = new ArrayList<String>();
		
		// SNMPTRAP監視情報 ファシリティIDリスト更新
		if(m_snmpTrapInfoMap != null){
			
			RepositoryControllerLocal repository;
			try {
				repository = RepositoryControllerUtil.getLocalHome().create();
				
				Set set = m_snmpTrapInfoMap.keySet();
				for (Iterator iter = set.iterator(); iter.hasNext();) {
					String monitorId = (String) iter.next();
					
					SnmpTrapFacilityInfo info = m_snmpTrapInfoMap.get(monitorId);
					if(info != null){
						
						String facilityId = info.getFacilityId();
						
						ArrayList<String> tmpFacilityIdList = null;
						if(repository.isNode(facilityId)){
							tmpFacilityIdList = new ArrayList<String>();
							tmpFacilityIdList.add(facilityId);
						}else if(facilityId.equals(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE)){
							
							//未登録ノードスコープの時は自分を検索対象ファシリティにします。
//							see com.clustercontrol.snmptrap.factory.TrapSNMP.snmpReceivedTrap
							tmpFacilityIdList = new ArrayList<String>();
							/*登録ノードをはじくために登録済みノードすべてのリストを追加*/
							tmpFacilityIdList = repository.getNodeFacilityIdList(FacilityTreeAttributeConstant.REGISTEREFD_SCOPE, RepositoryControllerBean.ALL);
												}
						else{
							tmpFacilityIdList = repository.getNodeFacilityIdList(facilityId, RepositoryControllerBean.ALL);
						}
						info.setFacilityIdList(tmpFacilityIdList);
						
						if(tmpFacilityIdList != null && tmpFacilityIdList.size() > 0){
							facilityIdList.addAll(tmpFacilityIdList);
						}
					} 
				}
				
				// リポジトリ情報更新
				updateRepositoryInfo(facilityIdList);
				return true;
				
			} catch (CreateException e) {
				m_log.error("updateCache(UpdateRepositoryInfo) : SNMPTRAP監視情報更新失敗。" + e.getMessage());
			} catch (FinderException e) {
				m_log.error("updateCache(UpdateRepositoryInfo) : SNMPTRAP監視情報更新失敗。" + e.getMessage());
			} catch (NamingException e) {
				m_log.error("updateCache(UpdateRepositoryInfo) : SNMPTRAP監視情報更新失敗。" + e.getMessage());
			}
			
			m_snmpTrapInfoMap = new ConcurrentHashMap<String, SnmpTrapFacilityInfo>();
			return false;
		}
		return true;
	}
	
	/**
	 * SNMPTRAP監視情報更新
	 * 
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public boolean updateSnmpTrapInfo() {
		
		m_log.debug("updateSnmpTrapInfo(): start");
		
		m_snmpTrapInfoMap = new ConcurrentHashMap<String, SnmpTrapFacilityInfo>();
		
		try{
			//セッションビーンに経由で、SNMPTRAP監視マップを取得します。
			//SNMPTRAP監視マップってなんだ？
			MonitorSnmpTrapRunManagementLocal management = MonitorSnmpTrapRunManagementUtil.getLocalHome().create();
			m_snmpTrapInfoMap = management.getSnmpTrapMap();
			
			/* デバッグ用コード*/
			/*Set set = m_snmpTrapInfoMap.keySet(); //setはMonitorIdの集合
			 m_log.debug("updateSnmpTrapInfo(): set.size() " + set.size());
			 
			 
			 for (Iterator iter = set.iterator(); iter.hasNext();) {
			 String monitorId = (String) iter.next();
			 
			 SnmpTrapFacilityInfo info = m_snmpTrapInfoMap.get(monitorId);
			 
			 m_log.error("updateSnmpTrapInfo() : monitorId" + monitorId);
			 m_log.error("updateSnmpTrapInfo() : chekcMode" + info.getCheckMode());
			 }*/
			
			return true;
			
		} catch (CreateException e) {
			m_log.error("updateSnmpTrapInfo() : SNMPTRAP監視情報更新失敗。" + e.getMessage());
		} catch (NamingException e) {
			m_log.error("updateSnmpTrapInfo() : SNMPTRAP監視情報更新失敗。" + e.getMessage());
		} catch (FinderException e) {
			m_log.error("updateSnmpTrapInfo() : SNMPTRAP監視情報更新失敗。" + e.getMessage());
		}
		return false;
	}
	
	/**
	 * リポジトリ情報更新
	 * 
	 * @return
	 */
	public boolean updateRepositoryInfo() {
		
		ArrayList<String> facilityIdList = new ArrayList<String>();
		
		// SNMPTRAP監視に設定されている全ファシリティID取得
		if(m_snmpTrapInfoMap != null){
			
			//監視設定の集合からkey(MonitorID)をとりだす。
			Set set = m_snmpTrapInfoMap.keySet();
			for (Iterator iter = set.iterator(); iter.hasNext();) {
				String monitorId = (String) iter.next();
				
				//ここのSNMP Trap 監視情報（１設定）をとりだす。
				SnmpTrapFacilityInfo info = m_snmpTrapInfoMap.get(monitorId);
				if(info != null){
					//監視設定からFacilityIDのリストを取り出しfacilityIdListに追加する。			
					ArrayList<String> tmpFacilityIdList = info.getFacilityIdList();
					if(tmpFacilityIdList != null && tmpFacilityIdList.size() > 0){
						facilityIdList.addAll(tmpFacilityIdList);
					}
				} 
			}
		}
		
		// 全部の監視設定から抽出したfacilityIDのリストで別のメソッド↓を呼ぶ
		boolean result = updateRepositoryInfo(facilityIdList);
		return result;
	}
	
	/**
	 * リポジトリ情報更新
	 * 
	 * @return
	 */
	public boolean updateRepositoryInfo(ArrayList<String> facilityIdList) {
		
		//IPAddressがキー、facilityIDのセットが入っているArrayListがObjectのHashmap
		m_ipAddressMap = new ConcurrentHashMap<String, ArrayList<String>>();
		m_hostNameMap = new ConcurrentHashMap<String, ArrayList<String>>();
		
		try {
			// ノードの属性取得
			if(facilityIdList != null && facilityIdList.size() > 0){
				
				//facilityIdのリストを元に、リポジトリに IPNETWORKNUMBER IPNETWORKNUMBERV6 NODENAME
				//を問い合わせに行く。結果はHashMap<facilityID , 3つ入ったオブジェクト>
				HashMap facilityAttrMap = m_repository.getNodeDetail(facilityIdList, m_attributeList);
				
				//帰ってきたオブジェクトからHapMapのキーであるfacilityIDのsetを取得
				Set keySet = facilityAttrMap.keySet();
				for (Iterator iter = keySet.iterator(); iter.hasNext();) {
					String facilityId = (String) iter.next();
					
					//facilityIDをキーにして属性のHaahMapを取得する。
					HashMap map = (HashMap)facilityAttrMap.get(facilityId);
					//属性のHashMapから各属性を読み出す。
					String ipNetworkNumber = (String)map.get(FacilityAttributeConstant.IPNETWORKNUMBER);
					String hostName = (String)facilityAttrMap.get(FacilityAttributeConstant.NODENAME);
					
					if(ipNetworkNumber != null && !"".equals(ipNetworkNumber)){
						//IPアドレスをキーにしてそのオブジェクトを取り出す。
						ArrayList<String> tmpFacilityIdList = m_ipAddressMap.get(ipNetworkNumber);
						if(tmpFacilityIdList == null){
							//未定義の場合には初期化
							tmpFacilityIdList = new ArrayList<String>();
						}
						if(!tmpFacilityIdList.contains(facilityId)){
							//まだそのfacilityIDが入っていなければ追加
							tmpFacilityIdList.add(facilityId);
						}
						//IpアドレスマップにipアドレスのキーとfacilityIDのArrayListで
						//putする。
						m_ipAddressMap.put(ipNetworkNumber, tmpFacilityIdList);
					}
					
					if(hostName != null && !"".equals(hostName)){
						//ホスト名をキーにしてfacilityIDのリストを取得
						ArrayList<String> tmpFacilityIdList = m_hostNameMap.get(hostName);
						
						if(tmpFacilityIdList == null){
							//未定義の場合には初期化
							tmpFacilityIdList = new ArrayList<String>();
						}
						if(!tmpFacilityIdList.contains(facilityId)){
							//まだそのfacilityIDが入っていなければ追加
							tmpFacilityIdList.add(facilityId);
						}
						tmpFacilityIdList.add(facilityId);
						m_hostNameMap.put(hostName, tmpFacilityIdList);
					}
				}
			}
			return true;
			
		} catch (FinderException e) {
			m_log.error("updateRepositoryInfo() : リポジトリ情報更新失敗。" + e.getMessage());
		} catch (NamingException e) {
			m_log.error("updateRepositoryInfo() : リポジトリ情報更新失敗。" + e.getMessage());
		}
		return false;
	}
	
	/**
	 * 指定されたホスト名のファシリティID一覧を取得
	 * 
	 * @param hostName ホスト名
	 * @return ファシリティID一覧
	 */
	public ArrayList<String> getFacilityIdListByHostName(String hostName) {
		
		if(m_hostNameMap != null){
			return m_hostNameMap.get(hostName);
		}
		return null;
	}
	
	/**
	 * 指定されたIPアドレスのファシリティID一覧を取得
	 * 
	 * @param address IPアドレス
	 * @return ファシリティID一覧
	 */
	public ArrayList<String> getFacilityIdListByIpAddress(String address) {
		
		if(m_ipAddressMap !=null){
			return m_ipAddressMap.get(address);
		}
		return null;
	}
	
	/**
	 * SNMPTRAP監視情報一覧を取得
	 * 
	 * @return SNMPTRAP監視情報
	 */
	public Map<String, SnmpTrapFacilityInfo> getSnmpTrapInfoMap() {
		
		return m_snmpTrapInfoMap;
	}
	
	/**
	 * SNMPTRAP監視情報一覧を取得
	 * 
	 * @return SNMPTRAP監視情報
	 */
	public Map<String, SnmpTrapFacilityInfo> getSnmpTrapInfoMap(String monitorId) {
		
		if(m_snmpTrapInfoMap != null){
			m_snmpTrapInfoMap.get(monitorId);
		}
		return null;
	}
}