/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.ping.factory;

import java.io.IOException;
import java.io.InputStream;
import java.io.StreamTokenizer;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.jms.JMSException;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.MonitorBlockConstant;
import com.clustercontrol.bean.PingRunCountConstant;
import com.clustercontrol.bean.PingRunIntervalConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.monitor.run.bean.MonitorNumericValueInfo;
import com.clustercontrol.monitor.run.factory.RunMonitorNumericValueType;
import com.clustercontrol.ping.bean.PingResult;
import com.clustercontrol.ping.ejb.entity.MonitorPingInfoLocal;
import com.clustercontrol.ping.ejb.entity.MonitorPingInfoPK;
import com.clustercontrol.ping.ejb.entity.MonitorPingInfoUtil;
import com.clustercontrol.ping.util.PingProperties;
import com.clustercontrol.ping.util.ReachAddress;
import com.clustercontrol.ping.util.ReachAddressFping;
import com.clustercontrol.priority.util.PriorityJudgment;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.util.Messages;
import com.sun.org.apache.xalan.internal.xsltc.runtime.Hashtable;

/**
 * ping監視を実行するクラス<BR>
 *
 * @version 2.1.0
 * @since 2.0.0
 */
public class RunMonitorPing extends RunMonitorNumericValueType {
	
	protected static Log m_log = LogFactory.getLog( RunMonitorPing.class );
	
	public static final String MESSAGE_ID_INFO = "001";
	public static final String MESSAGE_ID_WARNING = "002";
	public static final String MESSAGE_ID_CRITICAL = "003";
	public static final String MESSAGE_ID_UNKNOWN = "004";
	
	public static boolean fpingEnable;

	
	/** ping監視情報 */
	protected static final ArrayList<String> m_attributeList = new ArrayList<String>();
	static{
		m_attributeList.add(FacilityAttributeConstant.NODENAME);
		m_attributeList.add(FacilityAttributeConstant.IPNETWORKNUMBER);
		m_attributeList.add(FacilityAttributeConstant.IPNETWORKNUMBERV6);
	}
	
	/** ping監視情報 */
	protected MonitorPingInfoLocal m_ping = null;
	
	/** ping実行 */
//	protected ReachAddress m_reachability = null;
	
	/** 実行回数 */
	protected int m_runCount = PingRunCountConstant.TYPE_COUNT_01;
	
	/** 実行間隔（ミリ秒） */
	protected int m_runInterval = PingRunIntervalConstant.TYPE_SEC_01;
	
	/** タイムアウト（ミリ秒） */
	protected int m_pingTimeout;
	
	/** メッセージ */
	protected String m_message = null;
	
	/** 不明メッセージ */
	protected String m_unKnownMessage = null;
	
	/** オリジナルメッセージ */
	protected String m_messageOrg = null;
	
//	/** ping到達率（%） */
//	protected int m_reachRatio = 0;
	
	/** パケット紛失率（%） */
	protected int m_lost = 0;
	
	/** 応答平均時間（ミリ秒） */
	protected long m_average = 0;
	
	//ping実行
	ReachAddress m_reachability = null;
	
	//fping実行
	ReachAddressFping m_reachabilityFping = null;
	
	//fping利用時のfping出力メッセージ
	private ArrayList m_MsgOrg= null;
	
	//fping利用時のfpingエラー出力メッセージ
	private ArrayList m_MsgErr= null;
	
	
	private Hashtable m_Target;
	
	/**
	 * 
	 * コンストラクタ
	 * 
	 */
	public RunMonitorPing() throws NamingException, JMSException, CreateException{
		super();
		
		PingProperties.getProperties();
		fpingEnable=PingProperties.isM_fpingEnable();
	}
	
	/**
	 * ping数を取得を行います。<BR>
	 * 
	 * @param facilityId ファシリティID
	 * @return 値取得に成功した場合、true
	 */
	@Override
	@SuppressWarnings("unchecked")
	public boolean collect(String facilityId) {
		
		m_nodeDate = m_now.getTime();
		m_message = "";
		m_messageOrg = "";
//		m_reachRatio = 0;
		m_lost = 0;
		m_average = 0;
		
		
		if(m_reachability == null){
			m_reachability = new ReachAddress(m_runCount, m_runInterval, m_pingTimeout);
		}
		
		try{
			// ノードの属性取得
			HashMap facilityAttrMap = m_repository.getNodeDetail(facilityId, m_attributeList);
			
			String ipNetworkNumber = (String)facilityAttrMap.get(FacilityAttributeConstant.IPNETWORKNUMBER);
			String nodeName = (String)facilityAttrMap.get(FacilityAttributeConstant.NODENAME);
			
			boolean result = m_reachability.isReachable(ipNetworkNumber, nodeName);
			m_message = m_reachability.getMessage();
			m_messageOrg = m_reachability.getMessageOrg();
			if(result){
//				m_reachRatio = m_reachability.getReachRatio();
				m_lost = m_reachability.getLost();
				m_average = m_reachability.getAverage();
			}
			return result;
		}
		catch(FinderException e){
			m_log.debug("run():" + e.getMessage());
			
			m_message = Messages.getString("message.ping.4");
			m_messageOrg = e.getMessage();
			return false;
		}
		catch(NamingException e){
			m_log.debug("run():" + e.getMessage());
			
			m_message = Messages.getString("message.ping.4");
			m_messageOrg = e.getMessage();
			return false;
		}
	}
	
	/**
	 * 　fpingによるping数の取得を行います。<BR>
	 * @author Yosuke Miyamoto 
	 * @version 2.2.3 bata
	 * @param facilityId Ping対象のファシリティID(スコープ)collectのfacilityIDとは異なる
	 * @return 値取得に成功した場合、true
	 */
	public boolean collectFping(ArrayList facilityList, ArrayList priporityList) {
		
		//ノードの収集時刻を保存
		m_nodeDate = m_now.getTime();
		
		//まずは、データを作ります。
		// hosts[] →　IPアドレスリスト(String の配列)
		// node    → IPアドレスとノード名のリスト
		// target  → nodoのリスト

		ArrayList hosts = new ArrayList();
//		String[] hosts     = new String[facilityList.size()];
		m_Target = new Hashtable();
		
		String facilityId = null;
		String[] node;
		for(int index=0; index<facilityList.size(); index++){
			facilityId = (String)facilityList.get(index);
			if(facilityId != null && !"".equals(facilityId)){	
				
				node = new String[2];
				
				try{
					
					// ノードの属性取得
					HashMap facilityAttrMap = m_repository.getNodeDetail(facilityId, m_attributeList);
					
					node[0] = (String)facilityAttrMap.get(FacilityAttributeConstant.IPNETWORKNUMBER);
					node[1] = (String)facilityAttrMap.get(FacilityAttributeConstant.NODENAME);
				
					if (node[0] != null && !node[0].equals("")) {
						//IPアドレスを配列に入れていく。
						hosts.add(node[0]);
						
						//targetをつめていく。
						m_Target.put(facilityId,node);
					}
					
				}catch(FinderException e){
					m_log.error("collectFping():" + e.getMessage());
					
					m_message = Messages.getString("message.ping.4");
					m_messageOrg = e.getMessage();
					return false;
				}
				catch(NamingException e){
					m_log.error("collectFping():" + e.getMessage());
					
					m_message = Messages.getString("message.ping.4");
					m_messageOrg = e.getMessage();
					return false;
				}
			}	
		}
		
		if(m_reachabilityFping == null){
			m_reachabilityFping = new ReachAddressFping(m_runCount, m_runInterval, m_pingTimeout);
		}
		
		boolean result = m_reachabilityFping.isReachable(hosts);
		m_MsgErr = m_reachabilityFping.getM_errMsg();
		m_MsgOrg = m_reachabilityFping.getM_resultMsg();	
		
		return result;
	}
	
	/**
	 * 	fpingの標準出力から、IPアドレス毎の結果に整形します<BR>
	 * 
	 * 
	 * @param hosts      fpingしたアドレスのリスト
	 * @param message    出力メッセージ
	 * @param count      ping回数
	 * @return (IPアドレス 応答率、平均応答時間)のハッシュテーブル
	 */
	private Hashtable wrapUpFping(ArrayList messages,int count){
		
		Hashtable ret = new Hashtable();
		/**
		 * データセットとして必要なものは以下の5つ
		 * 分割したオリジナルメッセージからデータを作る。
		 * 
		 **/
		String msg;
		String msgOrg ;
		int lost;
		float average = 0;
		float reachRatio;
		
		//IPアドレスの正規表現パターン
		Pattern patternIp = Pattern.compile("^([0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}.*)");
		//fpingの出力を分割するパターン
		Pattern patternSp = Pattern.compile("(\\s:\\s|\\s)+");
		Matcher matcher;
		
		for(int index = 0 ; index < messages.size(); index++){
			
			//fpingの出力正式は　127.0.0.1 : 0.10 1.23
			matcher = patternIp.matcher((String) messages.get(index));
			
			//出力結果から結果内容をフィルタする。
			if(matcher.matches()){
				
				// String の分割する。
				String[] strs = patternSp.split((String) messages.get(index));
				
				/**以下の変数はメッセージを作成するために必要**/
				float max=0;         //最大応答時間(初期値0秒)
				float min=Float.MAX_VALUE; //最小応答時間(初期値Floatの最大値)
				int num=0;         //受信パケット
				
				//収集回数分だけ入っているデータを読み出す。
				for (int index1=1 ; index1 <=count; index1++ ){
					
					if(strs[index1].equals("-")){
						
					}else{
						
						//パケット数はcount up
						num++; 
						
						//最大応答時間のチェックと更新
						
						if(max< Float.parseFloat(strs[index1]))
							max =  Float.parseFloat(strs[index1]);
						
						//最小応答時間のチェックと更新
						if(min > Float.parseFloat(strs[index1]))
							min =  Float.parseFloat(strs[index1]);
						
						//平均応答時間
						average += Float.parseFloat(strs[index1]);
					}
				}
				
				//平均なので、和を回数で割っておきます。
				average /= count;
				
				StringBuffer buffer = new StringBuffer();
				
				buffer.append("Pinging " + strs[0]  + " (" + strs[0] + ") .\n\n");
				if (num == 0) {
					//受信パケットが0の時はlost→100 reach→0
					lost = 100;
					reachRatio=0;
					
					for (int index1=0 ; index1 < count; index1++ )
						// buffer.append("From " + strs[0] + " icmp_seq="+ index +" Destination Host Unreachable");
						buffer.append("Reply from " + strs[0] + " icmp_seq="+ index +" Destination Host Unreachable");
					
					buffer.append("\nPing statistics for " + strs[0] + ":\n");
					buffer.append( "Packets: Sent = " + count + ", Received = " + num + ", Lost = " + (count-num) + " (" + lost + "% loss),");
					
				} else {
					lost = (count - num) * 100 / count;
					reachRatio = num*100/count;
					
					buffer.append("\nPing statistics for " + strs[0] + ":\n");
					buffer.append( "Packets: Sent = " + count + ", Received = " + num + ", Lost = " + (count-num) + " (" + lost + "% loss),");
					buffer.append("Approximate round trip times in milli-seconds:\n");
					buffer.append("\tMinimum = " + min 
							+ "ms, Maximum = " + max 
							+ "ms, Average = " + average + "ms\n");
				}
				
				msgOrg =buffer.toString();
				msg= "Packets: Sent = " + count + ", Received = " + num + ", Lost = " + (count-num) + " (" + lost + "% loss)";

				PingResult res = new PingResult(strs[0],msg,msgOrg,lost,average, reachRatio);
				
				ret.put(strs[0],res); 
				
				msg="";
				msgOrg="" ;
				lost=100;
				average = 0;
				reachRatio=0;
			}
		}
		return ret;
	}
	
	/* (非 Javadoc)
	 * ping監視情報を設定
	 * @see com.clustercontrol.monitor.run.factory.OperationNumericValueInfo#setMonitorAdditionInfo()
	 */
	@Override
	public void setCheckInfo() throws FinderException, NamingException{
		
		// ping監視情報を取得
		MonitorPingInfoPK pk = new MonitorPingInfoPK(m_monitorId, m_monitorTypeId);
		m_ping = MonitorPingInfoUtil.getLocalHome().findByPrimaryKey(pk);
		
		// ping監視情報を設定
		if(m_ping.getRunCount() != null)
			m_runCount = m_ping.getRunCount().intValue();
		if(m_ping.getRunInterval() != null)
			m_runInterval = m_ping.getRunInterval().intValue();
		if(m_ping.getTimeout() != null)
			m_pingTimeout = m_ping.getTimeout().intValue();
	}
	
	
	/* (非 Javadoc)
	 * 判定結果を取得
	 * @see com.clustercontrol.monitor.run.factory.RunMonitorNumericValueType#getCheckResult(boolean)
	 */
	@Override
	public int getCheckResult(boolean ret) {
		// lowerLimit : パケット紛失率（％）
		// upperLimit : 時間（ミリ秒）
		
		int result = -1;
		MonitorNumericValueInfo info = null;
		
		// 値取得の成功時
		if(ret){
			
			// 通知をチェック
			info = (MonitorNumericValueInfo)m_judgementInfoList.get(Integer.valueOf(PriorityConstant.TYPE_INFO));
			if(m_lost <= info.getThresholdLowerLimit() && m_average <= info.getThresholdUpperLimit()){
				result = PriorityConstant.TYPE_INFO;
			}
			else {
				// 警告の範囲チェック
				info = (MonitorNumericValueInfo)m_judgementInfoList.get(Integer.valueOf(PriorityConstant.TYPE_WARNING));
				if(m_lost <= info.getThresholdLowerLimit() && m_average <= info.getThresholdUpperLimit()){
					result = PriorityConstant.TYPE_WARNING;
				}
				else{
					// 危険（通知・警告以外）
					result = PriorityConstant.TYPE_CRITICAL;
				}
			}
		}
		return result;
	}
	
	/* (非 Javadoc)
	 * ノード用メッセージIDを取得
	 * @see com.clustercontrol.monitor.run.factory.OperationMonitor#getMessageId(int)
	 */
	@Override
	public String getMessageId(int id) {
		
		if(id == PriorityConstant.TYPE_INFO){
			return MESSAGE_ID_INFO;
		}
		else if(id == PriorityConstant.TYPE_WARNING){
			return MESSAGE_ID_WARNING;
		}
		else if(id == PriorityConstant.TYPE_CRITICAL){
			return MESSAGE_ID_CRITICAL;
		}
		else{
			return MESSAGE_ID_UNKNOWN;
		}
	}
	
	/* (非 Javadoc)
	 * ノード用メッセージを取得
	 * @see com.clustercontrol.monitor.run.factory.OperationMonitor#getMessage(int)
	 */
	@Override
	public String getMessage(int id) {
		return m_message;
	}
	
	/* (非 Javadoc)
	 * ノード用オリジナルメッセージを取得
	 * @see com.clustercontrol.monitor.run.factory.OperationMonitor#getMessageOrg(int)
	 */
	@Override
	public String getMessageOrg(int id) {
		return m_messageOrg;
	}
	
	/* (非 Javadoc)
	 * スコープ用メッセージIDを取得
	 * @see com.clustercontrol.monitor.run.factory.RunMonitor#getMessageIdForScope(int)
	 */
	@Override
	public String getMessageIdForScope(int priority) {
		
		if(priority == PriorityConstant.TYPE_INFO){
			return MESSAGE_ID_INFO;
		}
		else if(priority == PriorityConstant.TYPE_WARNING){
			return MESSAGE_ID_WARNING;
		}
		else if(priority == PriorityConstant.TYPE_CRITICAL){
			return MESSAGE_ID_CRITICAL;
		}
		else{
			return MESSAGE_ID_UNKNOWN;
		}
	}
	
	
	
	/**
	 * ping監視を実行します。
	 * <p>
	 * <ol>
	 * <li>fpingに対応するためにRunMonitorのrunMonitorInfoをoverrideします。</li>
	 * <li>監視情報を取得し、保持します（{@link #setMonitorInfo(String, String)}）。</li>
	 * <li>判定情報を取得し、判定情報マップに保持します（{@link #setJudgementInfo()}）。</li>
	 * <li>チェック条件情報を取得し、保持します（{@link #setCheckInfo()}）。</li>
	 * <li>ファシリティ毎に監視を実行し、値を収集します。 （{@link #collect(String)}）。</li>
	 * <li>監視結果から、判定結果を取得します。 （{@link #getCheckResult(boolean)}）。</li>
	 * <li>監視結果から、重要度を取得します（{@link #getPriority(int)}）。</li>
	 * <li>監視結果を通知します（{@link #notify(boolean, String, int, Date)}）。</li>
	 * </ol>
	 * 
	 * @return 実行に成功した場合、</code> true </code>
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws JMSException
	 * @throws NamingException
	 * @throws CreateException
	 * 
	 * @see #setMonitorInfo(String, String)
	 * @see #setJudgementInfo()
	 * @see #setCheckInfo()
	 * @see #collect(String)
	 * @see #getCheckResult(boolean)
	 * @see #getPriority(int)
	 * @see #notify(boolean, String, int, Date)
	 */
	@SuppressWarnings("unchecked")
	public boolean runMonitorInfo()throws FinderException, RemoveException, JMSException, NamingException, CreateException {
		
		m_now = new Date();
		
		m_priorityMap = new HashMap();
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_INFO),		new ArrayList<String>());
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_WARNING),	new ArrayList<String>());
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_CRITICAL),	new ArrayList<String>());
		m_priorityMap.put(Integer.valueOf(PriorityConstant.TYPE_UNKNOWN),	new ArrayList<String>());
		
		try 
		{
			// 監視基本情報を設定
			boolean run = this.setMonitorInfo(m_monitorTypeId, m_monitorId);
			if(!run){
				// 処理終了
				return true;
			}
			
			// 判定情報を設定
			setJudgementInfo();
			
			// チェック条件情報を設定
			setCheckInfo();
			
			// ファシリティIDの配下全ての一覧を取得
			ArrayList facilityList = m_repository.getNodeFacilityIdList(m_facilityId, RepositoryControllerBean.ALL);
			
			//配下にノードがないということはノードの可能性があるので指定されたIDをセット
			if (facilityList.size() == 0) {
				if(m_repository.isNode(m_facilityId)){
					m_isNode = true;
					facilityList.add(m_facilityId);
				}
				else{
					return true;
				}
			}
			
			// ファシリティ毎に監視情報を収集
			ArrayList<Integer> priorityList = new ArrayList<Integer>();
			

			//fping利用フラグがfalseの時はver2.2までの方式でping監視を行う。
			if(!fpingEnable){
				
				String facilityId = null;
				for(int index=0; index<facilityList.size(); index++){
					facilityId = (String)facilityList.get(index);
					if(facilityId != null && !"".equals(facilityId)){
						
						// 監視値を収集
						boolean ret = collect(facilityId);
						
						// 監視値より判定結果を取得
						int checkResult = getCheckResult(ret);
						
						// 監視単位がスコープとノードの場合、スコープ用の情報を設定
						if(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock ||
								MonitorBlockConstant.TYPE_ALL == m_monitorBlock){
							
							// 判定結果よりノードの重要度を取得
							int priority = getPriority(checkResult);
							// スコープの重要度を設定
							priorityList.add(Integer.valueOf(priority));
							// 重要度別にファシリティIDを設定
							setPriorityMap(Integer.valueOf(priority), facilityId);
						}
						
						// スコープの値取得時刻を設定
						if(m_nodeDate > m_scopeDate){
							m_scopeDate = m_nodeDate;
						}
						
						// ノードのログ出力情報を送信 
						if(MonitorBlockConstant.TYPE_NODE == m_monitorBlock ||
								(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock && m_isNode) || 
								MonitorBlockConstant.TYPE_ALL == m_monitorBlock){
							
							// 監視管m_now = new Date();理へ通知
							notify(true, facilityId, checkResult, new Date(m_nodeDate));
						}
					}
				}
			}//fping利用フラグがtrueの時は新しいping監視を行う。
			else{
				//	 監視値を収集
				//   fpingを利用する場合には、ファシリティのリストを与えて一気にpingします。
				boolean ret = collectFping(facilityList,priorityList);
				
				
				if(ret){
					//データの整形を行います。
					Hashtable fpingResultSet = wrapUpFping(m_MsgErr,m_runCount);
					
					//最初に呼び出されたIPアドレスのリスト
					Iterator it = facilityList.iterator();
					
					String targetFacility;
					PingResult nodeResult;
					String[] node;
					
					while(it.hasNext()){
						
						//与えられたFacilityId個々に
						targetFacility = (String)it.next();
						//ノード名、IPアドレスを取得
						node = (String[]) m_Target.get(targetFacility);
						
						// ポーリングされていないノード(IPアドレス未記入のノード)を除く。
						if(node != null) {
							
							nodeResult = (PingResult)fpingResultSet.get(node[0]);
							
							if(nodeResult ==null){
								
								/**fping失敗で結果がありません。**/
								m_log.error("Fping no response.");
								
							}else{
								
								//結果データからイベントを生成するためにデータを読みだす。
								m_lost = nodeResult.getLost();
								//現行の使用の平均応答時間はmsec(long)のため変換する。
								m_average  = (long)nodeResult.getAverage();
								m_message  = (String)nodeResult.getMesseage();
								m_messageOrg = (String)nodeResult.getMesseageOrg();
								
								// 監視値より判定結果を取得
								int checkResult = getCheckResult(true);
								
								// 監視単位がスコープとノードの場合、スコープ用の情報を設定
								if(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock ||
										MonitorBlockConstant.TYPE_ALL == m_monitorBlock){
									
									// 判定結果よりノードの重要度を取得
									int priority = getPriority(checkResult);
									// スコープの重要度を設定
									priorityList.add(Integer.valueOf(priority));
									// 重要度別にファシリティIDを設定
									setPriorityMap(Integer.valueOf(priority), targetFacility);
								}
								
								// スコープの値取得時刻を設定
								if(m_nodeDate > m_scopeDate){
									m_scopeDate = m_nodeDate;
								}
								
								// ノードのログ出力情報を送信 
								if(MonitorBlockConstant.TYPE_NODE == m_monitorBlock ||
										(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock && m_isNode) || 
										MonitorBlockConstant.TYPE_ALL == m_monitorBlock){
									
									// 監視管理へ通知
									notify(true, targetFacility, checkResult, new Date(m_nodeDate));
								}
							}
						}
					}
				}
			}
			
			// スコープのログ出力情報を送信 
			if(!m_isNode){
				if(MonitorBlockConstant.TYPE_SCOPE == m_monitorBlock || 
						MonitorBlockConstant.TYPE_ALL == m_monitorBlock){
					
					Integer priority = PriorityJudgment.judgment(priorityList); 
					if(priority == null){
						priority = Integer.valueOf(m_failurePriority);
					}
					
					// 各ノードの内、最新日付を設定
					Date generationDate = null;
					if(m_scopeDate > 0){
						generationDate = new Date(m_scopeDate);
					}
					else{
						generationDate = m_now;
					}
					
					// 監視管理へ通知
					notify(false, m_facilityId, priority.intValue(), generationDate);					
				}
			}
			return true;
			
			
			
			
		} catch (FinderException e) {
			m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
			throw e;
		} catch (CreateException e) {
			m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
			throw e;
		} catch (JMSException e) {
			m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
			throw e;
		} catch (NamingException e) {
			m_log.error("run(): 監視対象ID:" + m_monitorTypeId + ",監視項目ID：" + m_monitorId  + ", " + e.getMessage());
			throw e;
		}
		
		
	}
	
	
	/**
	 * トランザクションを開始し、引数で指定された監視情報の監視を実行します。
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @param monitorId 監視項目ID
	 * @throws FinderException
	 * @throws CreateException 
	 * @throws RemoveException
	 * @throws JMSException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws SystemException
	 * 
	 * @see #runMonitorInfo()
	 */
	public void run(String monitorTypeId, String monitorId) throws FinderException, RemoveException, JMSException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, InvalidTransactionException, IllegalStateException, SystemException, CreateException {
		super.run(monitorTypeId,monitorId);
	}
	
}