/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performanceMGR.util;
import java.io.File;
import java.io.FileInputStream;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Properties;
import java.util.TreeMap;

import javax.ejb.EJBException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.performanceMGR.bean.CollectorItemInfo;
import com.clustercontrol.performanceMGR.bean.OIDInfo;

/**
 *SNMPで取得した値からグラフに必要な値への計算を行うクラス
 *
 * 
 * @version 1.0
 * @since 1.0
 */

public final class CalculationMethod {
	//ログ出力
	private static Log m_log = LogFactory.getLog( CalculationMethod.class );
	
	public static int NonDeviece = 0;	// デバイスを意識しない
	public static int SingleDevice = 1;	// デバイス別に値を求める
	public static int MultiDevice = 2;	// デバイスの合計値を求める
	
	// SNMPで取得する値の型がCOUNTER32であった場合の最大値
	private static long COUNTER32_MAX_VLUE = ((long)(Integer.MAX_VALUE))*2+1;
	
	private static HashMap m_OIDTable = new HashMap();
	private static String UPTIME_OID;
	private static Boolean initializeFlg = new Boolean(false);
	
	public static void initialize() {
		// 既に初期化済みの場合は何もしない。
		if(initializeFlg.booleanValue()){
			return;
		}
	
		// 性能値算出用設定ファイルを読み込む
	    Properties properties = new Properties();
		
		try {
			String homedir = System.getProperty("jboss.server.home.dir");
			String propertyFile = homedir + File.separator + "conf" + File.separator + "performance.properties";
			
			// プロパティファイルからキーと値のリストを読み込みます
			properties.load(new FileInputStream(propertyFile));
		} catch (Exception e) {
			// エラー処理
			m_log.error(e.getMessage(), e);
			return;
		}

		// UPTIMEを設定
		UPTIME_OID = properties.getProperty("collector.uptime.oid");
		
		// UPTIME_OIDの最後の"."より前の部分を抽出してUPTIME_BASEOIDとする
		StringBuffer buffer = new StringBuffer(UPTIME_OID);
//		UPTIME_BASEOID = buffer.delete(buffer.lastIndexOf("."), buffer.length()).toString();
		
		// 性能値の算出方法と算出に必要なOIDの設定をpropertiesファイルから読み込む
//		int itemNum = 0;
//		while(properties.getProperty("collector.item.code." + itemNum) != null){
//			itemNum++;
//		}
		
		int itemNum = Integer.parseInt(properties.getProperty("collector.item.num"));
		m_log.debug("Item Num : " + itemNum);
		
		for(int i=0; i<itemNum; i++) {
			String itemCode = properties.getProperty("collector.item.code." + i);		
			// itemCode が設定されていない場合は次のコードを読み込む
			if(itemCode == null){
				continue;
			}
			
			String operation = properties.getProperty("collector.method." + i);

			int OIDNum = 0;
			while(properties.getProperty("collector.oid." + i + '.' + OIDNum) != null){
				OIDNum++;
			}
			
			String[] oidBase = new String[OIDNum];
			String[] oidIndex = new String[OIDNum];
			Long[] defaultValue = new Long[OIDNum];
			int[] types = new int[OIDNum];
			
			for(int j=0; j < OIDNum; j++) {
				String fullOid = properties.getProperty("collector.oid." + i + '.' + j);
				
				// 最後の"."より前の部分を抽出してOIDとする
				buffer = new StringBuffer(fullOid);
				oidBase[j] = buffer.delete(buffer.lastIndexOf("."), buffer.length()).toString(); //$NON-NLS-1$

				// ":"より後ろの部分を値取得失敗時の適用値とする
				// ":"より後ろの部分が設定されている場合のみ設定する
				int lastIndes = fullOid.lastIndexOf(":");
				if (lastIndes == -1){
					// ":"を含まない場合
					// 最後の"."より後ろの部分を抽出してOIDのインデックスとする
					buffer = new StringBuffer(fullOid);
					oidIndex[j] = buffer.delete(0, buffer.lastIndexOf(".")+1).toString();

					// 値取得失敗時の適用値は設定しない
					defaultValue[j] = null;
				} else {
					// ":"を含む場合				
					// 最後の"."から":"までの部分を抽出してOIDのインデックスとする
					buffer = new StringBuffer(fullOid);
					oidIndex[j] = buffer.substring(buffer.lastIndexOf(".")+1, lastIndes);		

					// 値取得失敗時の適用値を設定する
					String defaultValueString = buffer.substring(lastIndes+1);
					defaultValue[j] = Long.parseLong(defaultValueString);
				}
				
				if("*".equals(oidIndex[j])){ //$NON-NLS-1$
					types[j] = MultiDevice;
				} else if ("?".equals(oidIndex[j])){ //$NON-NLS-1$
					types[j] = SingleDevice;
				} else {
					types[j] = NonDeviece;
				}				
				
				// デバッグ出力
				m_log.debug("initialize() "
						+ itemCode + ", "
						+ operation + ", \t"  
						+ oidBase[j] + ", "
						+ oidIndex[j] + ", "
						+ defaultValue[j] + ", "
						+ types[j]);
			}
			
			Definition definition = new Definition(
					itemCode,
					operation,
					oidBase,
					oidIndex,
					defaultValue,
					types
				);
			m_OIDTable.put(itemCode, definition);
		}
		
		// 初期化済みフラグをセットする。
		initializeFlg = new Boolean(true);
	}
	

//	/**
//	 * 収集対象のOIDの情報を返します。
//	 * 
//	 * @param itemCodeList 収集項目コードの配列
//	 * @param deviceIndexList デバイスインデックスの配列
//	 * @return
//	 */
//	public static OIDInfo[] getTargetOid(String[] itemCodeList, int[] deviceIndexList){
//		initialize();
//		
//		TreeMap oidInfoMap = new TreeMap();
//		
//		// アップタイムは必ず入れる（性能情報算出時の要素数の基準にするため）
//		oidInfoMap.put(CalculationMethod.getUPTimeOID(), CalculationMethod.getUPTimeOIDInfo());
//		
//		for(int i=0; i<itemCodeList.length; i++){
//			// 必要なOIDのセットを取得する
//			String[] oidBase = CalculationMethod.getOIDs(itemCodeList[i]);  // OIDを取得
//			Long[] defaultValues = CalculationMethod.getDefaultValues(itemCodeList[i]);  // 値取得失敗時の適用値を取得
//			String[] oidIndex = CalculationMethod.getOidIndex(itemCodeList[i]);  // OIDのインデックスを取得
//			
//			for(int j=0; j<oidBase.length; j++){
//				OIDInfo buffer = new OIDInfo();
//				buffer.setBaseOid(oidBase[j]);
//				buffer.setDefaultValue(defaultValues[j]);
//				
//				if(oidIndex[j].equals("*")){ 
//					String fullOID = oidBase[j] + "." + "*";
//					buffer.setAllIndex(true);
//					oidInfoMap.put(fullOID, buffer);
//					
//				} else if (oidIndex[j].equals("?")){
//					String fullOID = oidBase[j] + "." + deviceIndexList[i];  // OIDのインデックスとしてデバイスインデックスを設定 //$NON-NLS-1$
//					buffer.setOidIndex(deviceIndexList[i]);
//					buffer.setAllIndex(false);
//					oidInfoMap.put(fullOID, buffer);
//				} else {
//					String fullOID = oidBase[j] + "." + oidIndex[j];  // OIDのインデックスを設定 //$NON-NLS-1$
//					buffer.setOidIndex(Integer.parseInt(oidIndex[j]));
//					buffer.setAllIndex(false);
//					oidInfoMap.put(fullOID, buffer);  	// 指定のOIDのみを収集対象とするOIDリストに加える
//				}
//			}
//		}
//		
//		// マップに格納されている値を配列に代入する
//		OIDInfo[] targetOID = new OIDInfo[oidInfoMap.values().size()];
//		oidInfoMap.values().toArray(targetOID);
//		
//		return targetOID;
//	}
	
	/**
	 * 収集対象のOIDの情報を返します。
	 * 
	 * @param items 収集項目の配列
	 * @return
	 */
	public static OIDInfo[] getTargetOid(CollectorItemInfo[] items){
		initialize();
		
		TreeMap oidInfoMap = new TreeMap();
		
		// アップタイムは必ず入れる（性能情報算出時の要素数の基準にするため）
		oidInfoMap.put(CalculationMethod.getUPTimeOID(), CalculationMethod.getUPTimeOIDInfo());
		
		for(int i=0; i<items.length; i++){
			// 必要なOIDのセットを取得する
			String[] oidBase = CalculationMethod.getOIDs(items[i].getCollectorItemCode());  // OIDを取得
			Long[] defaultValues = CalculationMethod.getDefaultValues(items[i].getCollectorItemCode());  // 値取得失敗時の適用値を取得
			String[] oidIndex = CalculationMethod.getOidIndex(items[i].getCollectorItemCode());  // OIDのインデックスを取得
			
			for(int j=0; j<oidBase.length; j++){
				OIDInfo buffer = new OIDInfo();
				buffer.setBaseOid(oidBase[j]);
				buffer.setDefaultValue(defaultValues[j]);

                // デバイス・ファイルシステムのスコープ対応のためコメントアウト
                // performance.propertiesのにある*と?の処理を同じにする
//				if(oidIndex[j].equals("*")){ 
				if(oidIndex[j].equals("*") || oidIndex[j].equals("?")){ 
					String fullOID = oidBase[j] + "." + "*";
					buffer.setAllIndex(true);
					oidInfoMap.put(fullOID, buffer);
					
				// デバイス・ファイルシステムのスコープ対応のためコメントアウト	
//				} else if (oidIndex[j].equals("?")){
//					String fullOID = oidBase[j] + "." + items[i].getDeviceIndex();  // OIDのインデックスとしてデバイスインデックスを設定 //$NON-NLS-1$
//					buffer.setOidIndex(items[i].getDeviceIndex());
//					buffer.setAllIndex(false);
//					oidInfoMap.put(fullOID, buffer);
				} else {
					String fullOID = oidBase[j] + "." + oidIndex[j];  // OIDのインデックスを設定 //$NON-NLS-1$
					buffer.setOidIndex(Integer.parseInt(oidIndex[j]));
					buffer.setAllIndex(false);
					oidInfoMap.put(fullOID, buffer);  	// 指定のOIDのみを収集対象とするOIDリストに加える
				}
			}
		}
		
		// マップに格納されている値を配列に代入する
		OIDInfo[] targetOID = new OIDInfo[oidInfoMap.values().size()];
		oidInfoMap.values().toArray(targetOID);
		
		return targetOID;
	}
	
	/**
	 * 収集項目コードの性能値の計算に必要なOIDのリストを返す。（順序も保持）
	 */
	private static String[] getOIDs(String itemCode){
		initialize();
		return ((Definition)m_OIDTable.get(itemCode)).getOIDs();
	}

	/**
	 * 収集項目コードの性能値の計算に必要なOIDのインデックスのリストを返す。（順序も保持）
	 */
	private static String[] getOidIndex(String itemCode){
		initialize();
		return ((Definition)m_OIDTable.get(itemCode)).getOidIndex();
	}
	
	/**
	 * SNMPのuptimeのOID を取得します。
	 * @return uptimeのOID
	 */
	private static String getUPTimeOID(){
		initialize();
		return CalculationMethod.UPTIME_OID;
	}
	
	/**
	 * 収集項目コードの性能値の計算に必要なOIDの値取得失敗時の適用値を返す。（順序も保持）
	 */
	private static Long[] getDefaultValues(String itemCode){
		initialize();
		return ((Definition)m_OIDTable.get(itemCode)).getDefaultValues();
	}

	/**
	 * SNMPのuptimeのOID を取得します。
	 * @return uptimeのOIDInfo
	 */
	public static OIDInfo getUPTimeOIDInfo(){
		initialize();
		OIDInfo returnValue = new OIDInfo();
		
		// 最後の"."より前の部分を抽出する
		StringBuffer buffer = new StringBuffer(UPTIME_OID);
		returnValue.setBaseOid(buffer.delete(buffer.lastIndexOf("."), buffer.length()).toString()); //$NON-NLS-1$

		// 最後の"."より後ろの部分を抽出してOIDのインデックスとする
		buffer = new StringBuffer(UPTIME_OID);
		returnValue.setOidIndex(Integer.parseInt(buffer.delete(0, buffer.lastIndexOf(".")+1).toString())); //$NON-NLS-1$
		
		returnValue.setAllIndex(false);
		
		return returnValue;
	}
	
	/**
	 * 収集項目コードに対応する性能値の計算を行います。
	 * 収集時の性能値算出に使用します。
	 * 
	 * @param itemCode 収集項目コード
	 * @param deviceIndex　デバイスのインデックス
	 * @param mibValue　SNMPの値を持つハッシュマップ
	 * @return 計算された性能値
	 */
	public static double getPerformance(final String itemCode, final int deviceIndex, final HashMap mibValue){
		initialize();
		return ((Definition)m_OIDTable.get(itemCode)).calc(deviceIndex, mibValue);	
	}

	/**
	 * 各性能値の収集対象OIDと、性能値の算出方法を保持するインナークラス
	 */
	public static class Definition {
		private String m_itemCode;
		private Operation m_operation;
		private String[] m_oidBase;
		private String[] m_oidIndex;
		private Long[] m_defaultValue;
		private int m_types[];
		
		public Definition(String itemCode, String operationType, String baseOIDs[], String[] oidIndex, Long[] defaultValue, int[] types) {
			m_itemCode = itemCode;
			m_oidBase = baseOIDs;
			m_oidIndex = oidIndex;
			m_defaultValue = defaultValue;
			m_types = types;
			
			String operationClassName = "com.clustercontrol.performanceMGR.util.CalculationMethod$" + operationType;
			
			try{
				m_operation = (Operation)Class.forName(operationClassName).newInstance();
			} catch (ClassNotFoundException e){
				throw new EJBException(e);
			} catch (IllegalAccessException e){
				throw new EJBException(e);
			} catch (InstantiationException e){
				throw new EJBException(e);
			}
		}

		public synchronized double calc(final int deviceIndex, HashMap mibValueSet){	
			m_operation.setOIDBuffer(m_oidBase);
			m_operation.setOIDIndexBuffer(m_oidIndex);
			m_operation.setMibValueSetBuffer(mibValueSet);

			double returnValue;
			try {
				returnValue = m_operation.calc(deviceIndex);
			} catch (InvalidValueException e) {
				return Double.NaN;
			}

			m_operation.setOIDBuffer(null);
			m_operation.setOIDIndexBuffer(null);
			m_operation.setMibValueSetBuffer(null);
			
			return returnValue;
		}
		
		/**
		 * @return itemCode を戻します。
		 */
		public String getItemCode() {
			return m_itemCode;
		}
		/**
		 * @return OID を戻します。
		 */
		public String[] getOidIndex(){
			return m_oidIndex;
		}
		/**
		 * @return OID を戻します。
		 */
		public String getOidIndex(int index){
			return m_oidIndex[index];
		}
		/**
		 * @return OID を戻します。
		 */
		public String[] getOIDs(){
			return m_oidBase;
		}
		/**
		 * @return OID を戻します。
		 */
		public String getOIDs(int index){
			return m_oidBase[index];
		}
		/**
		 * @return types を戻します。
		 */
		public int[] getTypes() {
			return m_types;
		}
		/**
		 * @return types を戻します。
		 */
		public int getTypes(int index) {
			return m_types[index];
		}
		/**
		 * @return 値取得に失敗した場合の適用値を戻します。
		 */
		public Long[] getDefaultValues() {
			return m_defaultValue;
		}
	}

	private abstract static class Operation {
		private String[] m_OIDBuffer;
		private String[] m_OIDIndex;
		private HashMap<String, HashMap<Integer, long[]>> m_mibValueSetBuffer;
		
		public abstract double calc(final int deviceIndex) throws InvalidValueException;
	
		public void setOIDBuffer(String[] OIDBuffer){		
			m_OIDBuffer = OIDBuffer;
		}

		public void setOIDIndexBuffer(String[] OIDIndex){		
			m_OIDIndex = OIDIndex;
		}

		public String getOIDIndex(int index){		
			return m_OIDIndex[index];
		}
		
		public void setMibValueSetBuffer(HashMap<String, HashMap<Integer, long[]>> mibValueSetBuffer){
			m_mibValueSetBuffer = mibValueSetBuffer;
		}
		
		protected long getMibValue(int OIDIndex, int tableIndex, int timeIndex) throws InvalidValueException{
//			long[][] mibSet = (long[][])m_mibValueSetBuffer.get(m_OIDBuffer[OIDIndex]);
//			if(mibSet == null || mibSet[tableIndex] == null || mibSet.length <= tableIndex){
//				return 0;
//			} else {
//				return mibSet[tableIndex][timeIndex];
//			}

			HashMap<Integer, long[]> mibHashSet = m_mibValueSetBuffer.get(m_OIDBuffer[OIDIndex]);
			if(mibHashSet == null){
				throw new InvalidValueException();
			} else {
				long[] mibSet = mibHashSet.get(tableIndex);
				if(mibSet == null){
					throw new InvalidValueException();
				}
				
				return mibSet[timeIndex];
			}
		}
		
//		protected long[] getMibValueArray(int OIDIndex, int tableIndex){
//			long[][] mibSet = (long[][])m_mibValueSetBuffer.get(m_OIDBuffer[OIDIndex]);			
//			if(mibSet == null || mibSet[tableIndex] == null || mibSet.length <= tableIndex){
//				return null;
//			} else {
//				return mibSet[tableIndex];
//			}
//		}

		protected long getCurrentMibValue(int OIDIndex, int tableIndex) throws InvalidValueException{
//			long[][] mibSet = (long[][])m_mibValueSetBuffer.get(m_OIDBuffer[OIDIndex]);			
//			if(mibSet == null || mibSet[tableIndex] == null || mibSet.length <= tableIndex){
//				throw new InvalidValueException();
//			} else {
//				return mibSet[tableIndex][1];
//			}
			return getMibValue(OIDIndex, tableIndex, 1);
		}

		protected long getPreviousMibValue(int OIDIndex, int tableIndex) throws InvalidValueException{
//			long[][] mibSet = (long[][])m_mibValueSetBuffer.get(m_OIDBuffer[OIDIndex]);			
//			if(mibSet == null || mibSet[tableIndex] == null || mibSet.length <= tableIndex){
//				throw new InvalidValueException();
//			} else {
//				return mibSet[tableIndex][0];
//			}
			return getMibValue(OIDIndex, tableIndex, 0);	
		}

		
		/**
		 * OID Table Object の数を返す。
		 * @return
		 */
		public int getTableObjectCount(){
			return m_OIDBuffer.length;
		}
		
//		/**
//		 * 指定された OID Table Object に含まれる項目数を返す。
//		 * @param OIDIndex
//		 * @return
//		 */ 
//		protected int getOIDCountOfTableObject(int OIDIndex){
//			long[][] mibSet = (long[][])m_mibValueSetBuffer.get(m_OIDBuffer[OIDIndex]);		
//			return mibSet.length;
//		}
	}
	
	// 時間平均を百分率で求める計算
	public static class TimeMeanPercentage extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			long uptimeDiff;
			long valDirff;
			double perfData;
			
			// uptime が 0 の場合は算出不能
			if(getPreviousMibValue(0, 0) == 0){
				return Double.NaN;
			}
			
			uptimeDiff = getCurrentMibValue(0, 0) - getPreviousMibValue(0, 0);
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(uptimeDiff < 0){
				uptimeDiff = COUNTER32_MAX_VLUE + 1 + uptimeDiff; 
			}
			
			// 分母が0の場合（更新されていない場合）
			if(uptimeDiff == 0){
				return Double.NaN;
			}
			
			valDirff = getCurrentMibValue(1, deviceIndex) - getPreviousMibValue(1, deviceIndex);
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(valDirff < 0){
				valDirff = COUNTER32_MAX_VLUE + 1 + valDirff; 
			}
			
			perfData = ((double)valDirff)/uptimeDiff * 100.0D;
			
			return perfData;
		}
	}
	
	// 時間平均を求める計算
	public static class TimeMean extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			long uptimeDiff;
			long valDirff;
			double perfData;
			
			// uptime が 0 の場合は算出不能
			if(getPreviousMibValue(0, 0) == 0){
				return Double.NaN;
			}
			
			uptimeDiff = getCurrentMibValue(0, 0) - getPreviousMibValue(0, 0);
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(uptimeDiff < 0){
				uptimeDiff = COUNTER32_MAX_VLUE + 1 + uptimeDiff; 
			}
			
			// 分母が0の場合（更新されていない場合）
			if(uptimeDiff == 0){
				return Double.NaN;
			}
			
			valDirff = getCurrentMibValue(1, deviceIndex) - getPreviousMibValue(1, deviceIndex);
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(valDirff < 0){
				valDirff = COUNTER32_MAX_VLUE + 1 + valDirff; 
			}
			
			perfData = ((double)valDirff)/uptimeDiff * 100.0D;  // アップタイムは10ミリ秒単位のため
			
			return perfData;
		}
	}
	
	// 時間平均の合計を求める計算
	public static class TimeMeanSum extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			long uptimeDiff;
			long valDirff1;
			long valDirff2;
			double perfData;
			
			// uptime が 0 の場合は算出不能
			if(getPreviousMibValue(0, 0) == 0){
				return Double.NaN;
			}
			
			uptimeDiff = getCurrentMibValue(0, 0) - getPreviousMibValue(0, 0);
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(uptimeDiff < 0){
				uptimeDiff = COUNTER32_MAX_VLUE + 1 + uptimeDiff; 
			}
			
			// 分母が0の場合（更新されていない場合）
			if(uptimeDiff == 0){
				return Double.NaN;
			}
			
			valDirff1 = getCurrentMibValue(1, deviceIndex) - getPreviousMibValue(1, deviceIndex);
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(valDirff1 < 0){
				valDirff1 = COUNTER32_MAX_VLUE + 1 + valDirff1; 
			}
			valDirff2 = getCurrentMibValue(2, deviceIndex) - getPreviousMibValue(2, deviceIndex);
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(valDirff2 < 0){
				valDirff2 = COUNTER32_MAX_VLUE + 1 + valDirff2; 
			}
			
			perfData = ((double)(valDirff1 + valDirff2))/uptimeDiff * 100.0D;  // アップタイムは10ミリ秒単位のため
			
			return perfData;
		}
	}
	
	// 時間平均の百分率の余りを求める計算
	public static class InvTimeMeanPercentage extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{		
			long uptimeDiff;
			long valDirff;
			double perfData;
			
			uptimeDiff = getCurrentMibValue(0, 0) - getPreviousMibValue(0, 0);
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(uptimeDiff < 0){
				uptimeDiff = COUNTER32_MAX_VLUE + 1 + uptimeDiff; 
			}
			
			// 分母が0の場合（更新されていない場合）
			if(uptimeDiff == 0){
				return Double.NaN;
			}
			
			valDirff = getCurrentMibValue(1, deviceIndex) - getPreviousMibValue(1, deviceIndex);
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(valDirff < 0){
				valDirff = COUNTER32_MAX_VLUE + 1 + valDirff; 
			}
			
			perfData = ((double)valDirff)/uptimeDiff * 100.0D;
			
			return 100.0D - perfData;
		}
	}

	// 1番目の要素が全体に占める割合を計算
	public static class PercentageOfTotal  extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			long targetDiff;
			long diffTotal = 0;
			double perfData;
			
			targetDiff = getCurrentMibValue(0, deviceIndex) - getPreviousMibValue(0, deviceIndex);
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(targetDiff < 0){
				targetDiff = COUNTER32_MAX_VLUE + 1 + targetDiff; 
			}
			
			long diff;
			diffTotal = targetDiff;  // 上で求めた差分値を利用しその次の要素から計算を行う
			for(int i=1; i<getTableObjectCount(); i++){
				diff = getCurrentMibValue(i, deviceIndex) - getPreviousMibValue(i, deviceIndex);				
				// COUNTER32の上限を越えループした場合に最大値を足し込む
				if(diff < 0){
					diff = COUNTER32_MAX_VLUE + 1 + diff; 
				}
				diffTotal += diff;
			}
			
			// 分母が0の場合（更新されていない場合）
//			if(diffTotal == 0){
			if(diffTotal <= 100 && targetDiff == 0){  // どれかの値が更新されていない場合の対処
				return Double.NaN;
			}

			perfData = (double)targetDiff / diffTotal * 100.0D;
			
			return perfData;
		}
	}
	
	// 1番目と2番目の要素の合計が全体に占める割合を計算
	public static class SumPerTotal  extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			long targetDiff1;
			long targetDiff2;
			long diffTotal = 0;
			double perfData;

			targetDiff1 = getCurrentMibValue(0, deviceIndex) - getPreviousMibValue(0, deviceIndex);	// 1番目の要素
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(targetDiff1 < 0){
				targetDiff1 = COUNTER32_MAX_VLUE + 1 + targetDiff1; 
			}
			
			targetDiff2 = getCurrentMibValue(1, deviceIndex) - getPreviousMibValue(1, deviceIndex); // 2番目の要素
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(targetDiff2 < 0){
				targetDiff2 = COUNTER32_MAX_VLUE + 1 + targetDiff2; 
			}
			
			long diff;
			diffTotal = targetDiff1 + targetDiff2;
			for(int i=2; i<getTableObjectCount(); i++){
				diff = getCurrentMibValue(i, deviceIndex) - getPreviousMibValue(i, deviceIndex);
				// COUNTER32の上限を越えループした場合に最大値を足し込む
				if(diff < 0){
					diff = COUNTER32_MAX_VLUE + 1 + diff; 
				}
				
				diffTotal += diff;				
			}
			
			// 分母が0の場合（更新されていない場合）
			if(diffTotal == 0){
				return Double.NaN;
			}

			perfData = (double)(targetDiff1 + targetDiff2) / diffTotal * 100.0D;
			
			return perfData;
		}
	}
	
	// 1番目の要素以外が全体に占める割合を計算
	public static class InvPercentageOfTotal extends Operation {	
		public double calc(int deviceIndex) throws InvalidValueException{
			long targetDiff;
			long diffTotal = 0;
			double perfData;
			
			targetDiff = getCurrentMibValue(0, deviceIndex) - getPreviousMibValue(0, deviceIndex);
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(targetDiff < 0){
				targetDiff = COUNTER32_MAX_VLUE + 1 + targetDiff; 
			}
			
			long diff;
			diffTotal = targetDiff;  // 上で求めた差分値を利用しその次の要素から計算を行う
			for(int i=1; i<getTableObjectCount(); i++){
				diff = getCurrentMibValue(i, deviceIndex) - getPreviousMibValue(i, deviceIndex);
				// COUNTER32の上限を越えループした場合に最大値を足し込む
				if(diff < 0){
					diff = COUNTER32_MAX_VLUE + 1 + diff; 
				}
				
				diffTotal += diff;
			}
			
			// 分母が0の場合（更新されていない場合）
			if(diffTotal <= 100 && targetDiff == 0){  // どれかの値が更新されていない場合100%となることの対処
				return Double.NaN;
			}

			perfData = (double)targetDiff / diffTotal * 100.0D;
			
			if(perfData == 0){
				for(int i=0; i<getTableObjectCount(); i++){
					long cur = getCurrentMibValue(i, deviceIndex);
					long pre = getPreviousMibValue(i, deviceIndex);
				}
			}
			
			return 100.0D - perfData;
		}
	}
	
	// (1 - x/(y+z))*100を求める計算
	public static class InvPercentage12 extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			long total;
			double perfData;
			
			total = getCurrentMibValue(1, deviceIndex) + getCurrentMibValue(2, deviceIndex);
			
			if(total == 0){
				// エラー処理　おそらく値がとれていない
				return Double.NaN;
			}
			
			perfData = 100.0D - (double)getCurrentMibValue(0, deviceIndex) / total * 100.0D;
						
			return perfData;
		}
	}
	
	// 時間平均の合計を求める計算
	public static class TimeMeanPerTotal extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			long uptimeDiff;
			long totalDiff = 0;
			double perfData;	
			
			// uptime が 0 の場合は算出不能
			if(getPreviousMibValue(0, 0) == 0){
				return Double.NaN;
			}
			
			uptimeDiff = getCurrentMibValue(0, 0) - getPreviousMibValue(0, 0);
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(uptimeDiff < 0){
				uptimeDiff = COUNTER32_MAX_VLUE + 1 + uptimeDiff; 
			}
			
			// 分母が0の場合（更新されていない場合）
			if(uptimeDiff == 0){
				// エラー処理　おそらく値がとれていない
				return Double.NaN;
			}
			
			long diff = 0;
			// インデックス1からの差分の合計（インデックス0はuptime）
			for(int j=1; j<getTableObjectCount(); j++){
//				// テーブルインデックス1からの合計
//				for(int i=1; i<getOIDCountOfTableObject(j); i++){
//					diff = getCurrentMibValue(j, i) - getPreviousMibValue(j, i);
//					// COUNTER32の上限を越えループした場合に最大値を足し込む
//					if(diff < 0){
//						diff = COUNTER32_MAX_VLUE + 1 + diff; 
//					}
//					
//					totalDiff = totalDiff + diff;
//				}
				
//				 インデックス0の領域には合計値が格納されている。
				diff = getCurrentMibValue(j, 0) - getPreviousMibValue(j, 0);
				// COUNTER32の上限を越えループした場合に最大値を足し込む
				// 複数の値が同時に上限を越えた場合に対応するため、正の値となるまで足し込む
				while(diff < 0){
					diff = COUNTER32_MAX_VLUE + 1 + diff; 
				}
				
				totalDiff = totalDiff + diff;
			}
			
			perfData = (double)totalDiff/(double)uptimeDiff * 100.0D;  // アップタイムは10ミリ秒単位のため
			
			return perfData;			
		}
	}
	
	// 時間平均の合計を２で割る計算
	public static class TimeMeanPerTotalDividedBy2 extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			long uptimeDiff;
			long totalDiff = 0;
			double perfData;	
			
			// uptime が 0 の場合は算出不能
			if(getPreviousMibValue(0, 0) == 0){
				return Double.NaN;
			}
			
			uptimeDiff = getCurrentMibValue(0, 0) - getPreviousMibValue(0, 0);
			// COUNTER32の上限を越えループした場合に最大値を足し込む
			if(uptimeDiff < 0){
				uptimeDiff = COUNTER32_MAX_VLUE + 1 + uptimeDiff; 
			}
			
			// 分母が0の場合（更新されていない場合）
			if(uptimeDiff == 0){
				// エラー処理　おそらく値がとれていない
				return Double.NaN;
			}
			
			long diff = 0;
			// インデックス1からの差分の合計（インデックス0はuptime）
			for(int j=1; j<getTableObjectCount(); j++){
//				// テーブルインデックス1からの合計
//				for(int i=1; i<getOIDCountOfTableObject(j); i++){
//					diff = getCurrentMibValue(j, i) - getPreviousMibValue(j, i);
//					// COUNTER32の上限を越えループした場合に最大値を足し込む
//					if(diff < 0){
//						diff = COUNTER32_MAX_VLUE + 1 + diff; 
//					}
//					
//					totalDiff = totalDiff + diff;
//				}
				
//				 インデックス0の領域には合計値が格納されている。
				diff = getCurrentMibValue(j, 0) - getPreviousMibValue(j, 0);
				// COUNTER32の上限を越えループした場合に最大値を足し込む
				// 複数の値が同時に上限を越えた場合に対応するため、正の値となるまで足し込む
				while(diff < 0){
					diff = COUNTER32_MAX_VLUE + 1 + diff; 
				}
				
				totalDiff = totalDiff + diff;
			}
			
			perfData = (double)totalDiff/(double)uptimeDiff * 100.0D;  // アップタイムは10ミリ秒単位のため
			
			return perfData / 2;
		}
	}
	
	// (x-z)/(x+y) の百分率を求める計算
	public static class PercentageDiffAdd extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			long diff;
			long total;
			double perfData;
			
			diff = getCurrentMibValue(0, deviceIndex) - getCurrentMibValue(2, deviceIndex);
			
			total = getCurrentMibValue(0, deviceIndex) + getCurrentMibValue(1, deviceIndex);
			
			if(total == 0){
				// エラー処理　おそらく値がとれていない
				return Double.NaN;
			}
			
			perfData = (double)diff / total * 100.0D;
	
			return perfData;			
		}
	}

	// 1番目の要素の値を100で割った値を求める（1番目の要素の値を求めることに注意）
	public static class DividedBy100 extends Operation {
		public double calc(int deviceIndex) throws NumberFormatException, InvalidValueException{			
			return getCurrentMibValue(0, Integer.parseInt(getOIDIndex(0))) / 100.0D;			
		}
	}
	
	// 和を求める
	public static class Sum extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			return getCurrentMibValue(0, deviceIndex) + getCurrentMibValue(1, deviceIndex);			
		}
	}
	
	// 何も加工しない値を戻す
	public static class Non extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			return getCurrentMibValue(0, deviceIndex);			
		}
	}
	
	// 1番目の項目を2番目の項目で割ったものを百分率で戻す
	public static class Percentage extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			double perfData;
			
			if(getCurrentMibValue(1, deviceIndex) == 0) {
				return Double.NaN;				
			} else {
				perfData = (double)getCurrentMibValue(0, deviceIndex) / getCurrentMibValue(1, deviceIndex) * 100.D;
			}
				
			return perfData;			
		}
	}
	
	// 1番目の項目を2番目の項目で割ったものの残りの割合を百分率で戻す
	public static class InvPercentage extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			double perfData;
			
			if(getCurrentMibValue(1, deviceIndex) == 0) {
				return Double.NaN;				
			} else {
				perfData = 100.0D - (double)getCurrentMibValue(0, deviceIndex) / getCurrentMibValue(1, deviceIndex) * 100.D;
			}
				
			return perfData;			
		}
	}
	
	// 100-(val1+val2+val3)/val0)*100 を求める計算
	public static class InvPercentage31 extends Operation {
		public double calc(int deviceIndex) throws InvalidValueException{
			long total;
			double perfData;

			if(getCurrentMibValue(0, deviceIndex) == 0){
				// エラー処理　おそらく値がとれていない
				return Double.NaN;
			}
			
			total = getCurrentMibValue(1, deviceIndex) + getCurrentMibValue(2, deviceIndex) + getCurrentMibValue(3, deviceIndex);
			
			perfData = 100 - (double)total / getCurrentMibValue(0, deviceIndex) * 100.0D;
	
			return perfData;			
		}
	}
	
	public static class InvalidValueException extends Exception {
	}
}
