/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.logagent;

import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;

import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;
import javax.naming.InitialContext;

import com.clustercontrol.repository.bean.TopicConstant;
import com.clustercontrol.repository.message.UpdateRepositoryInfo;

/**
 * リポジトリ更新情報を受信クラス<BR>
 *
 * リポジトリ情報が更新されると、エージェントに更新情報が
 * JMS(Topic)を通じて送信される。
 * リポジトリ更新Topicへの接続と、メッセージの受信を行う。
 * 
 * @version 2.0.0
 * @since 2.0.0
 * 
 */
public class UpdateRepositoryInfoReceiveTopic implements MessageListener, ExceptionListener {
	private static final String TOPIC_CON_FACTORY = "ConnectionFactory";
	private static final long RETRY_INTERVAL = 10000;
	
	private TopicConnectionFactory m_factory;
	protected TopicConnection m_con;
	protected Topic m_topic;
	protected TopicSession m_session;
	protected TopicSubscriber m_subscriber;
	
	protected Agent m_agent;
	private Properties m_props;
	
	private boolean m_errFlg = false;
	
	/** メッセージサービス再接続用タイマー * */
	protected Timer m_timer = new Timer(true);
	
	/**
	 * コンストラクタ
	 * @param facilityIdList ファシリティIDのリスト　
	 * @param sendQueue　メッセージ送信用クラス
	 * @param props　プロパティファイル情報
	 */
	@SuppressWarnings("unchecked")
	public UpdateRepositoryInfoReceiveTopic(Agent agent, Properties props) {
		super();
		 
		m_props = props;
		m_agent = agent;
		 
		//接続処理
		initial();
	}
	
	
	/**
	 * 
	 * リポジトリ更新情報を受信したときに動作するメソッド<BR>
	 * 
	 */
	/* 
	 * トピック受信処理
	 * (non-Javadoc)
	 * @see javax.jms.MessageListener#onMessage(javax.jms.Message)
	 */
	public void onMessage(Message message) {
		
		if(message instanceof ObjectMessage){
			ObjectMessage objectMessage = (ObjectMessage)message;
			Object obj;
			try {
				obj = objectMessage.getObject();
			} catch (JMSException e) {
				return;
			}
			
			if(obj instanceof UpdateRepositoryInfo){
				
				m_agent.setFacility();
			
			}else{
				return;
			}
		}else{
			return;
		}
	}
	
	/* 通信エラーハンドラ
	 * (non-Javadoc)
	 * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
	 */
	public void onException(JMSException arg0) {
		setErrFlg(true);
		
		terminate();
	}
	
	/**
	 * @param errFlg
	 *            errFlg を設定。
	 */
	synchronized private void setErrFlg(boolean errFlg) {
		if (m_errFlg == false && errFlg == true) {
			
			m_timer.schedule(new ReSetupTask(), RETRY_INTERVAL,
					RETRY_INTERVAL);
			
		}
		m_errFlg = errFlg;
	}

	/**
	 *  
	 */
	synchronized private boolean reInitial() {
		boolean ret = false;
		
		terminate();
		
		if (initial()) {
			
			ret = true;
			
			//エラーフラグ解除
			setErrFlg(false);
			
		} else {
			
		}
		
		return ret;
	}
	/**
	 * サーバ接続の終了処理
	 *  
	 */
	public void terminate() {
		
		terminateSumscriber();
		
		
		try {
			if (m_session != null)
				m_session.close();
		} catch (JMSException e) {
		}
		
		try {
			if (m_con != null)
				m_con.close();
		} catch (JMSException e1) {
		}
	}
	/**
	 * トピック受信処理の終了 
	 */
	private void terminateSumscriber() {
		try {
			if (m_subscriber != null)
				m_subscriber.close();
		} catch (JMSException e) {
		}
	}
	
	
	
	/**
	 * 初期化処理
	 * JMSへの接続、トピック受信設定を行う
	 * @return
	 */
	private boolean initial() {
		
		InitialContext con = null;
		
		try {
			//InitialContextの生成
			con = new InitialContext(m_props);

			//コネクションファクトリ生成
			m_factory = (TopicConnectionFactory)con.lookup(TOPIC_CON_FACTORY);
			
			//コネクション生成
			m_con = m_factory.createTopicConnection();
			
			//セッション生成
			m_session = m_con.createTopicSession(false,
					Session.AUTO_ACKNOWLEDGE);
			
			//メッセージTopic取得
			m_topic = (Topic)con.lookup(TopicConstant.TOPIC_NAME_EXECUTE);
			
			
			//エラーハンドらセット
			m_con.setExceptionListener(this);
			
			m_con.start();
			
			//トピック接続開始
			initialTopic();
			
			
		} catch (Exception e) {
			setErrFlg(true);
			return false;
		} finally {
			try {
				if (con != null)
					con.close();
			} catch (Exception e1) {
			}
		}
		return true;
		
	}
	
	/**
	 * トピック受信設定
	 * @return
	 * @since
	 */
	private boolean initialTopic() {
		
		
		//現在のTopic受信を終了
		terminateSumscriber();
		
		//新しいファシリティIDでトピック受信開始
		try {
			m_subscriber = m_session.createSubscriber(m_topic);
			
			//コールバックを登録する
			m_subscriber.setMessageListener(this);

		} catch (Exception e) {
			setErrFlg(true);
			return false;
		} finally {
		}
		return true;
		
	}
	/**
	 * EJB再接続タイマータスク
	 * 通信エラーとなった場合に定周期で呼ばれ再接続を行う 
	 */
	protected class ReSetupTask extends TimerTask {
		
		/**
		 * コネクションクローズ
		 */
		public void run() {
			if (reInitial()) {
				//このタスクをタイマーから解除
				cancel();
			}
		}
		
	}
	
}
