/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.logagent;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ConfirmConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.logagent.util.EjbConnectionManager;
import com.clustercontrol.logtransfer.bean.LogTransferFileInfo;
import com.clustercontrol.monitor.message.LogOutputInfo;
import com.clustercontrol.repository.ejb.session.RepositoryController;


/**
 * ログ転送スレッドを管理するクラス<BR>
 * 
 * 転送対象ログファイル情報を受け取り、ログ転送スレッドを制御します。
 *  
 * @version 2.1.0
 * @since 2.1.0
 */
public class TransferLogManager {
	
	/** 起動しているログ転送スレッド  */
	private Hashtable<String, LogThreadManageInfo> m_runTransferLog = null;

	
	/** Queue送信  */
	private SendQueue m_sendQueue;
	
	/** ログ転送エージェントプロパティ */
	private Properties m_props;
	
	private EjbConnectionManager m_ejbConnectionManager;
	
	//ロガー
	private Log log = LogFactory.getLog(TransferLogManager.class);

	/**
	 * ログ転送管理情報クラス<BR>
	 *
	 */
	class LogThreadManageInfo{
		
		String m_logName;
		
		TransferLogThread m_thread;
			
		CopyOnWriteArrayList<LogTransferFileInfoFacility> m_logInfoList;

		public String getLogName() {
			return m_logName;
		}


		public TransferLogThread getThread() {
			return m_thread;
		}

		/**
		 * @param logName
		 */
		public LogThreadManageInfo(String logName) {
			super();
			// TODO 自動生成されたコンストラクター・スタブ
			m_logName = logName;
			m_logInfoList = new CopyOnWriteArrayList<LogTransferFileInfoFacility>();
		}

		public int size() {
			return m_logInfoList.size();
		}

		public void setLogInfoList(
				CopyOnWriteArrayList<LogTransferFileInfoFacility> logInfoList) {
			m_logInfoList = logInfoList;
		}

		public void setThread(TransferLogThread thread) {
			m_thread = thread;
		}

		public boolean add(String facilityId, String facilityPath, LogTransferFileInfo fileInfo) {
			
			return m_logInfoList.add(new LogTransferFileInfoFacility(facilityId, facilityPath, fileInfo));
		}

		public Iterator<LogTransferFileInfoFacility> iterator() {
			return m_logInfoList.iterator();
		}


		public boolean remove(Object arg0) {
			return m_logInfoList.remove(arg0);
		}

	
	}
	
	/**
	 * ログ転送情報（ファシリティIDを追加）クラス<BR>
	 *
	 */
	class LogTransferFileInfoFacility extends LogTransferFileInfo{
		
		/**
		 * 
		 */
		private static final long serialVersionUID = 1L;

		String m_facilityId;
		
		String m_facilityPath;
		
		LogTransferFileInfo m_fileInfo;

		/**
		 * @param facilityId
		 * @param fileInfo
		 */
		public LogTransferFileInfoFacility(String facilityId, String facilityPath, LogTransferFileInfo fileInfo) {
			m_facilityId = facilityId;
			m_facilityPath = facilityPath;
			m_fileInfo = fileInfo;
		}

		public int getExistenceFlg() {
			return m_fileInfo.getExistenceFlg();
		}

		public String getFilePath() {
			return m_fileInfo.getFilePath();
		}

		public int getRunInterval() {
			return m_fileInfo.getRunInterval();
		}

		public String getTransferId() {
			return m_fileInfo.getTransferId();
		}

		public int getValid() {
			return m_fileInfo.getValid();
		}

		public void setExistenceFlg(int arg0) {
			m_fileInfo.setExistenceFlg(arg0);
		}

		public void setFilePath(String arg0) {
			m_fileInfo.setFilePath(arg0);
		}

		public void setRunInterval(int arg0) {
			m_fileInfo.setRunInterval(arg0);
		}

		public void setTransferId(String arg0) {
			m_fileInfo.setTransferId(arg0);
		}

		public void setValid(int arg0) {
			m_fileInfo.setValid(arg0);
		}

		public String getFacilityId() {
			return m_facilityId;
		}
		
		public String getFacilityPath() {
			return m_facilityPath;
		}
		
	}
	

	/**
	 * コンストラクタ
	 * 
	 * @param ejbConnectionManager EJBコネクション管理
	 * @param sendQueue 監視管理Queue送信
	 * @param props ログ転送エージェントプロパティ
	 */
	public TransferLogManager(EjbConnectionManager ejbConnectionManager, SendQueue sendQueue, Properties props) {
		
		m_ejbConnectionManager = ejbConnectionManager;
		m_sendQueue = sendQueue;
		m_props = props;
		
		m_runTransferLog  = new Hashtable<String, LogThreadManageInfo>();
	}

	
	/**
	 * ログ転送スレッド停止（ファシリティ指定）します。<BR>
	 * 
	 * 指定されたファシリティIDのログ情報をログ転送管理情報から削除します。
	 * 
	 * @param facilityId ファシリティID
	 */
	@SuppressWarnings("unchecked")
	public void removeTransferLogInfo(String facilityId) {
		
		
		// 転送設定ID毎の設定を確認し、ログファイルの転送条件を決定する

//		Set keySet = m_runTransferLog.keySet();
		Set keySet = new HashSet(m_runTransferLog.keySet());
			
		for (Iterator iter = keySet.iterator(); iter.hasNext();) {
			
			int runInterval = Integer.MAX_VALUE;
			int existenceFlg = YesNoConstant.TYPE_NO;

			String logName = (String) iter.next();
			
			LogThreadManageInfo info = m_runTransferLog.get(logName);
			if(info == null){
				continue;
			}
			
			
			for (Iterator iterator = info.iterator(); iterator.hasNext();) {
				LogTransferFileInfoFacility logInfo = (LogTransferFileInfoFacility)iterator.next();
			
				//対象のファシリティIDの情報か
				if(facilityId.equals(logInfo.getFacilityId()))
				{
					//ログ情報の削除
					info.remove(logInfo);
					
				}else{
					// 最も短い「動作間隔（秒）」で実行
					int tmpRunInterval = logInfo.getRunInterval();
					if(tmpRunInterval < runInterval){
						runInterval = tmpRunInterval;
					}
					//ファイル存在チェック
					int tmpExistenceFlg = logInfo.getExistenceFlg();
					if(existenceFlg ==  YesNoConstant.TYPE_NO){
						existenceFlg = tmpExistenceFlg;
					}
					
				}
			}


			TransferLogThread thread = info.getThread();

			//すべての設定情報がなければ
			if(info.size() == 0){
				//ログ停止
				log.debug("thread stop : file path : " + logName);
				thread.requestStop();
				//転送情報も削除
				m_runTransferLog.remove(logName);
			}else{
				//それ以外は
				//動作周期更新
				thread.setCondition(existenceFlg, runInterval);
			}
			
		}
	
	}
	
	/**
	 * ログ転送スレッドを開始します。<BR>
	 * 
	 * @param list 転送対象ログファイル情報一覧
	 */
	public void setTransferLogInfo(String facilityId, ArrayList<LogTransferFileInfo> loglist) {
		

		
		// 転送設定ID毎の設定を確認し、ログファイルの転送条件を決定する

		for (Iterator iter = loglist.iterator(); iter.hasNext();) {
			
			int runInterval = Integer.MAX_VALUE;
			int existenceFlg = YesNoConstant.TYPE_NO;

			LogTransferFileInfo logInfo = (LogTransferFileInfo) iter.next();
			String filePath = logInfo.getFilePath();
			
			if(logInfo.getValid() == ValidConstant.TYPE_INVALID){
				//無効の場合削除
				LogThreadManageInfo logMnginfo = m_runTransferLog.get(filePath);
				if(logMnginfo == null){
					continue;
				}

				for (Iterator iterator = logMnginfo.iterator(); iterator.hasNext();) {
					LogTransferFileInfoFacility logInfoFacility = (LogTransferFileInfoFacility)iterator.next();
				
					//対象のファシリティIDの情報か
					if(facilityId.equals(logInfoFacility.getFacilityId()) &&
						logInfo.getTransferId().equals(logInfoFacility.getTransferId()) )
					{
						//ログ情報の削除
						log.debug("remove " + logInfoFacility.getFilePath() + " : " + logInfoFacility.getTransferId());
						logMnginfo.remove(logInfoFacility);
						
					}else{
						// 最も短い「動作間隔（秒）」で実行
						int tmpRunInterval = logInfoFacility.getRunInterval();
						if(tmpRunInterval < runInterval){
							runInterval = tmpRunInterval;
						}
						//ファイル存在チェック
						int tmpExistenceFlg = logInfoFacility.getExistenceFlg();
						if(existenceFlg ==  YesNoConstant.TYPE_NO){
							existenceFlg = tmpExistenceFlg;
						}
						
					}
				}
				

				TransferLogThread thread = logMnginfo.getThread();

				//すべての設定情報がなければ
				if(logMnginfo.size() == 0){
					//ログ停止
					log.debug("thread stop : file path : " + filePath);
					thread.requestStop();
					//転送情報も削除
					m_runTransferLog.remove(filePath);
				}else{
					//それ以外は
					//動作周期更新
					thread.setCondition(existenceFlg, runInterval);
				}
				
			
			}else{

				//有効の場合
				
				
				LogThreadManageInfo logMnginfo = m_runTransferLog.get(filePath);
				
				
				if(logMnginfo == null){
					//対応する情報がないので新規作成
					logMnginfo = new LogThreadManageInfo(filePath);
					
					String facilityPath = "";
					try {
						RepositoryController repository = m_ejbConnectionManager.getRepositoryController();
						facilityPath = repository.getFacilityPath(facilityId, null);
						
					} catch (Exception e) {
						log.error("setTransferLogInfo(): ファシリティパス取得時にエラーが発生しました。" + e.getMessage());
					}
					
					//ログ情報格納
					logMnginfo.add(facilityId,facilityPath,logInfo);
					
					//スレッド起こす
					TransferLogThread thread = new TransferLogThread(this,
																	  m_props,
																	  filePath,
																	  logInfo.getRunInterval(),
																	  logInfo.getExistenceFlg());
					thread.start();
					
					//スレッド情報格納
					logMnginfo.setThread(thread);
					
					// デバッグ出力
					if (log.isDebugEnabled()) {
						log.debug("put " + filePath + " : ");
						Iterator itr = logMnginfo.iterator();
						while (itr.hasNext()) {
							LogTransferFileInfoFacility info = (LogTransferFileInfoFacility) itr
									.next();
							log.debug("  " + info.getTransferId() + ", "
									+ info.getFacilityId() + ", "
									+ info.getFacilityPath());
						}
					}
					
					// ログ転送管理情報のMAPに登録
					m_runTransferLog.put(filePath,logMnginfo);
				
				
				
				}else{

					//対応する情報あり
					
					runInterval = logInfo.getRunInterval();
					existenceFlg = logInfo.getExistenceFlg();

					boolean add = true;
					for (Iterator iterator = logMnginfo.iterator(); iterator.hasNext();) {
						LogTransferFileInfoFacility logInfoFacility = (LogTransferFileInfoFacility)iterator.next();
					
						//対象のファシリティIDの情報か
						if(facilityId.equals(logInfoFacility.getFacilityId()) &&
							logInfo.getTransferId().equals(logInfoFacility.getTransferId()) )
						{
							//ログ情報内容書き換え
							
							logInfoFacility.setExistenceFlg(logInfo.getExistenceFlg());
							logInfoFacility.setRunInterval(logInfo.getRunInterval());
							 add = false;
						}

						// 最も短い「動作間隔（秒）」で実行
						int tmpRunInterval = logInfoFacility.getRunInterval();
						if(tmpRunInterval < runInterval){
							runInterval = tmpRunInterval;
						}
						//ファイル存在チェック
						int tmpExistenceFlg = logInfoFacility.getExistenceFlg();
						if(existenceFlg ==  YesNoConstant.TYPE_NO){
							existenceFlg = tmpExistenceFlg;
						}
					}

					//該当する情報がなければ、情報追加
					if(add){
						String facilityPath = "";
						try {
							RepositoryController repository = m_ejbConnectionManager.getRepositoryController();
							facilityPath = repository.getFacilityPath(facilityId, null);
							
						} catch (Exception e) {
							log.error("setTransferLogInfo(): ファシリティパス取得時にエラーが発生しました。" + e.getMessage());
						}
						
						//ログ情報格納
						logMnginfo.add(facilityId,facilityPath,logInfo);
					}

					TransferLogThread thread = logMnginfo.getThread();

					//動作周期更新
					thread.setCondition(existenceFlg, runInterval);
					
				
				}

			}

		}
		
	}
	
	
	/**
	 * 全てのログ転送スレッドを停止します。<BR>
	 * 
	 */
	public void stopTransfer() {
		
		synchronized (m_runTransferLog) {
			Set keySet = m_runTransferLog.keySet();
			for (Iterator iter = keySet.iterator(); iter.hasNext();) {
				String  filePath = (String) iter.next();

				LogThreadManageInfo info = m_runTransferLog.get(filePath);
				if(info == null){
					continue;
				}

				TransferLogThread thread = info.getThread();
				thread.requestStop();
			}
			m_runTransferLog.clear();
		}
	}
	
	
	/**
	 * 監視管理のJMSに情報を通知します。<BR>
	 * 
	 * @param priority 重要度
	 * @param app アプリケーション
	 * @param msgId メッセージID
	 * @param msg メッセージ
	 * @param msgOrg オリジナルメッセージ
	 */
	public void sendMessage(String filePath, int priority, String app, String msgId, String msg, String msgOrg) {
		
		//　転送設定ID毎、ファシリティID毎に通知
		LogThreadManageInfo info = m_runTransferLog.get(filePath);
		if(info == null){
			log.error(filePath+"エラー発生送信失敗");//@@どんな情報？？？
			return;
		}

		// ログ出力情報
		LogOutputInfo logOutput = new LogOutputInfo();
		logOutput.setPluginId(Agent.PLUGIN_ID);
		logOutput.setPriority(priority);
		logOutput.setApplication(app);
		logOutput.setMessageId(msgId);
		logOutput.setMessage(msg);
		logOutput.setMessageOrg(msgOrg);
		logOutput.setConfirmFlg(ConfirmConstant.TYPE_UNCONFIRMED);
		logOutput.setGenerationDate(new Date());
		logOutput.setEventLogFlg(true);
		logOutput.setStatusInfoFlg(false);
		
		for (Iterator iter = info.iterator(); iter.hasNext();) {
			LogTransferFileInfoFacility logInfo = (LogTransferFileInfoFacility) iter.next();
			
			logOutput.setMonitorId(logInfo.getTransferId());
			logOutput.setFacilityId(logInfo.getFacilityId());
			logOutput.setScopeText(logInfo.getFacilityPath());	

			this.m_sendQueue.put(logOutput);			
			
		}
		
	}


	/**
	 * ログ転送情報をリフレッシュする
	 * 起動時、再接続時にマネージャから送られてくるファイルリスト情報と
	 * キャッシュ上のファイルリスト情報を比較し、存在しないものがある場合は、
	 * キャッシュ上のファイルリストから削除する。
	 * 
	 * @param facilityId マネージャから送られてきたファシリティID
	 * @param logList マネージャから送られてきたファイル情報
	 */
	public void refreshRunTransferLog(String facilityId, ArrayList<LogTransferFileInfo> logList) {
		
		Set keySet = new HashSet<String>(m_runTransferLog.keySet());
		
		if(keySet != null && keySet.size() > 0) {
		
			for (Iterator itr = keySet.iterator(); itr.hasNext();) {
				
				String filePath = (String) itr.next();
				
				// ファイルパスを基に、LogThreadManagerInfoを取得
				LogThreadManageInfo info = m_runTransferLog.get(filePath);
				if(info == null){
					continue;
				}
				
				// ファイルパスが同じ内容の物が複数ある場合
				for(Iterator itr2 = info.iterator(); itr2.hasNext();) {
					LogTransferFileInfoFacility logFileInfoCache = (LogTransferFileInfoFacility)itr2.next();
										
					// キャッシュ上のファイル情報のファシリティIDと引数のファシリティIDを確認し、
					// 同じ場合のみ処理を続ける。
					if (!logFileInfoCache.getFacilityId().equals(facilityId)){
						continue;
					}
					
					boolean checkFlg = true;
					
					// 新しく取得したログ転送情報と既にHashtableに載っている情報の転送設定IDとファイルパスを比較し、
					// どちらか存在しないものがあれば、そのThreadを停止する。
					for(Iterator itr3 = logList.iterator(); itr3.hasNext();) {
						LogTransferFileInfo logFileInfoNew = (LogTransferFileInfo)itr3.next();
						
						// 転送設定IDを比較
						if(logFileInfoCache.getTransferId().equals(logFileInfoNew.getTransferId())){
							//ファイルパスを比較
							if(logFileInfoCache.getFilePath().equals(logFileInfoNew.getFilePath())){
								log.debug("match : transferID : "+logFileInfoCache.getTransferId()+
										" file path : "+logFileInfoCache.getFilePath()+
										" facilityID : "+logFileInfoCache.getFacilityId());
								checkFlg = false;
								break;
							}
						}
					}
					
					// 転送設定IDとファイルパスのどちらかが異なっていた場合、キャッシュ上から削除する
					if(checkFlg) {
						log.debug("remove : transferID : "+logFileInfoCache.getTransferId()+
								" file path : "+logFileInfoCache.getFilePath()+
								" facilityID : "+logFileInfoCache.getFacilityId());
						info.remove(logFileInfoCache);
						
						TransferLogThread thread = info.getThread();

						//すべての設定情報がなければ
						if(info.size() == 0){
							//ログ停止
							log.debug("thread stop : file path : " + filePath);
							thread.requestStop();
							//転送情報も削除
							m_runTransferLog.remove(filePath);
						}						
					}				
				}
			}
		}
	}
	
}
