/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.http.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.SchedulerException;

import com.clustercontrol.bean.MonitorConstant;
import com.clustercontrol.bean.RunIntervalConstant;
import com.clustercontrol.http.factory.AddMonitorHttp;
import com.clustercontrol.http.factory.AddMonitorHttpString;
import com.clustercontrol.http.factory.DeleteMonitorHttp;
import com.clustercontrol.http.factory.DeleteMonitorHttpString;
import com.clustercontrol.http.factory.ModifyMonitorHttp;
import com.clustercontrol.http.factory.ModifyMonitorHttpString;
import com.clustercontrol.http.factory.SelectMonitorHttp;
import com.clustercontrol.http.factory.SelectMonitorHttpString;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.factory.AddMonitor;
import com.clustercontrol.monitor.run.factory.DeleteMonitor;
import com.clustercontrol.monitor.run.factory.ModifyMonitor;
import com.clustercontrol.monitor.run.factory.SelectMonitor;

/**
 * HTTP監視を制御するSession Beanクラス<BR>
 * 
 * @ejb.bean name="MonitorHttpController"
 *           jndi-name="MonitorHttpController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=RepositoryController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorInfo"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 *     
 */
public abstract class MonitorHttpControllerBean implements SessionBean {
	
	protected static Log m_log = LogFactory.getLog( MonitorHttpControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * HTTP監視情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="HttpMonitorWrite"
	 *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 登録に成功した場合、true
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws ParseException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws SystemException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean addHttp(MonitorInfo info) throws InvalidTransactionException, IllegalStateException, CreateException, FinderException, SchedulerException, ParseException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, SystemException {
		
		Principal pri = m_context.getCallerPrincipal();
		
		// HTTP監視情報を登録
		AddMonitor http = null;
		if(info.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			http = new AddMonitorHttp();
		}
		else if(info.getMonitorType() == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			http = new AddMonitorHttpString();
		}
		else{
			return false;
		}
		return  http.add(info, pri.getName());
	}
	
	/**
	 * HTTP監視情報をマネージャ上で更新します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="HttpMonitorWrite"
	 *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 変更に成功した場合、true
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws CreateException
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws SchedulerException
	 * @throws ParseException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws SystemException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean modifyHttp(MonitorInfo info) throws InvalidTransactionException, IllegalStateException, CreateException, FinderException, RemoveException, SchedulerException, ParseException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, SystemException {
		
		Principal pri = m_context.getCallerPrincipal();
		
		// HTTP監視情報を更新
		ModifyMonitor http = null;
		if(info.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			http = new ModifyMonitorHttp();
		}
		else if(info.getMonitorType() == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			http = new ModifyMonitorHttpString();
		}
		else{
			return false;
		}
		return  http.modify(info, pri.getName());
	}
	
	/**
	 * HTTP監視情報をマネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="HttpMonitorWrite"
	 *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @param monitorType 監視判定タイプ
	 * @return 削除に成功した場合、true
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws SchedulerException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws SystemException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean deleteHttp(String monitorId, int monitorType) throws InvalidTransactionException, IllegalStateException, FinderException, RemoveException, SchedulerException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, SystemException {
		
		// HTTP監視情報を削除
		DeleteMonitor http = null;
		if(monitorType == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			http = new DeleteMonitorHttp();
		}
		else if(monitorType == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			http = new DeleteMonitorHttpString();
		}
		else{
			return false;
		}
		return  http.delete(MonitorConstant.STRING_HTTP, monitorId);
	}
	
	/**
	 * HTTP監視情報を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="HttpMonitorRead"
	 *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId 監視項目ID
	 * @param monitorType 監視判定タイプ
	 * @return HTTP監視情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public MonitorInfo getHttpInfo(String monitorId, int monitorType) throws CreateException, FinderException, SchedulerException, NamingException {
		
		// HTTP監視情報を取得
		SelectMonitor http = null;
		if(monitorType == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			http = new SelectMonitorHttp();
		}
		else if(monitorType == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			http = new SelectMonitorHttpString();
		}
		else{
			return null;
		}
		
		MonitorInfo info = http.getMonitor(MonitorConstant.STRING_HTTP, monitorId);
		return info;
	}
	
	/**
	 * HTTP監視一覧リストを取得します。<BR>
	 * 
	 * 戻り値のArrayListはArrayListのArrayListであり、
	 * 内部のArrayListは以下のように生成される。<BR>
	 * 
	 * <Pre>
	 * ArrayList info = new ArrayList();
	 *  info.add(monitor.getMonitorId());
	 *  info.add(monitor.getDescription());
	 *  info.add(facilityPath);
	 *  info.add(MonitorTypeConstant.typeToString(monitor.getMonitorType().intValue()));
	 *  info.add(monitor.getCalendarId());
	 *  info.add(RunIntervalConstant.typeToString(monitor.getRunInterval().intValue()));
	 *  info.add(schedule.getValid(monitor.getMonitorTypeId(), monitor.getMonitorId()));
	 *  info.add(monitor.getRegUser());
	 *  info.add(monitor.getRegDate() == null ? null:new Date(monitor.getRegDate().getTime()));
	 *  info.add(monitor.getUpdateUser());
	 *  info.add(monitor.getUpdateDate() == null ? null:new Date(monitor.getUpdateDate().getTime()));
	 *	
	 *	list.add(info);
	 * </li>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="HttpMonitorRead"
	 *     method-intf="Remote"
	 *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws NamingException 
	 * @throws SchedulerException 
	 * @throws FinderException 
	 * @throws CreateException 
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 */
	public ArrayList getHttpList() throws CreateException, FinderException, SchedulerException, NamingException {
		
		// HTTP監視一覧を取得
		SelectMonitorHttp http = new SelectMonitorHttp();
		ArrayList list = http.getMonitorList(MonitorConstant.STRING_HTTP);
		
		return list;
	}
	
	/**
	 * HTTP監視一覧のテーブル定義情報を取得します。<BR>
	 * 
	 * 戻り値のArrayListには、定義された順番に従ったTableColumnInfoが
	 * 含まれています。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @param locale ロケール
	 * @return TableColumnInfoの配列
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getHttpListTableDefine(Locale locale) {
		
		/** テーブル情報定義配列 */
		SelectMonitorHttp http = new SelectMonitorHttp();
		ArrayList tableDefine = http.getMonitorListTableDefine(locale);
		
		return tableDefine;
	}
	
	/**
	 * HTTP監視 文字列監視判定一覧のテーブル定義情報を取得します。<BR>
	 * 
	 * 戻り値のArrayListには、定義された順番に従ったTableColumnInfoが
	 * 含まれています
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @param locale ロケール
	 * @return TableColumnInfoの配列
	 * 
	 * @version 2.1.0
	 * 
	 *  @see com.clustercontrol.bean.TableColumnInfo
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getHttpStringValueListTableDefine(Locale locale) {
		
		/** テーブル情報定義配列 */
		SelectMonitorHttpString http = new SelectMonitorHttpString();
		ArrayList tableDefine = http.getStringValueListTableDefine(locale);
		
		return tableDefine;
	}
}
