/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.sql.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.SchedulerException;

import com.clustercontrol.bean.MonitorConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.factory.AddMonitor;
import com.clustercontrol.monitor.run.factory.DeleteMonitor;
import com.clustercontrol.monitor.run.factory.ModifyMonitor;
import com.clustercontrol.monitor.run.factory.SelectMonitor;
import com.clustercontrol.sql.factory.AddMonitorSql;
import com.clustercontrol.sql.factory.AddMonitorSqlString;
import com.clustercontrol.sql.factory.DeleteMonitorSql;
import com.clustercontrol.sql.factory.DeleteMonitorSqlString;
import com.clustercontrol.sql.factory.ModifyMonitorSql;
import com.clustercontrol.sql.factory.ModifyMonitorSqlString;
import com.clustercontrol.sql.factory.SelectMonitorSql;
import com.clustercontrol.sql.factory.SelectMonitorSqlString;

/**
 * SQL監視を制御するSession Bean <BR>
 * 
 * @ejb.bean name="MonitorSqlController"
 *           jndi-name="MonitorSqlController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=AccessController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=RepositoryController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyController"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorInfo"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 */
public abstract class MonitorSqlControllerBean implements SessionBean {
	
	protected static Log m_log = LogFactory.getLog( MonitorSqlControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * SQL監視情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SqlMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 登録に成功した場合、true
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws ParseException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws SystemException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean addSql(MonitorInfo info) throws InvalidTransactionException, IllegalStateException, CreateException, FinderException, SchedulerException, ParseException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, SystemException {
		
		Principal pri = m_context.getCallerPrincipal();
		
		// SQL監視情報を登録
		AddMonitor sql = null;
		if(info.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			sql = new AddMonitorSql();
		}
		else if(info.getMonitorType() == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			sql = new AddMonitorSqlString();
		}
		else{
			return false;
		}
		return  sql.add(info, pri.getName());
	}
	
	/**
	 * マネージャ上のSQL監視情報を更新します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SqlMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 変更に成功した場合、true
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws CreateException
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws SchedulerException
	 * @throws ParseException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws SystemException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean modifySql(MonitorInfo info) throws InvalidTransactionException, IllegalStateException, CreateException, FinderException, RemoveException, SchedulerException, ParseException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, SystemException {
		
		Principal pri = m_context.getCallerPrincipal();

		// SQL監視情報を更新
		ModifyMonitor sql = null;
		if(info.getMonitorType() == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			sql = new ModifyMonitorSql();
		}
		else if(info.getMonitorType() == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			sql = new ModifyMonitorSqlString();
		}
		else{
			return false;
		}
		return  sql.modify(info, pri.getName());
	}
	
	/**
	 * SQL監視情報をマネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SqlMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @param monitorType 監視判定タイプ
	 * @return 削除に成功した場合、true
	 * @throws InvalidTransactionException
	 * @throws IllegalStateException
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws SchedulerException
	 * @throws NamingException
	 * @throws NotSupportedException
	 * @throws HeuristicMixedException
	 * @throws HeuristicRollbackException
	 * @throws RollbackException
	 * @throws SystemException
	 */
	public boolean deleteSql(String monitorId, int monitorType) throws InvalidTransactionException, IllegalStateException, FinderException, RemoveException, SchedulerException, NamingException, NotSupportedException, HeuristicMixedException, HeuristicRollbackException, RollbackException, SystemException {

		// SQL監視情報を削除
		DeleteMonitor sql = null;
		if(monitorType == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			sql = new DeleteMonitorSql();
		}
		else if(monitorType == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			sql = new DeleteMonitorSqlString();
		}
		else{
			return false;
		}
		return  sql.delete(MonitorConstant.STRING_SQL, monitorId);
	}
	
	/**
	 * SQL監視情報を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SqlMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId 監視項目ID
	 * @param monitorType 監視判定タイプ
	 * @return SQL監視情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 * @see com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public MonitorInfo getSqlInfo(String monitorId, int monitorType) throws CreateException, FinderException, SchedulerException, NamingException {

		// SQL監視情報を取得
		SelectMonitor sql = null;
		if(monitorType == MonitorTypeConstant.TYPE_NUMERIC){
			// 数値監視の場合
			sql = new SelectMonitorSql();
		}
		else if(monitorType == MonitorTypeConstant.TYPE_STRING){
			// 文字列監視の場合
			sql = new SelectMonitorSqlString();
		}
		else{
			return null;
		}
		
		MonitorInfo info = sql.getMonitor(MonitorConstant.STRING_SQL, monitorId);
		return info;
	}

	/**
	 * SQL監視一覧リストを取得します。
	 * 
	 * 戻り値はArrayListのArrayListであり内部のArrayListは以下のように
	 * 値が格納される。
	 * 
	 * 
	 * 
	 * <Pre>
	 * ArrayList info = new ArrayList();
	 *  info.add(monitor.getMonitorId());
	 *  info.add(monitor.getDescription());
	 *  info.add(facilityPath);
	 *  info.add(MonitorTypeConstant.typeToString(monitor.getMonitorType().intValue()));
	 *  info.add(monitor.getCalendarId());
	 *  info.add(RunIntervalConstant.typeToString(monitor.getRunInterval().intValue()));
	 *  info.add(schedule.getValid(monitor.getMonitorTypeId(), monitor.getMonitorId()));
	 *  info.add(monitor.getRegUser());
	 *  info.add(monitor.getRegDate() == null ? null:new Date(monitor.getRegDate().getTime()));
	 *  info.add(monitor.getUpdateUser());
	 *  info.add(monitor.getUpdateDate() == null ? null:new Date(monitor.getUpdateDate().getTime()));
	 *
	 *  list.add(info);
	 * </Pre>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SqlMonitorRead"
     *     method-intf="Remote"
     *     
     * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws NamingException 
	 * @throws SchedulerException 
	 * @throws FinderException 
	 * @throws CreateException 
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 */
	public ArrayList getSqlList() throws CreateException, FinderException, SchedulerException, NamingException {

		// SQL監視一覧を取得
		SelectMonitorSql sql = new SelectMonitorSql();
		ArrayList list = sql.getMonitorList(MonitorConstant.STRING_SQL);

		return list;
	}
	
	/**
	 * SQL監視一覧のテーブル定義情報を取得します。<BR>
	 * 
	 * ArrayListには表示項目(String)が表示順に格納されています。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール
	 * @return TableColumnInfoの配列
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getSqlListTableDefine(Locale locale) {
		
		/** テーブル情報定義配列 */
		SelectMonitorSql sql = new SelectMonitorSql();
		ArrayList tableDefine = sql.getMonitorListTableDefine(locale);
		
		return tableDefine;
	}
	
	/**
	 * SQL監視 文字列監視判定一覧のテーブル定義情報を取得します。<BR>
	 * 
	 * ArrayListには表示項目(String)が表示順に格納されています。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param locale ロケール
	 * @return TableColumnInfoの配列
	 * 
	 * @version 2.1.0
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getSqlStringValueListTableDefine(Locale locale) {
		
		/** テーブル情報定義配列 */
		SelectMonitorSqlString sql = new SelectMonitorSqlString();
		ArrayList tableDefine = sql.getStringValueListTableDefine(locale);

		return tableDefine;
	}
}
