/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performanceMGR.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.performanceMGR.ejb.bmp.CollectorStatusBean;
import com.clustercontrol.performanceMGR.ejb.bmp.CollectorStatusDAO;
import com.clustercontrol.performanceMGR.ejb.bmp.CollectorStatusPK;


/**
 * 収集ステータス設定情報のDBアクセスを行うクラス
 * 
 * EJBのEntityBeanのCollectorStatusDAOの実装クラス
 * 
 * @version 1.0
 * @since 1.0
 */
public class CollectorStatusDAOImpl implements CollectorStatusDAO{
	//ログ出力
	protected static Log m_log = LogFactory.getLog( CollectorStatusDAO.class );
	
	/**
	 * コンストラクター
	 */
	public CollectorStatusDAOImpl(){

	}
	
	/**
	 * 初期化を行なう
	 */
	public void init() {
		
		m_log.trace("init CollectorStatusDAO");
	}
	
	/**
	 * EntityBeanで必要となるloadメソッド
	 * メモリ上のEntityBeanをDB上のデータに更新する
	 * @param pk 収集設定のPK
	 * @param ejb 同期を行うEntityBean
	 */
	public void load(CollectorStatusPK pk, CollectorStatusBean ejb)
	throws EJBException {
		
		m_log.debug("load() start : " + pk.getCollectorID());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM COLLECTOR_STATUS WHERE COLLECTORID = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.collectorID );
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setCollectorID(res.getString("COLLECTORID"));
				ejb.setRunStatus(res.getShort("RUN_STATUS"));			
				ejb.setErrorStatus(res.getShort("ERROR_STATUS"));
				ejb.setEndStatus(res.getShort("END_STATUS"));
				ejb.setCount(res.getInt("COUNT"));
				Timestamp timestamp = res.getTimestamp("LAST_COLLECT_DATE");
				if(timestamp != null){
					ejb.setLastCollectDate(new Date(timestamp.getTime()));
				} else {
					ejb.setLastCollectDate(null);
				}
			}else{	
				String msg = "CollectorStatus data is not found.";
				m_log.error("load() error : " + pk.getCollectorID() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.getCollectorID() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.getCollectorID() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end   : " + pk.getCollectorID());	
	}
	
	/**
	 *  EntityBeanで必要となるstoreメソッド
	 */
	public void store(CollectorStatusBean ejb) 
	throws EJBException{
		
		m_log.debug("store() start :" + ejb.getCollectorID());

		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "UPDATE COLLECTOR_STATUS SET " +
			"RUN_STATUS = ?," +
			"ERROR_STATUS = ?," +
			"END_STATUS = ?," +
			"COUNT = ?," +
			"LAST_COLLECT_DATE = ? " +
			"WHERE COLLECTORID = ?";
						
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setShort(1, (short)ejb.getRunStatus());			
			stmt.setShort(2, (short)ejb.getErrorStatus());
			stmt.setShort(3, (short)ejb.getEndStatus());
			stmt.setInt(4, ejb.getCount());
			
			Date lastCollectDate = ejb.getLastCollectDate();
			if(lastCollectDate != null){
				stmt.setTimestamp(5, new Timestamp(lastCollectDate.getTime()));
			} else {
				stmt.setTimestamp(5, null);
			}

			stmt.setString(6, ejb.getCollectorID());
			
			int row = stmt.executeUpdate();
			
			if (row != 1) {	
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getCollectorID() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getCollectorID() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getCollectorID() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		
		m_log.debug("store() end   :" + ejb.getCollectorID());		
	}
	
	/**
	 * EntityBeanで必要となるremoveメソッド
	 * DBのデータを削除する
	 * @param pk　削除を行うpkオブジェクト
	 */
	public void remove(CollectorStatusPK pk) throws EJBException {
		m_log.debug("remove()  start: " + pk.getCollectorID());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM COLLECTOR_STATUS WHERE COLLECTORID = ? ";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1,pk.collectorID );
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			
			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			
		}
		m_log.debug("remove() end : " + pk.getCollectorID());
	}
	
	/**
	 * EntityBeanで必要となるcreateメソッド
	 * DBに存在しないEntityBeanを生成する際にDBにインサートする
	 * @param ejb インサートするEntityBeanオブジェクト
	 * @return DB内でレコードを一意識別するpkオブジェクト
	 */
	public CollectorStatusPK create(CollectorStatusBean ejb) 
	throws CreateException, EJBException, DuplicateKeyException{
		
		m_log.debug("create() start : " + ejb.getCollectorID());

		Connection conn = null;
		CollectorStatusPK pk = new CollectorStatusPK();
		PreparedStatement stmt = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "INSERT INTO COLLECTOR_STATUS " +
			"(COLLECTORID, " +
			"RUN_STATUS," +
			"ERROR_STATUS," +
			"END_STATUS," +
			"COUNT," +
			"LAST_COLLECT_DATE)" +
			"values (?,?,?,?,?,?);";
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, ejb.getCollectorID() );
			stmt.setShort(2, (short)ejb.getRunStatus());
			stmt.setShort(3, (short)ejb.getErrorStatus());
			stmt.setShort(4, (short)ejb.getEndStatus());
			stmt.setInt(5, ejb.getCount());
			stmt.setTimestamp(6, null);  // まだ収集を開始していないため最終収集時刻はnullを設定する
			
			int row =	stmt.executeUpdate();
			
			if (row == 1) {
				/**
				 * CollectorStatusPKに値をつめます。
				 */
				pk = new CollectorStatusPK(ejb.getCollectorID());
			}
			else{
				
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getCollectorID() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			
			//Error Codel "23505" mean DuplicateKey
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getCollectorID() + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			m_log.error("create() error :" + ejb.toString() + " EJBException ");
			throw new CreateException(e.getMessage());
		
		} finally {
			
			//コネクション、結果セット、プリペアドステートメントのクロース
			try {
				if(stmt != null){
					stmt.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getCollectorID() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getCollectorID());
		return pk;
	}
	
	/**
	 *  EntityBeanで必要となるfindByPrimaryKeyメソッド
	 * クライアントから要求があったときにそのオブジェクトが既に生成されいるかをチェックする
	 * 存在していれば、一意識別子であるpkオブジェクトを返す。
	 * @param pk 一意識別子のpkオブジェクト
	 * @return pk 一意識別子のpkオブジェクト
	 * @throws FinderException
	 */
	public CollectorStatusPK findByPrimaryKey(CollectorStatusPK pk) 
	throws FinderException ,EJBException{
		
		m_log.debug("findbyPK() start : " + pk.getCollectorID());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "SELECT * FROM COLLECTOR_STATUS WHERE COLLECTORID = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			
			stmt.setString(1,pk.collectorID );
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			}
			else{
				String msg = "id " + pk.getCollectorID() + " not found.";
				m_log.error("findbyPK() error : " + pk.toString() + " SQLException");
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("findbyPK() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			
			//コネクション、結果セット、プリペアドステートメントのクローズ			
			try {
				if(res != null){
					res.close();
				}
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findbyPK() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		
		m_log.debug("findbyPK() end : " + pk.getCollectorID());
		}
	}
}
