/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.factory;

import java.sql.Timestamp;
import java.util.Calendar;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.StatusExpirationConstant;
import com.clustercontrol.monitor.ejb.entity.StatusInfoLocal;
import com.clustercontrol.monitor.ejb.entity.StatusInfoPK;
import com.clustercontrol.monitor.ejb.entity.StatusInfoUtil;
import com.clustercontrol.monitor.message.LogOutputInfo;

/**
 * ステータス情報を更新するクラス<BR>
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class OutputStatusInfo {

	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog( OutputStatusInfo.class );

	/**
	 * ステータス情報を出力します。<BR>
	 * 同じステータス情報が存在しない場合は、ステータス情報を作成します。
	 * 同じステータス情報がすでに存在する場合は、ステータス情報を更新します。
	 * 
	 * @param logOutput ログ出力情報
	 * @param outputDate 受信日時
	 * @return 成功した場合は、 <code>true</code>
	 */
	public boolean outputStatusInfo(LogOutputInfo logOutput, Timestamp outputDate) {
		
		boolean result = false;
		
		if(logOutput.isStatusInfoFlg())
		{
			StatusInfoLocal statusInfo = null;
			boolean insertFlg = false;
			
			try
			{
				// エンティティ情報の検索
				StatusInfoPK pk = new StatusInfoPK(logOutput.getMonitorId(), logOutput.getPluginId(),logOutput.getFacilityId());
				statusInfo = StatusInfoUtil.getLocalHome().findByPrimaryKey(pk);
			}
			catch (NamingException e)
			{
				m_log.error("outputStatusInfo():" + e.getMessage());
			}
			catch(FinderException e)
			{
				// 検索条件に合致するCMPエンティティが存在しない
				insertFlg = true;
			}
			
			try
			{
				// エンティティ情報の生成
				if(insertFlg)
				{
					insertStatusInfo(logOutput, outputDate);
					result = true;
				}
				//　エンティティ情報の更新
				else
				{
					updateStatusInfo(statusInfo, logOutput, outputDate);
					result = true;
				}
			}
			catch(NamingException e)
			{
				m_log.error("outputStatusInfo():" + e.getMessage());
			}
			catch(CreateException e)
			{
				m_log.error("outputStatusInfo():" + e.getMessage());
			}
			catch(EJBException e)
			{
				// NOT NULL のカラムにnullを送信した場合など
				m_log.error("outputStatusInfo():" + e.getMessage());
			}
		}
		return result;
	}
	
	/**
	 * ステータス情報を作成します。
	 * 
	 * @param logOutput ログ出力情報
	 * @param outputDate 受信日時
	 * @throws NamingException
	 * @throws CreateException
	 */
	public void  insertStatusInfo(LogOutputInfo logOutput, Timestamp outputDate) throws NamingException, CreateException{
	
		try
		{
			// 有効期限制御フラグ
			Integer expirationFlg = getExpirationFlg(logOutput);
			// 有効期限日時
			Timestamp expirationDate = null;
			
			// 有効期限日時
			if(expirationFlg != null){
		        expirationDate = getExpirationDate(logOutput, outputDate);
			}
			
			StatusInfoUtil.getLocalHome().create(
					logOutput.getFacilityId(),
					logOutput.getMonitorId(),
					logOutput.getPluginId(),
					logOutput.getApplication(),
					logOutput.getMessageId(),
					logOutput.getMessage(),
					new Integer(logOutput.getPriority()),
					new java.sql.Timestamp(logOutput.getGenerationDate().getTime()),
					outputDate,
					expirationFlg,
					expirationDate);
		}
		catch(NamingException e)
		{
			throw e;
		}
		catch(CreateException e)
		{
			throw e;
		}
	}
	
	/**
	 * ステータス情報を更新します。
	 * 
	 * @param statusInfo ステータス情報のローカルコンポーネントインターフェース
	 * @param logOutput ログ出力情報
	 * @param outputDate 受信日時
	 * @throws EJBException
	 */
	public void updateStatusInfo(StatusInfoLocal statusInfo, LogOutputInfo logOutput, Timestamp outputDate) throws EJBException{
	
		if(statusInfo != null)
		{
			try
			{
				// 有効期限制御フラグ
				Integer expirationFlg = getExpirationFlg(logOutput);
				// 有効期限日時
				Timestamp expirationDate = null;
				// 有効期限日時
				if(expirationFlg != null){
			        expirationDate = getExpirationDate(logOutput, outputDate);
				}
				
				statusInfo.setApplication(logOutput.getApplication());
				statusInfo.setMessageId(logOutput.getMessageId());
				statusInfo.setMessage(logOutput.getMessage());
			
				// 重要度が変更されていた場合、出力日時を更新する
				if (statusInfo.getPriority().intValue() != logOutput.getPriority()) {
					statusInfo.setGenerationDate(new java.sql.Timestamp(logOutput.getGenerationDate().getTime()));
				}
				
				statusInfo.setPriority(new Integer(logOutput.getPriority()));				
//				statusInfo.setGenerationDate(new java.sql.Timestamp(logOutput.getGenerationDate().getTime()));
				statusInfo.setOutputDate(outputDate);
				statusInfo.setExpirationFlg(expirationFlg);
				statusInfo.setExpirationDate(expirationDate);
			}
			catch(EJBException e)
			{
				// NOT NULL のカラムにnullを送信した場合など
				throw e;
			}
		}
	}
	
	/**
	 * 存続期間経過後の処理の制御フラグを返します。
	 * 
	 * @param logOutput
	 * @see com.clustercontrol.bean.StatusExpirationConstant
	 * @return 存続期間経過後の処理の制御フラグ
	 */
	private Integer getExpirationFlg(LogOutputInfo logOutput){

		// 有効期限制御フラグ
		Integer expirationFlg = null;

		if(logOutput != null)
		{
			if(StatusExpirationConstant.TYPE_DELETE  == logOutput.getStatusInvalidFlg()){
				expirationFlg = new Integer(logOutput.getStatusInvalidFlg());
			}
			if(StatusExpirationConstant.TYPE_UPDATE  == logOutput.getStatusInvalidFlg()){
				// 有効期間経過後の扱いが、更新されていない旨のメッセージに置き換える場合は、
				// 置換え後の重要度を設定する
				expirationFlg = new Integer(logOutput.getStatusUpdatePriority());	
			}
		}
		return expirationFlg;
	}
	
	/**
	 * 存続期間日時を返します。
	 * 
	 * @param logOutput ログ出力譲歩う
	 * @param outputDate 受信日時
	 * @return 存続期間日時
	 */
	private Timestamp getExpirationDate(LogOutputInfo logOutput, Timestamp outputDate){

		// 存続期間日時
		Timestamp expirationDate = null;

		if(logOutput != null)
		{
			if(logOutput.getStatusValidPeriod() > 0){
				Calendar cal = null;
				cal = Calendar.getInstance();
				cal.setTime(outputDate);
		        cal.add(Calendar.MINUTE, logOutput.getStatusValidPeriod());
		        expirationDate = new Timestamp(cal.getTimeInMillis());
			}
		}
		return expirationDate;
	}
}
