/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.factory;

import java.io.File;
import java.io.FileInputStream;
import java.sql.SQLException;
import java.sql.Time;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Properties;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.CommonTableTreeItem;
import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.bean.JudgmentObjectConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.bean.ViewListInfo;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.jobmanagement.bean.HistoryFilterPropertyConstant;
import com.clustercontrol.jobmanagement.bean.JobCommandInfo;
import com.clustercontrol.jobmanagement.bean.JobEndStatusInfo;
import com.clustercontrol.jobmanagement.bean.JobFileInfo;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobNotificationsInfo;
import com.clustercontrol.jobmanagement.bean.JobObjectInfo;
import com.clustercontrol.jobmanagement.bean.JobParameterInfo;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobTriggerTypeConstant;
import com.clustercontrol.jobmanagement.bean.JobWaitRuleInfo;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobEndInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobEndMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobFileInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobInfoPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeMasterLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;

/**
 * ジョブ情報を検索するクラスです。
 *
 * @version 2.4.0
 * @since 1.0.0
 */
public class SelectJob {
	/** ジョブ階層表示のセパレータ文字列 */
    public static final String SEPARATOR = ">";
    /** ログ出力のインスタンス */
    protected static Log m_log = LogFactory.getLog( SelectJob.class );
    /** 最大表示数 */
    private final static int MAX_DISPLAY_NUMBER = 500;
    /** ファイル転送ジョブ展開表示 */
	private static Boolean m_openForwardFileJob = null;
	/** リポジトリ */
	private RepositoryControllerLocal repository = null;
	
	/**
	 * コンストラクタ
	 * <P>
	 * ジョブ管理プロパティファイルから、ファイル転送ジョブ展開表示設定を取得します。
	 */
	public SelectJob(){
		if(m_openForwardFileJob == null){
			try {
			    Properties properties = new Properties();
				String homedir = System.getProperty("jboss.server.home.dir");
				String propertyFile = homedir + File.separator + "conf" + File.separator + "jobmanagement.properties";
				
				// プロパティファイルからキーと値のリストを読み込みます
				properties.load(new FileInputStream(propertyFile));
				
				String mode = properties.getProperty("open.forward.file.job", "false");
				if(mode.equalsIgnoreCase("true")){
					m_openForwardFileJob = Boolean.valueOf(true);
				}
				else {
					m_openForwardFileJob = Boolean.valueOf(false);
				}
			} catch (Exception e) {
				m_log.error("SelectJob():" + e.getMessage());
				m_openForwardFileJob = Boolean.valueOf(false);
			}
		}
	}
	
	/**
	 * ジョブツリー情報を取得します。
	 * <P>
	 * <ol>
	 * <li>ジョブツリー情報のルート(最上位)のインスタンスを作成します。</li>
	 * <li>ジョブリレーションマスタを親ジョブIDが"TOP"で検索し取得します。</li>
	 * <li>取得したジョブリレーションマスタの数、以下の処理を行います。</li>
	 *   <ol>
	 *   <li>ジョブリレーションマスタからジョブマスタを取得します。</li>
	 *   <li>ジョブマスタとジョブツリー情報のルートを渡して、ジョブツリー情報の作成を行います。</li>
	 *   </ol>
	 * </ol>
	 * 
	 * @param treeOnly true=ジョブ情報を含まない, false=ジョブ情報含む
	 * @param locale ロケール情報
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#createJobTree(JobMasterLocal, JobTreeItem, boolean)
	 */
	public JobTreeItem getJobTree(boolean treeOnly, Locale locale) throws FinderException, NamingException {
		
		//JobTreeItemの最上位インスタンスを作成
	    JobInfo info = new JobInfo("", JobConstant.STRING_COMPOSITE, JobConstant.TYPE_COMPOSITE);
		JobTreeItem tree = new JobTreeItem(null, info);
		
		//ジョブツリーのルートを生成
		info = new JobInfo("", Messages.getString("job", locale), JobConstant.TYPE_COMPOSITE);
		JobTreeItem item = new JobTreeItem(tree, info);
		
		//親ジョブIDが"TOP"のジョブリレーションを取得
		Collection ct = JobRelationMasterUtil.getLocalHome().findByParentJobId("TOP");
		
		Iterator itr = ct.iterator();
		while(itr.hasNext()){
		    //ジョブリレーションを取得
		    JobRelationMasterLocal children = (JobRelationMasterLocal)itr.next();
		    //ジョブリレーションからジョブを取得
		    JobMasterLocal childJob = children.getJobMaster();
		    //ジョブツリーを作成する
			createJobTree(childJob, item, treeOnly);
		}

		return tree;
	}
	
	/**
	 * ジョブツリー情報を作成します。
	 * 再帰呼び出しを行います。
	 * <P>
	 * <ol>
	 * <li>ジョブマスタを基に、ジョブ情報{@link com.clustercontrol.jobmanagement.bean.JobInfo}を作成します。</li>
	 * <li>ジョブツリー情報のインスタンスを作成します。</li>
	 * <li>ジョブリレーションマスタをジョブマスタのジョブIDで検索し取得します。</li>
	 * <li>取得したジョブリレーションマスタの数、以下の処理を行います。</li>
	 *   <ol>
	 *   <li>ジョブリレーションマスタからジョブマスタを取得します。</li>
	 *   <li>ジョブマスタとジョブツリー情報のルートを渡して、ジョブツリー情報の作成を行います。</li>
	 *   </ol>
	 * </ol>
	 * 
	 * @param job ジョブマスタ
	 * @param parent 親ジョブツリー情報
	 * @param treeOnly treeOnly true=ジョブ情報を含まない, false=ジョブ情報含む
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#createJobData(JobMasterLocal, boolean)
	 */
	protected void createJobTree(JobMasterLocal job, JobTreeItem parent, boolean treeOnly) throws FinderException, NamingException {
	    //JobTreeItemに格納するジョブ情報(JobInfo)を作成
		JobInfo info = createJobData(job, treeOnly);
		
		//JobTreeItemを作成
		JobTreeItem item = new JobTreeItem(parent, info);
		
		Collection collection = null;
		try {
		    //ジョブリレーションを親ジョブIDで検索
		    collection = JobRelationMasterUtil.getLocalHome().findByParentJobId(job.getJob_id());
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
			    //ジョブリレーションを取得
			    JobRelationMasterLocal children = (JobRelationMasterLocal)itr.next();
			    //ジョブリレーションからジョブを取得
			    JobMasterLocal childJob = children.getJobMaster();
			    //ジョブツリーを作成する
				createJobTree(childJob, item, treeOnly);
			}
		}
	}
	
	/**
	 * ジョブ情報{@link com.clustercontrol.jobmanagement.bean.JobInfo}を作成します。<BR>
	 * ジョブマスタを基に、ジョブ情報を作成します。
	 * 
	 * @param job ジョブマスタ
	 * @param treeOnly treeOnly true=ジョブ情報を含まない, false=ジョブ情報含む
	 * @return ジョブ情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	protected JobInfo createJobData(JobMasterLocal job, boolean treeOnly) throws FinderException, NamingException {

		//JobInfoを作成
	    JobInfo info = new JobInfo(job.getJob_id(), job.getJob_name(), job.getJob_type().intValue());
	    if(treeOnly){
	        return info;
	    }
	    
	    info.setDescription(job.getDescription());
	    info.setCreateTime(job.getReg_date());
	    info.setUpdateTime(job.getUpdate_date());
	    info.setCreateUser(job.getReg_user());
	    info.setUpdateUser(job.getUpdate_user());
	    
	    //待ち条件を取得
	    JobWaitRuleInfo waitRule = null;
	    JobStartMasterLocal start = job.getJobStartMaster();
	    if(start != null){
	        //待ち条件を取得
	        waitRule = new JobWaitRuleInfo();
	        waitRule.setSuspend(start.getSuspend().intValue());
	        waitRule.setCondition(start.getCondition_type().intValue());
	        waitRule.setEndCondition(start.getUnmatch_end_flg().intValue());
	        waitRule.setEndStatus(start.getUnmatch_end_status().intValue());
	        waitRule.setEndValue(start.getUnmatch_end_value().intValue());
	        waitRule.setSkip(start.getSkip().intValue());
	        waitRule.setSkipEndStatus(start.getSkip_end_status().intValue());
	        waitRule.setSkipEndValue(start.getSkip_end_value().intValue());
	        waitRule.setCalendar(start.getCalendar().intValue());
	        waitRule.setCalendarId(start.getCalendar_id());
	        waitRule.setCalendarEndValue(start.getCalendar_end_value().intValue());
	 
	        waitRule.setStart_delay(start.getStart_delay().intValue());
	        waitRule.setStart_delay_session(start.getStart_delay_session().intValue());
	        waitRule.setStart_delay_session_value(start.getStart_delay_session_value().intValue());
	        waitRule.setStart_delay_time(start.getStart_delay_time().intValue());
	        if(start.getStart_delay_time_value() instanceof Time)
	        	waitRule.setStart_delay_time_value(new Date(start.getStart_delay_time_value().getTime()));
	        waitRule.setStart_delay_condition_type(start.getStart_delay_condition_type().intValue());
	        waitRule.setStart_delay_notify(start.getStart_delay_notify().intValue());
	        waitRule.setStart_delay_notify_priority(start.getStart_delay_notify_priority().intValue());
	        waitRule.setStart_delay_operation(start.getStart_delay_operation().intValue());
	        waitRule.setStart_delay_operation_type(start.getStart_delay_operation_type().intValue());
	        waitRule.setStart_delay_operation_end_value(start.getStart_delay_operation_end_value().intValue());
	        
	        waitRule.setEnd_delay(start.getEnd_delay().intValue());
	        waitRule.setEnd_delay_session(start.getEnd_delay_session().intValue());
	        waitRule.setEnd_delay_session_value(start.getEnd_delay_session_value().intValue());
	        waitRule.setEnd_delay_job(start.getEnd_delay_job().intValue());
	        waitRule.setEnd_delay_job_value(start.getEnd_delay_job_value().intValue());
	        waitRule.setEnd_delay_time(start.getEnd_delay_time().intValue());
	        if(start.getEnd_delay_time_value() instanceof Time)
	        	waitRule.setEnd_delay_time_value(new Date(start.getEnd_delay_time_value().getTime()));
	        waitRule.setEnd_delay_condition_type(start.getEnd_delay_condition_type().intValue());
	        waitRule.setEnd_delay_notify(start.getEnd_delay_notify().intValue());
	        waitRule.setEnd_delay_notify_priority(start.getEnd_delay_notify_priority().intValue());
	        waitRule.setEnd_delay_operation(start.getEnd_delay_operation().intValue());
	        waitRule.setEnd_delay_operation_type(start.getEnd_delay_operation_type().intValue());
	        waitRule.setEnd_delay_operation_end_value(start.getEnd_delay_operation_end_value().intValue());
	        
	        
	        //待ち条件（ジョブ）を取得
			Collection startJobList = start.getJobStartJobMaster();
			ArrayList<JobObjectInfo> objectList = new ArrayList<JobObjectInfo>();
			if(startJobList != null && startJobList.size() > 0){
				Iterator itr = startJobList.iterator();
				while(itr.hasNext()){
				    JobStartJobMasterLocal startJob = (JobStartJobMasterLocal)itr.next();
					if(startJob != null){
					    JobObjectInfo objectInfo = new JobObjectInfo();
					    objectInfo.setJobId(startJob.getTarget_job_id());
					    try {
					        //対象ジョブを取得
                            JobMasterLocal targetJob = JobMasterUtil.getLocalHome().findByPrimaryKey(new JobMasterPK(startJob.getTarget_job_id()));
    					    objectInfo.setJobName(targetJob.getJob_name());
    					    objectInfo.setType(startJob.getTarget_job_type().intValue());
    					    objectInfo.setValue(startJob.getTarget_job_end_value().intValue());
    					    objectList.add(objectInfo);
					    } catch (FinderException e) {
					    	throw e;
                        } catch (NamingException e) {
                        	throw e;
                        }
					}
				}
			}
			
			//待ち条件（時刻）を取得
			JobStartTimeMasterLocal startTime = start.getJobStartTimeMaster();
			if(startTime != null){
			    JobObjectInfo objectInfo = new JobObjectInfo();
			    objectInfo.setType(JudgmentObjectConstant.TYPE_TIME);
			    objectInfo.setTime(new Date(startTime.getStart_time().getTime()));
			    objectList.add(objectInfo);
			}
			
			waitRule.setObject(objectList);
			info.setWaitRule(waitRule);
	    }
	    
	    //実行コマンドを取得
	    JobCommandInfo commandInfo = null;
	    JobCommandMasterLocal command = job.getJobCommandMaster();
	    if(command != null){
            commandInfo = new JobCommandInfo();
	        commandInfo.setFacilityID(command.getFacility_id());
	        commandInfo.setProcessingMethod(command.getProcess_mode().intValue());
	        commandInfo.setStartCommand(command.getStart_command());
	        commandInfo.setStopCommand(command.getStop_command());
	        commandInfo.setUser(command.getEffective_user());
	        commandInfo.setErrorEndFlg(command.getError_end_flg().intValue());
	        commandInfo.setErrorEndValue(command.getError_end_value().intValue());
            try {
                //リポジトリ(RepositoryControllerLocal)を取得
            	if(repository == null)
            		repository = RepositoryControllerUtil.getLocalHome().create();
                //ファシリティパスを取得
                commandInfo.setScope(repository.getFacilityPath(command.getFacility_id(), null));
            } catch (CreateException e) {
            } catch (NamingException e) {
            } catch (FinderException e) {
            }
	        info.setCommand(commandInfo);
	    }
	    
	    //ファイル転送を取得
	    JobFileInfo fileInfo = null;
	    JobFileMasterLocal file = job.getJobFileMaster();
	    if(file != null){
	    	fileInfo = new JobFileInfo();
	    	fileInfo.setProcessingMethod(file.getProcess_mode().intValue());
	    	fileInfo.setSrcFacilityID(file.getSrc_facility_id());
	    	fileInfo.setDestFacilityID(file.getDest_facility_id());
	    	fileInfo.setSrcFile(file.getSrc_file());
	    	fileInfo.setSrcWorkDir(file.getSrc_work_dir());
	    	fileInfo.setDestDirectory(file.getDest_directory());
	    	fileInfo.setDestWorkDir(file.getDest_work_dir());
	    	fileInfo.setCompressionFlg(file.getCompression_flg().intValue());
	    	fileInfo.setCheckFlg(file.getCheck_flg().intValue());
	    	fileInfo.setUser(file.getEffective_user());
            try {
                //リポジトリ(RepositoryControllerLocal)を取得
            	if(repository == null)
            		repository = RepositoryControllerUtil.getLocalHome().create();
                //ファシリティパスを取得
                fileInfo.setSrcScope(repository.getFacilityPath(file.getSrc_facility_id(), null));
                fileInfo.setDestScope(repository.getFacilityPath(file.getDest_facility_id(), null));
            } catch (CreateException e) {
            } catch (NamingException e) {
            } catch (FinderException e) {
            }
	        info.setFile(fileInfo);
	    }
	    
	    //通知メッセージを取得
	    ArrayList<JobNotificationsInfo> noticeList = new ArrayList<JobNotificationsInfo>();
	    Collection notices = job.getJobNoticeMaster();
	    if(notices != null){
	        Iterator itr = notices.iterator();
			while(itr.hasNext()){
			    JobNoticeMasterLocal notice = (JobNoticeMasterLocal)itr.next();
			    JobNotificationsInfo notificationsInfo = new JobNotificationsInfo();
			    notificationsInfo.setType(notice.getNotice_type().intValue());
			    notificationsInfo.setPriority(notice.getPriority().intValue());
			    notificationsInfo.setNotify_id(notice.getNotify_id());
			    noticeList.add(notificationsInfo);
			}
			info.setNotifications(noticeList);
	    }
	    
	    //終了状態を取得
	    ArrayList<JobEndStatusInfo> endList = new ArrayList<JobEndStatusInfo>();
	    Collection ends = job.getJobEndMaster();
	    if(ends != null){
	        Iterator itr = ends.iterator();
			while(itr.hasNext()){
			    JobEndMasterLocal end = (JobEndMasterLocal)itr.next();
			    JobEndStatusInfo endInfo = new JobEndStatusInfo();
			    endInfo.setType(end.getEnd_status().intValue());
			    endInfo.setValue(end.getEnd_value().intValue());
			    endInfo.setStartRangeValue(end.getEnd_value_from().intValue());
			    endInfo.setEndRangeValue(end.getEnd_value_to().intValue());
			    endList.add(endInfo);
			}
			info.setEndStatus(endList);
	    }
	    
	    //パラメータを取得
	    ArrayList<JobParameterInfo> paramList = new ArrayList<JobParameterInfo>();
	    Collection params = job.getJobParamMaster();
	    if(params != null){
	        Iterator itr = params.iterator();
			while(itr.hasNext()){
			    JobParamMasterLocal param = (JobParamMasterLocal)itr.next();
			    JobParameterInfo paramInfo = new JobParameterInfo();
			    paramInfo.setParamId(param.getParam_id());
			    paramInfo.setType(param.getParam_type().intValue());
			    paramInfo.setDescription(param.getDescription());
			    paramInfo.setValue(param.getValue());
			    paramList.add(paramInfo);
			}
			info.setParam(paramList);
	    }
	    
	    return info;
	}
	
	/**
	 * 親ジョブIDの配下の階層のみのジョブツリー情報を取得します。
	 * <P>
	 * <ol>
	 * <li>ジョブツリー情報のルート(最上位)のインスタンスを作成します。</li>
	 * <li>ジョブリレーションマスタを親ジョブIDで検索し取得します。</li>
	 * <li>取得したジョブリレーションマスタの数、以下の処理を行います。</li>
	 *   <ol>
	 *   <li>ジョブリレーションマスタからジョブマスタを取得します。</li>
	 *   <li>ジョブマスタを基に、ジョブツリー情報の作成を行います。</li>
	 *   </ol>
	 * </ol>
	 * 
	 * @param parentJobId 親ジョブID
	 * @param locale ロケール情報
	 * @return ジョブツリー情報{@link com.clustercontrol.jobmanagement.bean.JobTreeItem}の階層オブジェクト
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#createJobTree(JobMasterLocal, JobTreeItem, boolean)
	 */
	public JobTreeItem getJobTreeOneLevel(String parentJobId, Locale locale) throws FinderException, NamingException {
		
		//JobTreeItemの最上位インスタンスを作成
	    JobInfo info = new JobInfo("", JobConstant.STRING_COMPOSITE, JobConstant.TYPE_COMPOSITE);
		JobTreeItem tree = new JobTreeItem(null, info);
		
		//ジョブツリーのルートを生成
		info = new JobInfo("", Messages.getString("job", locale), JobConstant.TYPE_COMPOSITE);
		
		//FIXME 処理が閉じている
		JobTreeItem item = new JobTreeItem(tree, info);
		
		//親ジョブIDが"TOP"のジョブリレーションを取得
		Collection ct = JobRelationMasterUtil.getLocalHome().findByParentJobId(parentJobId);
		
		Iterator itr = ct.iterator();
		while(itr.hasNext()){
		    //ジョブリレーションを取得
		    JobRelationMasterLocal children = (JobRelationMasterLocal)itr.next();
		    //ジョブリレーションからジョブを取得
		    JobMasterLocal childJob = children.getJobMaster();
		    
		    //JobInfoを作成
		    JobInfo childInfo = new JobInfo(childJob.getJob_id(), childJob.getJob_name(), childJob.getJob_type().intValue());
		    //JobTreeItemを作成
		    @SuppressWarnings("unused")
		    JobTreeItem childItem = new JobTreeItem(item, childInfo);
		}

		return tree;
	}
	
	/**
	 * ジョブ履歴一覧情報を取得します。<BR>
	 * 表示履歴数を越えた場合は、表示履歴数分の履歴情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>全てにセッションを取得します。</li>
	 * <li>取得したセッション数、以下の処理を行います。</li>
	 *  <ol>
	 *  <li>セッションからセッションジョブを取得します。</li>
	 *  <li>セッションジョブからジョブ情報を取得します。</li>
	 *  <li>1セッションの情報をテーブルのカラム順（{@link com.clustercontrol.jobmanagement.bean.HistoryTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 *   <dl>
	 *   <dt>履歴情報一覧（Objectの2次元配列）</dt>
	 *   <dd>{ 履歴情報1 {カラム1の値, カラム2の値, … }, 履歴情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *   </dl>
	 *  </ol>
	 *  <li>履歴情報一覧，全履歴数を、ビュー一覧情報（{@link com.clustercontrol.bean.ViewListInfo}）にセットし返します。</li>
	 * </ol>
	 * 
	 * @param histories 表示履歴数
	 * @return ジョブ履歴一覧情報
	 * @throws FinderException
	 * @throws NamingException
	 * @throws SQLException
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 */
	@SuppressWarnings("unchecked")
	public ViewListInfo getHistoryList(int histories) throws FinderException, NamingException, SQLException {
		
		ViewListInfo list = new ViewListInfo();
		ArrayList history = new ArrayList();
		int total = 0;
		
		if(histories <= 0){
			histories = MAX_DISPLAY_NUMBER;
		}
		Integer limit = Integer.valueOf(histories + 1);
	    
	    //全セッションを取得
	    Collection sessionList = JobSessionUtil.getLocalHome().findHistory(
	    		null, null, null, null, null, null, null, null, limit);
	    if(sessionList != null){
			
			//履歴数をカウント
			if(sessionList.size() > histories){
				//最大表示件数より大きい場合
				total = JobSessionUtil.getLocalHome().countHistory(
						null, null, null, null, null, null, null, null);
			}
			else{
				total = sessionList.size();
			}
			
	        Iterator itr = sessionList.iterator();
			while(itr.hasNext()){
			    //セッションを取得
			    JobSessionLocal session = (JobSessionLocal)itr.next();
			    //セッションジョブを取得（セッションのセッションIDとジョブIDで検索）
			    JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
			            new JobSessionJobPK(session.getSession_id(), session.getJob_id()));
			    //セッションジョブからジョブを取得
			    JobInfoLocal jobInfo = sessionJob.getJobInfo();
			    //履歴一覧の１行を作成
			    ArrayList info = new ArrayList();
			    info.add(sessionJob.getStatus());
			    info.add(sessionJob.getEnd_status());
		        info.add(sessionJob.getEnd_value());
			    info.add(sessionJob.getSession_id());
			    info.add(sessionJob.getJob_id());
			    info.add(jobInfo.getJob_name());
			    info.add(jobInfo.getJob_type());
			    if(jobInfo.getJob_type().intValue() == JobConstant.TYPE_JOB){
			        JobCommandInfoLocal jobCommand = jobInfo.getJobCommandInfo();
			        if(jobCommand != null){
			            info.add(jobCommand.getFacility_id());
			            info.add(sessionJob.getScope_text());
			        }
			        else{
			            info.add(null);
			            info.add(null);
			        }
			    }
			    else{
			        info.add(null);
			        info.add(null);
			    }
			    info.add(session.getSchedule_date());
			    info.add(sessionJob.getStart_date());
			    info.add(sessionJob.getEnd_date());
		        if (session.getTrigger_info() != null && !session.getTrigger_info().equals("")) {
		        	info.add(JobTriggerTypeConstant.typeToString(session.getTrigger_type()));
		        	info.add(session.getTrigger_info());
		        }
		        else {
		        	info.add(null);
		        	info.add(null);
		        }
			    
			    history.add(info);
			    
				//取得した履歴を最大表示件数まで格納したら終了
				if(history.size() >= histories)
					break;
			}
	    }
	    list.setTotal(total);
	    list.setList(history);
	    
	    return list;
	}
	
	/**
	 * 検索条件に一致するジョブ履歴一覧情報を取得します。<BR>
	 * 表示履歴数を越えた場合は、表示履歴数分の履歴情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>検索条件に一致するセッションを取得します。</li>
	 * <li>取得したセッション数、以下の処理を行います。</li>
	 *  <ol>
	 *  <li>セッションからセッションジョブを取得します。</li>
	 *  <li>セッションジョブからジョブ情報を取得します。</li>
	 *  <li>1セッションの情報をテーブルのカラム順（{@link com.clustercontrol.jobmanagement.bean.HistoryTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 *   <dl>
	 *   <dt>履歴情報一覧（Objectの2次元配列）</dt>
	 *   <dd>{ 履歴情報1 {カラム1の値, カラム2の値, … }, 履歴情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 *  </ol>
	 *  <li>履歴情報一覧，全履歴数を、ビュー一覧情報（{@link com.clustercontrol.bean.ViewListInfo}）にセットし返します。</li>
	 * </ol>
	 * 
	 * @param property 検索条件
	 * @param histories 表示履歴数
	 * @return ジョブ履歴一覧情報
	 * @throws FinderException
	 * @throws NamingException
	 * @throws SQLException
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 */
	@SuppressWarnings("unchecked")
	public ViewListInfo getHistoryList(Property property, int histories) throws FinderException, NamingException, SQLException {
	    
		ArrayList values = null;
		//開始・再実行日時（自）取得
		values = PropertyUtil.getPropertyValue(property, HistoryFilterPropertyConstant.START_FROM_DATE);
		Date startFromDate = null;
		if(values.get(0) != null && values.get(0) instanceof Date){
		    startFromDate = (Date)values.get(0);
		}
		//開始・再実行日時（至）取得
		values = PropertyUtil.getPropertyValue(property, HistoryFilterPropertyConstant.START_TO_DATE);
		Date startToDate = null;
		if(values.get(0) != null && values.get(0) instanceof Date){
		    startToDate = (Date)values.get(0);
		}
		//終了・中断日時（自）取得
		values = PropertyUtil.getPropertyValue(property, HistoryFilterPropertyConstant.END_FROM_DATE);
		Date endFromDate = null;
		if(values.get(0) != null && values.get(0) instanceof Date){
		    endFromDate = (Date)values.get(0);
		}
		//終了・中断日時（至）取得
		values = PropertyUtil.getPropertyValue(property, HistoryFilterPropertyConstant.END_TO_DATE);
		Date endToDate = null;
		if(values.get(0) != null && values.get(0) instanceof Date){
		    endToDate = (Date)values.get(0);
		}

		//ジョブID取得
		values = PropertyUtil.getPropertyValue(property, HistoryFilterPropertyConstant.JOB_ID);
		String jobId = null;
        if(values.get(0) instanceof JobTreeItem){
            jobId = ((JobTreeItem)values.get(0)).getData().getId();
        }
        else if(values.get(0) instanceof String && ((String)values.get(0)).length() > 0){
            jobId = (String)values.get(0);
        }
		
		//状態取得
		values = PropertyUtil.getPropertyValue(property, HistoryFilterPropertyConstant.STATUS);
		Integer status = null;
		if(values.get(0) instanceof String){
		    String statusString = (String)values.get(0);
		    status = Integer.valueOf(StatusConstant.stringToType(statusString));
		}
		
		//実行契機種別取得
		values = PropertyUtil.getPropertyValue(property, HistoryFilterPropertyConstant.TRIGGER_TYPE);
		Integer triggerType = null;
		if(values.get(0) instanceof String){
		    String triggerTypeString = (String)values.get(0);
		    triggerType = Integer.valueOf(JobTriggerTypeConstant.stringToType(triggerTypeString));
		}
		
		//実行契機情報取得
		values = PropertyUtil.getPropertyValue(property, HistoryFilterPropertyConstant.TRIGGER_INFO);
		String triggerInfo = null;
		if(!"".equals((String) values.get(0))) {
			triggerInfo = (String) values.get(0);
		}
		
		ViewListInfo list = new ViewListInfo();
		ArrayList history = new ArrayList();
		int total = 0;
		
		if(histories <= 0){
			histories = MAX_DISPLAY_NUMBER;
		}
		Integer limit = Integer.valueOf(histories + 1);
	    
		//検索条件に該当するセッションを取得
		Collection sessionList = 
			JobSessionUtil.getLocalHome().findHistory(
					startFromDate, 
					startToDate, 
					endFromDate, 
					endToDate, 
					jobId, 
					status,
					triggerType,
					triggerInfo,
					limit);
	    
	    if(sessionList != null){
	        
	    	//履歴数をカウント
	    	if(sessionList.size() > histories){
	    		//最大表示件数より大きい場合
	    		total = JobSessionUtil.getLocalHome().countHistory(
	    				startFromDate, 
	    				startToDate, 
	    				endFromDate, 
	    				endToDate, 
	    				jobId,
	    				status,
	    				triggerType,
	    				triggerInfo);
	    	}
	    	else{
	    		total = sessionList.size();
	    	}
			
	        Iterator itr = sessionList.iterator();
			while(itr.hasNext()){
			    //セッションを取得
			    JobSessionLocal session = (JobSessionLocal)itr.next();
			    //セッションジョブを取得
			    JobSessionJobLocal sessionJob = 
			        JobSessionJobUtil.getLocalHome().findByPrimaryKey(
			                new JobSessionJobPK(session.getSession_id(), session.getJob_id()));
			    //ジョブを取得
			    JobInfoLocal jobInfo = sessionJob.getJobInfo();
			    //履歴一覧の１行を作成
			    ArrayList info = new ArrayList();
			    info.add(sessionJob.getStatus());
	            info.add(sessionJob.getEnd_status());
	            info.add(sessionJob.getEnd_value());
			    info.add(sessionJob.getSession_id());
			    info.add(sessionJob.getJob_id());
			    info.add(jobInfo.getJob_name());
			    info.add(jobInfo.getJob_type());
			    if(jobInfo.getJob_type().intValue() == JobConstant.TYPE_JOB){
			        JobCommandInfoLocal jobCommand = jobInfo.getJobCommandInfo();
			        if(jobCommand != null){
			            info.add(jobCommand.getFacility_id());
			            info.add(sessionJob.getScope_text());
			        }
			        else{
			            info.add(null);
			            info.add(null);
			        }
			    }
			    else{
			        info.add(null);
			        info.add(null);
			    }
			    info.add(session.getSchedule_date());
			    info.add(sessionJob.getStart_date());
			    info.add(sessionJob.getEnd_date());
		        if (session.getTrigger_info() != null && !session.getTrigger_info().equals("")) {
		        	info.add(JobTriggerTypeConstant.typeToString(session.getTrigger_type()));
		        	info.add(session.getTrigger_info());
		        }
		        else {
		        	info.add(null);
		        	info.add(null);
		        }
			    history.add(info);
			    
				//取得した履歴を最大表示件数まで格納したら終了
				if(history.size() >= histories)
					break;
			}
	    }
	    list.setTotal(total);
	    list.setList(history);
	    
	    return list;
	}
	
	/**
	 * セッションIDが一致するジョブ詳細一覧情報を取得します。
	 * <P>
	 * <ol>
	 * <li>セッションをセッションIDで検索し、取得します。</li>
	 * <li>セッションからセッションジョブを取得します。</li>
	 * <li>テーブルツリー情報のルート(最上位)のインスタンスを作成します。</li>
	 * <li>セッションジョブとテーブルツリー情報のルートを渡して、ジョブ詳細一覧情報の作成を行います。</li>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @return ジョブ詳細一覧情報
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#createDetailTree(JobSessionJobLocal, CommonTableTreeItem)
	 */
	public CommonTableTreeItem getDetailList(String sessionId) throws FinderException, NamingException {
	    //セッションをセッションIDで検索し取得
	    JobSessionLocal session = JobSessionUtil.getLocalHome().findByPrimaryKey(new JobSessionPK(sessionId));
	    //セッションジョブを取得
	    JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
	            new JobSessionJobPK(session.getSession_id(), session.getJob_id()));
	    
		//CommonTableTreeItemの最上位インスタンスを作成
	    CommonTableTreeItem tree = new CommonTableTreeItem(null, null);
	    //ジョブ詳細ツリーを作成
	    createDetailTree(sessionJob, tree);
	    
	    return tree;
	}
	
	/**
	 * ジョブ詳細ツリーを作成します。<BR>
	 * 再帰呼び出しを行います。
	 * <P>
	 * <ol>
	 * <li>セッションジョブを基に、ジョブ詳細情報（{@link ArrayList}）を作成します。</li>
	 * <li>テーブルツリー情報を作成します。</li>
	 * <li>セッションIDとジョブIDからジョブリレーション情報を取得します。</li>
	 * <li>ジョブリレーション情報の数、以下の処理を行います。</li>
	 *  <ol>
	 *  <li>ジョブリレーション情報から、セッションジョブを取得します。</li>
	 *  <li>セッションジョブとテーブルツリー情報を渡して、ジョブ詳細一覧情報の作成を行います。</li>
	 *  </ol>
	 * </ol>
	 * 
	 * @param sessionJob セッションジョブ
	 * @param parent 親テーブルツリー情報
	 * @throws FinderException 
	 * @throws NamingException 
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#createDetailTreeData(JobSessionJobLocal)
	 */
	protected void createDetailTree(
	        JobSessionJobLocal sessionJob, 
	        CommonTableTreeItem parent) throws FinderException, NamingException {
	    
	    //セッションジョブからジョブ詳細一覧の１行を作成
		ArrayList info = createDetailTreeData(sessionJob);
		
		//CommonTableTreeItemを作成
		CommonTableTreeItem item = new CommonTableTreeItem(parent, info);
		
		JobInfoLocal job = sessionJob.getJobInfo();
		if(m_openForwardFileJob.booleanValue() || job.getJob_type().intValue() != JobConstant.TYPE_FILEJOB){
		
			Collection collection = null;
			try {
			    //ジョブリレーションを親ジョブIDで検索
			    collection = JobRelationInfoUtil.getLocalHome().findByParentJobId(
			            sessionJob.getSession_id(), 
			            sessionJob.getJob_id());
			} catch (FinderException e) {
				throw e;
			} catch (NamingException e) {
				throw e;
			}
			
			if(collection != null && collection.size() > 0){
				Iterator itr = collection.iterator();
				while(itr.hasNext()){
				    //ジョブリレーションを取得
				    JobRelationInfoLocal children = (JobRelationInfoLocal)itr.next();
				    //セッションジョブを取得
				    JobSessionJobLocal childJob = children.getJobSessionJob();
				    //ジョブ詳細ツリーを作成
				    createDetailTree(childJob, item);
				}
			}
		}
	}
	
	/**
	 * セッションジョブを基にジョブ詳細情報を作成します。
	 * <p>
	 * <ol>
	 * <li>セッションジョブからジョブ情報を取得します。</li>
	 * <li>1セッションジョブの情報をテーブルのカラム順（{@link com.clustercontrol.jobmanagement.bean.JobDetailTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>ジョブ詳細情報を返します。</li>
	 * </ol>
	 * 
	 * @param sessionJob セッションジョブ
	 * @return ジョブ詳細情報
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList createDetailTreeData(JobSessionJobLocal sessionJob) {
	    //セッションジョブからジョブ詳細一覧の１行を作成
	    JobInfoLocal jobInfo = sessionJob.getJobInfo();
	    ArrayList info = new ArrayList();
	    info.add("");
	    info.add(sessionJob.getStatus());
        info.add(sessionJob.getEnd_status());
        info.add(sessionJob.getEnd_value());
	    info.add(sessionJob.getJob_id());
	    info.add(jobInfo.getJob_name());
	    info.add(jobInfo.getJob_type());
	    if(jobInfo.getJob_type().intValue() == JobConstant.TYPE_JOBNET ||
	            jobInfo.getJob_type().intValue() == JobConstant.TYPE_JOB){
	        JobCommandInfoLocal jobCommand = jobInfo.getJobCommandInfo();
	        if(jobCommand != null){
	            info.add(jobCommand.getFacility_id());
	            info.add(sessionJob.getScope_text());
	        }
	        else{
	            info.add(null);
	            info.add(null);
	        }
	        JobStartInfoLocal jobStart = jobInfo.getJobStartInfo();
	        if(jobStart != null){
	            JobStartTimeInfoLocal jobStartTime = jobStart.getJobStartTimeInfo();
	            if(jobStartTime != null){
	                info.add(jobStartTime.getStart_time());
	            }
	            else{
		            info.add(null);
	            }
	        }
	        else{
	            info.add(null);
	        }
	        
	    }
	    else{
	        info.add(null);
	        info.add(null);
	        info.add(null);
	    }
	    info.add(sessionJob.getStart_date());
	    info.add(sessionJob.getEnd_date());

	    return info;
	}
	
	/**
	 * ノード詳細一覧情報を取得します。
	 * <p>
	 * <ol>
	 * <li>セッションIDとジョブIDが一致するセッションジョブを取得します。</li>.
	 * <li>セッションジョブからジョブ情報を取得します。</li>
	 * <li>取得したジョブ情報の種別がジョブの場合、以下の処理を行います。</li>
	 *  <ol>
	 *  <li>1セッションジョブの情報をテーブルのカラム順（{@link com.clustercontrol.jobmanagement.bean.NodeDetailTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 *   <dl>
	 *   <dt>ノード詳細一覧情報（Objectの2次元配列）</dt>
	 *   <dd>{ ノード詳細情報1 {カラム1の値, カラム2の値, … }, ノード詳細情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *   </dl>
	 *  </ol>
	 * <li>取得したジョブ情報の種別がファイル転送ジョブの場合、ファイル転送ジョブのノード詳細一覧情報を作成します。</li>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @param locale ロケール情報
	 * @return ノード詳細一覧情報
	 * @throws FinderException
	 * @throws NamingException
	 * @throws CreateException
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getFileJobNodeDetailList(String, String, Locale)
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getNodeDetailList(String sessionId, String jobId, Locale locale) throws FinderException, NamingException, CreateException {
	    ArrayList nodeDetail = new ArrayList();
	    
	    if(sessionId != null && sessionId.length() > 0 && 
	            jobId != null && jobId.length() > 0){
	    
		    //セッションジョブをセッションIDとジョブIDで検索し、取得
		    JobSessionJobLocal sessionjob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
		            new JobSessionJobPK(sessionId, jobId));
		    
		    JobInfoLocal job = sessionjob.getJobInfo();
		    if(job.getJob_type().intValue() == JobConstant.TYPE_JOB){
		    	//ジョブの場合
		    	
			    //セッションジョブに関連するセッションノードを取得
			    Collection nodeList = sessionjob.getJobSessionNode();
			    
			    if(nodeList != null){
			        Iterator itr = nodeList.iterator();
					while(itr.hasNext()){
					    //セッションノードを取得
					    JobSessionNodeLocal sessionNode = (JobSessionNodeLocal)itr.next();
		
					    //ノード詳細一覧の１行を作成
					    ArrayList info = new ArrayList();
					    info.add(sessionNode.getStatus());
					    info.add(sessionNode.getEnd_value());
					    info.add(sessionNode.getFacility_id());
					    info.add(sessionNode.getNode_name());
					    info.add(sessionNode.getStart_date());
					    info.add(sessionNode.getEnd_date());
					    info.add(sessionNode.getMessage());
					    nodeDetail.add(info);
					}
			    }
		    }
		    else if(job.getJob_type().intValue() == JobConstant.TYPE_FILEJOB){
		    	//ファイル転送ジョブの場合
		    	
		    	//ファイル転送ジョブのノード詳細一覧取得
		    	nodeDetail = getFileJobNodeDetailList(sessionId, jobId, locale);
		    }
	    }
	    
	    return nodeDetail;
	}
	
	/**
	 * ファイル転送ジョブのノード詳細一覧情報を取得します。
	 * <p>
	 * <ol>
	 * <li>セッションIDとジョブIDが一致するセッションジョブを取得します。</li>.
	 * <li>セッションジョブからジョブ情報を取得します。</li>
	 * <li>取得したジョブ情報の種別がファイル転送ジョブの場合、以下の処理を行います。</li>
	 *  <ol>
	 *  <li>1セッションジョブの情報をテーブルのカラム順（{@link com.clustercontrol.jobmanagement.bean.NodeDetailTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 *   <dl>
	 *   <dt>ノード詳細一覧情報（Objectの2次元配列）</dt>
	 *   <dd>{ ノード詳細情報1 {カラム1の値, カラム2の値, … }, ノード詳細情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *   </dl>
	 *  </ol>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @param locale ロケール情報
	 * @return ファイル転送ジョブのノード詳細一覧情報
	 * @throws FinderException
	 * @throws NamingException
	 * @throws CreateException
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getFileJobDetailMessage(String, String, String, String, int, Locale)
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList getFileJobNodeDetailList(
			String sessionId, 
			String jobId, 
			Locale locale) throws FinderException, NamingException, CreateException {
		
	    ArrayList nodeDetail = new ArrayList();
	
	    if(sessionId != null && sessionId.length() > 0 && 
	            jobId != null && jobId.length() > 0){
	    
		    //セッションジョブをセッションIDとジョブIDで検索し、取得
		    JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
		            new JobSessionJobPK(sessionId, jobId));
		    
		    JobInfoLocal job = sessionJob.getJobInfo();
		    if(job.getJob_type().intValue() == JobConstant.TYPE_FILEJOB){
		    	//ファイル転送ジョブの場合
		    	JobFileInfoLocal file = job.getJobFileInfo();
		    	
				//リポジトリ(RepositoryControllerLocal)を取得
		    	if(repository == null)
		    		repository = RepositoryControllerUtil.getLocalHome().create();
	
			    //ジョブリレーションを親ジョブIDで検索
				Collection collection = JobRelationInfoUtil.getLocalHome().findByParentJobId(
			    		sessionJob.getSession_id(), 
			            sessionJob.getJob_id());

				if(collection != null && collection.size() > 0){
					Iterator itr = collection.iterator();
					while(itr.hasNext()){
					    //ジョブリレーションを取得
					    JobRelationInfoLocal children = (JobRelationInfoLocal)itr.next();
					    //セッションジョブを取得
					    JobSessionJobLocal childSessionJob = children.getJobSessionJob();
					    
					    JobInfoLocal childJob = childSessionJob.getJobInfo();
					    if(childJob.getJob_type().intValue() == JobConstant.TYPE_JOBNET){
							//ジョブIDからファシリティIDを抽出
						    String facilityId = childSessionJob.getJob_id();
						    facilityId = facilityId.replaceFirst(sessionJob.getJob_id() + "_", "");
						    
							//ファシリティパスを取得
			                String destFacilityName = repository.getFacilityPath(facilityId, null);
			                
						    //ノード詳細一覧の１行を作成
						    ArrayList info = new ArrayList();
						    info.add(childSessionJob.getStatus());
						    info.add(childSessionJob.getEnd_value());
						    info.add(facilityId);
						    info.add(childSessionJob.getScope_text());
						    info.add(childSessionJob.getStart_date());
						    info.add(childSessionJob.getEnd_date());
						    info.add(getFileJobDetailMessage(
						    		childJob.getSession_id(), 
						    		childJob.getJob_id(), 
					    			facilityId, 
					    			destFacilityName, 
						    		file.getCheck_flg().intValue(), 
						    		locale));	    
						    nodeDetail.add(info);
					    }
					}
				}
		    }
	    }
	    
	    return nodeDetail;
	}
	
	/**
	 * ファイル転送ジョブのノード詳細一覧情報のメッセージを取得します。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @param destFacilityId 受信ファシリティID
	 * @param destFacilityName 受信ファシリティ名
	 * @param checksum ファイルチェック
	 * @param locale ロケール情報
	 * @return メッセージ
	 * @throws FinderException
	 * @throws NamingException
	 */
	protected String getFileJobDetailMessage(
			String sessionId, 
			String jobId, 
			String destFacilityId, 
			String destFacilityName, 
			int checksum, 
			Locale locale) throws FinderException, NamingException {
		
		final String START = "_START";
		final String END = "_END";
		final String FILE = "_FILE";
		final String RTN = "\n";
		
	    StringBuilder message = new StringBuilder();
		
		if(sessionId != null && sessionId.length() > 0 && 
	            jobId != null && jobId.length() > 0){
	    
		    //ジョブリレーションを親ジョブIDで検索
			Collection collection = JobRelationInfoUtil.getLocalHome().findByStartDate(
					sessionId, 
					jobId);
			
			if(collection != null && collection.size() > 0){
				Iterator itr = collection.iterator();
				
				DateFormat dateFormat = DateFormat.getDateTimeInstance(DateFormat.LONG, DateFormat.MEDIUM, locale);
				HashMap<String, String> jobIdMap = new HashMap<String, String>();
				
				while(itr.hasNext()){
				    //ジョブリレーションを取得
				    JobRelationInfoLocal children = (JobRelationInfoLocal)itr.next();
				    //セッションジョブを取得
				    JobSessionJobLocal childSessionJob = children.getJobSessionJob();
				    JobInfoLocal job = childSessionJob.getJobInfo();
				    JobCommandInfoLocal command = job.getJobCommandInfo();

				    //ファイル転送が開始の場合
				    if(childSessionJob.getJob_id().indexOf(CreateFileJob.GET_CHECKSUM) != -1){
				    	//"_GET_CHECKSUM"を除いたジョブIDを取得
				    	String fileJobId = childSessionJob.getJob_id().replaceAll(CreateFileJob.GET_CHECKSUM, "");
				    	jobIdMap.put(fileJobId + FILE, command.getArgument());
				    	
				    	//ファイル転送が開始の場合
				    	if(childSessionJob.getStart_date() != null){
				    		String dateString = dateFormat.format(childSessionJob.getStart_date());
				    		String file = command.getArgument();
				    		Object[] args = { dateString, file, destFacilityName };
			                message.append(Messages.getString("message.job.48", args, locale));
			                message.append(RTN);
			                jobIdMap.put(fileJobId, START);
				    	}
				    
					    //ファイル転送が終了の場合
					    if(childSessionJob.getEnd_date() != null && 
					    		(childSessionJob.getEnd_status() == null || 
					    				childSessionJob.getEnd_status().intValue() != EndStatusConstant.TYPE_NORMAL)){
				    		String dateString = dateFormat.format(childSessionJob.getEnd_date());
				    		String file = command.getArgument();
				    		Object[] args = { dateString, file, destFacilityName };
			                message.append(Messages.getString("message.job.50", args, locale));
			                message.append(RTN);
			                jobIdMap.put(fileJobId, END);
					    }
				    }
				    else if(childSessionJob.getJob_id().indexOf(CreateFileJob.FORWARD) != -1){
				    	//"_FORWARD"を除いたジョブIDを取得
				    	String fileJobId = childSessionJob.getJob_id().replaceAll(CreateFileJob.FORWARD, "");

				    	//ファイル転送が開始の場合
				    	if(childSessionJob.getStart_date() != null && 
				    			checksum == YesNoConstant.TYPE_NO){
				    		
					    		String dateString = dateFormat.format(childSessionJob.getStart_date());
					    		String file = command.getArgument();
					    		Object[] args = { dateString, file, destFacilityName };
				                message.append(Messages.getString("message.job.48", args, locale));
				                message.append(RTN);
				                jobIdMap.put(fileJobId, START);
				    	}
				    	
				    
					    //ファイル転送が終了の場合
					    if(childSessionJob.getEnd_date() != null && 
					    		(jobIdMap.get(fileJobId) == null || 
					    				!jobIdMap.get(fileJobId).equals(END))){
				    		String dateString = dateFormat.format(childSessionJob.getEnd_date());
				    		String file = command.getArgument();
				    		Object[] args = { dateString, file, destFacilityName };
				    		
					    	if(childSessionJob.getEnd_status() != null && 
					    			childSessionJob.getEnd_status().intValue() == EndStatusConstant.TYPE_NORMAL){
					    		if(checksum == YesNoConstant.TYPE_NO){
						    		message.append(Messages.getString("message.job.49", args, locale));
						    		message.append(RTN);
						    		jobIdMap.put(fileJobId, END);
					    		}
					    	}
					    	else{
				                message.append(Messages.getString("message.job.50", args, locale));
				                message.append(RTN);
				                jobIdMap.put(fileJobId, END);
					    	}
					    }
					    
				    	if(checksum == YesNoConstant.TYPE_NO){
				    		jobIdMap.remove(fileJobId);
				    		jobIdMap.remove(fileJobId + FILE);
				    	}
				    }
				    else if(childSessionJob.getJob_id().indexOf(CreateFileJob.CHECK_CHECKSUM) != -1){
				    	//"_CHECK_CHECKSUM"を除いたジョブIDを取得
				    	String fileJobId = childSessionJob.getJob_id().replaceAll(CreateFileJob.CHECK_CHECKSUM, "");
				    	
					    //ファイル転送が終了の場合
					    if(childSessionJob.getEnd_date() != null && 
					    		(jobIdMap.get(fileJobId) == null || 
					    				!jobIdMap.get(fileJobId).equals(END))){
				    		String dateString = dateFormat.format(childSessionJob.getEnd_date());
				    		String file = jobIdMap.get(fileJobId + FILE);
				    		Object[] args = { dateString, file, destFacilityName };
				    		
					    	if(childSessionJob.getEnd_status() != null && 
					    			childSessionJob.getEnd_status().intValue() == EndStatusConstant.TYPE_NORMAL){
					    		message.append(Messages.getString("message.job.49", args, locale));
					    		message.append(RTN);
					    	}
					    	else{
				                message.append(Messages.getString("message.job.50", args, locale));
				                message.append(RTN);
				                jobIdMap.put(fileJobId, END);
					    	}
					    }
					    
			    		jobIdMap.remove(fileJobId);
			    		jobIdMap.remove(fileJobId + FILE);
				    }
				}
			}
	    }
		
		if(message.length() > 0)
			message.setLength(message.length() - 1);
	    
	    return message.toString();
	}
	
	/**
	 * ファイル転送一覧情報を取得します。
	 * <p>
	 * <ol>
	 * <li>セッションIDとジョブIDが一致するセッションジョブを取得します。</li>.
	 * <li>セッションジョブからジョブ情報を取得します。</li>
	 * <li>取得したジョブ情報の種別がファイル転送ジョブの場合、以下の処理を行います。</li>
	 *  <ol>
	 *  <li>1セッションジョブの情報をテーブルのカラム順（{@link com.clustercontrol.jobmanagement.bean.ForwardFileTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 *   <dl>
	 *   <dt>ファイル転送一覧情報（Objectの2次元配列）</dt>
	 *   <dd>{ ファイル転送情報1 {カラム1の値, カラム2の値, … }, ファイル転送情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *   </dl>
	 *  </ol>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @return ファイル転送一覧情報
	 * @throws FinderException
	 * @throws NamingException
	 * @throws CreateException
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#getForwardFileListInfo(String, String, String, String, String, String, int)
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getForwardFileList(String sessionId, String jobId) throws FinderException, NamingException, CreateException {
		ArrayList list = new ArrayList();
		
	    if(sessionId != null && sessionId.length() > 0 && 
	            jobId != null && jobId.length() > 0){
	    
		    //セッションジョブをセッションIDとジョブIDで検索し、取得
		    JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
		            new JobSessionJobPK(sessionId, jobId));
		    
		    JobInfoLocal job = sessionJob.getJobInfo();
		    if(job.getJob_type().intValue() == JobConstant.TYPE_FILEJOB){
		    	//ファイル転送ジョブの場合
		    	JobFileInfoLocal file = job.getJobFileInfo();
		    	
				//リポジトリ(RepositoryControllerLocal)を取得
		    	if(repository == null)
		    		repository = RepositoryControllerUtil.getLocalHome().create();
		    	
				//ファシリティパスを取得
                String srcFacilityName = repository.getFacilityPath(file.getSrc_facility_id(), null);
		    	
			    //ジョブリレーションを親ジョブIDで検索
				Collection collection = JobRelationInfoUtil.getLocalHome().findByParentJobId(
			    		sessionJob.getSession_id(), 
			            sessionJob.getJob_id());

				if(collection != null && collection.size() > 0){
					Iterator itr = collection.iterator();
					while(itr.hasNext()){
					    //ジョブリレーションを取得
					    JobRelationInfoLocal children = (JobRelationInfoLocal)itr.next();
					    //セッションジョブを取得
					    JobSessionJobLocal childSessionJob = children.getJobSessionJob();
					    
					    JobInfoLocal childJob = childSessionJob.getJobInfo();
					    if(childJob.getJob_type().intValue() == JobConstant.TYPE_JOBNET){
							//ジョブIDからファシリティIDを抽出
						    String facilityId = childSessionJob.getJob_id();
						    facilityId = facilityId.replaceFirst(sessionJob.getJob_id() + "_", "");
						    
							//ファシリティパスを取得
			                String destFacilityName = repository.getFacilityPath(facilityId, null);
			                
			                //ファイル転送一覧の１行を作成
						    ArrayList info = 
						    	getForwardFileListInfo(
						    			childJob.getSession_id(), 
						    			childJob.getJob_id(), 
						    			file.getSrc_facility_id(), 
						    			srcFacilityName, 
						    			facilityId, 
						    			destFacilityName, 
						    			file.getCheck_flg().intValue()); 
						    list.addAll(info);
					    }
					}
				}
		    }
	    }
	    
	    return list;
	}
	
	/**
	 * ファイル転送情報を取得します。
	 * <p>
	 * <ol>
	 * <li>セッションIDと親ジョブIDでジョブリレーション情報を取得します。</li>
	 * <li>取得したジョブリレーション情報の数、以下の処理を行います。</li>
	 *  <ol>
	 *  <li>ジョブリレーション情報からセッションジョブを取得します。</li>
	 *  <li>1セッションジョブの情報をテーブルのカラム順（{@link com.clustercontrol.jobmanagement.bean.ForwardFileTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 *   <dl>
	 *   <dt>ファイル転送一覧情報（Objectの配列）</dt>
	 *   <dd>ファイル転送情報 {カラム1の値, カラム2の値, … }</dd>
	 *   </dl>
	 *  </ol>
	 * </ol>
	 * 
	 * @param sessionId セッションID
	 * @param jobId 親ジョブID
	 * @param srcFacilityId 転送ファシリティID
	 * @param srcFacilityName 転送ファシリティ名
	 * @param destFacilityId 受信ファシリティID
	 * @param destFacilityName 受信ファシリティ名
	 * @param checksum ファイルチェック
	 * @return ファイル転送情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	@SuppressWarnings("unchecked")
	protected ArrayList getForwardFileListInfo(
			String sessionId, 
			String jobId, 
			String srcFacilityId, 
			String srcFacilityName, 
			String destFacilityId, 
			String destFacilityName, 
			int checksum) throws FinderException, NamingException {
		
		final String STATUS = "_STATUS";
		final String END_STATUS = "_END_STATUS";
		final String START_DATE = "_START_DATE";
		final String END_DATE = "_END_DATE";
		final String FILE = "_FILE";
		
		ArrayList list = new ArrayList();
		
		if(sessionId != null && sessionId.length() > 0 && 
				jobId != null && jobId.length() > 0){
			
			//ジョブリレーションを親ジョブIDで検索
			Collection collection = JobRelationInfoUtil.getLocalHome().findByStartDate(
					sessionId, 
					jobId);
			
			if(collection != null && collection.size() > 0){
				Iterator itr = collection.iterator();
				
				HashMap jobIdMap = new HashMap();
				
				while(itr.hasNext()){
					//ジョブリレーションを取得
					JobRelationInfoLocal children = (JobRelationInfoLocal)itr.next();
					//セッションジョブを取得
					JobSessionJobLocal childSessionJob = children.getJobSessionJob();
					JobInfoLocal job = childSessionJob.getJobInfo();
					JobCommandInfoLocal command = job.getJobCommandInfo();
					
					//ファイル転送が開始の場合
					if(childSessionJob.getJob_id().indexOf(CreateFileJob.GET_CHECKSUM) != -1){
						//"_GET_CHECKSUM"を除いたジョブIDを取得
						String fileJobId = childSessionJob.getJob_id().replaceAll(CreateFileJob.GET_CHECKSUM, "");
						
						jobIdMap.put(fileJobId + STATUS, childSessionJob.getStatus());
						jobIdMap.put(fileJobId + END_STATUS, childSessionJob.getEnd_status());
						jobIdMap.put(fileJobId + FILE, command.getArgument());
						jobIdMap.put(fileJobId + START_DATE, childSessionJob.getStart_date());
						jobIdMap.put(fileJobId + END_DATE, childSessionJob.getEnd_date());
					}
					else if(childSessionJob.getJob_id().indexOf(CreateFileJob.FORWARD) != -1){
						//"_FORWARD"を除いたジョブIDを取得
						String fileJobId = childSessionJob.getJob_id().replaceAll(CreateFileJob.FORWARD, "");
						
						if(checksum == YesNoConstant.TYPE_NO){
							jobIdMap.put(fileJobId + STATUS, childSessionJob.getStatus());
							jobIdMap.put(fileJobId + END_STATUS, childSessionJob.getEnd_status());
							jobIdMap.put(fileJobId + FILE, command.getArgument());
							jobIdMap.put(fileJobId + START_DATE, childSessionJob.getStart_date());
							jobIdMap.put(fileJobId + END_DATE, childSessionJob.getEnd_date());
						}
						else{
							if(childSessionJob.getEnd_status() != null && 
									childSessionJob.getEnd_status().intValue() != StatusConstant.TYPE_WAIT){
								jobIdMap.put(fileJobId + STATUS, childSessionJob.getStatus());
								jobIdMap.put(fileJobId + END_STATUS, childSessionJob.getEnd_status());
								jobIdMap.put(fileJobId + END_DATE, childSessionJob.getEnd_date());
							}
						}
						
						if(checksum == YesNoConstant.TYPE_NO){
							ArrayList info = new ArrayList();
							info.add(jobIdMap.get(fileJobId + STATUS));
							info.add(jobIdMap.get(fileJobId + END_STATUS));
							info.add(jobIdMap.get(fileJobId + FILE));
							info.add(srcFacilityId);
							info.add(srcFacilityName);
							info.add(destFacilityId);
							info.add(destFacilityName);
							info.add(jobIdMap.get(fileJobId + START_DATE));
							info.add(jobIdMap.get(fileJobId + END_DATE));
							list.add(info);
						}
					}
					else if(childSessionJob.getJob_id().indexOf(CreateFileJob.CHECK_CHECKSUM) != -1){
						//"_CHECK_CHECKSUM"を除いたジョブIDを取得
						String fileJobId = childSessionJob.getJob_id().replaceAll(CreateFileJob.CHECK_CHECKSUM, "");
						
						if(childSessionJob.getEnd_status() != null && 
								childSessionJob.getEnd_status().intValue() != StatusConstant.TYPE_WAIT){
							jobIdMap.put(fileJobId + STATUS, childSessionJob.getStatus());
							jobIdMap.put(fileJobId + END_STATUS, childSessionJob.getEnd_status());
							jobIdMap.put(fileJobId + END_DATE, childSessionJob.getEnd_date());
						}
						
						ArrayList info = new ArrayList();
						info.add(jobIdMap.get(fileJobId + STATUS));
						info.add(jobIdMap.get(fileJobId + END_STATUS));
						info.add(jobIdMap.get(fileJobId + FILE));
						info.add(srcFacilityId);
						info.add(srcFacilityName);
						info.add(destFacilityId);
						info.add(destFacilityName);
						info.add(jobIdMap.get(fileJobId + START_DATE));
						info.add(jobIdMap.get(fileJobId + END_DATE));
						list.add(info);
					}
				}
			}
		}
		
	    return list;
	}
	
	/**
	 * ジョブツリー情報を取得します。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @return ジョブツリー情報
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.bean.JobInfo
	 * @see com.clustercontrol.jobmanagement.factory.SelectJob#createSessionJobData(JobInfoLocal)
	 */
	@SuppressWarnings("unchecked")
	public JobTreeItem getSessionJobInfo(String sessionId, String jobId) throws FinderException, NamingException {
		//セッションジョブを取得
		JobSessionJobLocal sessionJob = 
			JobSessionJobUtil.getLocalHome().findByPrimaryKey(
	            new JobSessionJobPK(sessionId, jobId));
		
		JobInfoLocal job = sessionJob.getJobInfo();
		
		//JobTreeItemに格納するジョブ情報(JobInfo)を作成
		JobInfo info = createSessionJobData(job);
		
		//JobTreeItemを作成
		JobTreeItem item = new JobTreeItem(null, info);
	    
	    return item;
	}
	
	/**
	 * セッションジョブ情報作成
	 * 
	 * @param job ジョブ情報
	 * @return ジョブ情報
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.bean.JobInfo
	 */
	@SuppressWarnings("unchecked")
	protected JobInfo createSessionJobData(JobInfoLocal job) throws FinderException, NamingException {
		//JobInfoを作成
	    JobInfo info = new JobInfo(job.getJob_id(), job.getJob_name(), job.getJob_type().intValue());
	    
	    info.setDescription(job.getDescription());
	    info.setCreateTime(job.getReg_date());
	    info.setUpdateTime(job.getUpdate_date());
	    info.setCreateUser(job.getReg_user());
	    info.setUpdateUser(job.getUpdate_user());
	    
	    //待ち条件を取得
	    JobWaitRuleInfo waitRule = null;
	    JobStartInfoLocal start = job.getJobStartInfo();
	    if(start != null){
	        //待ち条件を取得
	        waitRule = new JobWaitRuleInfo();
	        waitRule.setSuspend(start.getSuspend().intValue());
	        waitRule.setCondition(start.getCondition_type().intValue());
	        waitRule.setEndCondition(start.getUnmatch_end_flg().intValue());
	        waitRule.setEndStatus(start.getUnmatch_end_status().intValue());
	        waitRule.setEndValue(start.getUnmatch_end_value().intValue());
	        waitRule.setSkip(start.getSkip().intValue());
	        waitRule.setSkipEndStatus(start.getSkip_end_status().intValue());
	        waitRule.setSkipEndValue(start.getSkip_end_value().intValue());
	        waitRule.setCalendar(start.getCalendar().intValue());
	        waitRule.setCalendarId(start.getCalendar_id());
	        waitRule.setCalendarEndValue(start.getCalendar_end_value().intValue());
	        
	        waitRule.setStart_delay(start.getStart_delay().intValue());
	        waitRule.setStart_delay_session(start.getStart_delay_session().intValue());
	        waitRule.setStart_delay_session_value(start.getStart_delay_session_value().intValue());
	        waitRule.setStart_delay_time(start.getStart_delay_time().intValue());
	        if(start.getStart_delay_time_value() instanceof Time)
	        	waitRule.setStart_delay_time_value(new Date(start.getStart_delay_time_value().getTime()));
	        waitRule.setStart_delay_condition_type(start.getStart_delay_condition_type().intValue());
	        waitRule.setStart_delay_notify(start.getStart_delay_notify().intValue());
	        waitRule.setStart_delay_notify_priority(start.getStart_delay_notify_priority().intValue());
	        waitRule.setStart_delay_operation(start.getStart_delay_operation().intValue());
	        waitRule.setStart_delay_operation_type(start.getStart_delay_operation_type().intValue());
	        waitRule.setStart_delay_operation_end_value(start.getStart_delay_operation_end_value().intValue());
	        
	        waitRule.setEnd_delay(start.getEnd_delay().intValue());
	        waitRule.setEnd_delay_session(start.getEnd_delay_session().intValue());
	        waitRule.setEnd_delay_session_value(start.getEnd_delay_session_value().intValue());
	        waitRule.setEnd_delay_job(start.getEnd_delay_job().intValue());
	        waitRule.setEnd_delay_job_value(start.getEnd_delay_job_value().intValue());
	        waitRule.setEnd_delay_time(start.getEnd_delay_time().intValue());
	        if(start.getEnd_delay_time_value() instanceof Time)
	        	waitRule.setEnd_delay_time_value(new Date(start.getEnd_delay_time_value().getTime()));
	        waitRule.setEnd_delay_condition_type(start.getEnd_delay_condition_type().intValue());
	        waitRule.setEnd_delay_notify(start.getEnd_delay_notify().intValue());
	        waitRule.setEnd_delay_notify_priority(start.getEnd_delay_notify_priority().intValue());
	        waitRule.setEnd_delay_operation(start.getEnd_delay_operation().intValue());
	        waitRule.setEnd_delay_operation_type(start.getEnd_delay_operation_type().intValue());
	        waitRule.setEnd_delay_operation_end_value(start.getEnd_delay_operation_end_value().intValue());
	        
	        
	        //待ち条件（ジョブ）を取得
			Collection startJobList = start.getJobStartJobInfo();
			ArrayList objectList = new ArrayList();
			if(startJobList != null && startJobList.size() > 0){
				Iterator itr = startJobList.iterator();
				while(itr.hasNext()){
					JobStartJobInfoLocal startJob = (JobStartJobInfoLocal)itr.next();
					if(startJob != null){
					    JobObjectInfo objectInfo = new JobObjectInfo();
					    objectInfo.setJobId(startJob.getTarget_job_id());
					    try {
					        //対象ジョブを取得
                            JobInfoLocal targetJob = 
                            	JobInfoUtil.getLocalHome().findByPrimaryKey(
                            			new JobInfoPK(startJob.getSession_id(), startJob.getTarget_job_id()));
    					    objectInfo.setJobName(targetJob.getJob_name());
    					    objectInfo.setType(startJob.getTarget_job_type().intValue());
    					    objectInfo.setValue(startJob.getTarget_job_end_value().intValue());
    					    objectList.add(objectInfo);
					    } catch (FinderException e) {
					    	throw e;
                        } catch (NamingException e) {
                        	throw e;
                        }
					}
				}
			}
			
			//待ち条件（時刻）を取得
			JobStartTimeInfoLocal startTime = start.getJobStartTimeInfo();
			if(startTime != null){
			    JobObjectInfo objectInfo = new JobObjectInfo();
			    objectInfo.setType(JudgmentObjectConstant.TYPE_TIME);
			    objectInfo.setTime(new Date(startTime.getStart_time().getTime()));
			    objectList.add(objectInfo);
			}
			
			waitRule.setObject(objectList);
			info.setWaitRule(waitRule);
	    }
	    
	    //実行コマンドを取得
	    JobCommandInfo commandInfo = null;
	    JobCommandInfoLocal command = job.getJobCommandInfo();
	    if(command != null){
            commandInfo = new JobCommandInfo();
	        commandInfo.setFacilityID(command.getFacility_id());
	        commandInfo.setProcessingMethod(command.getProcess_mode().intValue());
	        commandInfo.setStartCommand(command.getStart_command());
	        commandInfo.setStopCommand(command.getStop_command());
	        commandInfo.setUser(command.getEffective_user());
	        commandInfo.setErrorEndFlg(command.getError_end_flg().intValue());
	        commandInfo.setErrorEndValue(command.getError_end_value().intValue());
            try {
                //リポジトリ(RepositoryControllerLocal)を取得
            	if(repository == null)
            		repository = RepositoryControllerUtil.getLocalHome().create();
                //ファシリティパスを取得
                commandInfo.setScope(repository.getFacilityPath(command.getFacility_id(), null));
            } catch (CreateException e) {
            	//TODO CreateException 未実装
            } catch (NamingException e) {
            	//TODO NamingException 未実装
            } catch (FinderException e) {
            	//TODO FnderException 未実装
            }
	        info.setCommand(commandInfo);
	    }
	    
	    //ファイル転送を取得
	    JobFileInfo fileInfo = null;
	    JobFileInfoLocal file = job.getJobFileInfo();
	    if(file != null){
	    	fileInfo = new JobFileInfo();
	    	fileInfo.setProcessingMethod(file.getProcess_mode().intValue());
	    	fileInfo.setSrcFacilityID(file.getSrc_facility_id());
	    	fileInfo.setDestFacilityID(file.getDest_facility_id());
	    	fileInfo.setSrcFile(file.getSrc_file());
	    	fileInfo.setSrcWorkDir(file.getSrc_work_dir());
	    	fileInfo.setDestDirectory(file.getDest_directory());
	    	fileInfo.setDestWorkDir(file.getDest_work_dir());
	    	fileInfo.setCompressionFlg(file.getCompression_flg().intValue());
	    	fileInfo.setCheckFlg(file.getCheck_flg().intValue());
	    	fileInfo.setUser(file.getEffective_user());
            try {
                //リポジトリ(RepositoryControllerLocal)を取得
            	if(repository == null)
            		repository = RepositoryControllerUtil.getLocalHome().create();
                //ファシリティパスを取得
                fileInfo.setSrcScope(repository.getFacilityPath(file.getSrc_facility_id(), null));
                fileInfo.setDestScope(repository.getFacilityPath(file.getDest_facility_id(), null));
            } catch (CreateException e) {
            	//TODO CreateException 未実装
            } catch (NamingException e) {
            	//TODO NamingException 未実装
            } catch (FinderException e) {
            	//TODO FnderException 未実装
            }
	        info.setFile(fileInfo);
	    }
	    
	    //通知メッセージを取得
	    ArrayList<JobNotificationsInfo> noticeList = new ArrayList<JobNotificationsInfo>();
	    Collection notices = job.getJobNoticeInfo();
	    if(notices != null){
	        Iterator itr = notices.iterator();
			while(itr.hasNext()){
			    JobNoticeInfoLocal notice = (JobNoticeInfoLocal)itr.next();
			    JobNotificationsInfo notificationsInfo = new JobNotificationsInfo();
			    notificationsInfo.setType(notice.getNotice_type().intValue());
			    notificationsInfo.setPriority(notice.getPriority().intValue());
			    notificationsInfo.setNotify_id(notice.getNotify_id());
			    noticeList.add(notificationsInfo);
			}
			info.setNotifications(noticeList);
	    }
	    
	    //終了状態を取得
	    ArrayList<JobEndStatusInfo> endList = new ArrayList<JobEndStatusInfo>();
	    Collection ends = job.getJobEndInfo();
	    if(ends != null){
	        Iterator itr = ends.iterator();
			while(itr.hasNext()){
			    JobEndInfoLocal end = (JobEndInfoLocal)itr.next();
			    JobEndStatusInfo endInfo = new JobEndStatusInfo();
			    endInfo.setType(end.getEnd_status().intValue());
			    endInfo.setValue(end.getEnd_value().intValue());
			    endInfo.setStartRangeValue(end.getEnd_value_from().intValue());
			    endInfo.setEndRangeValue(end.getEnd_value_to().intValue());
			    endList.add(endInfo);
			}
			info.setEndStatus(endList);
	    }
	    
	    
	    //パラメータを取得
	    ArrayList<JobParameterInfo> paramList = new ArrayList<JobParameterInfo>();
	    Collection params = job.getJobParamInfo();
	    if(params != null){
	        Iterator itr = params.iterator();
			while(itr.hasNext()){
			    JobParamInfoLocal param = (JobParamInfoLocal)itr.next();
			    JobParameterInfo paramInfo = new JobParameterInfo();
			    paramInfo.setParamId(param.getParam_id());
			    paramInfo.setType(param.getParam_type().intValue());
			    paramInfo.setDescription(param.getDescription());
			    paramInfo.setValue(param.getValue());
			    paramList.add(paramInfo);
			}
			info.setParam(paramList);
	    }
	    
	    return info;
	}
}
