/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.bean.SystemParameterConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodePK;
import com.clustercontrol.jobmanagement.util.ParameterUtil;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;

/**
 * ジョブセッションノードのDAOインターフェースを実装するクラス<BR>
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobCommandInfoBean
 * @see com.clustercontrol.jobmanagement.dao.JobCommandInfoDAO
 */
public class JobSessionNodeDAOImpl implements JobSessionNodeDAO {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(JobSessionNodeDAOImpl.class);
	
	public void init() {
	}
	
	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionNodeDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodePK, com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeBean)
	 */
	public void load(JobSessionNodePK pk, JobSessionNodeBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_session_node WHERE session_id = ? AND job_id = ? AND facility_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getFacility_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setAgent_check(res.getInt("agent_check"));
				if(res.getTimestamp("check_date") != null)
					ejb.setCheck_date(new Date(res.getTimestamp("check_date").getTime()));
				else
					ejb.setCheck_date(null);
				if(res.getTimestamp("end_date") != null)
					ejb.setEnd_date(new Date(res.getTimestamp("end_date").getTime()));
				else
					ejb.setEnd_date(null);
				
				if(res.getString("end_value") != null){
					ejb.setEnd_value(res.getInt("end_value"));
				}else{
					ejb.setEnd_value(null);
				}
				
				
				ejb.setFacility_id(res.getString("facility_id"));
				ejb.setJob_id(res.getString("job_id"));
				ejb.setMessage(res.getString("message"));
				ejb.setNode_name(res.getString("node_name"));
				ejb.setResult(res.getString("result"));
				ejb.setRetry_count(res.getInt("retry_count"));
				ejb.setSession_id(res.getString("session_id"));
				if(res.getTimestamp("start_date") != null)
					ejb.setStart_date(new Date(res.getTimestamp("start_date").getTime()));
				else
					ejb.setStart_date(null);
				ejb.setStatus(res.getInt("status"));
				ejb.setStatus_buffer(res.getInt("status_buffer"));
			} else {
				String msg = "JobSessionNode data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}
	
	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionNodeDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeBean)
	 */
	public void store(JobSessionNodeBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getFacility_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_job_session_node SET ");
			sql.append("node_name = ?, ");
			sql.append("status = ?, ");
			sql.append("start_date = ?, ");
			sql.append("end_date = ?, ");
			sql.append("end_value = ?, ");
			sql.append("message = ?, ");
			sql.append("agent_check = ?, ");
			sql.append("check_date = ?, ");
			sql.append("retry_count = ?, ");
			sql.append("result = ?, ");
			sql.append("status_buffer = ? ");
			sql.append("WHERE session_id = ? ");
			sql.append("AND job_id = ? ");
			sql.append("AND facility_id = ?");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getNode_name());
			stmt.setInt(2, ejb.getStatus());
			if(ejb.getStart_date() instanceof java.util.Date)
				stmt.setTimestamp(3, new Timestamp(ejb.getStart_date().getTime()));
			else
				stmt.setTimestamp(3, null);
			if(ejb.getEnd_date() instanceof java.util.Date)
				stmt.setTimestamp(4, new Timestamp(ejb.getEnd_date().getTime()));
			else
				stmt.setTimestamp(4, null);
			if(ejb.getEnd_value() instanceof Integer)
				stmt.setInt(5, ejb.getEnd_value());
			else
				stmt.setNull(5, Types.INTEGER);
			stmt.setString(6, ejb.getMessage());
			if(ejb.getAgent_check() instanceof Integer)
				stmt.setInt(7, ejb.getAgent_check());
			else
				stmt.setNull(7, Types.INTEGER);
			if(ejb.getCheck_date() instanceof java.util.Date)
				stmt.setTimestamp(8, new Timestamp(ejb.getCheck_date().getTime()));
			else
				stmt.setTimestamp(8, null);
			if(ejb.getRetry_count() instanceof Integer)
				stmt.setInt(9, ejb.getRetry_count());
			else
				stmt.setNull(9, Types.INTEGER);
			stmt.setString(10, ejb.getResult());
			if(ejb.getStatus_buffer() instanceof Integer)
				stmt.setInt(11, ejb.getStatus_buffer());
			else
				stmt.setNull(11, Types.INTEGER);
			stmt.setString(12, ejb.getSession_id());
			stmt.setString(13, ejb.getJob_id());
			stmt.setString(14, ejb.getFacility_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getFacility_id() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getFacility_id() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getFacility_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getFacility_id());
	}
	
	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionNodeDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodePK)
	 */
	public void remove(JobSessionNodePK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_job_session_node WHERE session_id = ? AND job_id = ? AND facility_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getFacility_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}
	
	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionNodeDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeBean)
	 */
	public JobSessionNodePK create(JobSessionNodeBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getFacility_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		JobSessionNodePK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_session_node (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("facility_id, ");
			sql.append("node_name, ");
			sql.append("status, ");
			sql.append("start_date, ");
			sql.append("end_date, ");
			sql.append("end_value, ");
			sql.append("message, ");
			sql.append("agent_check, ");
			sql.append("check_date, ");
			sql.append("retry_count, ");
			sql.append("result, ");
			sql.append("status_buffer) ");
			sql.append("VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getSession_id());
			stmt.setString(2, ejb.getJob_id());
			stmt.setString(3, ejb.getFacility_id());
			stmt.setString(4, ejb.getNode_name());
			stmt.setInt(5, ejb.getStatus());
			if(ejb.getStart_date() instanceof java.util.Date)
				stmt.setTimestamp(6, new Timestamp(ejb.getStart_date().getTime()));
			else
				stmt.setTimestamp(6, null);
			if(ejb.getEnd_date() instanceof java.util.Date)
				stmt.setTimestamp(7, new Timestamp(ejb.getEnd_date().getTime()));
			else
				stmt.setTimestamp(7, null);
			if(ejb.getEnd_value() instanceof Integer)
				stmt.setInt(8, ejb.getEnd_value());
			else
				stmt.setNull(8, Types.INTEGER);
			stmt.setString(9, ejb.getMessage());
			if(ejb.getAgent_check() instanceof Integer)
				stmt.setInt(10, ejb.getAgent_check());
			else
				stmt.setNull(10, Types.INTEGER);
			if(ejb.getCheck_date() instanceof java.util.Date)
				stmt.setTimestamp(11, new Timestamp(ejb.getCheck_date().getTime()));
			else
				stmt.setTimestamp(11, null);
			if(ejb.getRetry_count() instanceof Integer)
				stmt.setInt(12, ejb.getRetry_count());
			else
				stmt.setNull(12, Types.INTEGER);
			stmt.setString(13, ejb.getResult());
			if(ejb.getStatus_buffer() instanceof Integer)
				stmt.setInt(14, ejb.getStatus_buffer());
			else
				stmt.setNull(14, Types.INTEGER);
			
			int row =	stmt.executeUpdate();
			pk = new JobSessionNodePK(
					ejb.getSession_id(),
					ejb.getJob_id(),
					ejb.getFacility_id());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getFacility_id() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getFacility_id() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getFacility_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getFacility_id());
		return pk;
	}
	
	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionNodeDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<JobSessionNodePK> ret = new ArrayList<JobSessionNodePK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_session_node";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobSessionNodePK pk= new JobSessionNodePK(
						res.getString("session_id"),
						res.getString("job_id"),
						res.getString("facility_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}
	
	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodePK)
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionNodeDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodePK)
	 */
	public JobSessionNodePK findByPrimaryKey(JobSessionNodePK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_session_node WHERE session_id = ? AND job_id = ? AND facility_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getFacility_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたセッションIDとジョブIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeBean#ejbFindBySessionIdAndJobId(java.lang.String, java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionNodeDAO#findBySessionIdAndJobId(java.lang.String, java.lang.String)
	 */
	public Collection findBySessionIdAndJobId(String sessionId, String jobId) throws FinderException {
		m_log.debug("findBySessionIdAndJobId() start : " + sessionId + ", " + jobId);
		
		ArrayList<JobSessionNodePK> ret = new ArrayList<JobSessionNodePK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_session_node WHERE session_id = ? AND job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, sessionId);
			stmt.setString(2, jobId);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobSessionNodePK pk= new JobSessionNodePK(
						res.getString("session_id"),
						res.getString("job_id"),
						res.getString("facility_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findBySessionIdAndJobId() error : " + sessionId + ", " + jobId + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findBySessionIdAndJobId() error : " + sessionId + ", " + jobId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findBySessionIdAndJobId() end : " + sessionId + ", " + jobId);
		return ret;
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたファシリティIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeBean#ejbFindByFacilityId(java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionNodeDAO#findByFacilityId(java.lang.String)
	 */
	public Collection findByFacilityId(String facilityId) throws FinderException {
		m_log.debug("findByFacilityId() start : " + facilityId);
		
		ArrayList<JobSessionNodePK> ret = new ArrayList<JobSessionNodePK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_session_node WHERE facility_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, facilityId);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobSessionNodePK pk= new JobSessionNodePK(
						res.getString("session_id"),
						res.getString("job_id"),
						res.getString("facility_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findByFacilityId() error : " + facilityId + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByFacilityId() error : " + facilityId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByFacilityId() end : " + facilityId);
		return ret;
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定された実行状態で、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeBean#ejbFindByStatus(java.lang.Integer)
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionNodeDAO#findByStatus(java.lang.Integer)
	 */
	public Collection findByStatus(Integer status) throws FinderException {
		m_log.debug("findByStatus() start : " + status);
		
		ArrayList<JobSessionNodePK> ret = new ArrayList<JobSessionNodePK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("SELECT * FROM cc_job_session_node ");
			sql.append("WHERE status = ?");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setInt(1, status);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobSessionNodePK pk= new JobSessionNodePK(
						res.getString("session_id"),
						res.getString("job_id"),
						res.getString("facility_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findByStatus() error : " + status + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByStatus() error : " + status + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByStatus() end : " + status);
		return ret;
	}

	/**
	 * ジョブセッションノードを、ジョブマスタとジョブコマンドマスタから作成します。
	 * 
	 * @param sessionId セッションID
	 * @param jobs ジョブIDのコレクション
	 * @throws CreateException
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeBean#ejbHomeMakeSession(String, Collection)
	 */
	@SuppressWarnings("unchecked")
	public void makeSession(String sessionId, Collection jobs) throws CreateException{
		m_log.debug("makeSession() start : " + sessionId );
		
		Connection conn = null;
		Connection connMaster = null;
		PreparedStatement stmt = null;
		PreparedStatement stmtMasster = null;
		String jobId = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			connMaster = ConnectionManager.getConnectionManager().getConnection();
			
			//リポジトリ(RepositoryControllerLocal)を取得
			RepositoryControllerLocal repository = 
				RepositoryControllerUtil.getLocalHome().create();
			
			//SQL文の定義(INSERT)
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_session_node (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("facility_id, ");
			sql.append("node_name, ");
			sql.append("status, ");
			sql.append("start_date, ");
			sql.append("end_date, ");
			sql.append("end_value, ");
			sql.append("message, ");
			sql.append("agent_check, ");
			sql.append("check_date, ");
			sql.append("retry_count, ");
			sql.append("result, ");
			sql.append("status_buffer) ");
			sql.append("VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
			
			//SQL文のセット(INSERT)
			stmt = conn.prepareStatement(sql.toString());
			stmt.setString(1, sessionId);
			stmt.setInt(5, StatusConstant.TYPE_WAIT);
			stmt.setTimestamp(6, null);
			stmt.setTimestamp(7, null);
			stmt.setNull(8, Types.INTEGER);
			stmt.setString(9, null);
			stmt.setNull(10, Types.INTEGER);
			stmt.setTimestamp(11, null);
			stmt.setInt(12, 0);
			stmt.setString(13, null);
			stmt.setNull(14, Types.INTEGER);
			
			//SQL文の定義(マスタ検索)
			StringBuilder sqlMaster = new StringBuilder();
			sqlMaster.append("SELECT ");
			sqlMaster.append("cc_job_mst.job_type, ");
			sqlMaster.append("cc_job_command_mst.facility_id ");
			sqlMaster.append("FROM cc_job_mst left join cc_job_command_mst ");
			sqlMaster.append("on  (cc_job_mst.job_id=cc_job_command_mst.job_id) ");
			sqlMaster.append("WHERE  cc_job_mst.job_id= ? ");
			
			//SQL文のセット(マスタ検索)
			stmtMasster = connMaster.prepareStatement(sqlMaster.toString());

			Iterator it = jobs.iterator();
			while(it.hasNext()){
				//ジョブID取得
				Hashtable ht = (Hashtable)it.next();
				jobId = (String)ht.get("jobId");
				
				//SQL文のセット(マスタ検索)
				stmtMasster.setString(1, jobId);
				//SQL実行(マスタ検索)
				ResultSet resMaster = stmtMasster.executeQuery();
				
				if (resMaster.next()) {
					//ジョブの場合
					if(resMaster.getInt("job_type") == JobConstant.TYPE_JOB){
						//ファシリティID取得
						String facilityId = resMaster.getString("facility_id");
						if(facilityId.equals(SystemParameterConstant.HEADER + 
								SystemParameterConstant.FACILITY_ID + 
								SystemParameterConstant.FOOTER)){
							facilityId = ParameterUtil.getSessionParameterValue(
									sessionId, 
									SystemParameterConstant.FACILITY_ID);
						}
						
						//SQL文の定義(JobSessionJobファシリティパス更新)
						StringBuilder updateSql = new StringBuilder();
						updateSql.append("UPDATE cc_job_session_job ");
						updateSql.append("SET scope_text = ? ");
						updateSql.append("WHERE session_id = ? ");
						updateSql.append("AND job_id = ?");
						
						//SQL文のセット(JobSessionJobファシリティパス更新)
						PreparedStatement updateStmt = conn.prepareStatement(updateSql.toString());
						updateStmt.setString(1, repository.getFacilityPath(facilityId, null));
						updateStmt.setString(2, sessionId);
						updateStmt.setString(3, jobId);
						//SQL実行(JobSessionJobファシリティパス更新)
						updateStmt.executeUpdate();
						updateStmt.close();
						
						//ノードのファシリティIDリスト取得
						ArrayList<String> nodeIdList = null;
						if(repository.isNode(facilityId)){
							//ファシリティIDがノードの場合
							nodeIdList = new ArrayList<String>();
							nodeIdList.add(facilityId);
						}
						else{
							//ファシリティIDがスコープの場合
							nodeIdList = repository.getNodeFacilityIdList(facilityId, RepositoryControllerBean.ALL);
						}
						
						if(nodeIdList instanceof ArrayList){
							for(int i = 0; i < nodeIdList.size(); i++){
								//ノード名を取得
								ArrayList<String> Attribute = new ArrayList<String>();
								Attribute.add(FacilityAttributeConstant.CN);
								HashMap map = repository.getNodeDetail((String)nodeIdList.get(i), Attribute);
								
								//SQL文のセット(INSERT)
								stmt.setString(2, jobId);
								stmt.setString(3, (String)nodeIdList.get(i));
								stmt.setString(4, (String)map.get(FacilityAttributeConstant.CN));
								
								//INSERT実行
								int row =	stmt.executeUpdate();
								if (row != 1) {
									String msg = "result row is not 1";
									m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException");
									throw new EJBException(msg);
								}
							}
						}
					}
				}
				resMaster.close();
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		} catch (CreateException e) {
			e.printStackTrace();
		} catch (NamingException e) {
			e.printStackTrace();
		} catch (FinderException e) {
			e.printStackTrace();
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
				if(stmtMasster != null){
					stmtMasster.close();
				}
				if(connMaster != null){
					connMaster.close();
				}
			} catch (SQLException e1) {
				m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("makeSession() end : " + sessionId );
	}
}
