/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Hashtable;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.bean.DelayNotifyConstant;
import com.clustercontrol.jobmanagement.bean.EndStatusCheckConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;

/**
 * ジョブセッションジョブのDAOインターフェースを実装するクラス<BR>
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobBean
 * @see com.clustercontrol.jobmanagement.dao.JobSessionJobDAO
 */
public class JobSessionJobDAOImpl implements JobSessionJobDAO {
	
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(JobSessionJobDAOImpl.class);
	
	public void init() {
	}
	
	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionJobDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK, com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobBean)
	 */
	public void load(JobSessionJobPK pk, JobSessionJobBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_session_job WHERE session_id = ? AND job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				if(res.getTimestamp("end_date") != null)
					ejb.setEnd_date(new Date(res.getTimestamp("end_date").getTime()));
				else
					ejb.setEnd_date(null);
				
				if(res.getString("end_status") != null){
					ejb.setEnd_status(res.getInt("end_status"));
				}else{
					ejb.setEnd_status(null);
				}
				
				if(res.getString("end_value") != null){
					ejb.setEnd_value(res.getInt("end_value"));
				}else{
					ejb.setEnd_value(null);
				}
				
				ejb.setEnd_staus_check_flg(res.getInt("end_staus_check_flg"));			
				ejb.setJob_id(res.getString("job_id"));
				ejb.setResult(res.getString("result"));
				ejb.setScope_text(res.getString("scope_text"));
				ejb.setSession_id(res.getString("session_id"));
				if(res.getTimestamp("start_date") != null)
					ejb.setStart_date(new Date(res.getTimestamp("start_date").getTime()));
				else
					ejb.setStart_date(null);
				ejb.setStatus(res.getInt("status"));
				ejb.setDelay_notify_flg(res.getInt("delay_notify_flg"));
				
			} else {
				String msg = "JobSessionJob data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}
	
	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionJobDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobBean)
	 */
	public void store(JobSessionJobBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getSession_id() + ", " + ejb.getJob_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_job_session_job SET ");
			sql.append("scope_text = ?, ");
			sql.append("status = ?, ");
			sql.append("start_date = ?, ");
			sql.append("end_date = ?, ");
			sql.append("end_value = ?, ");
			sql.append("end_status = ?, ");
			sql.append("result = ?, ");
			sql.append("end_staus_check_flg = ?, ");
			sql.append("delay_notify_flg = ? ");
			sql.append("WHERE session_id = ? ");
			sql.append("AND job_id = ?");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getScope_text());
			stmt.setInt(2, ejb.getStatus());
			if(ejb.getStart_date() instanceof java.util.Date)
				stmt.setTimestamp(3, new Timestamp(ejb.getStart_date().getTime()));
			else
				stmt.setTimestamp(3, null);
			if(ejb.getEnd_date() instanceof java.util.Date)
				stmt.setTimestamp(4, new Timestamp(ejb.getEnd_date().getTime()));
			else
				stmt.setTimestamp(4, null);
			if(ejb.getEnd_value() instanceof Integer)
				stmt.setInt(5, ejb.getEnd_value());
			else
				stmt.setNull(5, Types.INTEGER);
			if(ejb.getEnd_status() instanceof Integer)
				stmt.setInt(6, ejb.getEnd_status());
			else
				stmt.setNull(6, Types.INTEGER);
			stmt.setString(7, ejb.getResult());
			stmt.setInt(8, ejb.getEnd_staus_check_flg());
			stmt.setInt(9, ejb.getDelay_notify_flg());
			
			stmt.setString(10, ejb.getSession_id());
			stmt.setString(11, ejb.getJob_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : " + ejb.getSession_id() + ", " + ejb.getJob_id());
	}
	
	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionJobDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK)
	 */
	public void remove(JobSessionJobPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_job_session_job WHERE session_id = ? AND job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}
	
	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionJobDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobBean)
	 */
	public JobSessionJobPK create(JobSessionJobBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : " + ejb.getSession_id() + ", " + ejb.getJob_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		JobSessionJobPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_session_job (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("scope_text, ");
			sql.append("status, ");
			sql.append("start_date, ");
			sql.append("end_date, ");
			sql.append("end_value, ");
			sql.append("end_status, ");
			sql.append("result, ");
			sql.append("end_staus_check_flg, ");
			sql.append("delay_notify_flg) ");
			sql.append("VALUES (?,?,?,?,?,?,?,?,?,?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getSession_id());
			stmt.setString(2, ejb.getJob_id());
			stmt.setString(3, ejb.getScope_text());
			stmt.setInt(4, ejb.getStatus());
			if(ejb.getStart_date() instanceof java.util.Date)
				stmt.setTimestamp(5, new Timestamp(ejb.getStart_date().getTime()));
			else
				stmt.setTimestamp(5, null);
			if(ejb.getEnd_date() instanceof java.util.Date)
				stmt.setTimestamp(6, new Timestamp(ejb.getEnd_date().getTime()));
			else
				stmt.setTimestamp(6, null);
			if(ejb.getEnd_value() instanceof Integer)
				stmt.setInt(7, ejb.getEnd_value());
			else
				stmt.setNull(7, Types.INTEGER);
			if(ejb.getEnd_status() instanceof Integer)
				stmt.setInt(8, ejb.getEnd_status());
			else
				stmt.setNull(8, Types.INTEGER);
			stmt.setString(9, ejb.getResult());
			stmt.setInt(10, ejb.getEnd_staus_check_flg());
			stmt.setInt(11, ejb.getDelay_notify_flg());
			
			int row =	stmt.executeUpdate();
			pk = new JobSessionJobPK(
					ejb.getSession_id(),
					ejb.getJob_id());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getSession_id() + ", " + ejb.getJob_id());
		return pk;
	}
	
	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionJobDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<JobSessionJobPK> ret = new ArrayList<JobSessionJobPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_session_job";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobSessionJobPK pk= new JobSessionJobPK(
						res.getString("session_id"),
						res.getString("job_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}
	
	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionJobDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK)
	 */
	public JobSessionJobPK findByPrimaryKey(JobSessionJobPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_session_job WHERE session_id = ? AND job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたセッションIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobBean#ejbFindBySessionId(java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionJobDAO#findBySessionId(java.lang.String)
	 */
	public Collection findBySessionId(String sessionId) throws FinderException {
		m_log.debug("findBySessionId() start : " + sessionId);
		
		ArrayList<JobSessionJobPK> ret = new ArrayList<JobSessionJobPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_session_job WHERE session_id = ?";
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, sessionId);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobSessionJobPK pk= new JobSessionJobPK(
						res.getString("session_id"),
						res.getString("job_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findBySessionId() error : " + sessionId + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findBySessionId() error : " + sessionId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findBySessionId() end : " + sessionId);
		return ret;
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定された実行状態で、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobBean#ejbFindByStatus(java.lang.Integer)
	 * @see com.clustercontrol.jobmanagement.dao.JobSessionJobDAO#findByStatus(java.lang.Integer)
	 */
	public Collection findByStatus(Integer status) throws FinderException {
		m_log.debug("findByStatus() start : " + status);
		
		ArrayList<JobSessionJobPK> ret = new ArrayList<JobSessionJobPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("SELECT * FROM cc_job_session_job as a, cc_job_session as b ");
			sql.append("WHERE a.session_id = b.session_id AND a.job_id = b.job_id AND ");
			sql.append("a.status = ?");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setInt(1, status);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobSessionJobPK pk= new JobSessionJobPK(
						res.getString("session_id"),
						res.getString("job_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findByStatus() error : " + status + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByStatus() error : " + status + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByStatus() end : " + status);
		return ret;
	}
	
	/**
	 * ジョブセッションジョブを、ジョブマスタ、ジョブ待ち条件マスタ、ジョブコマンドマスタから作成します。
	 * 
	 * @param sessionId セッションID
	 * @param jobs ジョブIDのコレクション
	 * @throws CreateException
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobBean#ejbHomeMakeSession(String, Collection)
	 */
	public void makeSession(String sessionId, Collection jobs) throws CreateException{
		m_log.debug("makeSession() start : " + sessionId );
		
		Connection conn = null;
		Connection connMaster = null;
		PreparedStatement stmt = null;
		PreparedStatement stmtMaster = null;
		String jobId = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			connMaster = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義(INSERT)
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_session_job (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("scope_text, ");
			sql.append("status, ");
			sql.append("start_date, ");
			sql.append("end_date, ");
			sql.append("end_value, ");
			sql.append("end_status, ");
			sql.append("result, ");
			sql.append("end_staus_check_flg, ");
			sql.append("delay_notify_flg) ");
			sql.append("VALUES (?,?,?,?,?,?,?,?,?,?,?)");
			
			//SQL文のセット(INSERT)
			stmt = conn.prepareStatement(sql.toString());
			stmt.setString(1, sessionId);
			stmt.setTimestamp(5, null);
			stmt.setTimestamp(6, null);
			stmt.setNull(8, Types.INTEGER);
			stmt.setString(9, null);
			stmt.setInt(10, EndStatusCheckConstant.NO_WAIT_JOB);
			stmt.setInt(11, DelayNotifyConstant.NONE);
			
			//SQL文の定義(マスタ検索)
			StringBuilder sqlMaster = new StringBuilder();
			sqlMaster.append("SELECT ");
			sqlMaster.append("cc_job_mst.job_id, ");
			sqlMaster.append("cc_job_mst.job_type,");
			sqlMaster.append("cc_job_command_mst.facility_id, ");
			sqlMaster.append("cc_job_start_mst.suspend, ");
			sqlMaster.append("cc_job_start_mst.skip, ");
			sqlMaster.append("cc_job_start_mst.skip_end_value ");
			sqlMaster.append("FROM ");
			sqlMaster.append("(cc_job_mst left join cc_job_start_mst on cc_job_mst.job_id=cc_job_start_mst.job_id ) ");
			sqlMaster.append("left join cc_job_command_mst on cc_job_mst.job_id=cc_job_command_mst.job_id ");
			sqlMaster.append("WHERE  cc_job_mst.job_id= ? ");
			
			//SQL文のセット(マスタ検索)
			stmtMaster = connMaster.prepareStatement(sqlMaster.toString());
			
			//リポジトリ(RepositoryControllerLocal)を取得
			RepositoryControllerLocal repository = 
				RepositoryControllerUtil.getLocalHome().create();
			
			Iterator it = jobs.iterator();
			while(it.hasNext()){
				//ジョブID取得
				Hashtable ht = (Hashtable)it.next();
				jobId = (String)ht.get("jobId");
				
				//SQL文のセット(マスタ検索)
				stmtMaster.setString(1, jobId);
				//SQL実行(マスタ検索)
				ResultSet resMaster = stmtMaster.executeQuery();
				
				if (resMaster.next()) {
					//SQL文のセット(INSERT)
					stmt.setString(2, jobId);

					//ファシリティパスを設定
					if(resMaster.getString("facility_id") != null){
						String scopePath =  repository.getFacilityPath(resMaster.getString("facility_id"), null);
						stmt.setString(3, scopePath);
					}else{
						stmt.setString(3, null);
					}
					
					//実行状態・終了値を設定
					if(resMaster.getInt("suspend") == YesNoConstant.TYPE_YES){
						//保留
						stmt.setInt(4, StatusConstant.TYPE_RESERVING);
						stmt.setNull(7, Types.INTEGER);
					}
					else if(resMaster.getInt("skip") == YesNoConstant.TYPE_YES){
						//スキップ
						stmt.setInt(4, StatusConstant.TYPE_SKIP);
						stmt.setInt(7, resMaster.getInt("skip_end_value"));
					}
					else{
						//保留・スキップ以外
						stmt.setInt(4, StatusConstant.TYPE_WAIT);
						stmt.setNull(7, Types.INTEGER);
					}
					
					//INSERTを実行
					int row =	stmt.executeUpdate();
					if (row != 1) {
						String msg = "result row is not 1";
						m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException");
						throw new EJBException(msg);
					}
				}
				resMaster.close();
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		} catch (CreateException e) {
			e.printStackTrace();
		} catch (NamingException e) {
			e.printStackTrace();
		} catch (FinderException e) {
			e.printStackTrace();
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(stmtMaster != null){
					stmtMaster.close();
				}
				if(conn != null){
					conn.close();
				}
				if(connMaster != null){
					connMaster.close();
				}
			} catch (SQLException e1) {
				m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("makeSession() end : " + sessionId );
		
	}
}

