/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.util.apllog;

import java.text.MessageFormat;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ConfirmConstant;
import com.clustercontrol.bean.PriorityConstant;
import com.clustercontrol.monitor.message.LogOutputInfo;




/**
 * 
 * Hinemosの内部ログ（HinemosApp.log）の出力を行うクラス<BR>
 * 
 * Hinemos内部で発生する事象をログやHinemosのイベントとして
 * 処理します。
 * 
 * AplLogger
 * @version $Revision: 1.6 $
 * @since 
 */
public class AplLogger {


	public static final String INTERNAL_SCOPE="INTERNAL";
	public static final String INTERNAL_SCOPE_TEXT="Hinemos_Internal";
	
    private static final String PRIORITY_UNKNOWN = "unknown";
    private static final String PRIORITY_INFO = "info";
    private static final String PRIORITY_WARNING = "warning";
    private static final String PRIORITY_CRITICAL = "critical";
    private String m_pluginID;
    @SuppressWarnings("unused")
	private String m_aplID;
    private String m_aplName;
    static private SendQueue m_sendQueue = null;

    static private Map m_priorityMap = null;

    static private Map m_outPutMap = null;
    
	static Log FILE_LOGGER = LogFactory.getLog("hinemos.apllog");
	static Log log = LogFactory.getLog(AplLogger.class);

	
	
	/**	ログファイル出力用フォーマット「日付  プラグインID,アプリケーション,監視項目ID,メッセージID,ファシリティID,メッセージ,詳細メッセージ」 */
	private MessageFormat m_logfmt = new MessageFormat(
			"{0,date,yyyy/MM/dd HH:mm:ss}  {1},{2},{3},{4},{5},{6}");

	/**
	 * 出力先情報クラス<BR>
	 * ファイル出力／メッセージ出力のフラグを保持します。
	 * 
	 * <p>
	 * ファイル出力
	 * イベント出力
	 * 両方
	 * </p>
	 * 
	 * の３つの状態があります。
	 * 
	 * OutPutInfo
	 * @version $Revision: 1.6 $
	 * @since
	 */
	private class OutPutInfo{
	
	    @SuppressWarnings("unused")
		private static final String OUTPUT_BOTH = "both";
	    private static final String OUTPUT_FILE = "file";
	    private static final String OUTPUT_MSG = "msg";
	    
	    boolean isFileOutPut = true;
	    boolean isMsgOutPut = true;

        /**
         * @param isFileOutPut
         * @param isMsgOutPut
         */
        public OutPutInfo(String output) {

            if(output.equals(OUTPUT_FILE)){
                this.isFileOutPut = true;
                this.isMsgOutPut = false;
            }else if(output.equals(OUTPUT_MSG)){
                this.isFileOutPut = false;
                this.isMsgOutPut = true;
            }
                
        }
        
	    /**
         * @return FileOutPut が真かどうかを取得します。
         */
        public boolean isFileOutPut() {
            return isFileOutPut;
        }
        
        /**
         * @return isMsgOutPut が真かどうかを取得します。
         */
        public boolean isMsgOutPut() {
            return isMsgOutPut;
        }
	}
	
	
    /**
     * コンストラクタ<BR>
     * 
     * @param pluginID	プラグインID
     * @param aplId		アプリケーションID
     */
    public AplLogger(String pluginID, String aplId) {
        super();
        m_pluginID = pluginID;
        m_aplID = aplId;
        
        //メッセージ送信クラス初期化
        if( m_sendQueue == null ){
            m_sendQueue = new SendQueue("queue/clustercontrol/Monitor/EventLog");
        }
        
        //アプリケーション名取得
        m_aplName = PropertyUtil.getString(pluginID + "." + aplId);
        
        //重要度文字列取得
        if( m_priorityMap == null ){
            m_priorityMap = Collections.synchronizedMap(new HashMap());
            
            m_priorityMap.put(PRIORITY_CRITICAL, PropertyUtil.getString("priority.critical"));
            m_priorityMap.put(PRIORITY_WARNING, PropertyUtil.getString("priority.warning"));
            m_priorityMap.put(PRIORITY_INFO, PropertyUtil.getString("priority.info"));
            m_priorityMap.put(PRIORITY_UNKNOWN, PropertyUtil.getString("priority.unknown"));
            
        }
        
        //重要度別出力先取得
        if( m_outPutMap == null ){
            m_outPutMap = Collections.synchronizedMap(new HashMap());
            
            m_outPutMap.put(PRIORITY_CRITICAL, new OutPutInfo( PropertyUtil.getString("priority.critical.output")));
            m_outPutMap.put(PRIORITY_WARNING, 	new OutPutInfo( PropertyUtil.getString("priority.warning.output")));
            m_outPutMap.put(PRIORITY_INFO, 	new OutPutInfo( PropertyUtil.getString("priority.info.output")));
            m_outPutMap.put(PRIORITY_UNKNOWN, 	new OutPutInfo( PropertyUtil.getString("priority.unknown.output")));
            
        }
        
        
    }

    /**
     * メインクラス
     * @param args
     */
    public static void main(String[] args) {
        AplLogger apllog = new AplLogger("REP","rep");
        apllog.put("SYS","001");
        apllog.put("USR","001");
        apllog.put("USR","002");
        apllog.put("USR","003");
        
    }
    
    /**
     * 
     * ログを出力します。<BR>
     * 
     * @param moniterID		監視項目ID
     * @param msgID			メッセージID
     * @param msgArgs		メッセージ置換項目
     * @param detailMsg		詳細メッセージ
     * @since
     */
    public void put(String moniterID, String msgID, Object[] msgArgs, String detailMsg) {

        
        boolean isMsgOutPut = true;
        boolean isFileOutPut = true;

        //現在日時取得
        Date nowDate = new Date();

        //監視項目IDとプラグインIDでイベントメッセージ（リソースファイル）から
        StringBuffer keyBase = new StringBuffer();
        keyBase.append(m_pluginID);
        keyBase.append(".");
        keyBase.append(moniterID);
        keyBase.append(".");
        keyBase.append(msgID);

        //重要度を取得
        StringBuffer keyPriority = new StringBuffer();
        keyPriority.append(keyBase);
        keyPriority.append(".");
        keyPriority.append("priority");
        
        String priority = PropertyUtil.getString(keyPriority.toString());

        //メッセージを取得する。
        StringBuffer keyMsg = new StringBuffer();
        keyMsg.append(keyBase);
        keyMsg.append(".");
        keyMsg.append("message");
        
        //メッセージ項目値が指定されている場合、メッセージの項目を置換
        String msg = null;
        if(msgArgs != null && msgArgs.length != 0){
            msg = PropertyUtil.getString(keyMsg.toString(),msgArgs);
        }else{
            msg = PropertyUtil.getString(keyMsg.toString());
        }
        
        //重要度から出力先を決定して、メッセージ出力。
        OutPutInfo outPutInfo = (OutPutInfo)m_outPutMap.get(priority);
        if(outPutInfo != null){
            isMsgOutPut = outPutInfo.isMsgOutPut();
            isFileOutPut = outPutInfo.isFileOutPut();
        }
        
        
        //メッセージ出力
        if( isMsgOutPut ){
            
            if( !putMsg(nowDate,moniterID,msgID,getPriority(priority),msg,detailMsg) ){
                //メッセージ送信に失敗したら、ファイル出力
                isFileOutPut = true;
            }
        }
        
        //ファイル出力
        if( isFileOutPut ){
            String priorityStr = (String)m_priorityMap.get(priority);
            if( priorityStr == null ){
                priorityStr = (String)m_priorityMap.get(PRIORITY_UNKNOWN);
            }
            putFile(nowDate,moniterID,msgID,priorityStr,msg,detailMsg);
        }
        
        
    }
    /**
     * ログを出力します。<BR>
     * 
     * @param moniterID		監視項目ID
     * @param msgID			メッセージID
     * @since
     */
    public void put(String moniterID, String msgID) {

        put(moniterID, msgID,null,null);
    }
    /**
     * 
     * ログを出力します。<BR>
     * @param moniterID		監視項目ID
     * @param msgID			メッセージID
     * @param msgArgs		メッセージ置換項目
     * @since
     */
    public void put(String moniterID, String msgID, Object[] msgArgs) {
        put(moniterID, msgID,msgArgs,null);

    }
    /**
     * ログを出力します。<BR>
     * 
     * @param moniterID		監視項目ID
     * @param msgID			メッセージID
     * @param detailMsg		詳細メッセージ
     * @since
     */
    public void put(String moniterID, String msgID, String detailMsg) {
        put(moniterID, msgID,null,detailMsg);

    }
    
    /**
     * ログファイルへ出力します。<BR>
     * ファイルへのログを出力します。
     *  
     * @param genDate		日時
     * @param monitorId		監視項目ID
     * @param msgId			メッセージID
     * @param priority		重要度文字列
     * @param msg			メッセージ
     * @param detailMsg		詳細メッセージ
     * @since
     */
    public void putFile(Date genDate,String monitorId, String msgId, String priority, String msg, String detailMsg ) {

        //メッセージを編集
        Object[] args ={genDate,m_pluginID, m_aplName, monitorId, msgId, priority, msg, detailMsg};
        String logmsg = m_logfmt.format(args);
        
        //ファイル出力
        FILE_LOGGER.info(logmsg);
        
    }

    /**
     * イベントを出力します。<BR>
     * 
     * 事象をメッセージ情報クラスにセットして、メッセージ送信します。<BR>
     * 
     * @param genDate		日時
     * @param monitorId		監視項目ID
     * @param msgId			メッセージID
     * @param priority		重要度
     * @param msg			メッセージ
     * @param detailMsg		詳細メッセージ
     * @return				true:成功/false：失敗
     * @since
     */
    private boolean putMsg(Date genDate,String monitorId, String msgId, int priority, String msg, String detailMsg) {

//    	ログ出力情報
//    	・プラグインID			:コンストラクタ
//    	・監視項目ID			:コンストラクタ
//    	・ファシリティID		:SYSTEM 固定
//    	・スコープ				:なし？
//    	・アプリケーション		:コンストラクタ、リソースファイル
//    	・メッセージID			:パラメータ
//    	・メッセージ			:パラメータ、リソースファイル（重要度を除く）
//    	・オリジナルメッセージ	:パラメータ
//    	・重要度				:パラメータ、リソースファイル
//    	・確認済みフラグ		:NO
//    	・発生日時				:現在日時※メソッド呼び出し時に取得したもの
//    	・ログ出力フラグ		:true
//    	・ステータス出力フラグ	:false
//    	・除外期間（分）		:0
//    	・除外回数				:0
//    	・通知情報（送信先）	:なし
//

        
        //メッセージ情報作成
		LogOutputInfo logOutput = new LogOutputInfo();
		logOutput.setPluginId(m_pluginID);
		logOutput.setMonitorId(monitorId);
		logOutput.setFacilityId(INTERNAL_SCOPE);
		logOutput.setScopeText(INTERNAL_SCOPE_TEXT);
		logOutput.setApplication(m_aplName);
		logOutput.setMessageId(msgId);
		logOutput.setMessage(msg);
		logOutput.setMessageOrg(detailMsg == null ? "":detailMsg);
		logOutput.setPriority(priority);
		logOutput.setConfirmFlg( ConfirmConstant.booleanToType(false) );
		logOutput.setGenerationDate(genDate);
		logOutput.setEventLogFlg(true);
		logOutput.setStatusInfoFlg(false);
		logOutput.setExcludePeriod(0);
		logOutput.setExcludeNumber(0);
//		logOutput.setAddress(null);
		
		
        //メッセージ送信
        return m_sendQueue.put(logOutput);
    }

    /**
     * 
     * 文字列から、Priority区分を取得します。<BR>
     * 
     * @param priority
     * @since
     */
    private int getPriority(String priority) {

        int ret = PriorityConstant.TYPE_UNKNOWN;

        if(priority.equals(PRIORITY_CRITICAL)){
            ret = PriorityConstant.TYPE_CRITICAL;
        }else if(priority.equals(PRIORITY_WARNING)){
            ret = PriorityConstant.TYPE_WARNING;
        }else if(priority.equals(PRIORITY_INFO)){
            ret = PriorityConstant.TYPE_INFO;
        }
        
        return ret;
        
    }
}
