/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.agent;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CharsetEncoder;
import java.util.ArrayList;
import java.util.Date;
import java.util.Hashtable;
import java.util.List;
import java.util.Properties;
import java.util.StringTokenizer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.util.RunHistoryUtil;
import com.clustercontrol.jobmanagement.bean.CommandConstant;
import com.clustercontrol.jobmanagement.bean.RunStatusConstant;
import com.clustercontrol.jobmanagement.message.RunInstructionInfo;
import com.clustercontrol.jobmanagement.message.RunResultInfo;

/**
 * 公開鍵操作用のスレッドクラス<BR>
 * 
 * Hinemosのファイル転送では、ファイル転送に使用するsshの
 * 公開鍵をジョブ実行の中でやり取りします。<BR>
 * その一連の操作（鍵の取得、鍵の追加、鍵の削除）を
 * 行います。
 *
 * @version 2.1.0
 * @since 2.0.0
 */
public class PublicKeyThread extends AgentThread {
	private static final String PUBLIC_KEY = ".public.key";
	private static final String AUTHORIZED_KEY_PATH = ".authorized.keys.path";
	
	//ロガー
	private static Log log = LogFactory.getLog(PublicKeyThread.class);
	
	/**
     * コンストラクタ
     * 
	 * @param props
	 */
	public PublicKeyThread(
			RunInstructionInfo info, 
			SendQueue sendQueue, 
			Hashtable<String, Date> runHistory, 
			Properties props) {
		super(info, sendQueue, runHistory, props);
	}
	
	/**
	 * run()から呼び出される公開鍵操作のメソッド<BR>
	 * 
	 */
    /* (non-Javadoc)
     * @see java.lang.Runnable#run()
     */
    public void run() {
        log.debug("run start");
        
        Date startDate = new Date();
        
        //実行履歴に追加
		RunHistoryUtil.addRunHistory(m_info, m_runHistory, startDate);
		
        //---------------------------
        //-- 開始メッセージ送信
        //---------------------------
        
        //メッセージ作成
        RunResultInfo info = new RunResultInfo();
        info.setSessionId(m_info.getSessionId());
        info.setJobId(m_info.getJobId());
        info.setFacilityId(m_info.getFacilityId());
        info.setCommand(m_info.getCommand());
        info.setCommandType(m_info.getCommandType());
        info.setUser(m_info.getUser());
        info.setStatus(RunStatusConstant.START);
        info.setTime(startDate);
        
        log.info("run SessionID=" + m_info.getSessionId() + ", JobID=" + m_info.getJobId());
        
        //送信
        m_sendQueue.put(info);

    	if(m_info.getCommand().equals(CommandConstant.GET_PUBLIC_KEY)){
    		//公開鍵をagent.propertiesから取得します。
    		String key = m_props.getProperty(m_info.getUser().toLowerCase() + PUBLIC_KEY);
    	    log.debug("key:" + key);
    		if(key != null && key.length() > 0){
    			info.setStatus(RunStatusConstant.END);
    			info.setPublicKey(key);
    			info.setTime(new Date());
    			info.setErrorMessage("");
    			info.setMessage("");
    			info.setEndValue(0);
    		}
    		else{
    			//FIXME （エラー？）メッセージに何か入れよう
    			info.setStatus(RunStatusConstant.ERROR);
    			info.setPublicKey("");
    			info.setTime(new Date());
    			info.setErrorMessage("");
    			info.setMessage("");
    			info.setEndValue(-1);
    		}
    	}
    	else if(m_info.getCommand().equals(CommandConstant.ADD_PUBLIC_KEY)){
    		//公開鍵設定
    		if(addKey(m_info.getPublicKey())){
    			info.setStatus(RunStatusConstant.END);
    	        info.setTime(new Date());
    	        info.setErrorMessage("");
    	        info.setMessage("");
    	        info.setEndValue(0);
    		}
    		else{
    			//FIXME （エラー？）メッセージに何か入れよう
    			info.setStatus(RunStatusConstant.ERROR);
    	        info.setTime(new Date());
    	        info.setErrorMessage("");
    	        info.setMessage("");
    	        info.setEndValue(-1);
    		}
    	}
    	else if(m_info.getCommand().equals(CommandConstant.DELETE_PUBLIC_KEY)){
    		//公開鍵削除
    		if(deleteKey(m_info.getPublicKey())){
    			info.setStatus(RunStatusConstant.END);
		        info.setTime(new Date());
		        info.setErrorMessage("");
		        info.setMessage("");
		        info.setEndValue(0);
			}
			else{
				//FIXME （エラー？）メッセージに何か入れよう
				info.setStatus(RunStatusConstant.ERROR);
		        info.setTime(new Date());
		        info.setErrorMessage("");
		        info.setMessage("");
		        info.setEndValue(-1);
			}
    	}
        
    	//送信
		m_sendQueue.put(info);
		
        //実行履歴削除メッセージ送信
        sendDeleteHistory(info);
        
        log.debug("run end");
    }
	
	/**
     * 公開鍵をAuthorized_keyに追加します。<BR>
     * 
	 * @param publicKey
	 * @return
	 */
	public synchronized boolean addKey(String publicKey) {
		log.debug("add key start");
		
		//ファイル名取得
		String fileName = m_props.getProperty(m_info.getUser().toLowerCase() + AUTHORIZED_KEY_PATH);
		
		log.debug("faileName" + fileName);
		if(fileName == null || fileName.length() == 0)
			return false;
		
		//File取得
		File fi = new File(fileName);

		RandomAccessFile randomAccessFile = null;
		FileChannel channel = null;
		FileLock lock = null;
		boolean add = false;
		try {
			//RandomAccessFile作成
	        randomAccessFile = new RandomAccessFile(fi, "rw");
	        //FileChannel作成
	        channel = randomAccessFile.getChannel();

	        // ファイルをロック
	        lock = channel.lock();
	        //ファイルポジションを最後に移動
	        channel.position(channel.size());
	        
	        //追加文字列を取得
	        String writeData = "\n" + publicKey;
	        // ログ出力
	        log.debug("add key : " + writeData);
	        
	        //書き込み用バッファを作成
	        ByteBuffer buffer = ByteBuffer.allocate(512);
	        
	        //書き込み
	        buffer.clear();
	        buffer.put(writeData.getBytes());
	        buffer.flip();
	        channel.write(buffer);
	
	        add = true;
		} catch (Exception e) {
			log.error(e);
		} finally {
			try {
				if (channel != null) {
					channel.close();
				}
				if (randomAccessFile != null) {
					randomAccessFile.close();
				}
				if (lock != null) {
					//ロックをリリース
					lock.release();
				}
			} catch (Exception e) {
			}
		}
		
		return add;
	}
	
	/**
     *  公開鍵をAuthorized_keyから削除します。<BR>
     * 
	 * @param publicKey
	 * @return true : 成功　false:失敗
	 */
	public synchronized boolean deleteKey(String publicKey) {
		log.debug("delete key start");
		
        Charset charset = Charset.forName("UTF-8");
        CharsetEncoder encoder = charset.newEncoder();
        CharsetDecoder decoder = charset.newDecoder();
        
        //ファイル名取得
		String fileName = m_props.getProperty(m_info.getUser().toLowerCase() + AUTHORIZED_KEY_PATH);
		if(fileName == null || fileName.length() == 0)
			return false;
		
		//File取得
		File fi = new File(fileName);

		RandomAccessFile randomAccessFile = null;
		FileChannel channel = null;
		FileLock lock = null;
		boolean delete = false;
		try {
			//RandomAccessFile作成
	        randomAccessFile = new RandomAccessFile(fi, "rw");
	        //FileChannel作成
	        channel = randomAccessFile.getChannel();

	        // ファイルをロック
	        lock = channel.lock();
	        
	        //バッファを作成        
            ByteBuffer buffer = ByteBuffer.allocate((int)channel.size());

			//ファイル読み込み
        	channel.read(buffer);
            
        	// リミットの値を現在位置の値と等しくし、位置を0に設定
        	buffer.flip();
        	
        	//文字列に変換
        	String contents = decoder.decode(buffer).toString();
        	            
        	// デバッグ出力
        	log.debug("contents " + contents.length() + " : " + contents);
            
            //公開鍵を取得
            List<String> keyCheck = new ArrayList<String>();
            StringTokenizer tokenizer = new StringTokenizer(contents, "\n");
            while (tokenizer.hasMoreTokens()) {
                keyCheck.add(tokenizer.nextToken());
            }
            
            //引数の鍵と一致したものを削除
            int s = keyCheck.lastIndexOf(publicKey);
            if(s != -1){
	            // デバッグ出力
            	log.debug("remobe key : " + keyCheck.get(s));
	            keyCheck.remove(s);
			}

            //書き込み文字列の作成
            encoder.reset();
            buffer.clear();
            
            int i;
            if(keyCheck.size() > 0){
	            for (i = 0 ; i < keyCheck.size() - 1 ; i++) {
	                encoder.encode(CharBuffer.wrap((String)keyCheck.get(i) + "\n"), buffer, false);
	            }
	            encoder.encode(CharBuffer.wrap((String)keyCheck.get(i)), buffer, true);
            }
            
            //ファイル書き込み
            buffer.flip();
            channel.truncate(0);
            channel.position(0);
            channel.write(buffer);
	        
	        delete = true;
		} catch (IOException e) {
			log.error(e.getMessage(), e);
		} finally {
			try {
				if (channel != null) {
					channel.close();
				}
				if (randomAccessFile != null) {
					randomAccessFile.close();
				}
				//ロックをリリースする
				if (lock != null) {
					lock.release();
				}
			} catch (Exception e) {
			}
		}
		
		return delete;
	}
}
