/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.snmppoller;

import java.net.InetAddress;
import java.net.SocketException;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.opennms.protocols.snmp.SnmpEndOfMibView;
import org.opennms.protocols.snmp.SnmpHandler;
import org.opennms.protocols.snmp.SnmpNoSuchInstance;
import org.opennms.protocols.snmp.SnmpNoSuchObject;
import org.opennms.protocols.snmp.SnmpNull;
import org.opennms.protocols.snmp.SnmpObjectId;
import org.opennms.protocols.snmp.SnmpParameters;
import org.opennms.protocols.snmp.SnmpPduPacket;
import org.opennms.protocols.snmp.SnmpPduRequest;
import org.opennms.protocols.snmp.SnmpPeer;
import org.opennms.protocols.snmp.SnmpSession;
import org.opennms.protocols.snmp.SnmpSyntax;
import org.opennms.protocols.snmp.SnmpTimeTicks;
import org.opennms.protocols.snmp.SnmpUInt32;
import org.opennms.protocols.snmp.SnmpV2Error;
import org.opennms.protocols.snmp.SnmpVarBind;

import com.clustercontrol.sharedtable.bean.DataTable;

/**
 * SNMPのポーリングを実行するクラス
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class SnmpPoller implements SnmpHandler {
	protected static Log m_log = LogFactory.getLog(SnmpPoller.class);

	/** IPアドレス */
	private String m_ipAddress;

	// 収集を開始したOID
	private String m_oidText;

	private SnmpObjectId m_startOid;

	// 収集を終了するOID
	private SnmpObjectId m_stopOid;

	// 取得したデータを格納するテーブル
	DataTable m_dataTable;

	/** メッセージ */
	private String m_message = null;

	/** 待機フラグ */
	boolean m_waitFlg;

	/** 最期にポーリングをおこなった際のOID * */
	String lastOid;

	// エラーコード
	private int m_errorCode = 0;

	// エラーコード
	private final int Normal = 0; // 正常に動作いている場合

	private final int TimeOutError = 1; // 問い合わせがタイムアウトした場合

	private final int InternalError = 2; // タイムアウト以外のエラー

	/**
	 * メインルーチン　
	 * IPアドレスと　Hashtableを受け取り、
	 * ポーリングした結果をHashtableに代入する
	 */
	public void polling(InetAddress ipAddress, int port, int version,
			String community, int retries, int timeout, List oidList,
			DataTable table) {
		// デバッグ出力
		if (m_log.isDebugEnabled()) {
			m_log.debug("polling() start :" + ipAddress.toString());
			m_log.debug("Port      : " + port);
			m_log.debug("Version   : " + version);
			m_log.debug("Community : " + community);
			m_log.debug("Retries   : " + retries);
			m_log.debug("Timeout   : " + timeout);
		}

		m_errorCode = Normal;
		m_dataTable = table;

		m_ipAddress = ipAddress.toString();

		SnmpPeer peer = new SnmpPeer(ipAddress);
		if (port != -1)
			peer.setPort(port);

		if (timeout != -1)
			peer.setTimeout(timeout);

		if (retries != -1)
			peer.setRetries(retries);

		//
		// Initialize the peer
		//
		SnmpParameters parms = peer.getParameters();
		parms.setVersion(version);
		if (community != null)
			parms.setReadCommunity(community);

		//
		// Now create the session, set the initial request
		// and walk the tree!
		//
		SnmpSession session = null;
		try {
			session = new SnmpSession(peer);
		} catch (SocketException e) {
			m_log.error("polling() warning  :" + ipAddress.toString()
					+ " SocketException creating the SNMP session");
			m_errorCode = InternalError;
			return;
		}

		session.setDefaultHandler(this);

		try {
			Iterator itr = oidList.iterator();

			while (itr.hasNext()) {
				m_oidText = (String) itr.next();
				m_startOid = new SnmpObjectId(m_oidText);

				//
				// set the stop point
				//
				SnmpObjectId id = new SnmpObjectId(m_oidText);

				int[] ids = id.getIdentifiers();
				++ids[ids.length - 1];
				id.setIdentifiers(ids);
				m_stopOid = id;

				// build the first request
				// SnmpPduRequest pdu = new SnmpPduRequest(SnmpPduRequest.GET);
				SnmpPduRequest pdu = new SnmpPduRequest(m_oidText
						.endsWith(".0") ? SnmpPduPacket.GET
						: SnmpPduPacket.GETNEXT);
				pdu.setRequestId(SnmpPduPacket.nextSequence());

				SnmpObjectId oId = new SnmpObjectId(m_oidText);
				pdu.addVarBind(new SnmpVarBind(oId));

				synchronized (session) {
					m_waitFlg = true;
					session.send(pdu);

					if (m_waitFlg) {
						session.wait();
					}

					// timeout error が発生した場合は、次のOIDへのポーリングを実施せずに処理を中止する
					if (this.m_errorCode == TimeOutError) {
						break;
					}
				}
			}
		} catch (InterruptedException e) {
			m_log.error("polling() warning :" + ipAddress.toString()
					+ " polling failed at InterruptedException");
			m_errorCode = InternalError;
			return;
		} finally {
			try {

				session.close();

			} catch (Exception e) {
				/**
				 * FIXME
				 * Joe-SNMPの不具合の回避コード
				 * Joe-SNMPで不具合が修正されたら削除すること
				 */
				m_log.warn("polling():" + m_ipAddress + " Session close failed");

			}
		}

		// 例外処理(未実装)
		if (this.m_errorCode == TimeOutError) {

		} else if (this.m_errorCode == InternalError) {

		}

		// デバッグ出力
		if (m_log.isDebugEnabled()) {
			m_log.debug("polling() end :" + ipAddress.toString());
		}
	}

	public void snmpReceivedPdu(SnmpSession session, int cmd, SnmpPduPacket pdu) {
		// m_log.debug("snmpReceivedPdu() start");

		try {
			long time = System.currentTimeMillis(); // 取得時刻

			SnmpPduRequest req = null;
			if (pdu instanceof SnmpPduRequest) {
				req = (SnmpPduRequest) pdu;
			} else {
				m_log.error("polling() error :" + session.toString()
						+ " Received non-request pdu");
				synchronized (session) {
					m_errorCode = InternalError;
					session.notify();
				}
				return;
			}

			if (pdu.getCommand() != SnmpPduPacket.RESPONSE) {
				m_log.error("polling() error :" + session.toString()
								+ "  Received non-response command "
								+ pdu.getCommand());
				synchronized (session) {
					m_errorCode = InternalError;
					session.notify();
				}
				return;
			}

			// SnmpVarBind[] vars = pdu.toVarBindArray();

			if (req.getErrorStatus() != 0) {
				m_log.error("polling() error :" + session.toString()
						+ "  Error Status " + req.getErrorStatus());
				synchronized (session) {
					m_errorCode = InternalError;
					session.notify();
				}
				return;
			}

			// ポーリングはOID1つづつを繰り返し実行するため、SnmpVarBindは常にひとつ
			SnmpVarBind var = pdu.getVarBindAt(0);

			// ポーリングの終了判定
			// 条件：
			// 戻りの型が EndOfMibView の場合
			// 停止条件のOIDにマッチした場合は終了
			if (var.getValue().typeId() == SnmpEndOfMibView.ASNTYPE
					|| (m_stopOid != null && m_stopOid.compare(var.getName()) < 0)) {
				// デバッグ出力
				if (m_log.isDebugEnabled()) {
					if(var.getValue().typeId() == SnmpEndOfMibView.ASNTYPE){
						m_log.debug("get EndOfMibView " + m_ipAddress + " "
							+ var.getName().toString());
					} else {
						m_log.debug("stop " + m_ipAddress + " "
								+ var.getName().toString());
					}
				}

				// 収集を終了します
				synchronized (session) {
					session.notify();
				}
				return;
			}

			//
			// next pdu
			//
			SnmpPduRequest nxt = new SnmpPduRequest(SnmpPduPacket.GETNEXT);
			nxt.setRequestId(SnmpPduPacket.nextSequence());

			// デバッグ出力
			if (m_log.isDebugEnabled()) {
				m_log.debug("get " + m_ipAddress + " "
						+ var.getName().toString() + " " + time + " "
						+ var.getValue().getClass().getName() + " "
						+ var.getValue().toString());
			}

			if (m_startOid.isRootOf(var.getName())) {
				if (var.getValue() instanceof SnmpNoSuchInstance) {
					// 何もしない
				} else if (var.getValue() instanceof SnmpNoSuchObject) {
					// 何もしない
				} else if (var.getValue() instanceof SnmpV2Error) {
					// 何もしない
				} else if (var.getValue() instanceof SnmpNull) {
					// 何もしない
				} else if (var.getValue() instanceof SnmpTimeTicks) {
					// デバッグ
					if (m_log.isDebugEnabled()) {
						m_log.debug("set " + m_ipAddress + ","
								+ var.getName().toString() + "," + time + ","
								+ var.getValue().toString());
					}

					long i = ((SnmpUInt32) var.getValue()).getValue();
					m_dataTable.putValue(var.getName().toString(), time, String
							.valueOf(i));
				} else {
					// デバッグ
					if (m_log.isDebugEnabled()) {
						m_log.debug("set " + m_ipAddress + ","
								+ var.getName().toString() + "," + time + ","
								+ var.getValue().toString());
					}

					m_dataTable.putValue(var.getName().toString(), time, var
							.getValue().toString());
				}

				nxt.addVarBind(new SnmpVarBind(var.getName()));
			}

			if (nxt.getLength() != 0) {
				// m_log.debug("session.send() start");
				session.send(nxt, this);
			} else {
				// デバッグ
				if (m_log.isDebugEnabled()) {
					m_log.debug("outside of Start OID(" + m_startOid + ") "
							+ m_ipAddress + " " + var.getName());
				}
				
				// 収集を終了します
				synchronized (session) {
					session.notify();
				}
				return;
			}
		} catch (Exception e) { // 何か例外が生じた場合は収集を停止する
			m_log.error("InternalError ", e);
			// e.printStackTrace();

			// 収集を終了します
			synchronized (session) {
				m_errorCode = InternalError;
				session.notify();
			}
		}
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see org.opennms.protocols.snmp.SnmpHandler#snmpInternalError(org.opennms.protocols.snmp.SnmpSession,
	 *      int, org.opennms.protocols.snmp.SnmpSyntax)
	 */
	public void snmpInternalError(SnmpSession session, int err, SnmpSyntax pdu) {
		m_log.warn("snmpInternalError():" + session.toString()
				+ " snmpInternalError. The error code is " + err);

		synchronized (session) {
			m_errorCode = InternalError;
			m_message = "InternalError. The error code is " + err + ". IP:"
					+ m_ipAddress + " OID:" + m_oidText;

			// 収集中のOID配下のOIDの値をテーブルから削除
			removeTableValue();

			m_waitFlg = false;
			session.notify();
		}
	}

	/*
	 * (非 Javadoc)
	 * 
	 * @see org.opennms.protocols.snmp.SnmpHandler#snmpTimeoutError(org.opennms.protocols.snmp.SnmpSession,
	 *      org.opennms.protocols.snmp.SnmpSyntax)
	 */
	public void snmpTimeoutError(SnmpSession session, SnmpSyntax pdu) {
		m_log.warn("snmpTimeoutError():"
				+ session.getPeer().getPeer().toString() + " "
				+ ((SnmpPduRequest) pdu).toVarBindArray()[0].getName()
				+ " polling failed at TimeoutError");

		synchronized (session) {
			m_errorCode = TimeOutError;
			m_message = "TimeoutError." + " IP:"
					+ session.getPeer().getPeer().toString() + " OID:"
					+ ((SnmpPduRequest) pdu).toVarBindArray()[0].getName();

			// 収集中のOID配下のOIDの値をテーブルから削除
			removeTableValue();

			session.notify();
		}
	}

	/**
	 * 収集指定のOID配下のMIB値を全て削除する
	 * 
	 * 例)
	 * .1.3.6.1.2.1.2.2.1   を対象に収集している場合で
	 * .1.3.6.1.2.1.2.2.1.3 を収集の際にこのメソッドを呼び出した場合
	 * .1.3.6.1.2.1.2.2.1   配下のMIB値は全て削除されるため
	 * 既にテーブルに格納済みの
	 * .1.3.6.1.2.1.2.2.1.1 のMIB値および .1.3.6.1.2.1.2.2.1.2
	 * のMIB値を削除する
	 */
	private void removeTableValue() {
		// テーブルに格納されているOIDのセットを取得
		Set oidSet = m_dataTable.keySet();

		String[] fullOids = (String[]) oidSet
				.toArray(new String[oidSet.size()]);

		for (int i = 0; i < fullOids.length; i++) {

			String fullOid = fullOids[i];

			// もし先頭の部分が一致するならそのツリーの子孫と判定
			if (fullOid.startsWith(m_oidText)) {
				// テーブルに格納されている値を削除
				m_dataTable.removeValue(fullOid);
			}
		}
	}
}
