/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.factory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityInfo;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.bean.Property;
import com.clustercontrol.performanceMGR.bean.DeviceData;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.ejb.entity.DeviceInfoLocal;
import com.clustercontrol.repository.ejb.entity.DeviceInfoUtil;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityTreeLocal;
import com.clustercontrol.repository.ejb.entity.FacilityTreeUtil;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.util.ListSorter;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;

/**
 * ファシリティ検索クラス
 *
 * @version 1.0.0
 * @since 1.0.0
 */
public class SelectFacility {
	public static final String SEPARATOR = ">";
	
	/**
	 * ノード詳細を取得
	 * 
	 * 属性名と値のHashMapとして返す
	 * 
	 * @param facilityId
	 * @param attributes
	 * @return 属性名と値のHashMap
	 * @throws FinderException
	 * @throws NamingException
	 */
	public HashMap getNodeDetail(String facilityId, ArrayList attributes) throws FinderException, NamingException {
		HashMap map = new HashMap();
		try {
			FacilityLocal facility = FacilityUtil.getLocalHome().findByFacilityId(facilityId);
			
			for(int i = 0; i < attributes.size(); i++){
				//属性取得
				String attribute = (String)attributes.get(i);
				
				if(attribute.compareTo(FacilityAttributeConstant.FACILITYID) == 0){
					//ccFacilityId取得
					map.put(attribute, facility.getFacilityId());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.CN) == 0){
					//cn取得
					map.put(attribute, facility.getCn());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.DESCRIPTION) == 0){
					//description取得
					map.put(attribute, facility.getDescription());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.AGENTCONTROL) == 0){
					//ccAgentControl取得
					map.put(attribute, facility.getAgentControl());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.AGENTWATCH) == 0){
					//ccAgentWatch取得
					map.put(attribute, facility.getAgentWatch());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.AGENTWATCHSTATUS) == 0){
					//ccAgentWatchStatus取得
					map.put(attribute, facility.getAgentWatchStatus());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.CHARSET) == 0){
					//ccCharSet取得
					map.put(attribute, facility.getCharSet());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.CREATETIMESTAMP) == 0){
					//ccCreateTimestamp取得
					map.put(attribute, facility.getCreateTimestamp());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.CREATORSNAME) == 0){
					//ccCreatorsName取得
					map.put(attribute, facility.getCreatorsName());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.DHCPCLIENT) == 0){
					//ccDhcpClient取得
					map.put(attribute, facility.getDhcpClient());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.HOST) == 0){
					//host取得
					map.put(attribute, facility.getHost());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.IPNETWORKNUMBER) == 0){
					//ipNetworkNumber取得
					map.put(attribute, facility.getIpNetworkNumber());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.IPNETWORKNUMBERV6) == 0){
					//ccIpNetworkNumberV6取得
					map.put(attribute, facility.getIpNetworkNumberV6());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.IPPROTOCOLNUMBER) == 0){
					//ipProtocolNumber取得
					map.put(attribute, facility.getIpProtocolNumber());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.IPTYPE) == 0){
					//ccIpType取得
					map.put(attribute, facility.getIpType());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.LOGWATCH) == 0){
					//ccLogWatch取得
					map.put(attribute, facility.getLogWatch());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.MACHINE) == 0){
					//ccMachine取得
					map.put(attribute, facility.getMachine());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.MANAGERCONTACT) == 0){
					//ccManagerContact取得
					map.put(attribute, facility.getManagerContact());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.MANAGERNAME) == 0){
					//ccManagerName取得
					map.put(attribute, facility.getManagerName());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.MODIFIERSNAME) == 0){
					//ccModifiersName取得
					map.put(attribute, facility.getModifiersName());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.MODIFYTIMESTAMP) == 0){
					//ccModifyTimestamp取得
					map.put(attribute, facility.getModifyTimestamp());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.NODENAME) == 0){
					//ccNodeName取得
					map.put(attribute, facility.getNodeName());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.OSNAME) == 0){
					//ccOsName取得
					map.put(attribute, facility.getOsName());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.OSRELEASE) == 0){
					//ccOsRelease取得
					map.put(attribute, facility.getOsRelease());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.OSVERSION) == 0){
					//ccOsVersion取得
					map.put(attribute, facility.getOsVersion());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.PINGWATCH) == 0){
					//ccPingWatch取得
					map.put(attribute, facility.getPingWatch());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.PINGWATCHSTATUS) == 0){
					//ccPingWatchStatus取得
					map.put(attribute, facility.getPingWatchStatus());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.PLATFORM) == 0){
					//ccPlatform取得
					map.put(attribute, facility.getPlatform());
				}
				else if(attribute.compareTo(FacilityAttributeConstant.STATUSWATCH) == 0){
					//ccPlatform取得
					map.put(attribute, facility.getStatusWatch());
				}
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return map;
	}
	
	/**
	 * ノード詳細を取得
	 * 
	 * ファシリティIDと属性値(属性名と値のHashMap)のHashMapとして返す
	 * 
	 * @param facilityIdList
	 * @param attributes
	 * @return ファシリティIDと属性値(属性名と値のHashMap)のHashMap
	 * @throws NamingException
	 * @throws FinderException
	 */
	public HashMap getNodeDetail(ArrayList facilityIdList, ArrayList attributes) throws FinderException, NamingException {
		HashMap map = new HashMap();
		
		for(int i = 0; i < facilityIdList.size(); i++){
			String facilityId = (String)facilityIdList.get(i);
			HashMap detailMap = getNodeDetail(facilityId, attributes);
			map.put(facilityId, detailMap);
		}
		
		return map;
	}
	
	/**
	 * デバイス一覧を取得
	 * 
	 * DeviceDataの配列を返す
	 * 
	 * @param facilityId
	 * @return DeviceDataの配列
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getDeviceList(String facilityId) throws FinderException, NamingException {
		ArrayList list = null;
		try {
			//FacilityTreeの最上位エントリを取得
			Collection ct = DeviceInfoUtil.getLocalHome().findByFacilityId(facilityId);
			
			list = new ArrayList();
			Iterator itr = ct.iterator();
			while(itr.hasNext()){
				DeviceInfoLocal d = (DeviceInfoLocal)itr.next();
				DeviceData data = new DeviceData(
						d.getFacilityId(), 
						d.getSnmpOID(), 
						d.getDeviceIndex().intValue(), 
						d.getCn(), 
						d.getDeviceType());
				
				list.add(data);
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return list;
	}
	
	/**
	 * ノード一覧を取得
	 * 
	 * オブジェクトの2次元配列として返す
	 * 
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getNodeList() throws FinderException, NamingException {
		ArrayList list = null;
		try {
			//Facilityの最上位エントリを取得
			Collection ct = FacilityUtil.getLocalHome().findAllNode();
			
			list = new ArrayList();
			Iterator itr = ct.iterator();
			while(itr.hasNext()){
				FacilityLocal ft = (FacilityLocal)itr.next();
				
				ArrayList info = new ArrayList();
				info.add(ft.getFacilityId());
				info.add(ft.getCn());
				info.add(ft.getDescription());
				list.add(info);
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return list;
	}
	
	/**
	 * ノード一覧を取得
	 * 
	 * オブジェクトの2次元配列として返す
	 * 
	 * @param parentFacilityId
	 * @param level
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getNodeList(String parentFacilityId, int level) throws FinderException, NamingException {
		ArrayList list = null;
		try {
			Collection ct = null;
			if(level == RepositoryControllerBean.ALL){
				//FacilityTreeの最上位エントリを取得
				ct = FacilityTreeUtil.getLocalHome().findAllNode(parentFacilityId);
			}
			else{
				//FacilityTreeの最上位エントリを取得
				ct = FacilityTreeUtil.getLocalHome().findOneLevelNode(parentFacilityId);
			}
			
			HashMap map = new HashMap();
			list = new ArrayList();
			Iterator itr = ct.iterator();
			while(itr.hasNext()){
				FacilityTreeLocal ft = (FacilityTreeLocal)itr.next();
				
				if(map.get(ft.getFacilityId()) == null){
					map.put(ft.getFacilityId(), "1");
					
					FacilityLocal facility = ft.getFacility();
					ArrayList info = new ArrayList();
					info.add(facility.getFacilityId());
					info.add(facility.getCn());
					info.add(facility.getDescription());
					list.add(info);
				}
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return list;
	}
	
	/**
	 * ノード一覧を取得
	 * 
	 * オブジェクトの2次元配列として返す
	 * 
	 * @param property
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getNodeList(Property property) throws FinderException, NamingException {
		ArrayList list = null;
		try {
			ArrayList values = null;
			//ファシリティID取得
			values = PropertyUtil.getPropertyValue(property, NodeFilterProperty.FACILITY_ID);
			String facilityId = (String)values.get(0);
			//ファシリティ名取得
			values = PropertyUtil.getPropertyValue(property, NodeFilterProperty.FACILITY_NAME);
			String facilityName = (String)values.get(0);
			//説明取得
			values = PropertyUtil.getPropertyValue(property, NodeFilterProperty.DESCRIPTION);
			String description = (String)values.get(0);
			//IPアドレス取得
			values = PropertyUtil.getPropertyValue(property, NodeFilterProperty.IP_ADDRESS);
			String ipAddress = (String)values.get(0);
			//OS名取得
			values = PropertyUtil.getPropertyValue(property, NodeFilterProperty.OS_NAME);
			String osName = (String)values.get(0);
			//OSリリース取得
			values = PropertyUtil.getPropertyValue(property, NodeFilterProperty.OS_RELEASE);
			String osRelease = (String)values.get(0);
			//管理者取得
			values = PropertyUtil.getPropertyValue(property, NodeFilterProperty.ADMINISTRATOR);
			String managerName = (String)values.get(0);
			//連絡先取得
			values = PropertyUtil.getPropertyValue(property, NodeFilterProperty.CONTACT);
			String managerContact = (String)values.get(0);
			
			//Facilityの最上位から検索する
			Collection ct = 
				FacilityUtil.getLocalHome().findNode(
						facilityId, 
						facilityName, 
						description, 
						ipAddress, 
						osName, 
						osRelease, 
						managerName, 
						managerContact);
			
			list = new ArrayList();
			Iterator itr = ct.iterator();
			while(itr.hasNext()){
				FacilityLocal ft = (FacilityLocal)itr.next();
				
				ArrayList info = new ArrayList();
				info.add(ft.getFacilityId());
				info.add(ft.getCn());
				info.add(ft.getDescription());
				list.add(info);
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return list;
	}
	
	/**
	 * ファシリティIDリスト取得
	 * 
	 * ファシリティIDの配列として返す
	 * 
	 * @param parentFacilityId
	 * @param level
	 * @param sort
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getFacilityIdList(String parentFacilityId, int level, boolean sort) throws FinderException, NamingException {
		ArrayList list = null;
		try {
			Collection ct = null;
			if(level == RepositoryControllerBean.ALL){
				//FacilityTreeの最上位エントリを取得
				ct = FacilityTreeUtil.getLocalHome().findAll(parentFacilityId);
			}
			else{
				//FacilityTreeの最上位エントリを取得
				ct = FacilityTreeUtil.getLocalHome().findOneLevel(parentFacilityId);
			}
			
			HashMap map = new HashMap();
			list = new ArrayList();
			Iterator itr = ct.iterator();
			int i = 0;
			while(itr.hasNext()){
				if(i == 0){
					//findOneLevel()の結果には親ファシリティは含まれないので追加する
					if(i == 0 && parentFacilityId != null && parentFacilityId.compareTo("") != 0){
						if(map.get(parentFacilityId) == null){
							map.put(parentFacilityId, "1");
							
							list.add(parentFacilityId);
						}
					}
					i++;
				}
				
				FacilityTreeLocal ft = (FacilityTreeLocal)itr.next();
				
				if(map.get(ft.getFacilityId()) == null){
					map.put(ft.getFacilityId(), "1");
					
					list.add(ft.getFacilityId());
				}
			}
			
			//ソートする
			if(sort){
				ListSorter sorter = new ListSorter();
				sorter.sort(list);
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return list;
	}
	
	/**
	 * ファシリティIDリスト取得
	 * 
	 * ファシリティIDの配列として返す
	 * 
	 * @param parentFacilityId
	 * @param level
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getNodeFacilityIdList(String parentFacilityId, int level, boolean sort) throws FinderException, NamingException {
		ArrayList list = null;
		try {
			Collection ct = null;
			if(level == RepositoryControllerBean.ALL){
				//FacilityTreeの最上位エントリを取得
				ct = FacilityTreeUtil.getLocalHome().findAllNode(parentFacilityId);
			}
			else{
				//FacilityTreeの最上位エントリを取得
				ct = FacilityTreeUtil.getLocalHome().findOneLevelNode(parentFacilityId);
			}
			
			HashMap map = new HashMap();
			list = new ArrayList();
			Iterator itr = ct.iterator();
			while(itr.hasNext()){
				FacilityTreeLocal ft = (FacilityTreeLocal)itr.next();
				
				if(map.get(ft.getFacilityId()) == null){
					map.put(ft.getFacilityId(), "1");
					
					list.add(ft.getFacilityId());
				}
			}
			
			//ソートする
			if(sort){
				ListSorter sorter = new ListSorter();
				sorter.sort(list);
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return list;
	}
	
	/**
	 * ファシリティIDリスト取得
	 * 
	 * ファシリティIDの配列として返す
	 * 
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getNodeFacilityIdList(boolean sort) throws FinderException, NamingException {
		ArrayList list = null;
		try {
			//Facilityの最上位エントリを取得
			Collection ct = FacilityUtil.getLocalHome().findAllNode();
			
			list = new ArrayList();
			Iterator itr = ct.iterator();
			while(itr.hasNext()){
				FacilityLocal ft = (FacilityLocal)itr.next();
				list.add(ft.getFacilityId());
			}
			
			//ソートする
			if(sort){
				ListSorter sorter = new ListSorter();
				sorter.sort(list);
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return list;
	}
	
	/**
	 * スコープ一覧を取得
	 * 
	 * オブジェクトの2次元配列として返す
	 * 
	 * @param parentFacilityId
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getScopeList(String parentFacilityId) throws FinderException, NamingException {
		ArrayList list = null;
		try {
			//FacilityTreeの最上位エントリを取得
			Collection ct = FacilityTreeUtil.getLocalHome().findOneLevel(parentFacilityId);
			
			list = new ArrayList();
			Iterator itr = ct.iterator();
			while(itr.hasNext()){
				FacilityTreeLocal ft = (FacilityTreeLocal)itr.next();
				FacilityLocal facility = ft.getFacility();
				ArrayList info = new ArrayList();
				info.add(facility.getFacilityId());
				info.add(facility.getCn());
				info.add(facility.getDescription());
				list.add(info);
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return list;
	}
	
	/**
	 * Facilityを取得
	 * 
	 * @param facilityId
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public FacilityLocal getFacility(String facilityId) throws FinderException, NamingException {
		FacilityLocal facility = null;
		try {
			//Facilityを取得
			facility = FacilityUtil.getLocalHome().findByFacilityId(facilityId);
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return facility;
	}
	
	/**
	 * ノードを割当てたスコープ一覧を取得
	 * 
	 * @param facilityId
	 * @param level
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getNodeScopeList(String facilityId) throws FinderException, NamingException {
		ArrayList list = null;
		try {
			//FacilityTreeの最上位エントリを取得
			Collection ct = FacilityTreeUtil.getLocalHome().findByFacilityId(facilityId);
			FacilityTreeItem item = this.getFacilityTree(Locale.getDefault());
			
			list = new ArrayList();
			Iterator itr = ct.iterator();
			while(itr.hasNext()){
				FacilityTreeLocal ft = (FacilityTreeLocal)itr.next();
				String dn[] = ft.getDn().split(",");
				StringBuffer path = new StringBuffer();
				for(int i = 1; i < dn.length - 1; i++){
					String id = dn[i].replaceFirst("[\\s\\w]*=", "");
					FacilityLocal facility = FacilityUtil.getLocalHome().findByFacilityId(id);
					if(facility.getType() == FacilityConstant.TYPE_SCOPE){
						path.insert(0, SEPARATOR);
					}
					path.insert(0, facility.getCn());
				}
				ArrayList info = new ArrayList();
				info.add(path.toString());
				list.add(info);
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return list;
	}
	
	/**
	 * ファシリティツリー情報取得（ノード含む）
	 * 
	 * @param locale
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public FacilityTreeItem getFacilityTree(Locale locale) throws FinderException, NamingException {
		
		//FacilityTreeの最上位インスタンスを作成
		FacilityInfo info = new FacilityInfo();
		info.setFacilityName(FacilityConstant.STRING_COMPOSITE);
		info.setType(FacilityConstant.TYPE_COMPOSITE);
		FacilityTreeItem tree = new FacilityTreeItem(null, info);
		
		// コンポジットアイテムを生成して返します。
		info = new FacilityInfo();
		info.setFacilityName(Messages.getString("scope", locale));
		info.setType(FacilityConstant.TYPE_COMPOSITE);
		FacilityTreeItem item = new FacilityTreeItem(tree, info);
		
		try {
			//FacilityTreeの最上位エントリを取得
			Collection ct = FacilityTreeUtil.getLocalHome().findOneLevel();
			
			Iterator itr = ct.iterator();
			while(itr.hasNext()){
				FacilityTreeLocal ft = (FacilityTreeLocal)itr.next();
				createFacilityTree(ft, item);
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return tree;
	}
	
	/**
	 * ファシリティツリー情報取得（スコープのみ）
	 * 
	 * @param locale
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public FacilityTreeItem getScopeFacilityTree(Locale locale) throws FinderException, NamingException {
		
		//FacilityTreeの最上位インスタンスを作成
		FacilityInfo info = new FacilityInfo();
		info.setFacilityName(FacilityConstant.STRING_COMPOSITE);
		info.setType(FacilityConstant.TYPE_COMPOSITE);
		FacilityTreeItem tree = new FacilityTreeItem(null, info);
		
		// コンポジットアイテムを生成して返します。
		info = new FacilityInfo();
		info.setFacilityName(Messages.getString("scope", locale));
		info.setType(FacilityConstant.TYPE_COMPOSITE);
		FacilityTreeItem item = new FacilityTreeItem(tree, info);
		
		try {
			//FacilityTreeの最上位エントリを取得
			Collection ct = FacilityTreeUtil.getLocalHome().findOneLevelScope(null);
			
			Iterator itr = ct.iterator();
			while(itr.hasNext()){
				FacilityTreeLocal ft = (FacilityTreeLocal)itr.next();
				createScopeFacilityTree(ft, item);
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return tree;
	}
	
	/**
	 * FacilityTreeItem作成（ノード含む）
	 * 
	 * @param facilityTree
	 * @param parent
	 */
	protected void createFacilityTree(FacilityTreeLocal facilityTree, FacilityTreeItem parent) {
		FacilityLocal facility = null;
		try {
			facility = facilityTree.getFacility();
		} catch (FinderException e) {
			return;
		}
		
		FacilityInfo info = null;
		FacilityTreeItem item = null;
		if(facility != null){
			info = new FacilityInfo();
			info.setFacilityId(facility.getFacilityId());
			info.setFacilityName(facility.getCn());
			if(facility.getType() == FacilityConstant.TYPE_NODE){
				info.setType(FacilityConstant.TYPE_NODE);
			}
			else{
				info.setType(FacilityConstant.TYPE_SCOPE);
			}
			item = new FacilityTreeItem(parent, info);
		}
		
		Collection collection = null;
		try {
			collection = facilityTree.getChildren();
		} catch (FinderException e) {
			//TODO FinderException未実装
		}
		
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				FacilityTreeLocal children = (FacilityTreeLocal)itr.next();
				createFacilityTree(children, item);
			}
		}
	}
	
	/**
	 * FacilityTreeItem作成（スコープのみ）
	 * 
	 * @param facilityTree
	 * @param parent
	 */
	protected void createScopeFacilityTree(FacilityTreeLocal facilityTree, FacilityTreeItem parent) {
		FacilityLocal facility = null;
		try {
			facility = facilityTree.getFacility();
		} catch (FinderException e) {
			return;
		}
		
		FacilityInfo info = null;
		FacilityTreeItem item = null;
		if(facility != null){
			info = new FacilityInfo();
			info.setFacilityId(facility.getFacilityId());
			info.setFacilityName(facility.getCn());
			if(facility.getType() == FacilityConstant.TYPE_SCOPE){
				info.setType(FacilityConstant.TYPE_SCOPE);
				item = new FacilityTreeItem(parent, info);
			}
			else{
				return;
			}
		}
		
		Collection collection = null;
		try {
			collection = facilityTree.getChildren();
		} catch (FinderException e) {
			//TODO FinderException未実装
		}
		
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				FacilityTreeLocal children = (FacilityTreeLocal)itr.next();
				createScopeFacilityTree(children, item);
			}
		}
	}
	
	/**
	 * ファシリティパス取得
	 * 
	 * ファシリティのパスを返す
	 * 
	 * @param facilityId
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public String getFacilityPath(String facilityId, String parentFacilityId) throws FinderException, NamingException {
		try {
			Collection ct = FacilityTreeUtil.getLocalHome().findByFacilityId(facilityId);
			
			if(ct != null){
				FacilityTreeLocal tree = null;
				Iterator itr = ct.iterator();
				if(itr.hasNext()){
					tree = (FacilityTreeLocal)itr.next();
				}
				
				if(tree != null){
					StringBuffer buffer = new StringBuffer();
					if (tree.getTreeType().intValue() == FacilityConstant.TYPE_SCOPE) {
						//スコープのパスを取得
						if(parentFacilityId != null && parentFacilityId.length() > 0){
							//相対パスを取得
							FacilityTreeLocal parentTree = tree;
							while (parentTree != null) {
								FacilityLocal parent = parentTree.getFacility();
								if(parentFacilityId.compareTo(parent.getFacilityId()) == 0){
									if(buffer.length() == 0){
										buffer.insert(0, SEPARATOR);
									}
									break;
								}
								buffer.insert(0, SEPARATOR);
								buffer.insert(0, parent.getCn());
								parentTree = parentTree.getParent();
							}
						}
						else{
							//絶対パスを取得
							FacilityTreeLocal parentTree = tree;
							while (parentTree != null) {
								FacilityLocal parent = parentTree.getFacility();
								buffer.insert(0, SEPARATOR);
								buffer.insert(0, parent.getCn());
								parentTree = parentTree.getParent();
							}
						}
					}
					else{
						//ノードの場合、ノード名を取得
						FacilityLocal facility = tree.getFacility();
						buffer.append(facility.getCn());
					}
					
					return buffer.toString();
				}
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return "";
	}
	
	/**
	 * ファシリティIDリスト取得
	 * 
	 * ファシリティIDの配列として返す
	 * 
	 * @param hostName
	 * @param ipAddress
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getFacilityIdList(String hostName, String ipAddress) throws FinderException, NamingException {
		ArrayList list = null;
		try {
			HashMap map = new HashMap();
			list = new ArrayList();
			
			//IPアドレスが一致するFacilityを取得
			if(ipAddress != null && ipAddress.length() > 0){
				Collection ct = FacilityUtil.getLocalHome().findByIpAddress(ipAddress);
				Iterator itr = ct.iterator();
				while(itr.hasNext()){
					FacilityLocal facility = (FacilityLocal)itr.next();
					
					if(map.get(facility.getFacilityId()) == null){
						map.put(facility.getFacilityId(), "1");
						
						list.add(facility.getFacilityId());
					}
				}
			}
			
			//ノード名が一致するFacilityを取得
			if(hostName != null && hostName.length() > 0){
				Collection ct = FacilityUtil.getLocalHome().findByNodeName(hostName);
				Iterator itr = ct.iterator();
				while(itr.hasNext()){
					FacilityLocal facility = (FacilityLocal)itr.next();
					
					if(map.get(facility.getFacilityId()) == null){
						map.put(facility.getFacilityId(), "1");
						
						list.add(facility.getFacilityId());
					}
				}
			}
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		
		return list;
	}
	
	/**
	 * ノードチェック
	 * 
	 * ファシリティIDがノードかチェックし、結果をbooleanで返す
	 * 
	 * @param facilityId
	 * @return true:ノード/false:スコープ
	 * @throws FinderException
	 * @throws NamingException
	 */
	public boolean isNode(String facilityId) throws FinderException, NamingException {
		boolean node = false;
		try {
			//FacilityTreeの最上位エントリを取得
			FacilityLocal facility = FacilityUtil.getLocalHome().findByFacilityId(facilityId);
			
			if(facility.getType() == FacilityConstant.TYPE_NODE){
				node = true;
			}
			
		} catch (FinderException e) {
			throw e;
		} catch (NamingException e) {
			throw e;
		}
		return node;
	}
}
