/*
                                                                                                                                                                 
Copyright (C) 2006 NTT DATA Corporation
                                                                                                                                                                 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
                                                                                                                                                                 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
                                                                                                                                                                 
*/

package com.clustercontrol.performanceMGR.bean;

import java.lang.Double;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;

import javax.ejb.EJBException;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.performanceMGR.dao.RecordDataDAO;
import com.clustercontrol.performanceMGR.util.CalculationMethod;
import com.clustercontrol.performanceMGR.util.JNDIConnectionManager;
import com.clustercontrol.sharedtable.bean.ValueObject;
import com.clustercontrol.snmppoller.SnmpSharedTable;

/**
 * Facilityのノード実装クラス
 * 種別がノードであるファシリティの性能値を保持します。
 * ScopeTreeクラスの内部で使用することを目的としたクラスです。
 * 
 * @version 1.0
 * @since 1.0
 */
public class Node extends Facility {
	//	ログ出力
	protected static Log m_log = LogFactory.getLog( Node.class );
	
	private HashMap<String, long[][]> m_mibValueSet; // 性能値の計算に必要なMIB値の集合
	
	private long lastCollectTime;   // 最終更新時刻
		
	/**
	 *コンストラクター 
	 * @param facilityID
	 * @param faclityName
	 * @param ipAddress
	 */
	Node(final String facilityID, final String faclityName){
		super(facilityID, faclityName, Facility.NODE);
		
		m_mibValueSet = new HashMap();
	}
	
	/**
	 * 性能値を戻します。
	 * 
	 * @param  itemCode 収集項目コード
	 * @param  deviceIndex デバイス番号
	 * @return 性能値
	 */
	public double calcValue(final String itemCode, final int deviceIndex){
		double value = CalculationMethod.getPerformance(itemCode, deviceIndex, m_mibValueSet);
		
		m_log.debug("calcValue() : " + m_facilityID + " " + 
				itemCode + " " + deviceIndex + "  " + value);
		
		// 性能値が算出できた場合はバッファに保存する
		if(!Double.isNaN(value)){
			setCalcValueBuffer(new CollectorItemPK(itemCode, deviceIndex), value);
		}
		
		return value;
	}

	
	/**
	 * ポーラーから収集値を取得します。
	 * 
	 * @param oids 収集対象のOIDの配列
	 * @param interval 収集間隔
	 * @return 収集時刻（全てのノードで一致させるために最後に収集された値の収集時刻とする）
	 */
	public long fetchMibValue(final OIDInfo[] oids, int interval) {
		m_log.debug("fetchMibValue() start :");
		
		// 一度値を書き込んだOIDの値を2度書き込まないように書き込んだOIDを保持する
		HashSet<String> checkOid = new HashSet<String>();  
		
		// SNMP収集値の共有テーブルをルックアップします。
		SnmpSharedTable sst = null;
		try {
			InitialContext iniCtx = JNDIConnectionManager.getInitialContext();
			
			Object obj = iniCtx.lookup(SnmpPollerConstant.JNDI_NAME);

			// 取得したオブジェクトをキャストします。
			sst = (SnmpSharedTable)PortableRemoteObject.narrow(obj, SnmpSharedTable.class);
		} catch (NamingException e) {
			// エラー処理
			throw new EJBException(e.getMessage());
		}
		
		// 性能値の算出に必要なMIB値を取得
		for(int i=0; i<oids.length; i++){
			// デバイス全てを収集対象とするか否かで条件分岐
			if(!oids[i].isAllIndex()){
				
				String oid = oids[i].getOID();
				ValueObject value = sst.getValue(getFacilityID(), interval, oid);
				
				if(value == null){
					// エラー処理
					m_log.debug("Node : " + getFacilityID() + " can't get value " + oids[i].getOID());
					
					// 対象のOIDの値をまだセットしていない場合のみ処理を行う
					if(!checkOid.contains(oid)){	
						// 値取得に失敗している場合でも、取得失敗時の適用値が設定されている場合は、
						// その値をセットする
						Long defaultValue = oids[i].getDefaultValue();
						if(defaultValue != null){
							// 値をセットする
							setMibValueSet(oids[i].getBaseOid(), oids[i].getOidIndex(), defaultValue);

							// デバック出力
							if(m_log.isTraceEnabled()){
								m_log.trace("fetchMibValue() : " + getFacilityID()
										+ " set default value "	+ oid + " = " + defaultValue);
							}
						} else {
							// 値取得に失敗し、取得失敗時の適用値が設定されていない場合は、nullとする
							// CalculationMethodクラスでnullチェックし、nullの場合は、性能値をNaNとする
							m_mibValueSet.put(oids[i].getBaseOid(), null);
							
							// デバック出力
							if(m_log.isTraceEnabled()){
								m_log.trace("fetchMibValue() : " + getFacilityID()
										+ " set default value "	+ oid + " = " + defaultValue);
							}
						}
						checkOid.add(oid);  // 書き込み済みのOIDを設定する
					}
				} else {
					String mibValueString = (String)value.getValue();
					
					// デバッグ
					m_log.debug("Node : " + getFacilityID() + " " + oids[i].getOID() + " " + value);
					
					// 対象のOIDの値をまだセットしていない場合のみ処理を行う
					if(!checkOid.contains(oid)){	
						// 値をセットする
						setMibValueSet(oids[i].getBaseOid(), oids[i].getOidIndex(), Long.parseLong(mibValueString));
						checkOid.add(oid);  // 書き込み済みのOIDを設定する
						lastCollectTime = Math.max(lastCollectTime, value.getDate());
					}
				}
				
			} else {
				int j=0; // テーブルインデックスをインクリメントするための変数
				
				String oid = oids[i].getBaseOid() + "." + j;			
				
				// 対象のOIDの値をまだセットしていない場合のみ処理を行う
				if(!checkOid.contains(oid)){	
					// テーブルインデックスが0のものは存在しないため、ダミーデータを設定
					setMibValueSet(oids[i].getBaseOid(), j++, Long.MIN_VALUE);
					checkOid.add(oid);  // 書き込み済みのOIDを設定する
				}
				
				oid = oids[i].getBaseOid() + "." + j;
				ValueObject value = null;
				while((value = sst.getValue(getFacilityID(), interval, oid)) != null){
					// デバッグ
					m_log.debug("Node : " + getFacilityID() + " " + oids[i].getOID() + " " + value);
					
					// 対象のOIDの値をまだセットしていない場合のみ処理を行う
					if(!checkOid.contains(oid)){	
						// 値をセットする(インデックス0には値をいれずに、1から値を代入)
						setMibValueSet(oids[i].getBaseOid(), j, Long.parseLong((String)value.getValue()));
						checkOid.add(oid);  // 書き込み済みのOIDを設定する
						lastCollectTime = Math.max(lastCollectTime, value.getDate());
					}
					j++;  // インクリメント
					oid = oids[i].getBaseOid() + "." + j;
				}
				
				if(j == 1){
					// エラー処理
					m_log.debug("Node : " + getFacilityID() + " can't get value " + oids[i].getBaseOid() + ".*");
					
					// CalculationMethodクラスでnullチェックし、nullの場合は、性能値をNaNとする
					m_mibValueSet.put(oids[i].getBaseOid(), null);
				}
			}
		}
		
		m_log.debug("fetchMibValue() end :");
		return lastCollectTime;
	}
	
	/**
	 * 
	 * 最新のMIB値を配列に設定します。
	 *  
	 * @param baseOid 収集対象OIDの最後の"."より前を表す文字列
	 * @param tableIndex　収集対象OIDの最後の"."以下の数値
	 * @param mibValue　収集値
	 */
	private void setMibValueSet(final String baseOid, final int tableIndex, final long mibValue) {
		m_log.trace("setMibValueSet() :" + baseOid + " " + tableIndex + "  " + mibValue);
		long[][] mibValueSetList = (long[][])m_mibValueSet.get(baseOid);
		
		// 値を格納する配列が確保されていない場合
		if (mibValueSetList == null){
			mibValueSetList = new long[tableIndex+1][2];
			
			// テーブルに追加
			m_mibValueSet.put(baseOid, mibValueSetList);
		} else 	if (mibValueSetList.length <= tableIndex){	// 現在確保している領域よりも指定のインデックスの方が大きい場合
			// 新規に領域を確保	
			long[][]buffer = new long[tableIndex+1][2];
			
			// 新規に確保した領域に配列をコピー
			for(int i=0; i<mibValueSetList.length; i++){
				buffer[i] = mibValueSetList[i];
			}
			
			mibValueSetList = buffer;
			
			// テーブルに追加
			m_mibValueSet.put(baseOid, mibValueSetList);
		}
		
		// ディープコピー
		// 今まで保持していた値を前回収集の値として設定。
		mibValueSetList[tableIndex][0] = mibValueSetList[tableIndex][1];
		
		// 新規の値を設定。
		mibValueSetList[tableIndex][1] = mibValue;
	}
	
	/**
	 * 自分自身を返す。
	 * @return HashSet 
	 */
	public HashSet getNode(HashSet nodeSet){
		nodeSet.add(this);
		return nodeSet;
	}
	
	/**
	 * 自分自身をカウントして返す。
	 */
	public int getNodeCount(){
		return 1;
	}
	
	/**
	 * 現在保持しているMIB値をDBに出力する
	 *
	 */
	public void storeMibValue(String collectorId){
		m_log.debug("storeRowMibValue() :" + collectorId);

		// DBへ格納する日時情報
		Date date = new Date();  // 現在の時刻とする
		
		// MIB値を格納するためのテーブルにアクセス可能なオブジェクトを生成
		RecordDataDAO dao = new RecordDataDAO();
		
		// m_mibValueSet のキーはOID(最後のインデックスを除く)
		Iterator itr = m_mibValueSet.keySet().iterator();  

		while(itr.hasNext()){
			String baseOid = (String)itr.next();

			long[][] mibValueSetList = (long[][])m_mibValueSet.get(baseOid);

			for(int i=0; i<mibValueSetList.length; i++){
				String fullOid = baseOid + "." + i;
				
				dao.insertRecordData(
						collectorId, 
						fullOid,
						date,
						getFacilityID(),
						mibValueSetList[i][0]);
			}
		}
	}
}
