/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterPK;

/**
 * ジョブコマンド情報のDAOインターフェースを実装するクラスです。
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterBean
 * @see com.clustercontrol.jobmanagement.dao.JobRelationMasterDAO
 */
public class JobRelationMasterDAOImpl implements JobRelationMasterDAO {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(JobRelationMasterDAOImpl.class);
	
	public void init() {
	}
	
	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationMasterDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterPK, com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterBean)
	 */
	public void load(JobRelationMasterPK pk, JobRelationMasterBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_relation_mst WHERE job_id = ? AND parent_job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			stmt.setString(2, pk.getParent_job_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setJob_id(res.getString("job_id"));
				ejb.setParent_job_id(res.getString("parent_job_id"));
			} else {
				String msg = "JobRelationMaster data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}
	
	/**
	 * データベースの更新を行うのが基本だが、<BR>
	 * 全てのカラムが主キーになっているので、更新を行う必要はない。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationMasterDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterBean)
	 */
	public void store(JobRelationMasterBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getJob_id() + ", " + ejb.getParent_job_id());
		
		
		m_log.debug("store() end : " + ejb.getJob_id() + ", " + ejb.getParent_job_id());
	}
	
	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationMasterDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterPK)
	 */
	public void remove(JobRelationMasterPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_job_relation_mst WHERE job_id = ? AND parent_job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			stmt.setString(2, pk.getParent_job_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}
	
	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationMasterDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterBean)
	 */
	public JobRelationMasterPK create(JobRelationMasterBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : " + ejb.getJob_id() + ", " + ejb.getParent_job_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		JobRelationMasterPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_relation_mst (");
			sql.append("job_id, ");
			sql.append("parent_job_id) ");
			sql.append("VALUES (?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getJob_id());
			stmt.setString(2, ejb.getParent_job_id());
			
			int row =	stmt.executeUpdate();
			pk = new JobRelationMasterPK(
					ejb.getJob_id(),
					ejb.getParent_job_id());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getJob_id() + ", "
						+ ejb.getParent_job_id() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getJob_id() + ", "
						+ ejb.getParent_job_id() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getJob_id() + ", "
						+ ejb.getParent_job_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getJob_id() + ", " + ejb.getParent_job_id());
		return pk;
	}
	
	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationMasterDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<JobRelationMasterPK> ret = new ArrayList<JobRelationMasterPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_relation_mst";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobRelationMasterPK pk= new JobRelationMasterPK(
						res.getString("job_id"),
						res.getString("parent_job_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}
	
	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationMasterDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterPK)
	 */
	public JobRelationMasterPK findByPrimaryKey(JobRelationMasterPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_relation_mst WHERE job_id = ? AND parent_job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getJob_id());
			stmt.setString(2, pk.getParent_job_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "JobRelationMasterDAOImpl.findByPrimaryKey() : id=" + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定された親ジョブIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterBean#ejbFindByParentJobId(java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationMasterDAO#findByParentJobId(java.lang.String)
	 */
	public Collection findByParentJobId(String parent_job_id) throws FinderException {
		m_log.debug("findByParentJobId() start : " + parent_job_id);
		
		ArrayList<JobRelationMasterPK> ret = new ArrayList<JobRelationMasterPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_relation_mst WHERE parent_job_id = ? ORDER BY job_id ASC";
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, parent_job_id);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobRelationMasterPK pk= new JobRelationMasterPK(
						res.getString("job_id"),
						res.getString("parent_job_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findByParentJobId() error : " + parent_job_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByParentJobId() error : " + parent_job_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByParentJobId() end : " + parent_job_id);
		return ret;
	}
	
	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたジョブIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterBean#ejbFindByJobId(java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationMasterDAO#findByJobId(java.lang.String)
	 */
	public JobRelationMasterPK findByJobId(String job_id) throws FinderException {
		m_log.debug("findByJobId() start : " + job_id);
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_relation_mst WHERE job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, job_id);
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				JobRelationMasterPK pk= new JobRelationMasterPK(
						res.getString("job_id"),
						res.getString("parent_job_id")
				);
				return pk;
			} else {
				String msg = "JobRelationMasterDAOImpl.findByJobId() : id=" + job_id + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("findByJobId() error : " + job_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByJobId() error : " + job_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByJobId() end : " + job_id);
		}
	}
	
	/**
	 * 引数で指定されたジョブIDで、データベースを検索し、配下のジョブIDを取得します。
	 * ジョブIDのコレクションを返します。
	 * 
	 * @param jobId ジョブID
	 * @return ジョブIDのコレクション
	 * @throws FinderException
	 *
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterBean#ejbHomeGetJobsList(java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationMasterDAOImpl#getJobsListSub(String, ArrayList, PreparedStatement)
	 */
	public Collection getJobsList(String jobId) throws FinderException{
		m_log.debug("getJobsList() start : " + jobId);
		
		ArrayList<Hashtable> jobList = new ArrayList<Hashtable>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		Hashtable<String, String> ht = new Hashtable<String, String>();
		ht.put("parentJobId", "TOP");
		ht.put("jobId", jobId);
		
		m_log.debug("getJobsList() MasterJob id : " + jobId +", " + "TOP");
		jobList.add(ht);
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_relation_mst WHERE parent_job_id = ? ORDER BY job_id ASC";
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql);	
			stmt.setString(1, jobId);
			
			ArrayList<Hashtable> tmpJobList = new ArrayList<Hashtable>();
			
			res = stmt.executeQuery();
			while (res.next()) {
				Hashtable<String, String> ht2 = new Hashtable<String, String>();
				ht2.put("parentJobId", jobId);
				ht2.put("jobId", res.getString("job_id"));
				tmpJobList.add(ht2);
			}
			res.close();
			
			Iterator it = tmpJobList.iterator();
			while(it.hasNext()){
				Hashtable ht2 = (Hashtable) it.next();
				jobList.add(ht2);
				getJobsListSub((String)ht2.get("jobId"), jobList, stmt);
			}
		} catch (SQLException e) {
			m_log.error("getJobsList() error : " + jobId + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("getJobsList() error : " + jobId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("getJobsList() end : " + jobId);
		return jobList;
	}
	
	/**
	 * 引数で指定されたジョブIDで、データベースを検索し、配下のジョブIDを取得します。
	 * ジョブIDのリストを返します。
	 * 
	 * @param jobId ジョブID
	 * @param ret ジョブIDを格納するリスト
	 * @param stmt プリペアードステートメント(SQL文)
	 * @throws FinderException
	 */
	@SuppressWarnings("unchecked")
	public void getJobsListSub(String jobId, ArrayList ret,PreparedStatement stmt ) throws FinderException{
		
		m_log.debug("getJobsListSub() start : " + jobId);
		
		ArrayList tmp_ret = new ArrayList();
		ResultSet res = null;
		Hashtable ht  = null;
		
		try {
			stmt.setString(1, jobId);
			res = stmt.executeQuery();
			
			while(res.next()) {
				ht = new Hashtable();
				ht.put("parentJobId",jobId);
				ht.put("jobId",res.getString("job_id"));
				tmp_ret.add(ht);	
			}
			
			res.close();
			
			Iterator it = tmp_ret.iterator();
			
			while(it.hasNext()){
				ht = (Hashtable)it.next();
				ret.add(ht);	
				getJobsListSub((String)ht.get("jobId"),ret,stmt);
			}
			
			
		} catch (SQLException e) {
			m_log.error("getJobsListSub() error: " + jobId + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(res != null){
					res.close();}
				
			} catch (SQLException e1) {
				m_log.error("getJobsListSub() error : " + jobId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("getJobsListSub() end : " + jobId);
	}
	
	/**
	 * 引数で指定されたジョブIDで、データベースを検索し、最上位のジョブIDを取得します。
	 * 
	 * @param jobId ジョブID
	 * @return 最上位のジョブID
	 * @throws FinderException
	 *
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterBean#ejbHomeGetTopJobId(java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationMasterDAOImpl#getTopJobIdSub(String, PreparedStatement)
	 */
	public String getTopJobId(String jobId) throws FinderException{
		m_log.debug("getTopJobId() start : " + jobId);
		
		Connection conn = null;
		PreparedStatement stmt = null;
		String topJobId = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_relation_mst WHERE job_id = ?";
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			topJobId = getTopJobIdSub(jobId, stmt);

		} catch (SQLException e) {
			m_log.error("getTopJobId() error : " + jobId + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("getJobsList() error : " + jobId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("getTopJobId() end : " + jobId);
		return topJobId;
	}
	
	/**
	 * 引数で指定されたジョブIDで、データベースを検索し、最上位のジョブIDを取得します。
	 * 
	 * @param jobId ジョブID
	 * @param stmt プリペアードステートメント(SQL文)
	 * @return 最上位のジョブID
	 * @throws FinderException
	 */
	public String getTopJobIdSub(String jobId, PreparedStatement stmt) throws FinderException{
		
		m_log.debug("getTopJobIdSub() start : " + jobId);
		
		ResultSet res = null;
		String parentJobId = null;
		
		try {
			stmt.setString(1, jobId);
			res = stmt.executeQuery();
			
			if(res.next()) {
				parentJobId = res.getString("parent_job_id");
			}
			res.close();
			
			if(parentJobId != null && !parentJobId.equals("TOP")) {
				return getTopJobId(parentJobId);
			}

		} catch (SQLException e) {
			m_log.error("getTopJobIdSub() error: " + jobId + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(res != null){
					res.close();}
				
			} catch (SQLException e1) {
				m_log.error("getTopJobIdSub() error : " + jobId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		
		m_log.debug("getTopJobIdSub() end : " + jobId);
		
		return jobId;
	}
}
